interface JMAPSession {
  apiUrl: string;
  downloadUrl: string;
  uploadUrl: string;
  accounts: Record<string, {
    name: string;
    isPersonal: boolean;
    accountCapabilities: Record<string, unknown>;
  }>;
  primaryAccounts: Record<string, string>;
}

interface JMAPRequest {
  using: string[];
  methodCalls: [string, Record<string, unknown>, string][];
}

interface JMAPResponse {
  methodResponses: [string, Record<string, unknown>, string][];
  sessionState: string;
}

export class JMAPClient {
  private session: JMAPSession | null = null;
  private token: string;
  
  constructor(token: string) {
    this.token = token;
  }
  
  async getSession(): Promise<JMAPSession> {
    if (this.session) return this.session;
    
    const response = await fetch('https://api.fastmail.com/jmap/session', {
      headers: {
        'Authorization': `Bearer ${this.token}`,
      },
    });
    
    if (!response.ok) {
      throw new Error(`Failed to get session: ${response.statusText}`);
    }
    
    this.session = await response.json();
    return this.session!;
  }
  
  async call(methodCalls: [string, Record<string, unknown>, string][]): Promise<JMAPResponse> {
    const session = await this.getSession();
    
    const request: JMAPRequest = {
      using: [
        'urn:ietf:params:jmap:core',
        'urn:ietf:params:jmap:mail',
        'urn:ietf:params:jmap:submission',
      ],
      methodCalls,
    };
    
    const response = await fetch(session.apiUrl, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${this.token}`,
      },
      body: JSON.stringify(request),
    });
    
    if (!response.ok) {
      throw new Error(`JMAP call failed: ${response.statusText}`);
    }
    
    return response.json();
  }
  
  async getAccountId(): Promise<string> {
    const session = await this.getSession();
    return session.primaryAccounts['urn:ietf:params:jmap:mail'];
  }
}
