/**
 * Custom error classes with user-friendly messages and suggestions
 */

export class FastmailError extends Error {
  public readonly code: string;
  public readonly suggestion?: string;
  public readonly details?: unknown;

  constructor(
    message: string,
    code: string,
    suggestion?: string,
    details?: unknown
  ) {
    super(message);
    this.name = 'FastmailError';
    this.code = code;
    this.suggestion = suggestion;
    this.details = details;
  }

  toJSON() {
    return {
      error: this.message,
      code: this.code,
      suggestion: this.suggestion,
      details: this.details,
    };
  }
}

// Authentication errors
export class AuthenticationError extends FastmailError {
  constructor(type: 'email' | 'calendar', originalError?: string) {
    const messages = {
      email: {
        message: 'Failed to authenticate with Fastmail email API',
        suggestion: 'Check your FASTMAIL_API_TOKEN environment variable. Generate a new token at Fastmail Settings → Privacy & Security → Integrations → API tokens',
      },
      calendar: {
        message: 'Failed to authenticate with Fastmail calendar',
        suggestion: 'Check your FASTMAIL_USERNAME and FASTMAIL_PASSWORD environment variables. Use an app password, not your main password. Generate at Fastmail Settings → Privacy & Security → Integrations → App passwords',
      },
    };

    super(
      messages[type].message,
      'AUTH_ERROR',
      messages[type].suggestion,
      { originalError }
    );
    this.name = 'AuthenticationError';
  }
}

// Missing configuration errors
export class ConfigurationError extends FastmailError {
  constructor(missingVars: string[]) {
    const varList = missingVars.join(', ');
    super(
      `Missing required environment variable(s): ${varList}`,
      'CONFIG_ERROR',
      missingVars.includes('FASTMAIL_API_TOKEN')
        ? 'Set FASTMAIL_API_TOKEN for email operations. Get your token from Fastmail Settings → Privacy & Security → Integrations'
        : 'Set FASTMAIL_USERNAME and FASTMAIL_PASSWORD for calendar operations. Create an app password at Fastmail Settings → Privacy & Security → Integrations'
    );
    this.name = 'ConfigurationError';
  }
}

// Not found errors
export class NotFoundError extends FastmailError {
  constructor(
    resourceType: 'email' | 'mailbox' | 'calendar' | 'event' | 'thread' | 'reminder',
    resourceId: string
  ) {
    const suggestions: Record<string, string> = {
      email: `Use list_emails to see available emails, or search_emails to find specific ones`,
      mailbox: `Use list_mailboxes to see available folders/mailboxes`,
      calendar: `Use list_calendars to see available calendars`,
      event: `Use list_events with a date range to find events, or search_events to search by text`,
      thread: `The email thread may have been deleted or the email_id is incorrect. Use list_emails to verify`,
      reminder: `Use list_event_reminders to see reminders for an event`,
    };

    super(
      `${resourceType.charAt(0).toUpperCase() + resourceType.slice(1)} not found: ${resourceId}`,
      'NOT_FOUND',
      suggestions[resourceType]
    );
    this.name = 'NotFoundError';
  }
}

// Validation errors
export class ValidationError extends FastmailError {
  constructor(field: string, reason: string, expectedFormat?: string) {
    super(
      `Invalid ${field}: ${reason}`,
      'VALIDATION_ERROR',
      expectedFormat ? `Expected format: ${expectedFormat}` : undefined
    );
    this.name = 'ValidationError';
  }
}

// API errors
export class APIError extends FastmailError {
  constructor(
    operation: string,
    statusCode: number,
    statusText: string,
    responseBody?: string
  ) {
    let suggestion: string;
    
    if (statusCode === 401) {
      suggestion = 'Your authentication token may have expired. Generate a new one at Fastmail Settings';
    } else if (statusCode === 403) {
      suggestion = 'Your API token may not have sufficient permissions. Create a new token with full access';
    } else if (statusCode === 404) {
      suggestion = 'The requested resource does not exist or has been deleted';
    } else if (statusCode === 429) {
      suggestion = 'Rate limit exceeded. Wait a few seconds and try again';
    } else if (statusCode >= 500) {
      suggestion = 'Fastmail server error. Check https://fastmailstatus.com/ for service status';
    } else {
      suggestion = 'Check the API documentation or try the operation again';
    }

    super(
      `${operation} failed: ${statusCode} ${statusText}`,
      'API_ERROR',
      suggestion,
      { statusCode, responseBody }
    );
    this.name = 'APIError';
  }
}

// Bulk operation error (partial success)
export class BulkOperationError extends FastmailError {
  public readonly succeeded: string[];
  public readonly failed: { id: string; error: string }[];

  constructor(
    operation: string,
    succeeded: string[],
    failed: { id: string; error: string }[]
  ) {
    super(
      `${operation} partially completed: ${succeeded.length} succeeded, ${failed.length} failed`,
      'BULK_PARTIAL',
      'Review the failed items and retry them individually if needed',
      { succeeded, failed }
    );
    this.succeeded = succeeded;
    this.failed = failed;
    this.name = 'BulkOperationError';
  }
}

// Format error response for CLI output
export function formatError(error: unknown): {
  success: false;
  error: string;
  code?: string;
  suggestion?: string;
  details?: unknown;
} {
  if (error instanceof FastmailError) {
    return {
      success: false,
      error: error.message,
      code: error.code,
      suggestion: error.suggestion,
      details: error.details,
    };
  }

  if (error instanceof Error) {
    const message = error.message;
    
    if (message.includes('401') || message.includes('Unauthorized')) {
      return formatError(new AuthenticationError('email', message));
    }
    
    if (message.includes('FASTMAIL_API_TOKEN')) {
      return formatError(new ConfigurationError(['FASTMAIL_API_TOKEN']));
    }
    
    if (message.includes('FASTMAIL_USERNAME') || message.includes('FASTMAIL_PASSWORD')) {
      return formatError(new ConfigurationError(['FASTMAIL_USERNAME', 'FASTMAIL_PASSWORD']));
    }

    return {
      success: false,
      error: message,
      suggestion: 'If this error persists, check your environment variables and network connection',
    };
  }

  return {
    success: false,
    error: String(error),
  };
}
