#!/usr/bin/env bun

import { JMAPClient } from './jmap-client.js';
import { EmailTools } from './tools/email.js';
import { CalDAVClient } from './caldav-client.js';
import { CalendarTools } from './tools/calendar.js';
import { formatError, ValidationError, BulkOperationError } from './errors.js';

// Types
interface ToolResult {
  success: boolean;
  result?: unknown;
  error?: string;
}

// Lazy-initialized clients
let jmapClient: JMAPClient | null = null;
let emailTools: EmailTools | null = null;
let caldavClient: CalDAVClient | null = null;
let calendarTools: CalendarTools | null = null;

// Initialize JMAP clients (for email)
function getEmailTools(): EmailTools {
  if (!emailTools) {
    const token = process.env.FASTMAIL_API_TOKEN;
    if (!token) {
      throw new Error('FASTMAIL_API_TOKEN environment variable is required');
    }
    jmapClient = new JMAPClient(token);
    emailTools = new EmailTools(jmapClient);
  }
  return emailTools;
}

// Initialize CalDAV clients (for calendar)
function getCalendarTools(): CalendarTools {
  if (!calendarTools) {
    const username = process.env.FASTMAIL_USERNAME;
    const password = process.env.FASTMAIL_PASSWORD;
    if (!username || !password) {
      throw new Error('FASTMAIL_USERNAME and FASTMAIL_PASSWORD environment variables are required');
    }
    caldavClient = new CalDAVClient(username, password);
    calendarTools = new CalendarTools(caldavClient);
  }
  return calendarTools;
}

// Tool handler
async function handleTool(toolName: string, args: Record<string, unknown>): Promise<ToolResult> {
  try {
    // Normalize command: support both 'list_emails' and 'fastmail_list_emails' formats
    let normalizedToolName = toolName;
    if (!toolName.startsWith('fastmail_')) {
      normalizedToolName = `fastmail_${toolName}`;
    }

    switch (normalizedToolName) {
      // EMAIL TOOLS (9)
      case 'fastmail_list_mailboxes': {
        const email = getEmailTools();
        const result = await email.getMailboxes();
        return { success: true, result };
      }

      case 'fastmail_list_emails': {
        const email = getEmailTools();
        const mailboxId = args.mailbox_id as string | undefined;
        const limit = (args.limit as number) || 20;
        const result = await email.listEmails(mailboxId, limit);
        return { success: true, result };
      }

      case 'fastmail_get_email': {
        const email = getEmailTools();
        const emailId = args.email_id as string;
        if (!emailId) throw new Error('email_id is required');
        const result = await email.getEmail(emailId);
        return { success: true, result };
      }

      case 'fastmail_search_emails': {
        const email = getEmailTools();
        const query = args.query as string;
        if (!query) throw new Error('query is required');
        const limit = (args.limit as number) || 20;
        const result = await email.searchEmails(query, limit);
        return { success: true, result };
      }

      case 'fastmail_send_email': {
        const email = getEmailTools();
        const to = args.to as { name?: string; email: string }[];
        const subject = args.subject as string;
        const textBody = args.text_body as string;
        const htmlBody = args.html_body as string | undefined;
        const cc = args.cc as { name?: string; email: string }[] | undefined;
        const bcc = args.bcc as { name?: string; email: string }[] | undefined;
        const inReplyTo = args.in_reply_to as string | undefined;
        const references = args.references as string[] | undefined;

        if (!to || !subject || !textBody) {
          throw new Error('to, subject, and text_body are required');
        }

        const result = await email.sendEmail({
          to,
          cc,
          bcc,
          subject,
          textBody,
          htmlBody,
          inReplyTo,
          references,
        });
        return { success: true, result };
      }

      case 'fastmail_move_email': {
        const email = getEmailTools();
        const emailId = args.email_id as string;
        const targetMailboxId = args.target_mailbox_id as string;
        const sourceMailboxId = args.source_mailbox_id as string | undefined;

        if (!emailId || !targetMailboxId) {
          throw new Error('email_id and target_mailbox_id are required');
        }

        await email.moveToFolder(emailId, targetMailboxId, sourceMailboxId);
        return { success: true, result: { message: 'Email moved' } };
      }

      case 'fastmail_set_labels': {
        const email = getEmailTools();
        const emailId = args.email_id as string;
        const keywords = args.keywords as Record<string, boolean>;

        if (!emailId || !keywords) {
          throw new Error('email_id and keywords are required');
        }

        await email.setKeywords(emailId, keywords);
        return { success: true, result: { message: 'Labels set' } };
      }

      case 'fastmail_delete_email': {
        const email = getEmailTools();
        const emailId = args.email_id as string;

        if (!emailId) throw new Error('email_id is required');

        await email.deleteEmail(emailId);
        return { success: true, result: { message: 'Email deleted' } };
      }

      case 'fastmail_reply_email': {
        const email = getEmailTools();
        const emailId = args.email_id as string;
        const textBody = args.text_body as string;
        const htmlBody = args.html_body as string | undefined;
        const subject = args.subject as string | undefined;
        const cc = args.cc as { name?: string; email: string }[] | undefined;
        const bcc = args.bcc as { name?: string; email: string }[] | undefined;
        const replyAll = args.reply_all as boolean | undefined;

        if (!emailId || !textBody) {
          throw new Error('email_id and text_body are required');
        }

        // Get original email to extract sender info
        const originalEmail = await email.getEmail(emailId);
        const from = originalEmail.from?.[0];

        if (!from) {
          throw new Error('Could not find sender information in original email');
        }

        const to = [from];
        const inReplyTo = emailId;

        // If reply-all is enabled, include all recipients
        let additionalTo: { name?: string; email: string }[] = [];
        if (replyAll) {
          if (originalEmail.to) additionalTo.push(...originalEmail.to);
          if (originalEmail.cc) additionalTo.push(...originalEmail.cc);
        }

        const result = await email.sendEmail({
          to: [...to, ...additionalTo],
          cc,
          bcc,
          subject: subject || `Re: ${originalEmail.subject}`,
          textBody,
          htmlBody,
          inReplyTo,
        });

        return { success: true, result };
      }

      // CALENDAR TOOLS (8)
      case 'fastmail_list_calendars': {
        const calendar = getCalendarTools();
        const result = await calendar.listCalendars();
        return { success: true, result };
      }

      case 'fastmail_list_events': {
        const calendar = getCalendarTools();
        const calendarId = args.calendar_id as string | undefined;
        const startDate = args.start_date as string | undefined;
        const endDate = args.end_date as string | undefined;
        const result = await calendar.listEvents(calendarId, startDate, endDate);
        return { success: true, result };
      }

      case 'fastmail_get_event': {
        const calendar = getCalendarTools();
        const eventId = args.event_id as string;
        if (!eventId) throw new Error('event_id is required');
        const result = await calendar.getEvent(eventId);
        return { success: true, result };
      }

      case 'fastmail_create_event': {
        const calendar = getCalendarTools();
        const calendarId = args.calendar_id as string | undefined;
        const title = args.title as string;
        const start = args.start as string;
        const end = args.end as string;
        const description = args.description as string | undefined;
        const location = args.location as string | undefined;
        const allDay = args.all_day as boolean | undefined;
        const attendees = args.attendees as { name?: string; email: string }[] | undefined;
        const reminders = args.reminders as Array<{
          minutesBefore?: number;
          hoursBefore?: number;
          daysBefore?: number;
          action?: 'display' | 'audio' | 'email';
          description?: string;
        }> | undefined;

        if (!title || !start || !end) {
          throw new Error('title, start, and end are required');
        }

        const result = await calendar.createEvent(calendarId, {
          title,
          start,
          end,
          description,
          location,
          allDay,
          attendees,
          reminders,
        });
        return { success: true, result };
      }

      case 'fastmail_update_event': {
        const calendar = getCalendarTools();
        const eventId = args.event_id as string;
        const title = args.title as string | undefined;
        const start = args.start as string | undefined;
        const end = args.end as string | undefined;
        const description = args.description as string | undefined;
        const location = args.location as string | undefined;
        const allDay = args.all_day as boolean | undefined;
        const attendees = args.attendees as { name?: string; email: string }[] | undefined;

        if (!eventId) throw new Error('event_id is required');

        const updates: Record<string, unknown> = {};
        if (title !== undefined) updates.title = title;
        if (start !== undefined) updates.start = start;
        if (end !== undefined) updates.end = end;
        if (description !== undefined) updates.description = description;
        if (location !== undefined) updates.location = location;
        if (allDay !== undefined) updates.allDay = allDay;
        if (attendees !== undefined) updates.attendees = attendees;

        await calendar.updateEvent(eventId, updates);
        return { success: true, result: { message: 'Event updated' } };
      }

      case 'fastmail_delete_event': {
        const calendar = getCalendarTools();
        const eventId = args.event_id as string;

        if (!eventId) throw new Error('event_id is required');

        await calendar.deleteEvent(eventId);
        return { success: true, result: { message: 'Event deleted' } };
      }

      case 'fastmail_search_events': {
        const calendar = getCalendarTools();
        const query = args.query as string;
        const startDate = args.start_date as string | undefined;
        const endDate = args.end_date as string | undefined;

        if (!query) throw new Error('query is required');

        const result = await calendar.searchEvents(query, startDate, endDate);
        return { success: true, result };
      }

      // REMINDER TOOLS (4)
      case 'fastmail_create_recurring_event': {
        const calendar = getCalendarTools();
        const calendarId = args.calendar_id as string | undefined;
        const title = args.title as string;
        const start = args.start as string;
        const end = args.end as string;
        const recurrence = args.recurrence as 'daily' | 'weekly' | 'monthly' | 'yearly';
        const description = args.description as string | undefined;
        const location = args.location as string | undefined;
        const allDay = args.all_day as boolean | undefined;
        const attendees = args.attendees as { name?: string; email: string }[] | undefined;
        const recurrenceCount = args.recurrence_count as number | undefined;
        const recurrenceUntil = args.recurrence_until as string | undefined;
        const reminders = args.reminders as Array<{
          minutesBefore?: number;
          hoursBefore?: number;
          daysBefore?: number;
          action?: 'display' | 'audio' | 'email';
          description?: string;
        }> | undefined;

        if (!title || !start || !end || !recurrence) {
          throw new Error('title, start, end, and recurrence are required');
        }

        const result = await calendar.createRecurringEvent(calendarId, {
          title,
          start,
          end,
          recurrence,
          description,
          location,
          allDay,
          attendees,
          recurrenceCount,
          recurrenceUntil,
          reminders,
        });
        return { success: true, result };
      }

      case 'fastmail_add_event_reminder': {
        const calendar = getCalendarTools();
        const eventId = args.event_id as string;
        const minutesBefore = args.minutes_before as number | undefined;
        const hoursBefore = args.hours_before as number | undefined;
        const daysBefore = args.days_before as number | undefined;
        const action = args.action as 'display' | 'audio' | 'email' | undefined;
        const description = args.description as string | undefined;

        if (!eventId) throw new Error('event_id is required');

        const result = await calendar.addReminder(eventId, {
          minutesBefore,
          hoursBefore,
          daysBefore,
          action,
          description,
        });
        return { success: true, result };
      }

      case 'fastmail_remove_event_reminder': {
        const calendar = getCalendarTools();
        const eventId = args.event_id as string;
        const reminderId = args.reminder_id as string | undefined;

        if (!eventId) throw new Error('event_id is required');

        await calendar.removeReminder(eventId, reminderId);
        return { success: true, result: { message: 'Reminder removed' } };
      }

      case 'fastmail_list_event_reminders': {
        const calendar = getCalendarTools();
        const eventId = args.event_id as string;

        if (!eventId) throw new Error('event_id is required');

        const result = await calendar.listReminders(eventId);
        return { success: true, result };
      }

      case 'fastmail_create_event_with_reminder': {
        const calendar = getCalendarTools();
        const calendarId = args.calendar_id as string | undefined;
        const title = args.title as string;
        const start = args.start as string;
        const end = args.end as string;
        const description = args.description as string | undefined;
        const location = args.location as string | undefined;
        const allDay = args.all_day as boolean | undefined;
        const attendees = args.attendees as { name?: string; email: string }[] | undefined;
        const reminders = args.reminders as Array<{
          minutesBefore?: number;
          hoursBefore?: number;
          daysBefore?: number;
          action?: 'display' | 'audio' | 'email';
          description?: string;
        }> | undefined;

        if (!title || !start || !end) {
          throw new Error('title, start, and end are required');
        }

        if (!reminders || reminders.length === 0) {
          throw new Error('At least one reminder is required');
        }

        const result = await calendar.createEvent(calendarId, {
          title,
          start,
          end,
          description,
          location,
          allDay,
          attendees,
          reminders,
        });

         return { success: true, result };
       }

      case 'fastmail_get_thread': {
        const email = getEmailTools();
        const emailId = args.email_id as string;
        if (!emailId) throw new Error('email_id is required');
        const result = await email.getThread(emailId);
        return { success: true, result };
      }

      case 'fastmail_bulk_move_emails': {
        const email = getEmailTools();
        const emailIds = args.email_ids as string[];
        const targetMailboxId = args.target_mailbox_id as string;
        const sourceMailboxId = args.source_mailbox_id as string | undefined;
        
        if (!emailIds || !Array.isArray(emailIds) || emailIds.length === 0) {
          throw new Error('email_ids must be a non-empty array of email IDs');
        }
        if (!targetMailboxId) {
          throw new Error('target_mailbox_id is required');
        }
        
        const result = await email.bulkMoveToFolder(emailIds, targetMailboxId, sourceMailboxId);
        return { success: true, result };
      }

      case 'fastmail_bulk_set_labels': {
        const email = getEmailTools();
        const emailIds = args.email_ids as string[];
        const keywords = args.keywords as Record<string, boolean>;
        
        if (!emailIds || !Array.isArray(emailIds) || emailIds.length === 0) {
          throw new Error('email_ids must be a non-empty array of email IDs');
        }
        if (!keywords || typeof keywords !== 'object') {
          throw new Error('keywords must be an object with label:boolean pairs');
        }
        
        const result = await email.bulkSetKeywords(emailIds, keywords);
        return { success: true, result };
      }

      case 'fastmail_bulk_delete_emails': {
        const email = getEmailTools();
        const emailIds = args.email_ids as string[];
        
        if (!emailIds || !Array.isArray(emailIds) || emailIds.length === 0) {
          throw new Error('email_ids must be a non-empty array of email IDs');
        }
        
        const result = await email.bulkDeleteEmails(emailIds);
        return { success: true, result };
      }

      case 'fastmail_list_invitations': {
        const calendar = getCalendarTools();
        const result = await calendar.listInvitations();
        return { success: true, result };
      }

      case 'fastmail_respond_to_invitation': {
        const calendar = getCalendarTools();
        const eventId = args.event_id as string;
        const response = args.response as 'accept' | 'decline' | 'tentative';
        
        if (!eventId) throw new Error('event_id is required');
        if (!response || !['accept', 'decline', 'tentative'].includes(response)) {
          throw new Error('response must be one of: accept, decline, tentative');
        }
        
        await calendar.respondToInvitation(eventId, response);
        return { success: true, result: { message: `Invitation ${response}ed` } };
      }

        default:
         return { success: false, error: `Unknown tool: ${normalizedToolName}` };
    }
  } catch (error) {
    return formatError(error);
  }
}

// Help text
function showHelp(): void {
  console.log(`
Fastmail CLI - Email & Calendar Management

USAGE:
  bunx fastmail <command> [options]

COMMANDS:
  --help                     Show this help message
  --list                     List all available tools
  
  list_mailboxes
  list_emails [--mailbox-id ID] [--limit N]
  get_email --email-id ID
  search_emails --query TEXT [--limit N]
  send_email --to EMAIL [--to EMAIL...] --subject SUBJECT --text-body BODY [--html-body BODY]
  move_email --email-id ID --target-mailbox-id ID [--source-mailbox-id ID]
  set_labels --email-id ID --keywords JSON
  delete_email --email-id ID
  reply_email --email-id ID --text-body BODY [--subject SUBJECT] [--reply-all]
  
  list_calendars
  list_events [--calendar-id ID] [--start-date DATE] [--end-date DATE]
  get_event --event-id ID
  create_event --title TITLE --start DATE --end DATE [--description DESC] [--location LOC] [--all-day]
  update_event --event-id ID [--title TITLE] [--start DATE] [--end DATE]
  delete_event --event-id ID
  search_events --query TEXT [--start-date DATE] [--end-date DATE]
  
  create_recurring_event --title TITLE --start DATE --end DATE --recurrence FREQ [--reminders JSON]
  add_event_reminder --event-id ID [--minutes-before N] [--hours-before N] [--days-before N]
  remove_event_reminder --event-id ID [--reminder-id ID]
  list_event_reminders --event-id ID
  create_event_with_reminder --title TITLE --start DATE --end DATE [--reminders JSON]

ENVIRONMENT:
  FASTMAIL_API_TOKEN         Required for email operations
  FASTMAIL_USERNAME          Required for calendar operations
  FASTMAIL_PASSWORD          Required for calendar operations

OUTPUT:
  JSON format with structure: { success: boolean, result?: any, error?: string }
`);
}

// List available tools
function showList(): void {
  const tools = [
    // Email tools (10)
    'list_mailboxes',
    'list_emails',
    'get_email',
    'get_thread',
    'search_emails',
    'send_email',
    'move_email',
    'set_labels',
    'delete_email',
    'reply_email',
    // Bulk email tools (3)
    'bulk_move_emails',
    'bulk_set_labels',
    'bulk_delete_emails',
    // Calendar tools (10)
    'list_calendars',
    'list_events',
    'get_event',
    'create_event',
    'update_event',
    'delete_event',
    'search_events',
    'create_recurring_event',
    'list_invitations',
    'respond_to_invitation',
    // Reminder tools (4)
    'add_event_reminder',
    'remove_event_reminder',
    'list_event_reminders',
    'create_event_with_reminder',
  ];

  console.log('Available Tools (27 total):');
  console.log('\nEmail Tools (10):');
  tools.slice(0, 10).forEach((tool, i) => console.log(`  ${i + 1}. ${tool}`));
  
  console.log('\nBulk Email Tools (3):');
  tools.slice(10, 13).forEach((tool, i) => console.log(`  ${i + 11}. ${tool}`));

  console.log('\nCalendar Tools (10):');
  tools.slice(13, 23).forEach((tool, i) => console.log(`  ${i + 14}. ${tool}`));

  console.log('\nReminder Tools (4):');
  tools.slice(23).forEach((tool, i) => console.log(`  ${i + 24}. ${tool}`));
}

// Parse command line arguments
function parseArgs(args: string[]): { command: string; params: Record<string, unknown> } {
  const command = args[0];
  const params: Record<string, unknown> = {};

  for (let i = 1; i < args.length; i++) {
    const arg = args[i];

    if (arg.startsWith('--')) {
      const key = arg.slice(2);
      let value: unknown = true;

      // Handle key=value format
      if (key.includes('=')) {
        const [k, v] = key.split('=', 2);
        const param = k.replace(/-/g, '_');
        try {
          params[param] = JSON.parse(v);
        } catch {
          params[param] = v;
        }
      } else {
        // Handle next arg as value
        if (i + 1 < args.length && !args[i + 1].startsWith('--')) {
          const nextArg = args[i + 1];
          try {
            value = JSON.parse(nextArg);
          } catch {
            value = nextArg;
          }
          i++;
        }
        const param = key.replace(/-/g, '_');
        params[param] = value;
      }
    }
  }

  return { command, params };
}

// Parse JSON argument if provided as single argument
function parseJsonArg(args: string[]): { command: string; params: Record<string, unknown> } | null {
  if (args.length >= 2) {
    // Check if second argument looks like JSON (starts with { or [)
    const possibleJson = args[1];
    if (possibleJson && (possibleJson.startsWith('{') || possibleJson.startsWith('['))) {
      try {
        const params = JSON.parse(possibleJson);
        if (typeof params === 'object' && params !== null && !Array.isArray(params)) {
          return { command: args[0], params };
        }
      } catch {
        // Not valid JSON, fall through to normal parsing
      }
    }
  }
  return null;
}

// Main entry point
async function main(): Promise<void> {
  const args = process.argv.slice(2);

  if (args.length === 0 || args[0] === '--help') {
    showHelp();
    process.exit(0);
  }

  if (args[0] === '--list') {
    showList();
    process.exit(0);
  }

  // Try JSON parsing first, then fall back to normal parsing
  const jsonParsed = parseJsonArg(args);
  const { command, params } = jsonParsed || parseArgs(args);

  if (!command) {
    console.error('Error: No command specified');
    showHelp();
    process.exit(1);
  }

  const result = await handleTool(command, params);
  console.log(JSON.stringify(result, null, 2));

  process.exit(result.success ? 0 : 1);
}

main().catch((error) => {
  console.error('Fatal error:', error);
  process.exit(1);
});
