import { createDAVClient, DAVCalendar, DAVCalendarObject } from 'tsdav';

const CALDAV_SERVER = 'https://caldav.fastmail.com';

// Type for the client returned by createDAVClient
type CalDAVClientType = ReturnType<typeof createDAVClient> extends Promise<infer T> ? T : never;

export class CalDAVClient {
  private client: CalDAVClientType | null = null;
  private calendars: DAVCalendar[] | null = null;
  
  constructor(
    private username: string,
    private password: string
  ) {}
  
  async connect(): Promise<void> {
    if (this.client) return;
    
    this.client = await createDAVClient({
      serverUrl: CALDAV_SERVER,
      credentials: {
        username: this.username,
        password: this.password,
      },
      authMethod: 'Basic',
      defaultAccountType: 'caldav',
    });
    
    console.error('✅ Connected to Fastmail CalDAV');
  }
  
  async getCalendars(): Promise<DAVCalendar[]> {
    await this.connect();
    
    if (!this.calendars) {
      this.calendars = await this.client!.fetchCalendars();
    }
    
    return this.calendars;
  }
  
  async getCalendarById(calendarId: string): Promise<DAVCalendar | undefined> {
    const calendars = await this.getCalendars();
    return calendars.find(cal => cal.url === calendarId || cal.displayName === calendarId);
  }
  
  async fetchCalendarObjects(
    calendar: DAVCalendar,
    timeRange?: { start: string; end: string }
  ): Promise<DAVCalendarObject[]> {
    await this.connect();
    
    return await this.client!.fetchCalendarObjects({
      calendar,
      timeRange: timeRange || {
        start: new Date().toISOString(),
        end: new Date(Date.now() + 30 * 24 * 60 * 60 * 1000).toISOString(),
      },
    });
  }
  
  async createEvent(
    calendar: DAVCalendar,
    icsData: string,
    eventId: string
  ): Promise<void> {
    await this.connect();
    
    const eventUrl = `${calendar.url}${eventId}.ics`;
    
    await this.client!.createCalendarObject({
      calendar,
      filename: `${eventId}.ics`,
      iCalString: icsData,
    });
  }
  
  async updateEvent(
    calendar: DAVCalendar,
    event: DAVCalendarObject,
    icsData: string
  ): Promise<void> {
    await this.connect();
    
    await this.client!.updateCalendarObject({
      calendarObject: {
        ...event,
        data: icsData,
      },
    });
  }
  
  async deleteEvent(event: DAVCalendarObject): Promise<void> {
    await this.connect();
    
    await this.client!.deleteCalendarObject({
      calendarObject: event,
    });
  }
  
  getClient(): CalDAVClientType {
    if (!this.client) {
      throw new Error('CalDAV client not connected. Call connect() first.');
    }
    return this.client;
  }
}
