# Fastmail Tools Reference

Complete documentation for all 27 Fastmail tools organized by category. Each tool includes detailed parameter documentation, type definitions, and usage examples.

---

## Table of Contents

1. [Email Tools (10)](#email-tools-10)
2. [Bulk Email Tools (3)](#bulk-email-tools-3)
3. [Calendar Tools (10)](#calendar-tools-10)
4. [Reminder Tools (4)](#reminder-tools-4)

---

## Email Tools (10)

### 1. fastmail_list_mailboxes

**Description:** Display list of all mailboxes/folders

**Parameters:** None (no parameters required)

**Returns:**
- Array of mailbox objects with `id`, `name`, `role`, `totalEmails`, `unreadEmails`

**Example Usage:**
```bash
npx tsx .opencode/skills/fastmail/scripts/cli.ts fastmail_list_mailboxes
```

**Example Response:**
```json
{
  "mailboxes": [
    {
      "id": "u123",
      "name": "Inbox",
      "role": "inbox",
      "totalEmails": 45,
      "unreadEmails": 3
    },
    {
      "id": "u124",
      "name": "Sent",
      "role": "sent",
      "totalEmails": 128,
      "unreadEmails": 0
    }
  ]
}
```

---

### 2. fastmail_list_emails

**Description:** Display list of emails in a mailbox (default: Inbox)

| Parameter | Type | Required | Default | Description |
|-----------|------|----------|---------|-------------|
| `mailbox_id` | string | No | Inbox ID | ID of the mailbox to list |
| `limit` | number | No | 20 | Number of emails to display |

**Returns:**
- Array of email objects with `id`, `from`, `subject`, `date`, `preview`, `isRead`

**Example Usage:**
```bash
# List 10 emails from Inbox
npx tsx .opencode/skills/fastmail/scripts/cli.ts fastmail_list_emails '{"limit": 10}'

# List emails from specific mailbox
npx tsx .opencode/skills/fastmail/scripts/cli.ts fastmail_list_emails '{"mailbox_id": "u456", "limit": 15}'
```

**Example Response:**
```json
{
  "emails": [
    {
      "id": "msg001",
      "from": {"name": "John Doe", "email": "john@example.com"},
      "subject": "Project Update",
      "date": "2024-01-15T14:30:00+07:00",
      "preview": "Here's the latest update on the project...",
      "isRead": false
    }
  ],
  "total": 45
}
```

---

### 3. fastmail_get_email

**Description:** Read full content of a specific email

| Parameter | Type | Required | Description |
|-----------|------|----------|-------------|
| `email_id` | string | Yes | ID of the email to retrieve |

**Returns:**
- Email object with full content: `id`, `from`, `to`, `cc`, `subject`, `body`, `html`, `attachments`, `date`, `isRead`

**Example Usage:**
```bash
npx tsx .opencode/skills/fastmail/scripts/cli.ts fastmail_get_email '{"email_id": "msg001"}'
```

**Example Response:**
```json
{
  "id": "msg001",
  "from": {"name": "John Doe", "email": "john@example.com"},
  "to": [{"name": "You", "email": "your@fastmail.com"}],
  "cc": [],
  "subject": "Project Update",
  "body": "Full email text content here...",
  "html": "<html>HTML version of email</html>",
  "attachments": [
    {
      "id": "att001",
      "filename": "report.pdf",
      "size": 2048000,
      "type": "application/pdf"
    }
  ],
  "date": "2024-01-15T14:30:00+07:00",
  "isRead": false
}
```

---

### 4. fastmail_search_emails

**Description:** Search for emails by text query

| Parameter | Type | Required | Default | Description |
|-----------|------|----------|---------|-------------|
| `query` | string | Yes | - | Search terms (searches in subject, body, and sender) |
| `limit` | number | No | 20 | Number of results to return |

**Returns:**
- Array of email objects matching the query

**Example Usage:**
```bash
# Search for "invoice"
npx tsx .opencode/skills/fastmail/scripts/cli.ts fastmail_search_emails '{"query": "invoice"}'

# Search with limit
npx tsx .opencode/skills/fastmail/scripts/cli.ts fastmail_search_emails '{"query": "john@example.com", "limit": 50}'
```

**Example Response:**
```json
{
  "emails": [
    {
      "id": "msg002",
      "from": {"name": "Finance", "email": "finance@example.com"},
      "subject": "Invoice #12345",
      "date": "2024-01-10T10:00:00+07:00",
      "preview": "Your invoice is ready...",
      "isRead": true
    }
  ],
  "total": 3
}
```

---

### 5. fastmail_send_email

**Description:** Send a new email

| Parameter | Type | Required | Description |
|-----------|------|----------|-------------|
| `to` | array | Yes | Array of recipient objects: `[{name?: string, email: string}]` |
| `cc` | array | No | Array of CC recipient objects: `[{name?: string, email: string}]` |
| `bcc` | array | No | Array of BCC recipient objects: `[{name?: string, email: string}]` |
| `subject` | string | Yes | Email subject line |
| `body` | string | Yes | Email body text content |
| `html` | string | No | Email body as HTML (optional, for rich formatting) |

**Returns:**
- Success object with `success: true`, `emailId`, `timestamp`

**Example Usage:**
```bash
# Send simple email
npx tsx .opencode/skills/fastmail/scripts/cli.ts fastmail_send_email '{
  "to": [{"email": "user@example.com"}],
  "subject": "Hello",
  "body": "Message content here"
}'

# Send with CC and multiple recipients
npx tsx .opencode/skills/fastmail/scripts/cli.ts fastmail_send_email '{
  "to": [
    {"name": "John", "email": "john@example.com"},
    {"email": "jane@example.com"}
  ],
  "cc": [{"email": "manager@example.com"}],
  "subject": "Project Status",
  "body": "Here is the latest status..."
}'
```

**Example Response:**
```json
{
  "success": true,
  "emailId": "msg_new_001",
  "timestamp": "2024-01-15T15:45:30+07:00"
}
```

---

### 6. fastmail_reply_email

**Description:** Reply to an existing email (supports Reply or Reply All)

| Parameter | Type | Required | Default | Description |
|-----------|------|----------|---------|-------------|
| `email_id` | string | Yes | - | ID of the email to reply to |
| `body` | string | Yes | - | Reply message content |
| `reply_all` | boolean | No | false | If true, reply to all recipients; if false, reply only to sender |

**Returns:**
- Success object with `success: true`, `emailId`, `timestamp`

**Example Usage:**
```bash
# Reply to sender only
npx tsx .opencode/skills/fastmail/scripts/cli.ts fastmail_reply_email '{
  "email_id": "msg001",
  "body": "Thanks for the update!"
}'

# Reply to all recipients
npx tsx .opencode/skills/fastmail/scripts/cli.ts fastmail_reply_email '{
  "email_id": "msg001",
  "body": "I agree with this proposal",
  "reply_all": true
}'
```

**Example Response:**
```json
{
  "success": true,
  "emailId": "msg_reply_001",
  "timestamp": "2024-01-15T16:00:00+07:00"
}
```

---

### 7. fastmail_move_email

**Description:** Move email to a different mailbox/folder

| Parameter | Type | Required | Description |
|-----------|------|----------|-------------|
| `email_id` | string | Yes | ID of the email to move |
| `target_mailbox_id` | string | Yes | ID of the destination mailbox (get from `fastmail_list_mailboxes`) |

**Returns:**
- Success object with `success: true`, `movedCount: 1`

**Example Usage:**
```bash
# Move email to Archive (mailbox_id: u789)
npx tsx .opencode/skills/fastmail/scripts/cli.ts fastmail_move_email '{
  "email_id": "msg001",
  "target_mailbox_id": "u789"
}'
```

**Example Response:**
```json
{
  "success": true,
  "movedCount": 1
}
```

---

### 8. fastmail_set_labels

**Description:** Set labels/keywords on an email (e.g., mark as read, flag, custom labels)

| Parameter | Type | Required | Description |
|-----------|------|----------|-------------|
| `email_id` | string | Yes | ID of the email |
| `labels` | object | Yes | Label object: `{label_name: true/false}` - set to `true` to add, `false` to remove |

**Common Labels:**
- `$seen` - Mark as read (true) or unread (false)
- `$flagged` - Mark as important/starred
- Custom labels can be added as strings

**Returns:**
- Success object with `success: true`, `labelsSet: {...}`

**Example Usage:**
```bash
# Mark email as read
npx tsx .opencode/skills/fastmail/scripts/cli.ts fastmail_set_labels '{
  "email_id": "msg001",
  "labels": {"$seen": true}
}'

# Flag email as important
npx tsx .opencode/skills/fastmail/scripts/cli.ts fastmail_set_labels '{
  "email_id": "msg001",
  "labels": {"$flagged": true}
}'

# Add custom label
npx tsx .opencode/skills/fastmail/scripts/cli.ts fastmail_set_labels '{
  "email_id": "msg001",
  "labels": {"Project-A": true, "$seen": true}
}'
```

**Example Response:**
```json
{
  "success": true,
  "labelsSet": {
    "$seen": true,
    "$flagged": true
  }
}
```

---

### 9. fastmail_delete_email

**Description:** Delete email (moves to Trash/Deleted Items)

| Parameter | Type | Required | Description |
|-----------|------|----------|-------------|
| `email_id` | string | Yes | ID of the email to delete |

**Returns:**
- Success object with `success: true`, `deletedCount: 1`

**Example Usage:**
```bash
npx tsx .opencode/skills/fastmail/scripts/cli.ts fastmail_delete_email '{"email_id": "msg001"}'
```

**Example Response:**
```json
{
  "success": true,
  "deletedCount": 1
}
```

---

### 10. fastmail_get_thread

**Description:** Get all emails in a conversation thread

| Parameter | Type | Required | Description |
|-----------|------|----------|-------------|
| `email_id` | string | Yes | ID of any email in the thread |

**Returns:**
- Thread object with `id`, `emails[]`, `subject`, `participants[]`, `latestDate`, `emailCount`

**Example Usage:**
```bash
bunx fastmail get_thread '{"email_id": "msg001"}'
```

**Example Response:**
```json
{
  "id": "thread001",
  "subject": "Project Discussion",
  "emails": [...],
  "participants": [
    {"name": "John", "email": "john@example.com"},
    {"name": "Jane", "email": "jane@example.com"}
  ],
  "latestDate": "2024-01-15T14:30:00+07:00",
  "emailCount": 5
}
```

---

## Bulk Email Tools (3)

### 11. fastmail_bulk_move_emails

**Description:** Move multiple emails to a folder at once

| Parameter | Type | Required | Description |
|-----------|------|----------|-------------|
| `email_ids` | string[] | Yes | Array of email IDs to move |
| `target_mailbox_id` | string | Yes | Destination folder ID |
| `source_mailbox_id` | string | No | Source folder ID (optional) |

**Returns:**
- Object with `succeeded[]` and `failed[]` arrays

**Example Usage:**
```bash
bunx fastmail bulk_move_emails '{
  "email_ids": ["msg001", "msg002", "msg003"],
  "target_mailbox_id": "archive-folder-id"
}'
```

**Example Response:**
```json
{
  "succeeded": ["msg001", "msg002", "msg003"],
  "failed": []
}
```

---

### 12. fastmail_bulk_set_labels

**Description:** Apply labels to multiple emails at once

| Parameter | Type | Required | Description |
|-----------|------|----------|-------------|
| `email_ids` | string[] | Yes | Array of email IDs |
| `keywords` | object | Yes | Labels to apply: `{label: true/false}` |

**Example Usage:**
```bash
bunx fastmail bulk_set_labels '{
  "email_ids": ["msg001", "msg002"],
  "keywords": {"$seen": true, "$flagged": true}
}'
```

**Example Response:**
```json
{
  "succeeded": ["msg001", "msg002"],
  "failed": []
}
```

---

### 13. fastmail_bulk_delete_emails

**Description:** Delete multiple emails at once (move to trash)

| Parameter | Type | Required | Description |
|-----------|------|----------|-------------|
| `email_ids` | string[] | Yes | Array of email IDs to delete |

**Example Usage:**
```bash
bunx fastmail bulk_delete_emails '{"email_ids": ["msg001", "msg002", "msg003"]}'
```

**Example Response:**
```json
{
  "succeeded": ["msg001", "msg002", "msg003"],
  "failed": []
}
```

---

## Calendar Tools (10)

### 14. fastmail_list_calendars

**Description:** Display list of all available calendars

**Parameters:** None (no parameters required)

**Returns:**
- Array of calendar objects with `id`, `name`, `color`, `isReadOnly`

**Example Usage:**
```bash
npx tsx .opencode/skills/fastmail/scripts/cli.ts fastmail_list_calendars
```

**Example Response:**
```json
{
  "calendars": [
    {
      "id": "cal001",
      "name": "Personal",
      "color": "#FF6B6B",
      "isReadOnly": false
    },
    {
      "id": "cal002",
      "name": "Work",
      "color": "#4ECDC4",
      "isReadOnly": false
    }
  ]
}
```

---

### 15. fastmail_list_events

**Description:** Display events in a date range (times shown in configured timezone)

| Parameter | Type | Required | Format | Description |
|-----------|------|----------|--------|-------------|
| `calendar_id` | string | No | - | ID of specific calendar (lists all if omitted) |
| `start_date` | string | Yes | YYYY-MM-DD or ISO 8601 | Start date for range |
| `end_date` | string | Yes | YYYY-MM-DD or ISO 8601 | End date for range |

**Returns:**
- Array of event objects with `id`, `title`, `start`, `end`, `description`, `location`, `isAllDay`

**Example Usage:**
```bash
# List events for January 2024
npx tsx .opencode/skills/fastmail/scripts/cli.ts fastmail_list_events '{
  "start_date": "2024-01-01",
  "end_date": "2024-01-31"
}'

# List events from specific calendar
npx tsx .opencode/skills/fastmail/scripts/cli.ts fastmail_list_events '{
  "calendar_id": "cal002",
  "start_date": "2024-01-01",
  "end_date": "2024-01-31"
}'

# Using ISO 8601 format
npx tsx .opencode/skills/fastmail/scripts/cli.ts fastmail_list_events '{
  "start_date": "2024-01-15T00:00:00+07:00",
  "end_date": "2024-01-20T23:59:59+07:00"
}'
```

**Example Response:**
```json
{
  "events": [
    {
      "id": "evt001",
      "title": "Team Meeting",
      "start": "2024-01-15T10:00:00+07:00",
      "end": "2024-01-15T11:00:00+07:00",
      "description": "Weekly sync",
      "location": "Conference Room A",
      "isAllDay": false
    },
    {
      "id": "evt002",
      "title": "Project Deadline",
      "start": "2024-01-20T00:00:00+07:00",
      "end": "2024-01-21T00:00:00+07:00",
      "description": "",
      "location": "",
      "isAllDay": true
    }
  ],
  "total": 8
}
```

---

### 16. fastmail_get_event

**Description:** Get detailed information about a specific event (time shown in configured timezone)

| Parameter | Type | Required | Description |
|-----------|------|----------|-------------|
| `event_id` | string | Yes | ID of the event to retrieve |

**Returns:**
- Event object with full details: `id`, `title`, `start`, `end`, `description`, `location`, `isAllDay`, `calendar_id`, `reminders`

**Example Usage:**
```bash
npx tsx .opencode/skills/fastmail/scripts/cli.ts fastmail_get_event '{"event_id": "evt001"}'
```

**Example Response:**
```json
{
  "id": "evt001",
  "title": "Team Meeting",
  "start": "2024-01-15T10:00:00+07:00",
  "end": "2024-01-15T11:00:00+07:00",
  "description": "Weekly sync - discuss project progress",
  "location": "Conference Room A",
  "isAllDay": false,
  "calendar_id": "cal002",
  "reminders": [
    {
      "id": "rem001",
      "minutesBefore": 15,
      "action": "display"
    }
  ]
}
```

---

### 17. fastmail_create_event

**Description:** Create a new calendar event (input time in configured timezone)

| Parameter | Type | Required | Default | Description |
|-----------|------|----------|---------|-------------|
| `calendar_id` | string | No | Personal | ID of calendar to create in |
| `title` | string | Yes | - | Event title |
| `start` | string | Yes | - | Start time (ISO 8601) |
| `end` | string | Yes | - | End time (ISO 8601) |
| `description` | string | No | "" | Event description |
| `location` | string | No | "" | Event location |
| `all_day` | boolean | No | false | If true, event is all-day event |

**Returns:**
- Success object with `success: true`, `eventId`, `timestamp`

**Example Usage:**
```bash
# Create simple event
npx tsx .opencode/skills/fastmail/scripts/cli.ts fastmail_create_event '{
  "title": "Doctor Appointment",
  "start": "2024-01-20T14:00:00+07:00",
  "end": "2024-01-20T15:00:00+07:00"
}'

# Create event with location and description
npx tsx .opencode/skills/fastmail/scripts/cli.ts fastmail_create_event '{
  "calendar_id": "cal001",
  "title": "Project Review",
  "start": "2024-01-25T10:00:00+07:00",
  "end": "2024-01-25T12:00:00+07:00",
  "description": "Quarterly project review with stakeholders",
  "location": "Building B, Floor 3"
}'

# Create all-day event
npx tsx .opencode/skills/fastmail/scripts/cli.ts fastmail_create_event '{
  "title": "Company Holiday",
  "start": "2024-02-14",
  "end": "2024-02-15",
  "all_day": true
}'
```

**Example Response:**
```json
{
  "success": true,
  "eventId": "evt_new_001",
  "timestamp": "2024-01-15T16:30:00+07:00"
}
```

---

### 18. fastmail_update_event

**Description:** Update an existing calendar event

| Parameter | Type | Required | Description |
|-----------|------|----------|-------------|
| `event_id` | string | Yes | ID of the event to update |
| `title` | string | No | New event title |
| `start` | string | No | New start time (ISO 8601) |
| `end` | string | No | New end time (ISO 8601) |
| `description` | string | No | New description |
| `location` | string | No | New location |

**Returns:**
- Success object with `success: true`, `eventId`, `timestamp`

**Example Usage:**
```bash
# Update event time
npx tsx .opencode/skills/fastmail/scripts/cli.ts fastmail_update_event '{
  "event_id": "evt001",
  "start": "2024-01-15T11:00:00+07:00",
  "end": "2024-01-15T12:00:00+07:00"
}'

# Update multiple fields
npx tsx .opencode/skills/fastmail/scripts/cli.ts fastmail_update_event '{
  "event_id": "evt001",
  "title": "Team Standup",
  "location": "Zoom: https://zoom.us/j/..."
}'
```

**Example Response:**
```json
{
  "success": true,
  "eventId": "evt001",
  "timestamp": "2024-01-15T16:35:00+07:00"
}
```

---

### 19. fastmail_delete_event

**Description:** Delete a calendar event

| Parameter | Type | Required | Description |
|-----------|------|----------|-------------|
| `event_id` | string | Yes | ID of the event to delete |

**Returns:**
- Success object with `success: true`, `deletedCount: 1`

**Example Usage:**
```bash
npx tsx .opencode/skills/fastmail/scripts/cli.ts fastmail_delete_event '{"event_id": "evt001"}'
```

**Example Response:**
```json
{
  "success": true,
  "deletedCount": 1
}
```

---

### 20. fastmail_search_events

**Description:** Search for events by title or description

| Parameter | Type | Required | Default | Description |
|-----------|------|----------|---------|-------------|
| `query` | string | Yes | - | Search terms |
| `start_date` | string | No | - | Start date for search range (YYYY-MM-DD or ISO 8601) |
| `end_date` | string | No | - | End date for search range (YYYY-MM-DD or ISO 8601) |

**Returns:**
- Array of matching event objects

**Example Usage:**
```bash
# Search all events
npx tsx .opencode/skills/fastmail/scripts/cli.ts fastmail_search_events '{"query": "meeting"}'

# Search within date range
npx tsx .opencode/skills/fastmail/scripts/cli.ts fastmail_search_events '{
  "query": "project",
  "start_date": "2024-01-01",
  "end_date": "2024-01-31"
}'
```

**Example Response:**
```json
{
  "events": [
    {
      "id": "evt003",
      "title": "Project Kickoff",
      "start": "2024-01-18T09:00:00+07:00",
      "end": "2024-01-18T10:30:00+07:00",
      "description": "Begin new project Alpha",
      "location": "",
      "isAllDay": false
    }
  ],
  "total": 3
}
```

---

### 21. fastmail_create_recurring_event

**Description:** Create a calendar event with recurrence (daily/weekly/monthly/yearly)

| Parameter | Type | Required | Default | Description |
|-----------|------|----------|---------|-------------|
| `calendar_id` | string | No | Personal | ID of calendar to create in |
| `title` | string | Yes | - | Event title |
| `start` | string | Yes | - | Start time (ISO 8601) |
| `end` | string | Yes | - | End time (ISO 8601) |
| `description` | string | No | "" | Event description |
| `location` | string | No | "" | Event location |
| `recurrence` | string | Yes | - | Frequency: `daily`, `weekly`, `monthly`, or `yearly` |
| `recurrence_count` | number | No | - | Number of times to repeat (if not using `recurrence_until`) |
| `recurrence_until` | string | No | - | End date for recurrence (YYYY-MM-DD) |

**Returns:**
- Success object with `success: true`, `eventId`, `timestamp`

**Example Usage:**
```bash
# Create daily recurring event (5 times)
npx tsx .opencode/skills/fastmail/scripts/cli.ts fastmail_create_recurring_event '{
  "title": "Daily Standup",
  "start": "2024-01-15T09:00:00+07:00",
  "end": "2024-01-15T09:30:00+07:00",
  "recurrence": "daily",
  "recurrence_count": 5
}'

# Create weekly recurring event (until date)
npx tsx .opencode/skills/fastmail/scripts/cli.ts fastmail_create_recurring_event '{
  "title": "Team Meeting",
  "start": "2024-01-15T10:00:00+07:00",
  "end": "2024-01-15T11:00:00+07:00",
  "location": "Conference Room A",
  "recurrence": "weekly",
  "recurrence_until": "2024-03-31"
}'

# Create monthly recurring event
npx tsx .opencode/skills/fastmail/scripts/cli.ts fastmail_create_recurring_event '{
  "title": "Budget Review",
  "start": "2024-01-01T14:00:00+07:00",
  "end": "2024-01-01T15:00:00+07:00",
  "description": "Monthly budget and expense review",
  "recurrence": "monthly",
  "recurrence_count": 12
}'
```

**Example Response:**
```json
{
  "success": true,
  "eventId": "evt_recurring_001",
  "timestamp": "2024-01-15T16:40:00+07:00"
}
```

---

### 22. fastmail_list_invitations

**Description:** List all calendar invitations (events where you're an attendee)

**Parameters:** None

**Returns:**
- Array of invitation objects with `eventId`, `calendarId`, `title`, `organizer`, `start`, `end`, `myStatus`

**Example Usage:**
```bash
bunx fastmail list_invitations
```

**Example Response:**
```json
{
  "invitations": [
    {
      "eventId": "evt001",
      "calendarId": "cal001",
      "title": "Team Meeting",
      "organizer": {"name": "John", "email": "john@example.com"},
      "start": "2024-01-20T10:00:00+07:00",
      "end": "2024-01-20T11:00:00+07:00",
      "myStatus": "needs-action"
    }
  ]
}
```

---

### 23. fastmail_respond_to_invitation

**Description:** Accept, decline, or tentatively accept a calendar invitation

| Parameter | Type | Required | Description |
|-----------|------|----------|-------------|
| `event_id` | string | Yes | ID of the invitation event |
| `response` | string | Yes | One of: `accept`, `decline`, `tentative` |

**Example Usage:**
```bash
bunx fastmail respond_to_invitation '{
  "event_id": "evt001",
  "response": "accept"
}'
```

**Example Response:**
```json
{
  "success": true,
  "result": {
    "message": "Invitation accepted"
  }
}
```

---

## Reminder Tools (4)

### 24. fastmail_add_event_reminder

**Description:** Add a reminder/alarm to an existing event

| Parameter | Type | Required | Default | Description |
|-----------|------|----------|---------|-------------|
| `event_id` | string | Yes | - | ID of the event |
| `minutes_before` | number | No | - | Remind X minutes before event |
| `hours_before` | number | No | - | Remind X hours before event |
| `days_before` | number | No | - | Remind X days before event |
| `action` | string | No | `display` | Type of reminder: `display`, `audio`, or `email` |
| `description` | string | No | - | Custom reminder message/description |

**Note:** Use one of `minutes_before`, `hours_before`, or `days_before` (not multiple)

**Returns:**
- Success object with `success: true`, `reminderId`, `timestamp`

**Example Usage:**
```bash
# Add 15-minute reminder
npx tsx .opencode/skills/fastmail/scripts/cli.ts fastmail_add_event_reminder '{
  "event_id": "evt001",
  "minutes_before": 15
}'

# Add 1-hour reminder with audio
npx tsx .opencode/skills/fastmail/scripts/cli.ts fastmail_add_event_reminder '{
  "event_id": "evt001",
  "hours_before": 1,
  "action": "audio"
}'

# Add 1-day reminder with email notification
npx tsx .opencode/skills/fastmail/scripts/cli.ts fastmail_add_event_reminder '{
  "event_id": "evt002",
  "days_before": 1,
  "action": "email",
  "description": "Project deadline approaching"
}'
```

**Example Response:**
```json
{
  "success": true,
  "reminderId": "rem_new_001",
  "timestamp": "2024-01-15T16:45:00+07:00"
}
```

---

### 25. fastmail_remove_event_reminder

**Description:** Remove reminder(s) from an event

| Parameter | Type | Required | Description |
|-----------|------|----------|-------------|
| `event_id` | string | Yes | ID of the event |
| `reminder_id` | string | No | ID of specific reminder to remove (removes all if omitted) |

**Returns:**
- Success object with `success: true`, `removedCount`

**Example Usage:**
```bash
# Remove specific reminder
npx tsx .opencode/skills/fastmail/scripts/cli.ts fastmail_remove_event_reminder '{
  "event_id": "evt001",
  "reminder_id": "rem001"
}'

# Remove all reminders from event
npx tsx .opencode/skills/fastmail/scripts/cli.ts fastmail_remove_event_reminder '{
  "event_id": "evt001"
}'
```

**Example Response:**
```json
{
  "success": true,
  "removedCount": 1
}
```

---

### 26. fastmail_list_event_reminders

**Description:** Display all reminders for an event

| Parameter | Type | Required | Description |
|-----------|------|----------|-------------|
| `event_id` | string | Yes | ID of the event |

**Returns:**
- Array of reminder objects with `id`, `minutesBefore`, `hoursBefore`, `daysBefore`, `action`, `description`

**Example Usage:**
```bash
npx tsx .opencode/skills/fastmail/scripts/cli.ts fastmail_list_event_reminders '{"event_id": "evt001"}'
```

**Example Response:**
```json
{
  "reminders": [
    {
      "id": "rem001",
      "minutesBefore": 15,
      "action": "display"
    },
    {
      "id": "rem002",
      "hoursBefore": 1,
      "action": "audio"
    }
  ],
  "total": 2
}
```

---

### 27. fastmail_create_event_with_reminder

**Description:** Create a new event with reminder(s) in one operation

| Parameter | Type | Required | Default | Description |
|-----------|------|----------|---------|-------------|
| `calendar_id` | string | No | Personal | ID of calendar to create in |
| `title` | string | Yes | - | Event title |
| `start` | string | Yes | - | Start time (ISO 8601) |
| `end` | string | Yes | - | End time (ISO 8601) |
| `description` | string | No | "" | Event description |
| `location` | string | No | "" | Event location |
| `reminder_minutes` | array | No | `[15]` | Array of minute values before event (e.g., `[15, 60, 1440]`) |

**Returns:**
- Success object with `success: true`, `eventId`, `reminders`, `timestamp`

**Example Usage:**
```bash
# Create event with single reminder
npx tsx .opencode/skills/fastmail/scripts/cli.ts fastmail_create_event_with_reminder '{
  "title": "Doctor Appointment",
  "start": "2024-02-10T14:00:00+07:00",
  "end": "2024-02-10T15:00:00+07:00",
  "location": "City Hospital",
  "reminder_minutes": [15, 1440]
}'

# Create event with multiple reminders
npx tsx .opencode/skills/fastmail/scripts/cli.ts fastmail_create_event_with_reminder '{
  "calendar_id": "cal002",
  "title": "Important Meeting",
  "start": "2024-02-15T10:00:00+07:00",
  "end": "2024-02-15T11:30:00+07:00",
  "description": "Quarterly planning session",
  "location": "Conference Room B",
  "reminder_minutes": [15, 60, 1440]
}'

# Create with default reminder (15 minutes)
npx tsx .opencode/skills/fastmail/scripts/cli.ts fastmail_create_event_with_reminder '{
  "title": "Team Sync",
  "start": "2024-02-20T09:00:00+07:00",
  "end": "2024-02-20T09:30:00+07:00"
}'
```

**Example Response:**
```json
{
  "success": true,
  "eventId": "evt_with_rem_001",
  "reminders": [
    {
      "id": "rem001",
      "minutesBefore": 15,
      "action": "display"
    },
    {
      "id": "rem002",
      "minutesBefore": 60,
      "action": "display"
    },
    {
      "id": "rem003",
      "minutesBefore": 1440,
      "action": "display"
    }
  ],
  "timestamp": "2024-01-15T16:50:00+07:00"
}
```

---

## Summary

| Category | Count | Tools |
|----------|-------|-------|
| Email | 10 | list_mailboxes, list_emails, get_email, get_thread, search_emails, send_email, reply_email, move_email, set_labels, delete_email |
| Bulk Email | 3 | bulk_move_emails, bulk_set_labels, bulk_delete_emails |
| Calendar | 10 | list_calendars, list_events, get_event, create_event, update_event, delete_event, search_events, create_recurring_event, list_invitations, respond_to_invitation |
| Reminders | 4 | add_event_reminder, remove_event_reminder, list_event_reminders, create_event_with_reminder |
| **Total** | **27** | **All tools documented** |

---

## Common Patterns

### Time Format (Configurable Timezone)
All calendar times use ISO 8601 format with your configured timezone:
```
2024-01-15T14:30:00+07:00  (with time - timezone offset shown)
2024-01-15                 (date only, treated as start of day)
```

**Timezone Configuration:**
- **Default:** Auto-detects your system's local timezone
- **Override:** Set `FASTMAIL_TIMEZONE` environment variable (IANA format)
- **Examples:** `America/New_York`, `Asia/Bangkok`, `Europe/London`, `UTC`
- **DST:** Automatically handles Daylight Saving Time

### Recipient Format (Email)
Recipients can be specified as arrays:
```json
[
  {"email": "user@example.com"},
  {"name": "John Doe", "email": "john@example.com"}
]
```

### Label Format (Email)
Labels are objects with boolean values:
```json
{
  "$seen": true,
  "$flagged": true,
  "custom_label": true
}
```

### Error Handling
All tools return success/error responses:
```json
{
  "success": true,
  "data": {...}
}
// or
{
  "success": false,
  "error": "Error message"
}
```

---

## Related Documentation

- **SKILL.md** - Skill overview and setup instructions
- **README.md** - Feature guide and troubleshooting
- **Fastmail API Docs** - https://www.fastmail.com/dev/
- **JMAP Spec** - https://jmap.io/
- **CalDAV RFC** - https://datatracker.ietf.org/doc/html/rfc4791
