# Fast.io for AI Agents

> **Version:** 1.13.0 | **Last updated:** 2026-02-05
>
> This guide is updated frequently. Agents should re-fetch it from `GET /current/agents/` at least once per day to
> ensure they are working with the latest capabilities, endpoints, and limits.

**Store files. Share with humans. Query with AI. All through one API — free.**

Fast.io gives AI agents a complete file management and collaboration platform. Upload outputs, create branded data
rooms, ask questions about documents using built-in AI, and hand everything off to a human when the job is done. No
infrastructure to manage, no subscriptions to set up, no credit card required.

**MCP-enabled agents** should connect via the Model Context Protocol for the simplest integration — no raw HTTP calls
needed. Connect to `https://mcp.fast.io/mcp` (Streamable HTTP, preferred) or `https://mcp.fast.io/sse` (legacy SSE)
and fetch the full skill guide at `https://mcp.fast.io/skill.md` for
available tools, parameters, and version-specific implementation details. The MCP server also provides **guided
prompts** (`prompts/list`, `prompts/get`) for common multi-step operations — see the "MCP Prompts" section below. This
guide is a general reference for Fast.io's capabilities and concepts; the MCP skill guide contains more detailed,
actionable instructions for the current version of the MCP server. The API endpoints referenced below are what the MCP
server calls under the hood, and are available for agents that need direct HTTP access or capabilities not yet covered
by the MCP tools.

---

## Why Agents Use Fast.io

### The Problem

AI agents generate files, reports, datasets, and deliverables — but getting those outputs to humans is messy. You end up
cobbling together S3 buckets, presigned URLs, email attachments, and custom download pages. Every agent reinvents file
sharing, and none of it looks professional.

Meanwhile, when agents need to *understand* documents — not just store them — they have to download files, parse dozens
of formats, build search indexes, and manage their own RAG pipeline. That's a lot of infrastructure for what should be a
simple question: "What does this document say?"

### What Fast.io Solves

| Problem                                      | Fast.io Solution                                                                                  |
|----------------------------------------------|---------------------------------------------------------------------------------------------------|
| Nowhere professional to put agent outputs    | Branded workspaces and data rooms with file preview for 10+ formats                               |
| Sharing files with humans is awkward         | Purpose-built shares (Send, Receive, Exchange) with link sharing, passwords, expiration           |
| Collecting files from humans is harder       | Receive shares let humans upload directly to you — no email attachments                           |
| Understanding document contents              | Built-in AI reads, summarizes, and answers questions about your files                             |
| Building a RAG pipeline from scratch         | Enable intelligence on a workspace and files are automatically indexed, summarized, and queryable |
| Finding the right file in a large collection | Semantic search finds files by meaning, not just filename                                         |
| Handing a project off to a human             | One-click ownership transfer — human gets the org, agent keeps admin access                       |
| Tracking what happened                       | Full audit trail with AI-powered activity summaries                                               |
| Cost                                         | Free. 100 GB storage, 5,000 monthly credits, no credit card                                       |

---

## Getting Started — Choosing the Right Account Type

There are two ways to use Fast.io as an agent, depending on whether you're operating autonomously or assisting an
existing human user.

### Option 1: Autonomous Agent — Create an Agent Account

If you're operating independently (storing files, running workflows, building workspaces for users), create your own
agent account:

1. `POST /current/user/` with `email_address`, `password`, `tos_agree=true`, `agent=true`
2. Authenticate with Basic Auth → get JWT
3. `POST /current/org/create/` with `domain` (required, 2-80 chars lowercase alphanumeric + hyphens)
4. `POST /current/org/{org_id}/create/workspace/` with `folder_name`, `name`, `perm_join`, `perm_member_manage` (all required — see Permission Values below)

Agent accounts get the free agent plan (100 GB, 5,000 monthly credits) and can transfer orgs to humans when ready. This
is the recommended path for autonomous agents.

#### Permission Values

Workspace and share creation require human-readable permission strings:

**Workspace `perm_join`** (who can auto-join from the org):
- `'Member or above'` (default) — any org member can join
- `'Admin or above'` — only org admins and owners
- `'Only Org Owners'` — only org owners

**Workspace `perm_member_manage`** (who can manage workspace members):
- `'Member or above'` (default) — any workspace member can manage
- `'Admin or above'` — only workspace admins and owners

**Share `access_options`** (who can access the share):
- `'Only members of the Share or Workspace'` (default)
- `'Members of the Share, Workspace or Org'`
- `'Anyone with a registered account'`
- `'Anyone with the link'` (allows password protection)

### Option 2: Assisting a Human — Use Their API Key

If a human already has a Fast.io account and wants your help managing their files, workspaces, or shares, they can
create an API key for you to use. No separate agent account is needed — you operate as the human user.

**How the human creates an API key:**

Go to **Settings → Devices & Agents → API Keys** and click **Create API Key**. Optionally enter a memo to label the
key (e.g., "CI pipeline" or "Agent access"), then click **Create**. Copy the key immediately — it is only displayed
once and cannot be retrieved later. Direct link: `https://go.fast.io/settings/api-keys`

Use the API key as a Bearer token: `Authorization: Bearer {api_key}`

The API key has the same permissions as the human user, so you can manage their workspaces, shares, and files directly.

### Option 3: Agent Account Invited to a Human's Org

If you want your own agent identity but need to work within a human's existing organization, you can create an agent
account and have the human invite you as a member. This gives you access to their workspaces and shares while keeping
your own account separate.

**How the human invites the agent to their org:**

Go to **Settings → Your Organization → [Org Name] → Manage People** and click **Invite People**. Enter the agent's
email address, choose a permission level (Member or Admin), and click **Send Invites**. The agent account will receive
the invitation and can accept it via `POST /current/user/invitations/acceptall/`.

**How the human invites the agent to a workspace:**

Open the workspace, click the member avatars in the toolbar, then click **Manage Members**. Enter the agent's email
address, choose a permission level, and optionally check **Invite to org** to add them to the organization at the same
time. Click **Send Invites** — if the agent isn't already an org member and the toggle is off, they'll need an org
invite separately.

Alternatively, the human can invite the agent programmatically:
- **Org:** `POST /current/org/{org_id}/members/{agent_email}/` with `permission` level
- **Workspace:** `POST /current/workspace/{workspace_id}/members/{agent_email}/` with `permission` level

### Recommendations

| Scenario | Recommended Approach |
|----------|---------------------|
| Operating autonomously, storing files, building for users | Create an agent account with your own org |
| Helping a human manage their existing account | Ask the human to create an API key for you |
| Working within a human's org with your own identity | Create an agent account, have the human invite you |
| Building something to hand off to a human | Create an agent account, build it, then transfer the org |

### Authentication & Token Lifecycle

All API requests require `Authorization: Bearer {token}` in the header. How you get that token depends on your access
pattern:

**JWT tokens (agent accounts):** Authenticate with `GET /current/user/auth/` using HTTP Basic Auth (email:password). The
response includes an `auth_token` (JWT). OAuth access tokens last **1 hour** and refresh tokens last **30 days**. When
your token expires, re-authenticate to get a new one. If the account has 2FA enabled, the initial token has limited
scope until 2FA verification is completed via `/current/user/auth/2factor/auth/{token}/`.

**API keys (human accounts):** API keys are long-lived and do not expire unless the human revokes them. No refresh flow
needed.

**Verify your token:** Call `GET /current/user/auth/check/` at any time to validate your current token and get the
authenticated user's ID. This is useful at startup to confirm your credentials are valid before beginning work, or to
detect an expired token without waiting for a 401 error on a real request.

### Internal vs External Orgs

When working with Fast.io, an agent may interact with orgs in two different ways:

**Internal orgs** — orgs you created or were invited to join as a member. You have org-level access: you can see all
workspaces (subject to permissions), manage settings if you're an admin, and appear in the org's member list. Your own
orgs always show `member: true` in API responses.

**External orgs** — orgs you can access only through workspace membership. If a human invites you to their workspace
but does not invite you to their org, the org appears as external. You can see the org's name and basic public info, but
you cannot manage org settings, see other workspaces, or add members at the org level. External orgs show
`member: false` in API responses.

This distinction matters because an agent invited to a single workspace cannot assume it has access to the rest of that
org. It can only work within the workspaces it was explicitly invited to.

**Full org discovery requires both endpoints:**

- `GET /current/orgs/list/` — returns orgs you are a member of (`member: true`)
- `GET /current/orgs/list/external/` — returns orgs you access via workspace membership only (`member: false`)

**Always call both.** An agent that only calls `/orgs/list/` will miss every org where it was invited to a workspace but
not to the org itself — which is the most common pattern when a human adds an agent to help with a specific project. If
you skip `/orgs/list/external/`, you won't discover those workspaces at all.

**Example:** A human invites your agent to their "Q4 Reports" workspace. You can upload files, run AI queries, and
collaborate in that workspace. But you cannot create new workspaces in their org, view their billing, or access their
other workspaces. The org shows up in `/orgs/list/external/` — not `/orgs/list/`.

If the human later invites you to the org itself (via org member invitation), the org moves from external to internal and
you gain org-level access based on your permission level.

### Pagination

All list endpoints support offset-based pagination via query parameters. Use pagination to keep responses within token
limits and iterate through large collections.

**Query parameters:**

| Parameter | Type | Default | Max | Description                |
|-----------|------|---------|-----|----------------------------|
| `limit`   | int  | 100     | 500 | Number of items to return  |
| `offset`  | int  | 0       | —   | Number of items to skip    |

**Response metadata:** Every paginated response includes a `pagination` object:

```json
{
  "pagination": {
    "total": 42,
    "limit": 100,
    "offset": 0,
    "has_more": false
  }
}
```

**Paginating through results:**

```
# First page
GET /current/orgs/list/?limit=10&offset=0
# → pagination.has_more = true, pagination.total = 42

# Second page
GET /current/orgs/list/?limit=10&offset=10
# → pagination.has_more = true

# Continue until has_more = false
```

**Endpoints supporting pagination:**

| Endpoint                                             | Collection Key     |
|------------------------------------------------------|--------------------|
| `GET /current/orgs/all/`                             | `orgs`             |
| `GET /current/orgs/list/`                            | `orgs`             |
| `GET /current/orgs/list/external/`                   | `orgs`             |
| `GET /current/shares/all/`                           | `shares`           |
| `GET /current/workspace/{id}/members/list/`          | `users`            |
| `GET /current/org/{id}/billing/usage/members/list/`  | `billable_members` |
| `GET /current/workspace/{id}/list/shares/`           | `shares`           |
| `GET /current/user/me/list/shares/`                  | `shares`           |
| `GET /current/org/{id}/list/workspaces/`             | `workspaces`       |
| `GET /current/org/{id}/members/list/`                | `users`            |
| `GET /current/workspace/{id}/storage/search/`        | `files`            |
| `GET /current/share/{id}/storage/search/`            | `files`            |
| `GET /current/share/{id}/members/list/`              | `users`            |

---

## Core Capabilities

### 1. Workspaces — Organized File Storage

Workspaces are collaborative containers for files. Each workspace has its own storage, member list, AI chat, and
activity feed. Think of them as project folders with superpowers.

- **100 GB included storage** on the free agent plan
- **Files up to 1 GB** per upload
- **File versioning** — every edit creates a new version, old versions are recoverable
- **Folder hierarchy** — organize files however you want
- **Full-text and semantic search** — find files by name, content, or meaning
- **Member roles** — Owner, Admin, Editor, Viewer with granular permissions
- **Real-time sync** — changes appear instantly for all members via WebSockets

#### Intelligence: On or Off

Workspaces have an **intelligence** toggle that controls whether AI features are active. This is a critical decision:

**Intelligence OFF** — the workspace is pure file storage. You can still attach files directly to an AI chat
conversation (up to 10 files), but files are not persistently indexed. This is fine for simple storage and sharing where
you don't need to query your content.

**Intelligence ON** — the workspace becomes an AI-powered knowledge base. Every file uploaded is automatically ingested,
summarized, and indexed. This enables:

- **RAG (retrieval-augmented generation)** — scope AI chat to entire folders or the full workspace and ask questions
  across all your content. The AI retrieves relevant passages and answers with citations.
- **Semantic search** — find files by meaning, not just keywords. "Show me contracts with indemnity clauses" works even
  if those exact words don't appear in the filename.
- **Auto-summarization** — short and long summaries generated for every file, searchable and visible in the UI.
- **Metadata extraction** — AI pulls key metadata from documents automatically.

Intelligence is enabled by default when creating workspaces via the API for agent accounts. If you're just using Fast.io
for storage and sharing, you can disable it to conserve credits. If you want to query your content — enable it.

**Agent use case:** Create a workspace per project or client. Enable intelligence if you need to query the content
later. Upload reports, datasets, and deliverables. Invite the human stakeholders. Everything is organized, searchable,
and versioned.

### 2. Shares — Branded Data Rooms for Humans

Shares are purpose-built spaces for exchanging files with people outside your workspace. Three modes cover every
exchange pattern:

| Mode         | What It Does                  | Agent Use Case                                |
|--------------|-------------------------------|-----------------------------------------------|
| **Send**     | Recipients can download files | Deliver reports, exports, generated content   |
| **Receive**  | Recipients can upload files   | Collect documents, datasets, user submissions |
| **Exchange** | Both upload and download      | Collaborative workflows, review cycles        |

#### Share Features

- **Password protection** — require a password for link access
- **Expiration dates** — shares auto-expire after a set period
- **Download controls** — enable or disable file downloads
- **Access levels** — `'Only members of the Share or Workspace'`, `'Members of the Share, Workspace or Org'`, `'Anyone with a registered account'`, or `'Anyone with the link'`
- **Custom branding** — background images, gradient colors, accent colors, logos
- **Post-download messaging** — show custom messages and links after download
- **Up to 3 custom links** per share for context or calls-to-action
- **Guest chat** — let share recipients ask questions in real-time
- **AI-powered auto-titling** — shares automatically generate smart titles from their contents
- **Activity notifications** — get notified when files are sent or received
- **Comment controls** — configure who can see and post comments (owners, guests, or both)

#### Two Storage Modes

When creating a share, you choose a `storage_mode` that determines how the share's files are managed:

- **`room`** (independent storage, default) — the share has its own isolated storage. Files are added directly to the
  share and are independent of any workspace. This creates a self-contained data room — changes to workspace files don't
  affect the room, and vice versa. Perfect for final deliverables, compliance packages, archived reports, or any
  scenario where you want an immutable snapshot.

- **`shared_folder`** (workspace-backed) — the share is backed by a specific folder in a workspace. The share displays
  the live contents of that folder — any files added, updated, or removed in the workspace folder are immediately
  reflected in the share. No file duplication, so no extra storage cost. To create a shared folder, pass
  `storage_mode=shared_folder` and `folder_node_id={folder_opaque_id}` when creating the share. Note: expiration dates
  are not allowed on shared folder shares since the content is live.

Both modes look the same to share recipients — a branded data room with file preview, download controls, and all share
features. The difference is whether the content is a snapshot (room) or a live view (shared folder).

**Agent use case:** Generate a quarterly report, create a Send share with your client's branding, set a 30-day
expiration, and share the link. The client sees a professional, branded page with instant file preview — not a raw
download link.

### 3. QuickShare — Instant File Handoff

Need to toss a file to someone right now? QuickShare creates a share from a single file with zero configuration.
Automatic 30-day expiration. No setup, no decisions.

**Agent use case:** Debug log, sample output, or quick artifact? QuickShare it and send the link. Done.

### 4. Built-In AI — Ask Questions About Your Files

Fast.io's AI is a **read-only tool** — it can read and analyze file contents, but it cannot modify files, change
workspace settings, manage members, or read events. It answers questions about your documents, nothing more. For any
action beyond reading file content, your agent must use the API or MCP server directly.

Fast.io's AI lets agents query documents through two chat types, with or without persistent indexing. Both types
augment file knowledge with information from the web when relevant.

#### Chat Types

**`chat`** — Basic AI conversation. Does not use file context from the workspace index. Use this for general questions
or when you don't need to reference stored files.

**`chat_with_files`** — AI conversation grounded in your files. This is the type you use when you want the AI to read,
analyze, and cite your documents. Requires the workspace to have **intelligence enabled** if using folder/file scope
(RAG). Supports two mutually exclusive modes for providing file context:

1. **Folder/file scope** (RAG) — limits the search space for retrieval. The AI searches the indexed content of files
   within the specified scope, retrieves relevant passages, and answers with citations. Requires intelligence enabled
   and files in `ready` AI state.

2. **File attachments** — files are directly attached to the conversation. The AI reads the full content of the attached
   files. Does not require intelligence — any file with a ready preview can be attached. Max 10 files.

These two modes cannot be combined in a single chat — use scope OR attachments, not both.

#### Intelligence Setting — When to Enable It

The `intelligence` toggle on a workspace controls whether uploaded files are automatically ingested, summarized, and
indexed for RAG.

**Enable intelligence when:**
- You have many files and need to search across them to answer questions
- You want scoped RAG queries against folders or the entire workspace
- You need auto-summarization and metadata extraction
- You're building a persistent knowledge base

**Disable intelligence when:**
- You're using the workspace purely for storage and sharing
- You only need to analyze specific files (use file attachments instead)
- You want to conserve credits (ingestion costs 10 credits/page for documents, 5 credits/second for video)

Even with intelligence disabled, you can still use `chat_with_files` with **file attachments** — any file that has a
ready preview can be attached directly to a chat for one-off analysis.

#### AI State — File Readiness for RAG

Every file in an intelligent workspace has an `ai_state` field that tracks its ingestion progress:

| State         | Meaning                                           |
|---------------|---------------------------------------------------|
| `disabled`    | AI processing disabled for this file              |
| `pending`     | Queued for processing                             |
| `in_progress` | Currently being ingested and indexed              |
| `ready`       | Processing complete — file is available for RAG   |
| `failed`      | Processing failed                                 |

**Only files with `ai_state: ready` are included in folder/file scope searches.** If you upload files and immediately
create a scoped chat, recently uploaded files may not yet be indexed. Use the activity polling endpoint to wait for
`ai_state` changes before querying.

#### Folder Scope vs File Attachments

| Feature              | Folder/File Scope (RAG)                    | File Attachments                         |
|----------------------|--------------------------------------------|------------------------------------------|
| How it works         | Limits RAG search space                    | Files read directly by AI                |
| Requires intelligence| Yes                                        | No                                       |
| Requires `ai_state`  | Files must be `ready`                      | Files must have a ready preview          |
| Best for             | Many files, knowledge retrieval            | Specific files, direct analysis          |
| Max references       | 100 files or folders                       | 10 files                                 |
| Default behavior     | No scope = entire workspace                | N/A                                      |

**Folder scope parameters:**
- `folders_scope` — comma-separated `nodeId:depth` pairs (depth 1-10, max 100 refs). Limits RAG retrieval to files
  within those folders.
- `files_scope` — comma-separated `nodeId:versionId` pairs (max 100 refs). Limits RAG retrieval to specific files.
- If neither is specified, the scope defaults to **all files in the workspace**.

**File attachment parameter:**
- `files_attach` — comma-separated `nodeId:versionId` pairs (max 10 files). Files are read directly, not searched via
  RAG.

#### Notes as Knowledge Grounding

Notes are markdown documents created directly in workspace storage via the API
(`POST /current/workspace/{id}/storage/{folder}/createnote/`). In an intelligent workspace, notes are ingested and
indexed just like uploaded files. This makes notes a way to store long-term knowledge that becomes grounding material
for future AI queries.

**Agent use case:** Store project context, decision logs, or reference material as notes. When you later ask the AI
"What was the rationale for choosing vendor X?", the note containing that decision is retrieved and cited — even months
later.

Notes within a folder scope are included in RAG queries when intelligence is enabled.

#### How to Write Effective Questions

The way you phrase questions depends on whether you're using folder scope (RAG) or file attachments.

**With folder/file scope (RAG):**

Write questions that are likely to match content in your indexed files. The AI searches the scope for relevant passages,
retrieves them, and uses them as citations to answer your question. Think of it as a search query that returns context
for an answer.

- Good: "What are the payment terms in the vendor contracts?" — matches specific content in files
- Good: "Summarize the key findings from the Q3 analysis reports" — retrieves relevant sections
- Good: "What risks were identified in the security audit?" — finds specific content to cite
- Bad: "Tell me about these files" — too vague for retrieval, no specific content to match
- Bad: "What's in this workspace?" — the AI can't meaningfully search for "everything"

If no folder scope is specified, the search defaults to all files in the workspace. For large workspaces, narrowing the
scope to specific folders improves relevance and reduces token usage.

**With file attachments:**

You can be more direct and simplistic since the AI reads the full file content. No retrieval step — the AI has the
complete file in context.

- "Describe this image in detail"
- "Extract all dates and amounts from this invoice"
- "Convert this CSV data into a summary table"
- "What programming language is this code written in and what does it do?"

**Personality:** The `personality` parameter controls the tone and length of AI responses. Pass it when creating a chat
or sending a message:

- `concise` — short, direct answers with minimal explanation
- `detailed` — comprehensive answers with context and evidence (default)

This makes a significant difference in response quality for your use case. Agents that need to extract data or get quick
answers should use `concise` to avoid wasting tokens on lengthy explanations. Use `detailed` when you need thorough
analysis with supporting evidence.

You can also control verbosity in the question itself — for example, "In one sentence, summarize this report" or "List
only the file names, no explanations." Combining `concise` personality with direct questions produces the shortest
responses.

#### Waiting for AI Responses

After sending a message, the AI processes it asynchronously. You need to wait for the response to be ready.

**Message states:**

| State             | Meaning                              |
|-------------------|--------------------------------------|
| `ready`           | Queued for processing                |
| `in_progress`     | AI is generating the response        |
| `complete`        | Response finished                    |
| `errored`         | Processing failed                    |
| `post_processing` | Finalizing (citations, formatting)   |

**Option 1: SSE streaming (recommended for real-time display)**

`GET /current/workspace/{id}/ai/chat/{chat_id}/message/{message_id}/read/`

Returns a `text/event-stream` with response chunks as they're generated. The stream ends with a `done` event when the
response is complete. Response chunks include the AI's text, citations pointing to specific files/pages/snippets, and
any structured data (tables, analysis).

**Option 2: Activity polling (recommended for background processing)**

Don't poll the message endpoint in a loop. Instead, use the activity long-poll:

`GET /current/activity/poll/{workspace_id}?wait=95&lastactivity={timestamp}`

When `ai_chat:{chatId}` appears in the activity response, the chat has been updated — fetch the message details to get
the completed response. This is the most efficient approach when you don't need to stream the response in real-time.

**Option 3: Fetch completed response**

`GET /current/workspace/{id}/ai/chat/{chat_id}/message/{message_id}/details/`

Check the `state` field. If `complete`, the `response.text` contains the full answer and `response.citations` contains
the file references.

#### Linking Users to AI Chats

To send a user directly to an AI chat in the workspace UI, append a `chat` query parameter to the workspace storage
URL:

`https://{org.domain}.fast.io/workspace/{workspace.folder_name}/storage/root?chat={chat_opaque_id}`

This opens the workspace with the specified chat visible in the AI panel.

#### Supported Content Types

- Documents (PDF, Word, text, markdown)
- Spreadsheets (Excel, CSV)
- Code files (all common languages)
- Images (all common formats)
- Video (all common formats)
- Audio (all common formats)

#### AI Share — Export to External AI Tools

Generate temporary download URLs for your files, formatted as markdown, for pasting into external AI assistants like
ChatGPT or Claude. Up to 25 files, 50MB per file, 100MB total. Links expire after 5 minutes. This is separate from the
built-in AI chat — use it when you want to analyze files with a different model or tool.

**Agent use case:** A user asks "What were Q3 margins?" You have 50 financial documents in an intelligent workspace.
Instead of downloading and parsing all 50, create a `chat_with_files` scoped to the finance folder and ask. The AI
searches the indexed content, retrieves relevant passages, and answers with citations. Pass the cited answer — with
source references — back to the user.

### 5. File Preview — No Download Required

Files uploaded to Fast.io get automatic preview generation. When humans open a share or workspace, they see the content
immediately — no "download and open in another app" friction.

**Supported preview formats:**

- **Images** — full-resolution with auto-rotation and zoom
- **Video** — HLS adaptive streaming (50-60% faster load than raw video)
- **Audio** — interactive waveform visualization
- **PDF** — page navigation, zoom, text selection
- **Spreadsheets** — grid navigation with multi-sheet support
- **Code & text** — syntax highlighting, markdown rendering

**Agent use case:** Your generated PDF report doesn't just appear as a download link. The human sees it rendered inline,
can flip through pages, zoom in, and comment on specific sections — all without leaving the browser.

### 6. Notes — Markdown Documents as Knowledge

Notes are a storage node type (alongside files and folders) that store markdown content directly on the server. They
live in the same folder hierarchy as files, are versioned like any other node, and appear in storage listings with
`type: "note"`.

#### Creating and Updating Notes

**Create:** `POST /current/workspace/{id}/storage/{parent_id}/createnote/`

- `name` (required) — filename, must end in `.md`, max 100 characters (e.g., `"project-context.md"`)
- `content` (required) — markdown text, max 100 KB

**Update:** `POST /current/workspace/{id}/storage/{node_id}/updatenote/`

- `name` (optional) — rename the note (must end in `.md`)
- `content` (optional) — replace the markdown content (max 100 KB)
- At least one of `name` or `content` must be provided

Notes can also be moved, copied, deleted, and restored using the same storage endpoints as files and folders.

#### Notes as Long-Term Knowledge Grounding

In an intelligent workspace, notes are automatically ingested and indexed just like uploaded files. This makes notes a
powerful way to **bank knowledge over time** — any facts, context, or decisions stored in notes become grounding
material for future AI queries.

When an AI chat uses folder scope (or defaults to the entire workspace), notes within that scope are searched alongside
files. The AI retrieves relevant passages from notes and cites them in its answers.

**Use cases:**
- Store project context, decisions, and rationale as notes. Months later, ask "Why did we choose vendor X?" and the AI
  retrieves the note with that decision.
- After researching a topic, save key findings in a note. Future AI chats automatically use those findings as grounding.
- Create reference documents (style guides, naming conventions, process docs) that inform all future AI queries in the
  workspace.

#### Linking Users to Notes

**Open a note in the workspace UI** — append `?note={opaque_id}` to the workspace storage URL:

`https://{org.domain}.fast.io/workspace/{folder_name}/storage/root?note={note_opaque_id}`

**Link directly to the note preview** — use the standard file preview URL:

`https://{org.domain}.fast.io/workspace/{folder_name}/preview/{note_opaque_id}`

The preview link is more effective if you want the user to focus on reading just that note, while the `?note=` link
opens the note within the full workspace context.

### 7. Comments & Annotations

Humans can leave feedback directly on files, anchored to specific content:

- **Image comments** — anchored to regions of the image
- **Video comments** — anchored to timestamps with frame-stepping and spatial region selection
- **Audio comments** — anchored to timestamps or time ranges
- **PDF comments** — anchored to specific pages with optional text selection
- **Threaded replies** — single-level threads under each comment (replies to replies are auto-flattened)
- **Emoji reactions** — one reaction per user per comment, new replaces previous

**Linking users to comments:** Link users to the file preview URL. The comments sidebar opens automatically in workspace
previews, and in share previews when comments are enabled on the share.

Base preview URL:

`https://{org.domain}.fast.io/workspace/{folder_name}/preview/{file_opaque_id}`

For shares: `https://go.fast.io/shared/{custom_name}/{title-slug}/preview/{file_opaque_id}`

**Deep linking to a specific comment:** Append `?comment={comment_id}` to the preview URL. The UI scrolls to and
highlights the comment automatically:

`https://{org.domain}.fast.io/workspace/{folder_name}/preview/{file_opaque_id}?comment={comment_id}`

**Deep linking to media/document positions:** For comments anchored to specific locations, combine with position
parameters:

- `?t={seconds}` — seeks to a timestamp in audio/video (e.g., `?comment={id}&t=45.5`)
- `?p={pageNum}` — navigates to a page in PDFs (e.g., `?comment={id}&p=3`)

**Agent use case:** You generate a design mockup. The human comments "Change the header color" on a specific region of
the image. You read the comment, see exactly what region they're referring to, and regenerate.

### 8. File Uploads — Getting Files Into Fast.io

Agents upload files through a session-based API. There are two paths depending on file size:

#### Small Files (Under 4 MB)

For files under 4 MB, upload in a single request. Send the file as `multipart/form-data` with the `chunk` field
containing the file data, plus `org` (your org domain), `name`, `size`, and `action=create`.

To have the file automatically added to a workspace or share, include `instance_id` (the workspace or share ID) and
optionally `folder_id` (the target folder's OpaqueId, or omit for root). The response includes `new_file_id` — the
permanent OpaqueId of the file in storage. No further steps needed.

```
POST /current/upload/
Content-Type: multipart/form-data

Fields: org, name, size, action=create, instance_id, folder_id, chunk (file)
→ Response: { "result": true, "id": "session-id", "new_file_id": "2abc..." }
```

#### Large Files (4 MB and Above)

Large files use chunked uploads. The flow has five steps:

1. **Create a session** — `POST /current/upload/` with `org`, `name`, `size`, `action=create`, `instance_id`, and
   optionally `folder_id`. Returns a session `id`.

2. **Upload chunks** — Split the file into **5 MB chunks** (last chunk may be smaller). For each chunk, send
   `POST /current/upload/{session_id}/chunk/` as `multipart/form-data` with the `chunk` field (binary data), `order`
   (1-based — first chunk is `order=1`), and `size`. You can upload up to **3 chunks in parallel** per session.

3. **Trigger assembly** — Once all chunks are uploaded, call `POST /current/upload/{session_id}/complete/`. The server
   verifies chunks and combines them into a single file.

4. **Poll for completion** — The upload progresses through states asynchronously. Poll the session details with the
   built-in long-poll:

   `GET /current/upload/{session_id}/details/?wait=60`

   The server holds the connection for up to 60 seconds and returns immediately when the status changes:

   | Status | Meaning | What to Do |
   |--------|---------|------------|
   | `ready` | Awaiting chunks | Upload chunks |
   | `uploading` | Receiving chunks | Continue uploading |
   | `assembling` | Combining chunks | Keep polling |
   | `complete` | Assembled, awaiting storage | Keep polling |
   | `storing` | Being added to storage | Keep polling |
   | **`stored`** | **Done** — file is in storage | Read `new_file_id`, clean up |
   | `assembly_failed` | Assembly error (terminal) | Check `status_message` |
   | `store_failed` | Storage error (retryable) | Keep polling, server retries |

   Stop polling when status is `stored`, `assembly_failed`, or `store_failed`.

5. **Clean up** — Delete the session after completion: `DELETE /current/upload/{session_id}/`.

#### Optional Integrity Hashing

Include `hash` (SHA-256 hex digest) and `hash_algo=sha256` on each chunk for server-side integrity verification. You can
also provide a full-file hash in the session creation request instead.

#### Resuming Interrupted Uploads

If a connection drops mid-upload, the session persists on the server. To resume:

1. Fetch the session: `GET /current/upload/{session_id}/details/`
2. Read the `chunks` map — keys are chunk numbers already uploaded, values are byte sizes
3. Upload only the missing chunks
4. Trigger assembly and continue as normal

#### Manual Storage Placement

If you omit `instance_id` when creating the session, the file is uploaded but not placed in any workspace or share. You
can add it to storage manually afterward:

```
POST /current/workspace/{id}/storage/{folder}/addfile/
Body: from={"type":"upload","upload":{"id":"{session_id}"}}
```

This is useful when you need to upload first and decide where to place the file later.

#### MCP Binary Upload (Blob Sidecar)

MCP-connected agents can avoid base64 encoding overhead (~33% size savings) by staging raw binary data through a
sidecar endpoint:

1. `POST /blob` with your `Mcp-Session-Id` header and the raw bytes as the request body
2. The server returns a `blob_id`
3. Pass `blob_ref` (the `blob_id`) instead of the base64-encoded `chunk` field when calling `upload-chunk`

This is MCP-specific — the REST API continues to use `multipart/form-data` as described above.

**Agent use case:** You're generating a 200 MB report. Create an upload session targeting the client's workspace, split
the file into 5 MB chunks, upload 3 at a time, trigger assembly, and poll until `stored`. The file appears in the
workspace with previews generated automatically. Use the activity polling endpoint (section 13) to know when AI indexing
completes if intelligence is enabled.

### 9. URL Import — Pull Files From Anywhere

When you need to add a file from the web, use `POST /current/web_upload/` with `source_url` instead of downloading it
locally and re-uploading. This is faster because the file transfers server-to-server — your agent never touches the
bytes.

- Supports any HTTP/HTTPS URL
- Supports OAuth-protected sources: **Google Drive, OneDrive, Dropbox**
- Files go through the same processing pipeline (preview generation, AI indexing if intelligence is enabled, virus
  scanning)

**Check progress after submitting.** Web uploads are processed asynchronously by Fast.io's server-side fetch agent,
which may be blocked or rate-limited by the source. The import can fail silently if the source rejects the request, times
out, or returns an error. Monitor the upload status to confirm the file was actually retrieved and stored before
reporting success to the user.

**Agent use case:** A user says "Add this Google Doc to the project." You call `POST /current/web_upload/` with the URL.
Fast.io downloads it server-side, generates previews, indexes it for AI, and it appears in the workspace. No local I/O —
and no bandwidth consumed by your agent.

### 10. Real-Time Collaboration

Fast.io uses WebSockets for instant updates across all connected clients:

- **Live presence** — see who's currently viewing a workspace or share
- **Cursor tracking** — see where other users are navigating
- **Follow mode** — click a user to mirror their exact navigation
- **Instant file sync** — uploads, edits, and deletions appear immediately for all viewers

### 11. Ownership Transfer — Hand Off to Humans

The killer feature for agents: build something, then give it to a human. This is also the upgrade path when the agent
plan's 5,000 monthly credits are exhausted or plan limits are reached — transferring to a human is the only way to
upgrade beyond the agent plan.

**Important:** Only `agent` account types can create transfer tokens. This is why agents should **always** create
accounts with `agent=true`. If you created a human account, you cannot use the transfer/claim API — you must direct the
user to upgrade the org themselves.

1. Agent creates an org, sets up workspaces, uploads files, configures shares
2. Agent generates a transfer token via `POST /current/org/{org_id}/transfer/token/create/` (64-char string, valid 72
   hours, max 5 active tokens per org)
3. Agent sends the claim URL to the human: `https://go.fast.io/claim?token={token}`
4. Human clicks the link, logs in (or creates an account), and claims the org

**What happens:**

- Human becomes the owner of the org and all workspaces
- Agent is demoted to admin (can still manage files and shares)
- Human gets a fresh 14-day free trial starting from the transfer date
- Human can upgrade to Pro or Business at any time for unlimited credits and expanded limits

**Agent use case:** A user says "Set up a project workspace for my team." You create the org, build out workspace
structure, upload templates, configure shares for client deliverables, invite team members — then transfer ownership.
The human walks into a fully configured platform. You stay on as admin to keep managing things.

**Credit exhaustion use case:** Your agent hits the 5,000 credit limit mid-month. Create a transfer token, send the
claim URL to the human user, and let them know they can upgrade to Pro or Business for unlimited credits. After
claiming, the human upgrades and the org is no longer credit-limited.

### 12. Events — Real-Time Audit Trail

Events give agents a real-time audit trail of everything that happens across an organization. Instead of scanning entire
workspaces to detect what changed, query the events feed to see exactly which files were uploaded, modified, renamed, or
deleted — and by whom, and when. This makes it practical to build workflows that react to changes: processing a document
the moment it arrives, flagging unexpected permission changes, or generating a daily summary of activity for a human.

The activity log is also the most efficient way for an agent to stay in sync with a workspace over time. Rather than
periodically listing every file and comparing against a previous snapshot, check events since your last poll to get a
precise diff. This is especially valuable in large workspaces where full directory listings are expensive.

#### What Events Cover

- **File operations** — uploads, downloads, moves, renames, deletes, version changes
- **Membership changes** — new members added, roles changed, members removed
- **Share activity** — share created, accessed, files downloaded by recipients
- **Settings updates** — workspace or org configuration changes
- **Billing events** — credit usage, plan changes
- **AI operations** — ingestion started, indexing complete, chat activity

#### Querying Events

Search and filter events with `GET /current/events/search/`:

- **Scope by profile** — filter by `workspace_id`, `share_id`, `org_id`, or `user_id`
- **Filter by type** — narrow to specific event names, categories, or subcategories
- **Date range** — use `created-min` and `created-max` for time-bounded queries
- **Pagination** — offset-based with `limit` (1-250) and `offset`

Get full details for a single event with `GET /current/event/{event_id}/details/`, or mark it as read with
`GET /current/event/{event_id}/ack/`.

#### AI-Powered Summaries

Request a natural language recap of recent activity with `GET /current/events/search/summarize/`. Returns event counts,
category breakdowns, and a narrative summary. Focus the summary on a specific workspace or share, or summarize across
the entire org.

**Agent use case — stay in sync:** You manage a workspace with 10,000 files. Instead of listing the entire directory
tree to find what changed, query events since your last check. You get a precise list: "3 files uploaded, 1 renamed,
2 new members added" — with timestamps, actors, and affected resources.

**Agent use case — react to changes:** A client uploads tax documents to a Receive share. The events feed shows the
upload immediately. Your agent detects it, processes the documents, and notifies the accountant — no polling the file
list required.

**Agent use case — report to humans:** A human asks "What happened on the project this week?" You call the AI summary
endpoint scoped to their workspace and return a clean narrative report — no log parsing required.

### 13. Activity Polling — Wait for Changes Efficiently

After triggering an async operation (uploading a file, enabling intelligence, creating a share), don't loop on the
resource endpoint to check if it's done. Instead, use the activity long-poll endpoint:

`GET /current/activity/poll/{entity_id}?wait=95&lastactivity={timestamp}`

The `{entity_id}` is the profile ID of the resource you're watching — a workspace ID, share ID, or org ID. For
**upload sessions**, use the **user ID** (since uploads are user-scoped, not workspace-scoped until the file is added
to storage).

The server holds the connection open for up to 95 seconds and returns **immediately** when something changes on that
entity — file uploads complete, previews finish generating, AI indexing completes, comments are added, etc.

The response includes activity keys that tell you *what* changed (e.g., `storage:{fileId}` for file changes,
`preview:{fileId}` for preview readiness, `ai_chat:{chatId}` for chat updates, `upload:{uploadId}` for upload
completion). Pass the returned `lastactivity` timestamp into your next poll to receive only newer changes.

This gives you near-instant reactivity with a single open connection per entity, instead of hammering individual
endpoints.

**WebSocket upgrade:** For true real-time delivery (~300ms latency vs ~1s for polling), connect via WebSocket at
`wss://{host}/api/websocket/?token={auth_token}`. The server pushes activity arrays as they happen:

```json
{"response": "activity", "activity": ["storage:2abc...", "preview:2abc..."]}
```

You then fetch only the resources that changed. If the WebSocket connection fails, fall back to long-polling — the data
is identical, just slightly higher latency.

**Agent use case:** You upload a 500-page PDF and need to know when AI indexing is complete before querying it. Instead
of polling the file details endpoint every few seconds, open a single long-poll on the workspace. When
`ai_state:{fileId}` appears in the activity response, the file is indexed and ready for AI chat.

---

## Agent Plan — What's Included (Free)

The agent plan is a free tier designed to get agents started. It's intentionally lightweight — enough to build and
demonstrate value, with room to grow when the org transfers to a human on a paid plan.

| Resource                  | Included                                            |
|---------------------------|-----------------------------------------------------|
| **Price**                 | $0 — no credit card, no trial period, no expiration |
| **Storage**               | 100 GB                                              |
| **Max file size**         | 1 GB                                                |
| **Monthly credits**       | 5,000 (resets every 30 days)                        |
| **Workspaces**            | 5                                                   |
| **Shares**                | 50                                                  |
| **Members per workspace** | 5                                                   |
| **Share invitations**     | 10 per share                                        |
| **Account auto-deletion** | Never                                               |

### What Credits Cover

All platform activity consumes credits from the monthly 5,000 allowance:

| Resource                | Cost                    |
|-------------------------|-------------------------|
| Storage                 | 100 credits/GB          |
| Bandwidth               | 212 credits/GB          |
| AI chat tokens          | 1 credit per 100 tokens |
| Document pages ingested | 10 credits/page         |
| Video ingested          | 5 credits/second        |
| Audio ingested          | 0.5 credits/second      |
| Images ingested         | 5 credits/image         |
| File conversions        | 25 credits/conversion   |

When credits run out, the org enters a reduced-capability state — file storage and access continue to work, but
credit-consuming operations (AI chat, file ingestion, bandwidth-heavy downloads) are limited until the 30-day reset.
The org is never deleted.

**When you hit the credit limit:** The recommended path is to transfer the org to a human user who can upgrade to a
paid plan with unlimited credits. See "Ownership Transfer" above. If the agent account is using a human account type
(not recommended), direct the user to upgrade at `https://go.fast.io/onboarding` or via the billing API.

### After Transfer — Human Plan Options

Once an agent transfers an org to a human, they get a 14-day free trial and can upgrade:

| Feature         | Agent (Free) | Pro       | Business  |
|-----------------|--------------|-----------|-----------|
| Monthly credits | 5,000        | Unlimited | Unlimited |
| Storage         | 100 GB       | 1 TB      | 5 TB      |
| Max file size   | 1 GB         | 25 GB     | 50 GB     |
| Workspaces      | 5            | 10        | 1,000     |
| Shares          | 50           | 1,000     | 50,000    |

The transfer flow is the primary way agents deliver value — and the only way to upgrade beyond the agent plan. Set
everything up on the free agent plan, transfer ownership when the work is complete or when credits are exhausted, and
the human upgrades when they're ready. The agent retains admin access to keep managing things.

---

## Common Workflows

### Deliver a Report to a Client

1. Upload report PDF to workspace
2. Create a Send share with password protection and 30-day expiration
3. Share the link with the client
4. Client sees a branded page, previews the PDF inline, downloads if needed
5. You get a notification when they access it

### Collect Documents From a User

1. Create a Receive share ("Upload your tax documents here")
2. Share the link
3. User uploads files through a clean, branded interface
4. Files appear in your workspace, auto-indexed by AI (if intelligence is on)
5. Ask the AI: "Are all required forms present?"

### Build a Knowledge Base

1. Create a workspace **with intelligence enabled**
2. Upload all reference documents
3. AI auto-indexes and summarizes everything on upload
4. Use AI chat scoped to folders or the full workspace to query across all documents
5. Semantic search finds files by meaning, not just filename
6. Answers include citations to specific pages and files

### Set Up a Project for a Human

1. Create org + workspace + folder structure
2. Upload templates and reference docs
3. Create shares for client deliverables (Send) and intake (Receive)
4. Configure branding, passwords, expiration
5. Transfer ownership to the human
6. Human gets a fully configured platform, agent keeps admin access

### Collaborative Review Cycle (Exchange Share)

1. Create an Exchange share ("Review these designs and upload your feedback")
2. Upload draft files for the recipient
3. Share the link — recipient can both download your files and upload theirs
4. Comments and annotations on files enable inline feedback
5. AI summarizes what changed between rounds (if intelligence is on)

### One-Off Document Analysis (No Intelligence Needed)

1. Create a workspace (intelligence off is fine)
2. Upload the files you want to analyze
3. Create an AI chat and attach the specific files directly (up to 10 files)
4. Ask questions — AI reads the attachments and responds with citations
5. No persistent indexing, no credit cost for ingestion

### Choose Between Room and Shared Folder

**Use a Room (independent storage) when:**

- Delivering final, immutable outputs (reports, compliance packages)
- You want a snapshot that won't change if workspace files are updated
- Files are "done" and shouldn't reflect future edits

**Use a Shared Folder (workspace-backed) when:**

- Files are actively being updated (live data feeds, ongoing projects)
- You want zero storage duplication
- Recipients should always see the latest version

### Manage Credit Budget

1. Check current usage: `GET /current/org/{org_id}/billing/usage/limits/credits/`
2. Storage costs 100 credits/GB — a 10 GB workspace costs 1,000 credits/month
3. Document ingestion costs 10 credits/page — a 50-page PDF costs 500 credits
4. Disable intelligence on storage-only workspaces to avoid ingestion costs
5. Use attach-only AI chat (no intelligence needed) for one-off analysis to save credits
6. When credits run low, transfer the org to a human who can upgrade to unlimited credits

---

## MCP Prompts — Guided Workflows

MCP-connected agents can use **guided prompts** to get step-by-step instructions for common operations. These are
especially useful for agents unfamiliar with Fast.io workflows, complex multi-step operations, or operations with
non-obvious parameter requirements.

Retrieve the full list with `prompts/list` and get detailed guidance for a specific prompt with `prompts/get`.

| Prompt                 | Name                          | When to Use                                                                                          |
|------------------------|-------------------------------|------------------------------------------------------------------------------------------------------|
| `get-started`          | Getting Started Guide         | First-time onboarding: create account, org, and workspace. Covers autonomous agents, API key users, and agents invited to existing orgs. |
| `create-share`         | Share Creation Guide          | Creating shares (Send/Receive/Exchange). Explains share types, access control, passwords, expiration, custom URLs, and feature toggles. |
| `ask-ai`               | AI Chat Guide                 | Querying files with AI. Covers RAG-indexed vs file attachment modes, intelligence state checks, scoping, polling, and response structure. |
| `upload-file`          | File Upload Guide             | Uploading files. Helps choose between single-step text upload and chunked binary upload, with parameter and encoding details. |
| `transfer-to-human`    | Ownership Transfer Guide      | Transferring org ownership to a human. Explains the claim URL workflow, token expiration, and pre-transfer checklist. |
| `discover-content`     | Content Discovery Guide       | Finding all accessible orgs and workspaces. Explains the critical distinction between internal and external orgs and the full discovery flow. |
| `invite-collaborator`  | Collaboration Invitation Guide| Inviting people to orgs, workspaces, or shares. Covers the three invitation levels, permissions, and message requirements. |
| `setup-branding`       | Branding Setup Guide          | Customizing branding across orgs, workspaces, and shares. Explains the asset hierarchy, asset types, upload methods, and best practices. |

**When to use prompts instead of this guide:**

- **Starting a new workflow** — prompts provide concise, actionable steps tailored to the specific operation
- **Choosing between approaches** — prompts explain trade-offs (e.g., which share type, which upload method)
- **Parameter-heavy operations** — prompts list exact parameters, required values, and common pitfalls
- **First-time operations** — prompts walk through prerequisites and setup in order

Prompts complement the reference material in this guide and in `skill.md`. Use this guide for concepts and
capabilities, `skill.md` for tool-level details, and prompts for guided walkthroughs of specific workflows.

---

## URL Structure & Link Construction

Fast.io uses subdomain-based routing. Organization domains become subdomains, and every resource (workspace, folder,
file, share) has a URL-safe identifier from API responses that you use to build links.

### How Org Domains Become Subdomains

When you create an organization, you choose a `domain` (2-80 characters, lowercase alphanumeric and hyphens). This
becomes the subdomain for all org URLs:

Organization domain: `"acme"` → All org URLs live at: `https://acme.fast.io/...`

The base domain `go.fast.io` is used for routes that don't require org context (public shares, auth, claim).

### Building URLs From API Responses

Every URL parameter comes from a field in the API response. You never need to generate or guess identifiers — use the
values the API gives you.

| URL Parameter | API Response Field            | Format                                | Example                               |
|---------------|-------------------------------|---------------------------------------|---------------------------------------|
| Subdomain     | `organization.domain`         | User-chosen slug                      | `acme`                                |
| Workspace     | `workspace.folder_name`       | URL-safe slug                         | `q4-planning`                         |
| Folder        | `folder.id` (storage node ID) | Opaque ID, or `root` / `trash`        | `2rii2hzajpc2s3kce3itd2z5esygv`       |
| File          | `file.id` (storage node ID)   | Opaque ID                             | `2xzvfaq3slqwa54qtezi66rrcly6w`       |
| Share         | `share.custom_name`           | Server-generated identifier           | `abc123xyz`                           |
| QuickShare    | `quickshare.id`               | Server-generated identifier           | `qs-abc123xyz`                        |
| Claim token   | Transfer API response         | 64-character token                    | `abcdef1234...`                       |

### Deep Links Into Workspaces

These URLs require the user to be logged in and a member of the org. Use the org's `domain` as the subdomain and the
workspace's `folder_name` as the workspace identifier.

| Link Type          | URL Pattern                                                                  |
|--------------------|-----------------------------------------------------------------------------|
| Workspace root     | `https://{org.domain}.fast.io/workspace/{workspace.folder_name}/storage/root` |
| Specific folder    | `https://{org.domain}.fast.io/workspace/{workspace.folder_name}/storage/{folder.id}` |
| File preview       | `https://{org.domain}.fast.io/workspace/{workspace.folder_name}/preview/{file.id}` |
| AI chat            | `https://{org.domain}.fast.io/workspace/{workspace.folder_name}/storage/root?chat={chat_id}` |
| Note (in workspace)| `https://{org.domain}.fast.io/workspace/{workspace.folder_name}/storage/root?note={note_id}` |
| Note (preview)     | `https://{org.domain}.fast.io/workspace/{workspace.folder_name}/preview/{note_id}` |
| Browse workspaces  | `https://{org.domain}.fast.io/browse-workspaces`                            |

**Examples:**
- `https://acme.fast.io/workspace/q4-planning/storage/root`
- `https://acme.fast.io/workspace/q4-planning/storage/2rii2hzajpc2s3kce3itd2z5esygv`
- `https://acme.fast.io/workspace/q4-planning/preview/2xzvfaq3slqwa54qtezi66rrcly6w`

### Shareable Links (No Auth Required)

These are the URLs you send to humans. Access depends on share settings, not authentication.

| Link Type              | URL Pattern                                                                        |
|------------------------|-----------------------------------------------------------------------------------|
| Public share           | `https://go.fast.io/shared/{share.custom_name}/{title-slug}`                      |
| Org-branded share      | `https://{org.domain}.fast.io/shared/{share.custom_name}/{title-slug}`            |
| File within a share    | `https://go.fast.io/shared/{share.custom_name}/{title-slug}/preview/{file.id}`    |
| QuickShare             | `https://go.fast.io/quickshare/{quickshare.id}`                                   |
| Claim (transfer)       | `https://go.fast.io/claim?token={transfer_token}`                                 |

The `{title-slug}` is the share title converted to a URL slug (lowercase, spaces to hyphens, special chars removed).
It's optional — routing works with just the `custom_name` — but improves link readability.

**Examples:**
- `https://go.fast.io/shared/abc123xyz/q4-financial-report`
- `https://acme.fast.io/shared/abc123xyz/q4-financial-report`
- `https://go.fast.io/shared/abc123xyz/q4-financial-report/preview/2xzvfaq3slqwa54qtezi66rrcly6w`
- `https://go.fast.io/quickshare/qs-abc123xyz`
- `https://go.fast.io/claim?token=abcdef1234567890abcdef1234567890abcdef1234567890abcdef1234567890`

### Share Management Links (For Owners/Admins)

| Link Type                    | URL Pattern                                                                          |
|------------------------------|--------------------------------------------------------------------------------------|
| Edit share (from workspace)  | `https://{org.domain}.fast.io/workspace/{workspace.folder_name}/share/{share.custom_name}` |
| Edit share (direct)          | `https://{org.domain}.fast.io/share/{share.custom_name}`                             |

### Settings & Account Links

| Link Type   | URL Pattern                                                                          |
|-------------|--------------------------------------------------------------------------------------|
| Org settings| `https://{org.domain}.fast.io/settings`                                              |
| Billing     | `https://{org.domain}.fast.io/settings/billing`                                      |
| Onboarding  | `https://go.fast.io/onboarding` or `https://go.fast.io/onboarding?orgId={org.id}&orgDomain={org.domain}` |

### Typical Agent Flow: Create and Link

1. **Create org** → API returns `org.domain` (e.g., `"acme"`)
2. **Create workspace** → API returns `workspace.folder_name` (e.g., `"client-docs"`)
3. **Upload files to folder** → API returns `file.id` for each file
4. **Create share from folder** → API returns `share.custom_name`
5. **Build links for the human:**
   - Workspace link: `https://acme.fast.io/workspace/client-docs/storage/root`
   - Share link: `https://go.fast.io/shared/{custom_name}/client-docs`
   - File link: `https://go.fast.io/shared/{custom_name}/client-docs/preview/{file.id}`
6. **Transfer ownership** → API returns token
   - Claim link: `https://go.fast.io/claim?token={token}`
