---
name: FastAPI
description: Build fast, production-ready Python APIs with type hints, validation, and async support.
metadata: {"clawdbot":{"emoji":"⚡","requires":{"bins":["python3"]},"os":["linux","darwin","win32"]}}
---

# FastAPI Patterns

## Async Traps
- Mixing sync database drivers (psycopg2, PyMySQL) in async endpoints blocks the event loop — use async drivers (asyncpg, aiomysql) or run sync code in `run_in_executor`
- `time.sleep()` in async endpoints blocks everything — use `await asyncio.sleep()` instead
- CPU-bound work in async endpoints starves other requests — offload to `ProcessPoolExecutor` or background workers
- Async endpoints calling sync functions that do I/O still block — the entire call chain must be async

## Pydantic Validation
- Default values in models become shared mutable state: `items: list = []` shares the same list across requests — use `Field(default_factory=list)`
- `Optional[str]` doesn't make a field optional in the request — add `= None` or use `Field(default=None)`
- Pydantic v2 uses `model_validate()` not `parse_obj()`, and `model_dump()` not `.dict()` — v1 methods are deprecated
- Use `Annotated[str, Field(min_length=1)]` for reusable validated types instead of repeating constraints

## Dependency Injection
- Dependencies run on every request by default — use `lru_cache` on expensive dependencies or cache in app.state for singletons
- `Depends()` without an argument reuses the type hint as the dependency — clean but can confuse readers
- Nested dependencies form a DAG — if A depends on B and C, and both B and C depend on D, D runs once (cached per-request)
- `yield` dependencies for cleanup (DB sessions, file handles) — code after yield runs even if the endpoint raises

## Lifespan and Startup
- `@app.on_event("startup")` is deprecated — use `lifespan` async context manager
- Store shared resources (DB pool, HTTP client) in `app.state` during lifespan, not as global variables
- Lifespan runs once per worker process — with 4 Uvicorn workers you get 4 DB pools

```python
from contextlib import asynccontextmanager
@asynccontextmanager
async def lifespan(app):
    app.state.db = await create_pool()
    yield
    await app.state.db.close()
app = FastAPI(lifespan=lifespan)
```

## Request/Response
- Return `dict` from endpoints, not Pydantic models directly — FastAPI handles serialization and it's faster
- Use `status_code=201` on POST endpoints returning created resources — 200 is the default but semantically wrong
- `Response` with `media_type="text/plain"` for non-JSON responses — returning a string still gets JSON-encoded otherwise
- Set `response_model_exclude_unset=True` to omit None fields from response — cleaner API output

## Error Handling
- `raise HTTPException(status_code=404)` — don't return Response objects for errors, it bypasses middleware
- Custom exception handlers with `@app.exception_handler(CustomError)` — but remember they don't catch HTTPException
- Use `detail=` for user-facing messages, log the actual error separately — don't leak stack traces

## Background Tasks
- `BackgroundTasks` runs after the response is sent but still in the same process — not suitable for long-running jobs
- Tasks execute sequentially in order added — don't assume parallelism
- If a background task fails, the client never knows — add your own error handling and alerting

## Security
- `OAuth2PasswordBearer` is for documentation only — it doesn't validate tokens, you must implement that in the dependency
- CORS middleware must come after exception handlers in middleware order — or errors won't have CORS headers
- `Depends(get_current_user)` in path operation, not in router — dependencies on routers affect all routes including health checks

## Testing
- `TestClient` runs sync even for async endpoints — use `httpx.AsyncClient` with `ASGITransport` for true async testing
- Override dependencies with `app.dependency_overrides[get_db] = mock_db` — cleaner than monkeypatching
- `TestClient` context manager ensures lifespan runs — without `with TestClient(app) as client:` startup/shutdown hooks don't fire
