//! Sitemap and page structure analysis tool

use crate::{
    error::{BrowserError, Result},
    tools::{Tool, ToolContext, ToolResult},
};
use schemars::JsonSchema;
use serde::{Deserialize, Serialize};

/// Parameters for sitemap analysis
#[derive(Debug, Clone, Serialize, Deserialize, JsonSchema)]
pub struct SitemapParams {
    /// Base URL of the site to analyze
    pub url: String,

    /// Also analyze page structure (headings, sections, nav)
    #[serde(default)]
    pub analyze_structure: bool,

    /// Maximum number of pages to analyze for structure (default: 5)
    #[serde(default = "default_max_pages")]
    pub max_pages: usize,

    /// Maximum number of sitemaps to parse (default: 10, useful for sites with many sitemaps)
    #[serde(default = "default_max_sitemaps")]
    pub max_sitemaps: usize,
}

fn default_max_pages() -> usize {
    5
}

fn default_max_sitemaps() -> usize {
    10
}

/// Result of sitemap analysis
#[derive(Debug, Clone, Serialize, Deserialize, JsonSchema)]
pub struct SitemapResult {
    /// Base URL that was analyzed
    pub base_url: String,

    /// Contents of robots.txt if found
    #[serde(skip_serializing_if = "Option::is_none")]
    pub robots_txt: Option<String>,

    /// List of sitemap URLs discovered
    pub sitemaps: Vec<String>,

    /// List of page URLs found in sitemaps
    pub pages: Vec<String>,

    /// Page structure analysis results (if analyze_structure was true)
    #[serde(default, skip_serializing_if = "Vec::is_empty")]
    pub page_structures: Vec<PageStructure>,
}

/// Structure analysis of a single page
#[derive(Debug, Clone, Serialize, Deserialize, JsonSchema)]
pub struct PageStructure {
    /// URL of the page
    pub url: String,

    /// Page title
    pub title: String,

    /// Heading hierarchy
    pub headings: Vec<Heading>,

    /// Navigation links found in nav/header elements
    pub nav_links: Vec<NavLink>,

    /// Semantic sections (main, article, section, aside, footer)
    pub sections: Vec<Section>,

    /// Main content area information
    #[serde(skip_serializing_if = "Option::is_none")]
    pub main_content: Option<MainContent>,

    /// Meta information
    #[serde(default)]
    pub meta: Meta,
}

/// A heading element
#[derive(Debug, Clone, Serialize, Deserialize, JsonSchema)]
pub struct Heading {
    /// Heading level (1-6)
    pub level: u8,

    /// Heading text content
    pub text: String,
}

/// A navigation link
#[derive(Debug, Clone, Serialize, Deserialize, JsonSchema)]
pub struct NavLink {
    /// Link text
    pub text: String,

    /// Link href
    pub href: String,
}

/// A semantic section element
#[derive(Debug, Clone, Serialize, Deserialize, JsonSchema)]
pub struct Section {
    /// HTML tag name
    pub tag: String,

    /// Element ID if present
    #[serde(skip_serializing_if = "Option::is_none")]
    pub id: Option<String>,

    /// Element class if present
    #[serde(skip_serializing_if = "Option::is_none")]
    pub class: Option<String>,

    /// ARIA role if present
    #[serde(skip_serializing_if = "Option::is_none")]
    pub role: Option<String>,
}

/// Main content area information
#[derive(Debug, Clone, Serialize, Deserialize, JsonSchema)]
pub struct MainContent {
    /// HTML tag of the main content container
    pub tag: String,

    /// Element ID if present
    #[serde(skip_serializing_if = "Option::is_none")]
    pub id: Option<String>,

    /// Approximate word count
    pub word_count: usize,
}

/// Page meta information
#[derive(Debug, Clone, Default, Serialize, Deserialize, JsonSchema)]
pub struct Meta {
    /// Meta description
    #[serde(skip_serializing_if = "Option::is_none")]
    pub description: Option<String>,

    /// Meta keywords
    #[serde(skip_serializing_if = "Option::is_none")]
    pub keywords: Option<String>,

    /// Canonical URL
    #[serde(skip_serializing_if = "Option::is_none")]
    pub canonical: Option<String>,

    /// OpenGraph title
    #[serde(skip_serializing_if = "Option::is_none")]
    pub og_title: Option<String>,
}

/// JavaScript code for extracting page structure
const STRUCTURE_JS: &str = r#"
(function() {
    var structure = {
        url: window.location.href,
        title: document.title,
        headings: [],
        nav_links: [],
        sections: [],
        main_content: null,
        meta: {}
    };

    // Extract headings hierarchy
    var headings = document.querySelectorAll('h1, h2, h3, h4, h5, h6');
    for (var i = 0; i < headings.length && i < 50; i++) {
        structure.headings.push({
            level: parseInt(headings[i].tagName.charAt(1)),
            text: headings[i].innerText.trim().substring(0, 200)
        });
    }

    // Extract nav links
    var navs = document.querySelectorAll('nav a, header a, [role="navigation"] a');
    var seenLinks = new Set();
    for (var i = 0; i < navs.length && structure.nav_links.length < 30; i++) {
        var href = navs[i].getAttribute('href');
        var text = navs[i].innerText.trim();
        if (href && text && !seenLinks.has(href)) {
            seenLinks.add(href);
            structure.nav_links.push({ text: text.substring(0, 100), href: href });
        }
    }

    // Extract semantic sections
    var sections = document.querySelectorAll('main, article, section, aside, footer');
    for (var i = 0; i < sections.length && i < 20; i++) {
        var el = sections[i];
        structure.sections.push({
            tag: el.tagName.toLowerCase(),
            id: el.id || null,
            class: el.className ? el.className.substring(0, 100) : null,
            role: el.getAttribute('role') || null
        });
    }

    // Check for main content area
    var main = document.querySelector('main, [role="main"], #main, #content, .main-content');
    if (main) {
        structure.main_content = {
            tag: main.tagName.toLowerCase(),
            id: main.id || null,
            word_count: main.innerText.split(/\s+/).length
        };
    }

    // Extract meta information
    var metaDesc = document.querySelector('meta[name="description"]');
    if (metaDesc) structure.meta.description = metaDesc.getAttribute('content');

    var metaKeywords = document.querySelector('meta[name="keywords"]');
    if (metaKeywords) structure.meta.keywords = metaKeywords.getAttribute('content');

    var canonical = document.querySelector('link[rel="canonical"]');
    if (canonical) structure.meta.canonical = canonical.getAttribute('href');

    var ogTitle = document.querySelector('meta[property="og:title"]');
    if (ogTitle) structure.meta.og_title = ogTitle.getAttribute('content');

    return JSON.stringify(structure);
})()
"#;

/// JavaScript code for extracting URLs from sitemap XML
const EXTRACT_URLS_JS: &str = r#"
(function() {
    var urls = [];
    // Try XML sitemap format
    var locs = document.querySelectorAll('loc');
    for (var i = 0; i < locs.length; i++) {
        urls.push(locs[i].textContent.trim());
    }
    // Also check for sitemap index
    var sitemaps = document.querySelectorAll('sitemap loc');
    for (var i = 0; i < sitemaps.length; i++) {
        urls.push('SITEMAP:' + sitemaps[i].textContent.trim());
    }
    return JSON.stringify(urls);
})()
"#;

/// JavaScript code for checking if page is a valid sitemap
const CHECK_SITEMAP_JS: &str = r#"
(function() {
    var text = document.body?.innerText || '';
    if (text.includes('<urlset') || text.includes('<sitemapindex') ||
        document.querySelector('urlset, sitemapindex')) {
        return 'valid';
    }
    return 'invalid';
})()
"#;

#[derive(Default)]
pub struct SitemapTool;

impl Tool for SitemapTool {
    type Params = SitemapParams;

    fn name(&self) -> &str {
        "sitemap"
    }

    fn execute_typed(&self, params: SitemapParams, context: &mut ToolContext) -> Result<ToolResult> {
        let base_url = params.url.trim_end_matches('/');
        let sitemap_urls = vec![
            format!("{}/sitemap.xml", base_url),
            format!("{}/sitemap_index.xml", base_url),
            format!("{}/sitemap-index.xml", base_url),
        ];

        let mut result = SitemapResult {
            base_url: base_url.to_string(),
            robots_txt: None,
            sitemaps: Vec::new(),
            pages: Vec::new(),
            page_structures: Vec::new(),
        };

        let tab = context.session.tab()?;

        // Try to fetch robots.txt first
        let robots_url = format!("{}/robots.txt", base_url);
        context.session.navigate(&robots_url)?;
        context.session.wait_for_navigation()?;

        let robots_js = r#"document.body?.innerText || document.documentElement?.innerText || ''"#;
        if let Ok(eval_result) = tab.evaluate(robots_js, false) {
            if let Some(value) = &eval_result.value {
                if let Some(text) = value.as_str() {
                    if !text.contains("404") && !text.is_empty() && text.len() < 50000 {
                        result.robots_txt = Some(text.to_string());
                        // Extract sitemap URLs from robots.txt
                        for line in text.lines() {
                            let line = line.trim();
                            if line.to_lowercase().starts_with("sitemap:") {
                                let sitemap_url = line[8..].trim().to_string();
                                if !result.sitemaps.contains(&sitemap_url) {
                                    result.sitemaps.push(sitemap_url);
                                }
                            }
                        }
                    }
                }
            }
        }

        // Try common sitemap URLs if none found in robots.txt
        if result.sitemaps.is_empty() {
            for sitemap_url in &sitemap_urls {
                context.session.navigate(sitemap_url)?;
                context.session.wait_for_navigation()?;

                if let Ok(eval_result) = tab.evaluate(CHECK_SITEMAP_JS, false) {
                    if let Some(value) = &eval_result.value {
                        if value.as_str() == Some("valid") {
                            result.sitemaps.push(sitemap_url.clone());
                            break;
                        }
                    }
                }
            }
        }

        // Parse sitemap(s) for URLs (limited by max_sitemaps)
        let mut sitemaps_parsed = 0;
        let mut sitemap_queue = result.sitemaps.clone();

        while let Some(sitemap_url) = sitemap_queue.first().cloned() {
            if sitemaps_parsed >= params.max_sitemaps {
                break;
            }
            sitemap_queue.remove(0);
            sitemaps_parsed += 1;

            context.session.navigate(&sitemap_url)?;
            context.session.wait_for_navigation()?;

            if let Ok(eval_result) = tab.evaluate(EXTRACT_URLS_JS, false) {
                if let Some(value) = &eval_result.value {
                    if let Some(json_str) = value.as_str() {
                        if let Ok(urls) = serde_json::from_str::<Vec<String>>(json_str) {
                            for url in urls {
                                if url.starts_with("SITEMAP:") {
                                    let nested_sitemap = url.trim_start_matches("SITEMAP:").to_string();
                                    if !result.sitemaps.contains(&nested_sitemap) {
                                        result.sitemaps.push(nested_sitemap.clone());
                                        sitemap_queue.push(nested_sitemap);
                                    }
                                } else if !result.pages.contains(&url) {
                                    result.pages.push(url);
                                }
                            }
                        }
                    }
                }
            }
        }

        // Analyze page structure if requested
        if params.analyze_structure {
            let mut pages_to_analyze: Vec<String> = result
                .pages
                .iter()
                .take(params.max_pages.saturating_sub(1))
                .cloned()
                .collect();

            // Also analyze the homepage (insert at beginning)
            pages_to_analyze.insert(0, base_url.to_string());
            pages_to_analyze.truncate(params.max_pages);

            for page_url in &pages_to_analyze {
                context.session.navigate(page_url)?;
                context.session.wait_for_navigation()?;

                if let Ok(eval_result) = tab.evaluate(STRUCTURE_JS, false) {
                    if let Some(value) = &eval_result.value {
                        if let Some(json_str) = value.as_str() {
                            if let Ok(structure) = serde_json::from_str::<PageStructure>(json_str) {
                                result.page_structures.push(structure);
                            }
                        }
                    }
                }
            }
        }

        Ok(ToolResult::success_with(&result))
    }
}

/// Standalone function for sitemap analysis (used by CLI)
pub fn analyze_sitemap(
    session: &crate::browser::BrowserSession,
    url: &str,
    analyze_structure: bool,
    max_pages: usize,
    max_sitemaps: usize,
) -> Result<SitemapResult> {
    let mut context = ToolContext::new(session);
    let params = SitemapParams {
        url: url.to_string(),
        analyze_structure,
        max_pages,
        max_sitemaps,
    };

    let tool = SitemapTool;
    let result = tool.execute_typed(params, &mut context)?;

    result
        .data
        .and_then(|v| serde_json::from_value(v).ok())
        .ok_or_else(|| BrowserError::ToolExecutionFailed {
            tool: "sitemap".to_string(),
            reason: "Failed to parse sitemap result".to_string(),
        })
}
