#!/usr/bin/env python3
# /// script
# requires-python = ">=3.13"
# dependencies = [
#     "aiounifi>=88",
#     "aiohttp>=3.9.0",
#     "python-dotenv>=1.0.0",
# ]
# ///
"""
UniFi Network Controller CLI - Comprehensive management tool

Uses aiounifi library for robust async communication with UniFi controllers.
Supports UDM Pro/SE, Dream Machine, Cloud Key Gen2+, and self-hosted controllers.

SYSTEM & SITES:
    uv run unifi.py sites                     # List all sites
    uv run unifi.py sysinfo                   # System information
    uv run unifi.py health                    # Site health status

DEVICES (APs, Switches, Gateways):
    uv run unifi.py devices                   # List all devices
    uv run unifi.py device MAC                # Device details
    uv run unifi.py restart MAC               # Restart device
    uv run unifi.py restart MAC --hard        # Hard restart (cycles PoE)
    uv run unifi.py upgrade MAC               # Upgrade device firmware
    uv run unifi.py locate MAC                # Blink LED to locate
    uv run unifi.py unlocate MAC              # Stop LED blinking
    uv run unifi.py led MAC on|off|default    # Set LED status
    uv run unifi.py led MAC on --color=#FF0000 --brightness=50

SWITCH PORTS:
    uv run unifi.py ports                     # List all switch ports
    uv run unifi.py port MAC PORT_IDX         # Port details
    uv run unifi.py port-enable MAC PORT_IDX  # Enable switch port
    uv run unifi.py port-disable MAC PORT_IDX # Disable switch port
    uv run unifi.py poe MAC PORT_IDX auto|off|passthrough|24v  # Set PoE mode
    uv run unifi.py power-cycle MAC PORT_IDX  # Power cycle PoE port

SMART POWER (PDU/Outlets):
    uv run unifi.py outlets                   # List all outlets
    uv run unifi.py outlet MAC IDX on|off     # Control outlet relay
    uv run unifi.py outlet-cycle MAC IDX on|off  # Enable/disable auto-cycle

CLIENTS:
    uv run unifi.py clients                   # List active clients
    uv run unifi.py clients-all               # List all clients (incl. offline)
    uv run unifi.py client MAC                # Client details
    uv run unifi.py block MAC                 # Block client
    uv run unifi.py unblock MAC               # Unblock client
    uv run unifi.py reconnect MAC             # Kick/reconnect client
    uv run unifi.py forget MAC [MAC2...]      # Forget client(s)

WIFI NETWORKS:
    uv run unifi.py wlans                     # List wireless networks
    uv run unifi.py wlan ID                   # WLAN details
    uv run unifi.py wlan-enable ID            # Enable WLAN
    uv run unifi.py wlan-disable ID           # Disable WLAN
    uv run unifi.py wlan-password ID NEWPASS  # Change WLAN password
    uv run unifi.py wlan-qr ID                # Generate WiFi QR code (PNG)

PORT FORWARDING:
    uv run unifi.py port-forwards             # List port forwarding rules
    uv run unifi.py port-forward ID           # Port forward details

TRAFFIC RULES:
    uv run unifi.py traffic-rules             # List traffic rules
    uv run unifi.py traffic-rule ID           # Traffic rule details
    uv run unifi.py traffic-rule-enable ID    # Enable traffic rule
    uv run unifi.py traffic-rule-disable ID   # Disable traffic rule
    uv run unifi.py traffic-rule-toggle ID on|off  # Toggle traffic rule

TRAFFIC ROUTES:
    uv run unifi.py traffic-routes            # List traffic routes
    uv run unifi.py traffic-route ID          # Traffic route details
    uv run unifi.py traffic-route-enable ID   # Enable traffic route
    uv run unifi.py traffic-route-disable ID  # Disable traffic route

FIREWALL:
    uv run unifi.py firewall-policies         # List firewall policies
    uv run unifi.py firewall-policy ID        # Firewall policy details
    uv run unifi.py firewall-zones            # List firewall zones
    uv run unifi.py firewall-zone ID          # Firewall zone details

DPI (Deep Packet Inspection):
    uv run unifi.py dpi-apps                  # List DPI restriction apps
    uv run unifi.py dpi-app ID                # DPI app details
    uv run unifi.py dpi-app-enable ID         # Enable DPI app restriction
    uv run unifi.py dpi-app-disable ID        # Disable DPI app restriction
    uv run unifi.py dpi-groups                # List DPI restriction groups
    uv run unifi.py dpi-group ID              # DPI group details

HOTSPOT VOUCHERS:
    uv run unifi.py vouchers                  # List vouchers
    uv run unifi.py voucher-create --duration=60 --quota=1 --note="Guest"
    uv run unifi.py voucher-delete ID         # Delete voucher

EVENTS:
    uv run unifi.py events                    # Stream events (Ctrl+C to stop)

RAW API:
    uv run unifi.py raw GET /stat/health      # Raw API GET request
    uv run unifi.py raw POST /cmd/devmgr '{"cmd":"restart","mac":"..."}'
"""

from __future__ import annotations

import argparse
import asyncio
import json
import os
import sys
from dataclasses import dataclass
from typing import Any, Self

import aiohttp
import aiounifi
from aiounifi.controller import Controller
from aiounifi.models.api import ApiRequest
from aiounifi.models.configuration import Configuration
from aiounifi.models.device import (
    DevicePowerCyclePortRequest,
    DeviceRestartRequest,
    DeviceSetLedStatus,
    DeviceSetOutletCycleEnabledRequest,
    DeviceSetOutletRelayRequest,
    DeviceSetPoePortModeRequest,
    DeviceSetPortEnabledRequest,
)
from aiounifi.models.voucher import Voucher
from aiounifi.models.wlan import WlanChangePasswordRequest
from dotenv import load_dotenv


@dataclass
class DeviceLocateRequest(ApiRequest):
    """Request object for device locate mode."""

    @classmethod
    def create(cls, mac: str, locate: bool) -> Self:
        """Enable or disable device locate mode."""
        cmd = "set-locate" if locate else "unset-locate"
        return cls(method="post", path="/cmd/devmgr", data={"cmd": cmd, "mac": mac})


load_dotenv()


def get_config() -> tuple[str, str, str, int, str, bool]:
    """Get configuration from environment variables."""
    host = os.environ.get("UNIFI_HOST", "").rstrip("/")
    username = os.environ.get("UNIFI_USERNAME", "")
    password = os.environ.get("UNIFI_PASSWORD", "")
    site = os.environ.get("UNIFI_SITE", "default")
    is_udm = os.environ.get("UNIFI_IS_UDM", "true").lower() == "true"

    if not all([host, username, password]):
        print("ERROR: Missing config in .env", file=sys.stderr)
        print("Required: UNIFI_HOST, UNIFI_USERNAME, UNIFI_PASSWORD", file=sys.stderr)
        sys.exit(1)

    # Extract host and port
    if "://" in host:
        host = host.split("://")[1]

    port = 443 if is_udm else 8443
    if ":" in host:
        host, port_str = host.rsplit(":", 1)
        port = int(port_str)

    return host, username, password, port, site, is_udm


def output(data: Any, raw: bool = False) -> None:
    """Print JSON output."""
    if raw:
        print(json.dumps(data, indent=2, default=str))
    elif hasattr(data, "raw"):
        print(json.dumps(data.raw, indent=2, default=str))
    elif isinstance(data, dict):
        print(json.dumps(data, indent=2, default=str))
    elif isinstance(data, list):
        print(
            json.dumps(
                [d.raw if hasattr(d, "raw") else d for d in data],  # type: ignore[union-attr]
                indent=2,
                default=str,
            )
        )
    else:
        print(json.dumps(data, indent=2, default=str))


def output_table(
    items: list[Any], keys: list[str], headers: list[str] | None = None
) -> None:
    """Print items as a formatted table."""
    if not items:
        print("No items found.")
        return

    headers = headers or keys
    rows: list[list[str]] = []
    for item in items:
        row: list[str] = []
        for key in keys:
            val: Any = ""
            if hasattr(item, key):  # type: ignore[arg-type]
                val = getattr(item, key)  # type: ignore[arg-type]
            elif hasattr(item, "raw") and key in item.raw:  # type: ignore[union-attr]
                val = item.raw[key]  # type: ignore[union-attr]
            elif isinstance(item, dict) and key in item:
                val = item[key]  # type: ignore[index]
            row.append(str(val) if val is not None else "")  # type: ignore[arg-type]
        rows.append(row)

    # Calculate column widths
    widths = [len(h) for h in headers]
    for row in rows:
        for i, cell in enumerate(row):
            widths[i] = max(widths[i], len(cell))

    # Print header
    header_line = " | ".join(h.ljust(widths[i]) for i, h in enumerate(headers))
    print(header_line)
    print("-" * len(header_line))

    # Print rows
    for row in rows:
        print(" | ".join(cell.ljust(widths[i]) for i, cell in enumerate(row)))


async def get_controller(session: aiohttp.ClientSession) -> Controller:
    """Create and login to controller."""
    host, username, password, port, site, _ = get_config()

    controller = Controller(
        Configuration(
            session,
            host,
            username=username,
            password=password,
            port=port,
            site=site,
            ssl_context=False,  # Disable SSL verification for self-signed certs
        )
    )

    try:
        await asyncio.wait_for(controller.login(), timeout=15)
    except aiounifi.LoginRequired:
        print("ERROR: Login failed - check credentials", file=sys.stderr)
        sys.exit(1)
    except aiounifi.Unauthorized:
        print("ERROR: Unauthorized - invalid credentials", file=sys.stderr)
        sys.exit(1)
    except TimeoutError:
        print("ERROR: Connection timeout", file=sys.stderr)
        sys.exit(1)
    except aiounifi.RequestError as e:
        print(f"ERROR: Connection failed - {e}", file=sys.stderr)
        sys.exit(1)
    except aiounifi.errors.ResponseError as e:
        if "429" in str(e) or "LIMIT_REACHED" in str(e):
            print(
                "ERROR: Rate limited - too many login attempts. Wait a few minutes.",
                file=sys.stderr,
            )
        else:
            print(f"ERROR: API error - {e}", file=sys.stderr)
        sys.exit(1)

    return controller


async def cmd_sites(ctrl: Controller, args: argparse.Namespace) -> None:
    """List all sites."""
    await ctrl.sites.update()
    if args.json:
        output(list(ctrl.sites.values()))
    else:
        output_table(
            list(ctrl.sites.values()),
            ["name", "description"],
            ["Name", "Description"],
        )


async def cmd_sysinfo(ctrl: Controller, args: argparse.Namespace) -> None:
    """Get system information."""
    await ctrl.system_information.update()
    # SystemInformationHandler stores data differently
    data: dict[str, Any] = getattr(ctrl.system_information, "raw", None) or {}
    output(data, raw=True)


async def cmd_health(ctrl: Controller, args: argparse.Namespace) -> None:
    """Get site health."""
    from dataclasses import dataclass

    from aiounifi.models.api import ApiRequest as AR

    @dataclass
    class HealthRequest(AR):
        @classmethod
        def create(cls) -> "HealthRequest":
            return cls(method="get", path="/stat/health")

    response = await ctrl.request(HealthRequest.create())
    output(response, raw=True)


async def cmd_devices(ctrl: Controller, args: argparse.Namespace) -> None:
    """List all devices."""
    await ctrl.devices.update()
    if args.json:
        output(list(ctrl.devices.values()))
    else:
        output_table(
            list(ctrl.devices.values()),
            ["mac", "name", "model", "ip", "state", "version"],
            ["MAC", "Name", "Model", "IP", "State", "Version"],
        )


async def cmd_device(ctrl: Controller, args: argparse.Namespace) -> None:
    """Get device details."""
    await ctrl.devices.update()
    mac = args.mac.lower()
    device = ctrl.devices.get(mac)
    if not device:
        print(f"ERROR: Device {mac} not found", file=sys.stderr)
        sys.exit(1)
    output(device)


async def cmd_restart(ctrl: Controller, args: argparse.Namespace) -> None:
    """Restart a device."""
    mac = args.mac.lower()
    soft = not args.hard
    response = await ctrl.request(DeviceRestartRequest.create(mac, soft=soft))
    output(response, raw=True)


async def cmd_upgrade(ctrl: Controller, args: argparse.Namespace) -> None:
    """Upgrade device firmware."""
    mac = args.mac.lower()
    response = await ctrl.devices.upgrade(mac)
    output(response, raw=True)


async def cmd_locate(ctrl: Controller, args: argparse.Namespace) -> None:
    """Enable device locate mode (blink LED)."""
    mac = args.mac.lower()
    response = await ctrl.request(DeviceLocateRequest.create(mac, locate=True))
    output(response, raw=True)


async def cmd_unlocate(ctrl: Controller, args: argparse.Namespace) -> None:
    """Disable device locate mode."""
    mac = args.mac.lower()
    response = await ctrl.request(DeviceLocateRequest.create(mac, locate=False))
    output(response, raw=True)


async def cmd_led(ctrl: Controller, args: argparse.Namespace) -> None:
    """Set device LED status."""
    await ctrl.devices.update()
    mac = args.mac.lower()
    device = ctrl.devices.get(mac)
    if not device:
        print(f"ERROR: Device {mac} not found", file=sys.stderr)
        sys.exit(1)

    response = await ctrl.request(
        DeviceSetLedStatus.create(
            device,
            status=args.status,
            brightness=args.brightness,
            color=args.color,
        )
    )
    output(response, raw=True)


async def cmd_ports(ctrl: Controller, args: argparse.Namespace) -> None:
    """List all switch ports."""
    await ctrl.devices.update()
    if args.json:
        output(list(ctrl.ports.values()))
    else:
        ports = list(ctrl.ports.values())
        output_table(
            ports,
            ["name", "port_idx", "speed", "poe_enable", "up"],
            ["Name", "Port", "Speed", "PoE", "Up"],
        )


async def cmd_port(ctrl: Controller, args: argparse.Namespace) -> None:
    """Get port details."""
    await ctrl.devices.update()
    mac = args.mac.lower()
    port_key = f"{mac}_{args.port_idx}"
    port = ctrl.ports.get(port_key)
    if not port:
        print(f"ERROR: Port {port_key} not found", file=sys.stderr)
        sys.exit(1)
    output(port)


async def cmd_poe(ctrl: Controller, args: argparse.Namespace) -> None:
    """Set PoE mode for a switch port."""
    await ctrl.devices.update()
    mac = args.mac.lower()
    device = ctrl.devices.get(mac)
    if not device:
        print(f"ERROR: Device {mac} not found", file=sys.stderr)
        sys.exit(1)

    response = await ctrl.request(
        DeviceSetPoePortModeRequest.create(
            device, port_idx=args.port_idx, mode=args.mode
        )
    )
    output(response, raw=True)


async def cmd_power_cycle(ctrl: Controller, args: argparse.Namespace) -> None:
    """Power cycle a PoE port."""
    mac = args.mac.lower()
    response = await ctrl.request(
        DevicePowerCyclePortRequest.create(mac, args.port_idx)
    )
    output(response, raw=True)


async def cmd_port_enable(ctrl: Controller, args: argparse.Namespace) -> None:
    """Enable a switch port."""
    await ctrl.devices.update()
    mac = args.mac.lower()
    device = ctrl.devices.get(mac)
    if not device:
        print(f"ERROR: Device {mac} not found", file=sys.stderr)
        sys.exit(1)

    response = await ctrl.request(
        DeviceSetPortEnabledRequest.create(device, port_idx=args.port_idx, enabled=True)
    )
    output(response, raw=True)


async def cmd_port_disable(ctrl: Controller, args: argparse.Namespace) -> None:
    """Disable a switch port."""
    await ctrl.devices.update()
    mac = args.mac.lower()
    device = ctrl.devices.get(mac)
    if not device:
        print(f"ERROR: Device {mac} not found", file=sys.stderr)
        sys.exit(1)

    response = await ctrl.request(
        DeviceSetPortEnabledRequest.create(
            device, port_idx=args.port_idx, enabled=False
        )
    )
    output(response, raw=True)


async def cmd_outlets(ctrl: Controller, args: argparse.Namespace) -> None:
    """List all outlets."""
    await ctrl.devices.update()
    if args.json:
        output(list(ctrl.outlets.values()))
    else:
        output_table(
            list(ctrl.outlets.values()),
            ["name", "index", "relay_state"],
            ["Name", "Index", "Relay State"],
        )


async def cmd_outlet(ctrl: Controller, args: argparse.Namespace) -> None:
    """Set outlet relay state."""
    await ctrl.devices.update()
    mac = args.mac.lower()
    device = ctrl.devices.get(mac)
    if not device:
        print(f"ERROR: Device {mac} not found", file=sys.stderr)
        sys.exit(1)

    state = args.state.lower() == "on"
    response = await ctrl.request(
        DeviceSetOutletRelayRequest.create(device, args.idx, state)
    )
    output(response, raw=True)


async def cmd_outlet_cycle(ctrl: Controller, args: argparse.Namespace) -> None:
    """Set outlet auto-cycle enabled state."""
    await ctrl.devices.update()
    mac = args.mac.lower()
    device = ctrl.devices.get(mac)
    if not device:
        print(f"ERROR: Device {mac} not found", file=sys.stderr)
        sys.exit(1)

    state = args.state.lower() == "on"
    response = await ctrl.request(
        DeviceSetOutletCycleEnabledRequest.create(device, args.idx, state)
    )
    output(response, raw=True)


async def cmd_clients(ctrl: Controller, args: argparse.Namespace) -> None:
    """List active clients."""
    await ctrl.clients.update()
    if args.json:
        output(list(ctrl.clients.values()))
    else:
        output_table(
            list(ctrl.clients.values()),
            ["mac", "name", "ip", "hostname"],
            ["MAC", "Name", "IP", "Hostname"],
        )


async def cmd_clients_all(ctrl: Controller, args: argparse.Namespace) -> None:
    """List all clients (including offline)."""
    await ctrl.clients_all.update()
    if args.json:
        output(list(ctrl.clients_all.values()))
    else:
        output_table(
            list(ctrl.clients_all.values()),
            ["mac", "name", "ip", "hostname"],
            ["MAC", "Name", "IP", "Hostname"],
        )


async def cmd_client(ctrl: Controller, args: argparse.Namespace) -> None:
    """Get client details."""
    await ctrl.clients.update()
    await ctrl.clients_all.update()
    mac = args.mac.lower()

    client = ctrl.clients.get(mac) or ctrl.clients_all.get(mac)
    if not client:
        print(f"ERROR: Client {mac} not found", file=sys.stderr)
        sys.exit(1)
    output(client)


async def cmd_block(ctrl: Controller, args: argparse.Namespace) -> None:
    """Block a client."""
    mac = args.mac.lower()
    response = await ctrl.clients.block(mac)
    output(response, raw=True)


async def cmd_unblock(ctrl: Controller, args: argparse.Namespace) -> None:
    """Unblock a client."""
    mac = args.mac.lower()
    response = await ctrl.clients.unblock(mac)
    output(response, raw=True)


async def cmd_reconnect(ctrl: Controller, args: argparse.Namespace) -> None:
    """Reconnect (kick) a client."""
    mac = args.mac.lower()
    response = await ctrl.clients.reconnect(mac)
    output(response, raw=True)


async def cmd_forget(ctrl: Controller, args: argparse.Namespace) -> None:
    """Forget client(s)."""
    macs = [m.lower() for m in args.macs]
    response = await ctrl.clients.remove_clients(macs)
    output(response, raw=True)


async def cmd_wlans(ctrl: Controller, args: argparse.Namespace) -> None:
    """List wireless networks."""
    await ctrl.wlans.update()
    if args.json:
        output(list(ctrl.wlans.values()))
    else:
        output_table(
            list(ctrl.wlans.values()),
            ["id", "name", "enabled", "security"],
            ["ID", "Name", "Enabled", "Security"],
        )


async def cmd_wlan(ctrl: Controller, args: argparse.Namespace) -> None:
    """Get WLAN details."""
    await ctrl.wlans.update()
    wlan = ctrl.wlans.get(args.id)
    if not wlan:
        print(f"ERROR: WLAN {args.id} not found", file=sys.stderr)
        sys.exit(1)
    output(wlan)


async def cmd_wlan_enable(ctrl: Controller, args: argparse.Namespace) -> None:
    """Enable a WLAN."""
    await ctrl.wlans.update()
    wlan = ctrl.wlans.get(args.id)
    if not wlan:
        print(f"ERROR: WLAN {args.id} not found", file=sys.stderr)
        sys.exit(1)
    response = await ctrl.wlans.enable(wlan)
    output(response, raw=True)


async def cmd_wlan_disable(ctrl: Controller, args: argparse.Namespace) -> None:
    """Disable a WLAN."""
    await ctrl.wlans.update()
    wlan = ctrl.wlans.get(args.id)
    if not wlan:
        print(f"ERROR: WLAN {args.id} not found", file=sys.stderr)
        sys.exit(1)
    response = await ctrl.wlans.disable(wlan)
    output(response, raw=True)


async def cmd_wlan_qr(ctrl: Controller, args: argparse.Namespace) -> None:
    """Generate QR code for WLAN."""
    await ctrl.wlans.update()
    wlan = ctrl.wlans.get(args.id)
    if not wlan:
        print(f"ERROR: WLAN {args.id} not found", file=sys.stderr)
        sys.exit(1)

    qr_bytes = ctrl.wlans.generate_wlan_qr_code(wlan)
    filename = args.output or f"{wlan.name.replace(' ', '_')}_wifi.png"
    with open(filename, "wb") as f:
        f.write(qr_bytes)
    print(f"QR code saved to: {filename}")


async def cmd_wlan_password(ctrl: Controller, args: argparse.Namespace) -> None:
    """Change WLAN password."""
    response = await ctrl.request(
        WlanChangePasswordRequest.create(args.id, args.password)
    )
    output(response, raw=True)


async def cmd_port_forwards(ctrl: Controller, args: argparse.Namespace) -> None:
    """List port forwarding rules."""
    await ctrl.port_forwarding.update()
    if args.json:
        output(list(ctrl.port_forwarding.values()))
    else:
        output_table(
            list(ctrl.port_forwarding.values()),
            ["id", "name", "destination_port", "forward_ip", "forward_port", "enabled"],
            ["ID", "Name", "Dst Port", "Forward To", "Fwd Port", "Enabled"],
        )


async def cmd_port_forward(ctrl: Controller, args: argparse.Namespace) -> None:
    """Get port forward details."""
    await ctrl.port_forwarding.update()
    rule = ctrl.port_forwarding.get(args.id)
    if not rule:
        print(f"ERROR: Port forward {args.id} not found", file=sys.stderr)
        sys.exit(1)
    output(rule)


async def cmd_traffic_rules(ctrl: Controller, args: argparse.Namespace) -> None:
    """List traffic rules."""
    await ctrl.traffic_rules.update()
    if args.json:
        output(list(ctrl.traffic_rules.values()))
    else:
        output_table(
            list(ctrl.traffic_rules.values()),
            ["id", "description", "enabled"],
            ["ID", "Description", "Enabled"],
        )


async def cmd_traffic_rule(ctrl: Controller, args: argparse.Namespace) -> None:
    """Get traffic rule details."""
    await ctrl.traffic_rules.update()
    rule = ctrl.traffic_rules.get(args.id)
    if not rule:
        print(f"ERROR: Traffic rule {args.id} not found", file=sys.stderr)
        sys.exit(1)
    output(rule)


async def cmd_traffic_rule_enable(ctrl: Controller, args: argparse.Namespace) -> None:
    """Enable a traffic rule."""
    await ctrl.traffic_rules.update()
    rule = ctrl.traffic_rules.get(args.id)
    if not rule:
        print(f"ERROR: Traffic rule {args.id} not found", file=sys.stderr)
        sys.exit(1)
    response = await ctrl.traffic_rules.enable(rule)
    output(response, raw=True)


async def cmd_traffic_rule_disable(ctrl: Controller, args: argparse.Namespace) -> None:
    """Disable a traffic rule."""
    await ctrl.traffic_rules.update()
    rule = ctrl.traffic_rules.get(args.id)
    if not rule:
        print(f"ERROR: Traffic rule {args.id} not found", file=sys.stderr)
        sys.exit(1)
    response = await ctrl.traffic_rules.disable(rule)
    output(response, raw=True)


async def cmd_traffic_rule_toggle(ctrl: Controller, args: argparse.Namespace) -> None:
    """Toggle a traffic rule on or off."""
    await ctrl.traffic_rules.update()
    rule = ctrl.traffic_rules.get(args.id)
    if not rule:
        print(f"ERROR: Traffic rule {args.id} not found", file=sys.stderr)
        sys.exit(1)
    state = args.state.lower() == "on"
    response = await ctrl.traffic_rules.toggle(rule, state)
    output(response, raw=True)


async def cmd_traffic_routes(ctrl: Controller, args: argparse.Namespace) -> None:
    """List traffic routes."""
    await ctrl.traffic_routes.update()
    if args.json:
        output(list(ctrl.traffic_routes.values()))
    else:
        output_table(
            list(ctrl.traffic_routes.values()),
            ["id", "description", "enabled"],
            ["ID", "Description", "Enabled"],
        )


async def cmd_traffic_route(ctrl: Controller, args: argparse.Namespace) -> None:
    """Get traffic route details."""
    await ctrl.traffic_routes.update()
    route = ctrl.traffic_routes.get(args.id)
    if not route:
        print(f"ERROR: Traffic route {args.id} not found", file=sys.stderr)
        sys.exit(1)
    output(route)


async def cmd_traffic_route_enable(ctrl: Controller, args: argparse.Namespace) -> None:
    """Enable a traffic route."""
    await ctrl.traffic_routes.update()
    route = ctrl.traffic_routes.get(args.id)
    if not route:
        print(f"ERROR: Traffic route {args.id} not found", file=sys.stderr)
        sys.exit(1)
    response = await ctrl.traffic_routes.enable(route)
    output(response, raw=True)


async def cmd_traffic_route_disable(ctrl: Controller, args: argparse.Namespace) -> None:
    """Disable a traffic route."""
    await ctrl.traffic_routes.update()
    route = ctrl.traffic_routes.get(args.id)
    if not route:
        print(f"ERROR: Traffic route {args.id} not found", file=sys.stderr)
        sys.exit(1)
    response = await ctrl.traffic_routes.disable(route)
    output(response, raw=True)


async def cmd_firewall_policies(ctrl: Controller, args: argparse.Namespace) -> None:
    """List firewall policies."""
    await ctrl.firewall_policies.update()
    if args.json:
        output(list(ctrl.firewall_policies.values()))
    else:
        output_table(
            list(ctrl.firewall_policies.values()),
            ["id", "name", "action", "enabled"],
            ["ID", "Name", "Action", "Enabled"],
        )


async def cmd_firewall_policy(ctrl: Controller, args: argparse.Namespace) -> None:
    """Get firewall policy details."""
    await ctrl.firewall_policies.update()
    policy = ctrl.firewall_policies.get(args.id)
    if not policy:
        print(f"ERROR: Firewall policy {args.id} not found", file=sys.stderr)
        sys.exit(1)
    output(policy)


async def cmd_firewall_zones(ctrl: Controller, args: argparse.Namespace) -> None:
    """List firewall zones."""
    await ctrl.firewall_zones.update()
    if args.json:
        output(list(ctrl.firewall_zones.values()))
    else:
        output_table(
            list(ctrl.firewall_zones.values()),
            ["id", "name", "zone_key"],
            ["ID", "Name", "Zone Key"],
        )


async def cmd_firewall_zone(ctrl: Controller, args: argparse.Namespace) -> None:
    """Get firewall zone details."""
    await ctrl.firewall_zones.update()
    zone = ctrl.firewall_zones.get(args.id)
    if not zone:
        print(f"ERROR: Firewall zone {args.id} not found", file=sys.stderr)
        sys.exit(1)
    output(zone)


async def cmd_dpi_apps(ctrl: Controller, args: argparse.Namespace) -> None:
    """List DPI restriction apps."""
    await ctrl.dpi_apps.update()
    if args.json:
        output(list(ctrl.dpi_apps.values()))
    else:
        output_table(
            list(ctrl.dpi_apps.values()),
            ["id", "enabled", "blocked"],
            ["ID", "Enabled", "Blocked"],
        )


async def cmd_dpi_app(ctrl: Controller, args: argparse.Namespace) -> None:
    """Get DPI app details."""
    await ctrl.dpi_apps.update()
    app = ctrl.dpi_apps.get(args.id)
    if not app:
        print(f"ERROR: DPI app {args.id} not found", file=sys.stderr)
        sys.exit(1)
    output(app)


async def cmd_dpi_app_enable(ctrl: Controller, args: argparse.Namespace) -> None:
    """Enable a DPI app restriction."""
    response = await ctrl.dpi_apps.enable(args.id)
    output(response, raw=True)


async def cmd_dpi_app_disable(ctrl: Controller, args: argparse.Namespace) -> None:
    """Disable a DPI app restriction."""
    response = await ctrl.dpi_apps.disable(args.id)
    output(response, raw=True)


async def cmd_dpi_groups(ctrl: Controller, args: argparse.Namespace) -> None:
    """List DPI restriction groups."""
    await ctrl.dpi_groups.update()
    if args.json:
        output(list(ctrl.dpi_groups.values()))
    else:
        output_table(
            list(ctrl.dpi_groups.values()),
            ["id", "name"],
            ["ID", "Name"],
        )


async def cmd_dpi_group(ctrl: Controller, args: argparse.Namespace) -> None:
    """Get DPI group details."""
    await ctrl.dpi_groups.update()
    group = ctrl.dpi_groups.get(args.id)
    if not group:
        print(f"ERROR: DPI group {args.id} not found", file=sys.stderr)
        sys.exit(1)
    output(group)


async def cmd_vouchers(ctrl: Controller, args: argparse.Namespace) -> None:
    """List vouchers."""
    await ctrl.vouchers.update()
    if args.json:
        output(list(ctrl.vouchers.values()))
    else:
        output_table(
            list(ctrl.vouchers.values()),
            ["id", "code", "quota", "note", "used"],
            ["ID", "Code", "Quota", "Note", "Used"],
        )


async def cmd_voucher_create(ctrl: Controller, args: argparse.Namespace) -> None:
    """Create a voucher."""
    from datetime import timedelta

    voucher = Voucher(
        {
            "quota": args.quota,
            "duration": int(timedelta(minutes=args.duration).total_seconds()),
            "qos_usage_quota": args.usage_quota,
            "qos_rate_max_up": args.rate_up,
            "qos_rate_max_down": args.rate_down,
            "note": args.note or "",
        }
    )
    response = await ctrl.vouchers.create(voucher)
    output(response, raw=True)


async def cmd_voucher_delete(ctrl: Controller, args: argparse.Namespace) -> None:
    """Delete a voucher."""
    await ctrl.vouchers.update()
    voucher = ctrl.vouchers.get(args.id)
    if not voucher:
        print(f"ERROR: Voucher {args.id} not found", file=sys.stderr)
        sys.exit(1)
    response = await ctrl.vouchers.delete(voucher)
    output(response, raw=True)


async def cmd_events(ctrl: Controller, args: argparse.Namespace) -> None:
    """Stream events from the controller."""
    print("Streaming events (Ctrl+C to stop)...", file=sys.stderr)

    def event_callback(message: Any) -> None:
        print(json.dumps({"message": str(message)}, indent=2, default=str))

    ctrl.messages.subscribe(event_callback)  # type: ignore[arg-type]

    try:
        await ctrl.start_websocket()
    except asyncio.CancelledError:
        pass


async def cmd_raw(ctrl: Controller, args: argparse.Namespace) -> None:
    """Make a raw API request."""

    @dataclass
    class RawRequest(ApiRequest):
        @classmethod
        def create(
            cls, method: str, path: str, data: dict[str, Any] | None = None
        ) -> "RawRequest":
            return cls(method=method.lower(), path=path, data=data)

    req_data: dict[str, Any] | None = None
    if args.data:
        req_data = json.loads(args.data)

    response = await ctrl.request(RawRequest.create(args.method, args.path, req_data))
    output(response, raw=True)


def build_parser() -> argparse.ArgumentParser:
    """Build argument parser."""
    parser = argparse.ArgumentParser(
        description="UniFi Network Controller CLI",
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog=__doc__,
    )
    parser.add_argument("--json", action="store_true", help="Output as JSON")
    subparsers = parser.add_subparsers(dest="command", help="Command to run")

    # Sites & System
    subparsers.add_parser("sites", help="List all sites")
    subparsers.add_parser("sysinfo", help="System information")
    subparsers.add_parser("health", help="Site health status")

    # Devices
    subparsers.add_parser("devices", help="List all devices")

    p = subparsers.add_parser("device", help="Device details")
    p.add_argument("mac", help="Device MAC address")

    p = subparsers.add_parser("restart", help="Restart device")
    p.add_argument("mac", help="Device MAC address")
    p.add_argument("--hard", action="store_true", help="Hard restart (cycles PoE)")

    p = subparsers.add_parser("upgrade", help="Upgrade device firmware")
    p.add_argument("mac", help="Device MAC address")

    p = subparsers.add_parser("locate", help="Enable device locate (blink LED)")
    p.add_argument("mac", help="Device MAC address")

    p = subparsers.add_parser("unlocate", help="Disable device locate")
    p.add_argument("mac", help="Device MAC address")

    p = subparsers.add_parser("led", help="Set device LED status")
    p.add_argument("mac", help="Device MAC address")
    p.add_argument("status", choices=["on", "off", "default"], help="LED status")
    p.add_argument("--color", help="LED color (hex, e.g. #FF0000)")
    p.add_argument("--brightness", type=int, help="LED brightness (0-100)")

    # Ports
    subparsers.add_parser("ports", help="List all switch ports")

    p = subparsers.add_parser("port", help="Port details")
    p.add_argument("mac", help="Switch MAC address")
    p.add_argument("port_idx", type=int, help="Port index")

    p = subparsers.add_parser("poe", help="Set PoE mode")
    p.add_argument("mac", help="Switch MAC address")
    p.add_argument("port_idx", type=int, help="Port index")
    p.add_argument(
        "mode", choices=["auto", "off", "passthrough", "24v"], help="PoE mode"
    )

    p = subparsers.add_parser("power-cycle", help="Power cycle PoE port")
    p.add_argument("mac", help="Switch MAC address")
    p.add_argument("port_idx", type=int, help="Port index")

    p = subparsers.add_parser("port-enable", help="Enable switch port")
    p.add_argument("mac", help="Switch MAC address")
    p.add_argument("port_idx", type=int, help="Port index")

    p = subparsers.add_parser("port-disable", help="Disable switch port")
    p.add_argument("mac", help="Switch MAC address")
    p.add_argument("port_idx", type=int, help="Port index")

    # Outlets
    subparsers.add_parser("outlets", help="List all outlets")

    p = subparsers.add_parser("outlet", help="Set outlet relay state")
    p.add_argument("mac", help="Device MAC address")
    p.add_argument("idx", type=int, help="Outlet index")
    p.add_argument("state", choices=["on", "off"], help="Relay state")

    p = subparsers.add_parser("outlet-cycle", help="Set outlet auto-cycle")
    p.add_argument("mac", help="Device MAC address")
    p.add_argument("idx", type=int, help="Outlet index")
    p.add_argument("state", choices=["on", "off"], help="Cycle enabled")

    # Clients
    subparsers.add_parser("clients", help="List active clients")
    subparsers.add_parser("clients-all", help="List all clients")

    p = subparsers.add_parser("client", help="Client details")
    p.add_argument("mac", help="Client MAC address")

    p = subparsers.add_parser("block", help="Block client")
    p.add_argument("mac", help="Client MAC address")

    p = subparsers.add_parser("unblock", help="Unblock client")
    p.add_argument("mac", help="Client MAC address")

    p = subparsers.add_parser("reconnect", help="Reconnect client")
    p.add_argument("mac", help="Client MAC address")

    p = subparsers.add_parser("forget", help="Forget client(s)")
    p.add_argument("macs", nargs="+", help="Client MAC address(es)")

    # WLANs
    subparsers.add_parser("wlans", help="List wireless networks")

    p = subparsers.add_parser("wlan", help="WLAN details")
    p.add_argument("id", help="WLAN ID")

    p = subparsers.add_parser("wlan-enable", help="Enable WLAN")
    p.add_argument("id", help="WLAN ID")

    p = subparsers.add_parser("wlan-disable", help="Disable WLAN")
    p.add_argument("id", help="WLAN ID")

    p = subparsers.add_parser("wlan-qr", help="Generate WiFi QR code")
    p.add_argument("id", help="WLAN ID")
    p.add_argument("-o", "--output", help="Output filename")

    p = subparsers.add_parser("wlan-password", help="Change WLAN password")
    p.add_argument("id", help="WLAN ID")
    p.add_argument("password", help="New password")

    # Port Forwarding
    subparsers.add_parser("port-forwards", help="List port forwarding rules")

    p = subparsers.add_parser("port-forward", help="Port forward details")
    p.add_argument("id", help="Port forward ID")

    # Traffic Rules
    subparsers.add_parser("traffic-rules", help="List traffic rules")

    p = subparsers.add_parser("traffic-rule", help="Traffic rule details")
    p.add_argument("id", help="Rule ID")

    p = subparsers.add_parser("traffic-rule-enable", help="Enable traffic rule")
    p.add_argument("id", help="Rule ID")

    p = subparsers.add_parser("traffic-rule-disable", help="Disable traffic rule")
    p.add_argument("id", help="Rule ID")

    p = subparsers.add_parser("traffic-rule-toggle", help="Toggle traffic rule")
    p.add_argument("id", help="Rule ID")
    p.add_argument("state", choices=["on", "off"], help="State")

    # Traffic Routes
    subparsers.add_parser("traffic-routes", help="List traffic routes")

    p = subparsers.add_parser("traffic-route", help="Traffic route details")
    p.add_argument("id", help="Route ID")

    p = subparsers.add_parser("traffic-route-enable", help="Enable traffic route")
    p.add_argument("id", help="Route ID")

    p = subparsers.add_parser("traffic-route-disable", help="Disable traffic route")
    p.add_argument("id", help="Route ID")

    # Firewall
    subparsers.add_parser("firewall-policies", help="List firewall policies")

    p = subparsers.add_parser("firewall-policy", help="Firewall policy details")
    p.add_argument("id", help="Policy ID")

    subparsers.add_parser("firewall-zones", help="List firewall zones")

    p = subparsers.add_parser("firewall-zone", help="Firewall zone details")
    p.add_argument("id", help="Zone ID")

    # DPI
    subparsers.add_parser("dpi-apps", help="List DPI restriction apps")

    p = subparsers.add_parser("dpi-app", help="DPI app details")
    p.add_argument("id", help="DPI app ID")

    p = subparsers.add_parser("dpi-app-enable", help="Enable DPI app restriction")
    p.add_argument("id", help="DPI app ID")

    p = subparsers.add_parser("dpi-app-disable", help="Disable DPI app restriction")
    p.add_argument("id", help="DPI app ID")

    subparsers.add_parser("dpi-groups", help="List DPI restriction groups")

    p = subparsers.add_parser("dpi-group", help="DPI group details")
    p.add_argument("id", help="DPI group ID")

    # Vouchers
    subparsers.add_parser("vouchers", help="List vouchers")

    p = subparsers.add_parser("voucher-create", help="Create voucher")
    p.add_argument("--duration", type=int, default=60, help="Duration in minutes")
    p.add_argument("--quota", type=int, default=1, help="Number of uses")
    p.add_argument("--usage-quota", type=int, help="Usage quota in MB")
    p.add_argument("--rate-up", type=int, help="Upload rate limit in Kbps")
    p.add_argument("--rate-down", type=int, help="Download rate limit in Kbps")
    p.add_argument("--note", help="Note/description")

    p = subparsers.add_parser("voucher-delete", help="Delete voucher")
    p.add_argument("id", help="Voucher ID")

    # Events
    subparsers.add_parser("events", help="Stream events")

    # Raw API
    p = subparsers.add_parser("raw", help="Raw API request")
    p.add_argument(
        "method", choices=["GET", "POST", "PUT", "DELETE"], help="HTTP method"
    )
    p.add_argument("path", help="API path (e.g. /stat/health)")
    p.add_argument("data", nargs="?", help="JSON data for POST/PUT")

    return parser


COMMANDS = {
    "sites": cmd_sites,
    "sysinfo": cmd_sysinfo,
    "health": cmd_health,
    "devices": cmd_devices,
    "device": cmd_device,
    "restart": cmd_restart,
    "upgrade": cmd_upgrade,
    "locate": cmd_locate,
    "unlocate": cmd_unlocate,
    "led": cmd_led,
    "ports": cmd_ports,
    "port": cmd_port,
    "poe": cmd_poe,
    "power-cycle": cmd_power_cycle,
    "port-enable": cmd_port_enable,
    "port-disable": cmd_port_disable,
    "outlets": cmd_outlets,
    "outlet": cmd_outlet,
    "outlet-cycle": cmd_outlet_cycle,
    "clients": cmd_clients,
    "clients-all": cmd_clients_all,
    "client": cmd_client,
    "block": cmd_block,
    "unblock": cmd_unblock,
    "reconnect": cmd_reconnect,
    "forget": cmd_forget,
    "wlans": cmd_wlans,
    "wlan": cmd_wlan,
    "wlan-enable": cmd_wlan_enable,
    "wlan-disable": cmd_wlan_disable,
    "wlan-qr": cmd_wlan_qr,
    "wlan-password": cmd_wlan_password,
    "port-forwards": cmd_port_forwards,
    "port-forward": cmd_port_forward,
    "traffic-rules": cmd_traffic_rules,
    "traffic-rule": cmd_traffic_rule,
    "traffic-rule-enable": cmd_traffic_rule_enable,
    "traffic-rule-disable": cmd_traffic_rule_disable,
    "traffic-rule-toggle": cmd_traffic_rule_toggle,
    "traffic-routes": cmd_traffic_routes,
    "traffic-route": cmd_traffic_route,
    "traffic-route-enable": cmd_traffic_route_enable,
    "traffic-route-disable": cmd_traffic_route_disable,
    "firewall-policies": cmd_firewall_policies,
    "firewall-policy": cmd_firewall_policy,
    "firewall-zones": cmd_firewall_zones,
    "firewall-zone": cmd_firewall_zone,
    "dpi-apps": cmd_dpi_apps,
    "dpi-app": cmd_dpi_app,
    "dpi-app-enable": cmd_dpi_app_enable,
    "dpi-app-disable": cmd_dpi_app_disable,
    "dpi-groups": cmd_dpi_groups,
    "dpi-group": cmd_dpi_group,
    "vouchers": cmd_vouchers,
    "voucher-create": cmd_voucher_create,
    "voucher-delete": cmd_voucher_delete,
    "events": cmd_events,
    "raw": cmd_raw,
}


async def main() -> None:
    """Main entry point."""
    parser = build_parser()
    args = parser.parse_args()

    if not args.command:
        parser.print_help()
        sys.exit(0)

    handler = COMMANDS.get(args.command)
    if not handler:
        print(f"Unknown command: {args.command}", file=sys.stderr)
        sys.exit(1)

    async with aiohttp.ClientSession(
        cookie_jar=aiohttp.CookieJar(unsafe=True)
    ) as session:
        ctrl = await get_controller(session)
        await handler(ctrl, args)


if __name__ == "__main__":
    try:
        asyncio.run(main())
    except KeyboardInterrupt:
        pass
