#!/usr/bin/env python3
"""
evoagentx_cli.py - CLI wrapper for EvoAgentX workflow operations

This script provides a command-line interface for EvoAgentX operations.
If EvoAgentX is not installed, it provides installation guidance.

Usage:
  python3 evoagentx_cli.py check              # Check if EvoAgentX is installed
  python3 evoagentx_cli.py install            # Show installation instructions
  python3 evoagentx_cli.py create-workflow    # Create a basic workflow template
  python3 evoagentx_cli.py evolve             # Evolve an existing workflow
  python3 evoagentx_cli.py status             # Check EvoAgentX status
"""

import sys
import os
import json
import argparse
from pathlib import Path


def check_evoagentx():
    """Check if EvoAgentX is installed"""
    try:
        import evoagentx
        return True
    except ImportError:
        return False


def install_command(args):
    """Show installation instructions"""
    print("""
🚀 EvoAgentX Installation
========================

EvoAgentX is an open-source framework for evolving agentic workflows.

Installation Steps:
------------------
1. Install Python 3.8+ and pip
2. Install EvoAgentX:

   pip install evoagentx

   Or from source:
   git clone https://github.com/EvoAgentX/EvoAgentX.git
   cd EvoAgentX
   pip install -e .

3. Verify installation:

   python3 -c "import evoagentx; print(evoagentx.__version__)"

Documentation:
--------------
- GitHub: https://github.com/EvoAgentX/EvoAgentX
- Docs: https://evoagentx.github.io/EvoAgentX/
- Paper: https://arxiv.org/abs/2507.03616

Quick Start:
------------
Once installed, use EvoAgentX in your OpenClaw agent:

from evoagentx import Workflow, Agent
from evoagentx.evolution import EvolutionEngine

class MyWorkflow(Workflow):
    async def execute(self, context):
        # Your workflow logic
        return result

# Evolve the workflow
engine = EvolutionEngine()
optimized = await engine.evolve(workflow=MyWorkflow())
""")


def create_workflow_template(args):
    """Create a basic workflow template file"""
    template = '''"""
{workflow_name}.py - EvoAgentX Workflow
Generated by evoagentx-workflow skill
"""

from evoagentx import Workflow, Agent


class {workflow_name}(Workflow):
    """
    {description}
    """
    
    async def execute(self, context):
        """
        Main workflow execution
        
        Args:
            context: Execution context with inputs
            
        Returns:
            Workflow result
        """
        # Step 1: Initialize
        self.log("Starting workflow...")
        
        # Step 2: Execute core logic
        # TODO: Add your workflow logic here
        result = await self.process(context)
        
        # Step 3: Return results
        return {{
            "status": "success",
            "result": result
        }}
    
    async def process(self, context):
        """Core processing logic"""
        # Implement your specific logic
        pass


# For standalone testing
if __name__ == "__main__":
    import asyncio
    
    workflow = {workflow_name}()
    result = asyncio.run(workflow.execute({{}}))
    print(result)
'''

    workflow_name = args.name or "MyWorkflow"
    description = args.description or "A self-evolving agent workflow"
    
    filename = f"{workflow_name.lower()}.py"
    content = template.format(
        workflow_name=workflow_name,
        description=description
    )
    
    with open(filename, 'w') as f:
        f.write(content)
    
    print(f"✅ Created workflow template: {filename}")
    print(f"   Edit this file to implement your workflow logic.")
    print(f"   Run with: python3 {filename}")


def check_status(args):
    """Check EvoAgentX installation status"""
    print("🔍 Checking EvoAgentX status...\n")
    
    installed = check_evoagentx()
    
    if installed:
        import evoagentx
        print(f"✅ EvoAgentX is installed")
        print(f"   Version: {evoagentx.__version__}")
        print(f"   Location: {evoagentx.__file__}")
        
        # Check for optional dependencies
        try:
            import openai
            print(f"✅ OpenAI integration available")
        except ImportError:
            print(f"⚠️  OpenAI integration not available (pip install openai)")
        
        print("\n✅ Ready to use EvoAgentX workflows!")
    else:
        print(f"❌ EvoAgentX is not installed")
        print(f"\nTo install, run:")
        print(f"   python3 evoagentx_cli.py install")
        print(f"   pip install evoagentx")


def evolve_workflow(args):
    """Evolve an existing workflow (requires EvoAgentX)"""
    if not check_evoagentx():
        print("❌ EvoAgentX is not installed.")
        print("   Run: python3 evoagentx_cli.py install")
        return
    
    print(f"🔄 Evolving workflow: {args.workflow_file}")
    print("\n⚠️  This is a template command.")
    print("   Full evolution requires EvoAgentX to be installed and configured.")
    print("\nTo evolve a workflow programmatically:")
    print("""
from evoagentx.evolution import EvolutionEngine
from your_workflow import YourWorkflow

engine = EvolutionEngine()
optimized = await engine.evolve(
    workflow=YourWorkflow(),
    iterations=10,
    evaluation_criteria={"accuracy": 0.95}
)
""")


def show_examples(args):
    """Show usage examples"""
    print("""
📝 EvoAgentX Workflow Examples
==============================

Example 1: Research Workflow
------------------------------
from evoagentx import Workflow

class ResearchWorkflow(Workflow):
    async def execute(self, context):
        # Search
        results = await self.tools.web_search(context.query)
        
        # Extract
        content = await self.tools.web_fetch(results[0])
        
        # Synthesize
        summary = await self.llm.summarize(content)
        
        return summary

Example 2: Evolving the Workflow
--------------------------------
from evoagentx.evolution import EvolutionEngine

engine = EvolutionEngine()
optimized = await engine.evolve(
    workflow=ResearchWorkflow(),
    dataset=test_queries,
    metric="comprehensiveness"
)

Example 3: Multi-Agent Workflow
-------------------------------
from evoagentx import Agent, Workflow

class MultiAgentWorkflow(Workflow):
    async def execute(self, context):
        # Create specialized agents
        researcher = Agent(name="researcher", tools=["web_search"])
        writer = Agent(name="writer", tools=["text_generation"])
        
        # Execute workflow
        research = await researcher.run(context.topic)
        article = await writer.run(research)
        
        return article

For more examples, see:
https://github.com/EvoAgentX/EvoAgentX/tree/main/examples
""")


def main():
    parser = argparse.ArgumentParser(
        description="EvoAgentX Workflow CLI - Manage self-evolving agent workflows"
    )
    subparsers = parser.add_subparsers(dest="command", help="Commands")
    
    # Check command
    subparsers.add_parser("check", help="Check if EvoAgentX is installed")
    
    # Install command
    subparsers.add_parser("install", help="Show installation instructions")
    
    # Create command
    create_parser = subparsers.add_parser("create-workflow", help="Create workflow template")
    create_parser.add_argument("--name", default="MyWorkflow", help="Workflow class name")
    create_parser.add_argument("--description", default="A workflow", help="Workflow description")
    
    # Evolve command
    evolve_parser = subparsers.add_parser("evolve", help="Evolve a workflow")
    evolve_parser.add_argument("workflow_file", help="Python file containing workflow")
    
    # Status command
    subparsers.add_parser("status", help="Check EvoAgentX status")
    
    # Examples command
    subparsers.add_parser("examples", help="Show usage examples")
    
    args = parser.parse_args()
    
    if args.command == "check" or args.command == "status":
        check_status(args)
    elif args.command == "install":
        install_command(args)
    elif args.command == "create-workflow":
        create_workflow_template(args)
    elif args.command == "evolve":
        evolve_workflow(args)
    elif args.command == "examples":
        show_examples(args)
    else:
        parser.print_help()
        print("\n" + "="*50)
        print("Quick Start:")
        print("  python3 evoagentx_cli.py status     # Check installation")
        print("  python3 evoagentx_cli.py install    # Get install help")
        print("  python3 evoagentx_cli.py examples   # See examples")


if __name__ == "__main__":
    main()
