# Everclaw — Troubleshooting Guide

Common errors, their causes, and how to fix them.

---

## "session not found"

**Cause:** You're passing `session_id` and/or `model_id` as JSON body fields instead of HTTP headers.

**Fix:** These MUST be HTTP headers on the `/v1/chat/completions` request:

```bash
# CORRECT — headers
curl -s -u "admin:$COOKIE_PASS" "http://localhost:8082/v1/chat/completions" \
  -H "session_id: 0xYOUR_SESSION_ID" \
  -H "model_id: 0xYOUR_MODEL_ID" \
  -d '{"model":"kimi-k2.5","messages":[...],"stream":false}'

# WRONG — body fields (will fail)
curl -s ... -d '{"session_id":"0x...","model_id":"0x...","model":"kimi-k2.5","messages":[...]}'
```

This is the **#1 most common mistake** with the Morpheus proxy-router.

---

## "dial tcp: missing address"

**Cause:** You opened a session using a **bid ID** instead of a **model ID**.

**Fix:** Always open sessions via the model ID endpoint:

```bash
# CORRECT — model ID
curl -X POST "http://localhost:8082/blockchain/models/0xMODEL_ID/session" ...

# WRONG — bid ID (causes address resolution failure)
curl -X POST "http://localhost:8082/blockchain/bids/0xBID_ID/session" ...
```

The provider endpoint can only be resolved from the model ID path.

---

## "ERC20: transfer amount exceeds balance"

**Cause:** Your wallet doesn't have enough MOR to stake for the session. MOR is locked (staked) when sessions are opened.

**Fix:**
1. Check your balance: `bash skills/everclaw/scripts/balance.sh`
2. Close old/unused sessions to reclaim staked MOR: `bash skills/everclaw/scripts/session.sh close 0xSESSION_ID`
3. Wait for MOR to return to wallet (1-2 blocks)
4. Try opening the session again with a shorter duration

---

## "api adapter not found"

**Cause:** The model is not configured in `models-config.json`.

**Fix:** Add an entry for the model to `~/morpheus/models-config.json`:

```json
{
  "0xMODEL_ID_HERE": {
    "modelName": "model-name",
    "apiType": "openai",
    "apiURL": ""
  }
}
```

Then restart the proxy-router.

---

## Sessions Lost After Router Restart

**Cause:** This is normal behavior. The proxy-router does not persist session state across restarts. Sessions exist on the blockchain, but the router's in-memory mapping is lost.

**Fix:** Re-open sessions after every restart:

```bash
bash skills/everclaw/scripts/session.sh open kimi-k2.5:web 3600
```

The blockchain still has your session, and staked MOR is still locked until the session expires or is closed.

---

## MorpheusUI Conflicts

**Cause:** The MorpheusUI desktop app manages its own proxy-router process. Running both simultaneously causes port conflicts and the UI may kill your headless router.

**Fix:**
- Don't run MorpheusUI and headless proxy-router at the same time
- If you need to switch, fully quit one before starting the other
- Check for rogue processes: `pgrep -f proxy-router`

---

## ETH_NODE_ADDRESS Missing or Empty

**Cause:** The `ETH_NODE_ADDRESS` variable is not set in `.env` or not exported. The router silently connects to an empty string and all blockchain operations fail with cryptic errors.

**Fix:**

1. Verify it's in `~/morpheus/.env`:
   ```bash
   grep ETH_NODE_ADDRESS ~/morpheus/.env
   ```

2. Make sure it's being exported:
   ```bash
   export ETH_NODE_ADDRESS=https://base-mainnet.public.blastapi.io
   ```

3. Restart the router after fixing.

---

## Provider Timeout / Connection Failed

**Cause:** The provider for that model may be offline, overloaded, or unreachable.

**Fix:**
1. Try a different model — different models have different providers
2. Check available models: `curl -s -u "admin:$COOKIE_PASS" http://localhost:8082/blockchain/models | jq .`
3. Close the current session and open a new one (may connect to a different provider)
4. Wait a few minutes and retry

---

## "cannot connect to the model" Warnings at Startup

**Cause:** Normal behavior. At startup, the router attempts to verify connectivity to all configured models. Remote models may not respond to health checks.

**Fix:** No fix needed. These warnings are informational. Inference will still work once you open a session and send a request.

---

## Router Won't Start — Port Already in Use

**Cause:** Another process is using port 8082.

**Fix:**

```bash
# Find what's using the port
lsof -i :8082

# Kill the process if it's a stale router
kill <PID>

# Or change the port in .env
# WEB_ADDRESS=0.0.0.0:8083
```

---

## "insufficient funds for gas"

**Cause:** Your wallet doesn't have enough ETH on Base for gas fees.

**Fix:**
- Bridge ETH to Base mainnet
- Session open/close costs ~0.0001-0.001 ETH in gas
- Keep at least 0.01 ETH on Base for operations

---

## Cookie File Issues

**Cause:** The `.cookie` file is auto-generated by the proxy-router on first start. If it's missing or corrupted, API auth fails.

**Fix:**
1. Check it exists: `cat ~/morpheus/.cookie`
2. Format should be: `admin:<random_password>`
3. If missing, restart the router — it auto-generates the file
4. If corrupted, delete it and restart: `rm ~/morpheus/.cookie && bash skills/everclaw/scripts/start.sh`

---

## Slow Response or Hanging Requests

**Cause:** Large models (qwen3-235b, qwen3-coder-480b) take longer to generate responses. The P2P network adds some latency.

**Fix:**
- Use `"stream": true` for faster first-token experience
- Set appropriate timeouts in your client
- Use smaller/faster models (glm-4.7-flash, llama-3.3-70b) for quick tasks
- Check router logs for errors: `tail -f ~/morpheus/data/logs/router-stdout.log`

---

## General Debugging Checklist

1. ✅ Is the proxy-router running? `pgrep -f proxy-router`
2. ✅ Is the health check passing? `curl -s -u "admin:$(cat ~/morpheus/.cookie | cut -d: -f2)" http://localhost:8082/healthcheck`
3. ✅ Is `ETH_NODE_ADDRESS` set? `grep ETH_NODE_ADDRESS ~/morpheus/.env`
4. ✅ Is `models-config.json` populated? `cat ~/morpheus/models-config.json | jq 'keys'`
5. ✅ Do you have MOR? `bash skills/everclaw/scripts/balance.sh`
6. ✅ Do you have ETH for gas? (check balance output)
7. ✅ Is there an active session for your model? `bash skills/everclaw/scripts/session.sh list`
8. ✅ Are you using HTTP headers (not body) for session_id/model_id?
9. ✅ Check router logs: `tail -50 ~/morpheus/data/logs/router-stdout.log`
