---
name: everclaw
version: 0.9.7
description: AI inference you own, forever powering your OpenClaw agents via the Morpheus decentralized network. Stake MOR tokens, access Kimi K2.5 and 30+ models, and maintain persistent inference by recycling staked MOR. Includes Morpheus API Gateway bootstrap for zero-config startup, OpenAI-compatible proxy with auto-session management, automatic retry with fresh sessions, OpenAI-compatible error classification to prevent cooldown cascades, multi-key auth profile rotation for Venice API keys, Gateway Guardian v4 with billing-aware escalation, through-OpenClaw inference probes, proactive Venice DIEM credit monitoring, circuit breaker for stuck sub-agents, and nuclear self-healing restart, always-on proxy-router with launchd auto-restart, smart session archiver to prevent dashboard overload, bundled security skills, zero-dependency wallet management via macOS Keychain, x402 payment client for agent-to-agent USDC payments, and ERC-8004 agent registry reader for discovering trustless agents on Base.
homepage: https://everclaw.com
metadata:
  openclaw:
    emoji: "♾️"
    requires:
      bins: ["curl", "node"]
      env:
        - name: WALLET_PRIVATE_KEY
          optional: true
          description: "Morpheus wallet private key — injected at runtime from 1Password or macOS Keychain. NEVER stored on disk."
        - name: ETH_NODE_ADDRESS
          optional: true
          default: "https://base-mainnet.public.blastapi.io"
          description: "Base mainnet RPC endpoint for blockchain operations."
        - name: OP_SERVICE_ACCOUNT_TOKEN
          optional: true
          description: "1Password service account token (retrieved from macOS Keychain at runtime)."
    credentials:
      - name: "Wallet Private Key"
        storage: "macOS Keychain or 1Password (never on disk)"
        required: false
        description: "Required only for local P2P inference (MOR staking). Not needed for API Gateway mode."
      - name: "Morpheus API Gateway Key"
        storage: "openclaw.json providers config"
        required: false
        description: "Free API key from app.mor.org. Community bootstrap key included for initial setup."
    network:
      outbound:
        - host: "api.mor.org"
          purpose: "Morpheus API Gateway — model inference and session management"
        - host: "base-mainnet.public.blastapi.io"
          purpose: "Base L1 RPC — blockchain transactions (session open/close, MOR staking)"
        - host: "provider.mor.org"
          purpose: "Morpheus P2P network — direct inference via staked sessions"
        - host: "api.venice.ai"
          purpose: "Venice API — primary inference provider (when configured)"
      local:
        - port: 8082
          purpose: "Morpheus proxy-router (Go binary) — blockchain session management"
        - port: 8083
          purpose: "Morpheus-to-OpenAI proxy (Node.js) — translates OpenAI API to proxy-router"
    persistence:
      services:
        - name: "com.morpheus.router"
          purpose: "Proxy-router for Morpheus P2P inference"
          mechanism: "launchd KeepAlive (macOS)"
        - name: "com.morpheus.proxy"
          purpose: "OpenAI-compatible proxy translating to Morpheus"
          mechanism: "launchd KeepAlive (macOS)"
        - name: "ai.openclaw.guardian"
          purpose: "Gateway health watchdog with billing-aware escalation"
          mechanism: "launchd StartInterval (macOS)"
      directories:
        - "~/morpheus/ — proxy-router binary, config, session data"
        - "~/.openclaw/workspace/skills/everclaw/ — skill files"
        - "~/.openclaw/logs/ — guardian logs"
    install:
      method: "git clone (recommended) or clawhub install everclaw-inference"
      note: "curl | bash installer available but users should review scripts before executing. All scripts are open source at github.com/profbernardoj/everclaw."
    tags: ["inference", "everclaw", "morpheus", "mor", "decentralized", "ai", "blockchain", "base", "persistent", "fallback", "guardian", "security"]
---

# ♾️ Everclaw — AI Inference You Own, Forever Powering Your OpenClaw Agents

*Powered by [Morpheus AI](https://mor.org)*

Access Kimi K2.5, Qwen3, GLM-4, Llama 3.3, and 10+ models with inference you own. Everclaw connects your OpenClaw agent to the Morpheus P2P network — stake MOR tokens, open sessions, and recycle your stake for persistent, self-sovereign access to AI.

> 📦 **ClawHub:** `clawhub install everclaw-inference` — [clawhub.ai/DavidAJohnston/everclaw-inference](https://clawhub.ai/DavidAJohnston/everclaw-inference)
>
> ⚠️ **Name Collision Warning:** A different product ("Everclaw Vault") uses the bare `everclaw` slug on ClawHub. **Always use `everclaw-inference`** — never `clawhub install everclaw` or `clawhub update everclaw`. See `CLAWHUB_WARNING.md` for details.

## How It Works

1. **Get MOR tokens** on Base — swap from ETH/USDC via Uniswap or Aerodrome (see below)
2. You run a **proxy-router** (Morpheus Lumerin Node) locally as a consumer
3. The router connects to Base mainnet and discovers model providers
4. You **stake MOR tokens** to open a session with a provider (MOR is locked, not spent)
5. You send inference requests to `http://localhost:8082/v1/chat/completions`
6. When the session ends, your **MOR is returned** (minus tiny usage fees)
7. Re-stake the returned MOR into new sessions → persistent inference you own

## Getting MOR Tokens

You need MOR on Base to stake for inference. If you already have ETH, USDC, or USDT on Base:

```bash
# Swap ETH for MOR
bash skills/everclaw/scripts/swap.sh eth 0.01

# Swap USDC for MOR
bash skills/everclaw/scripts/swap.sh usdc 50
```

Or swap manually on a DEX:
- **Uniswap:** [MOR/ETH on Base](https://app.uniswap.org/explore/tokens/base/0x7431ada8a591c955a994a21710752ef9b882b8e3)
- **Aerodrome:** [MOR swap on Base](https://aerodrome.finance/swap?from=eth&to=0x7431ada8a591c955a994a21710752ef9b882b8e3)

Don't have anything on Base yet? Buy ETH on Coinbase, withdraw to Base, then swap to MOR. See `references/acquiring-mor.md` for the full guide.

**How much do you need?** MOR is staked, not spent — you get it back. 50-100 MOR is enough for daily use. 0.005 ETH covers months of Base gas fees.

## Architecture

```
Agent → proxy-router (localhost:8082) → Morpheus P2P Network → Provider → Model
                ↓
         Base Mainnet (MOR staking, session management)
```

---

## 1. Installation

### Option A: ClawHub (Easiest)

```bash
clawhub install everclaw-inference
```

To update: `clawhub update everclaw-inference`

⚠️ **Use `everclaw-inference`** — not `everclaw`. The bare `everclaw` slug belongs to a different, unrelated product on ClawHub.

### Option B: One-Command Installer

The safe installer handles fresh installs, updates, and ClawHub collision detection:

```bash
# Fresh install
curl -fsSL https://raw.githubusercontent.com/profbernardoj/everclaw/main/scripts/install-everclaw.sh | bash

# Or if you already have the skill:
bash skills/everclaw/scripts/install-everclaw.sh

# Check for updates
bash skills/everclaw/scripts/install-everclaw.sh --check
```

### Option C: Manual Git Clone

```bash
git clone https://github.com/profbernardoj/everclaw.git ~/.openclaw/workspace/skills/everclaw
```

To update: `cd ~/.openclaw/workspace/skills/everclaw && git pull`

### Install the Morpheus Router

After cloning, install the proxy-router:

```bash
bash skills/everclaw/scripts/install.sh
```

This downloads the latest proxy-router release for your OS/arch, extracts it to `~/morpheus/`, and creates initial config files.

### Manual Installation

1. Go to [Morpheus-Lumerin-Node releases](https://github.com/MorpheusAIs/Morpheus-Lumerin-Node/releases)
2. Download the release for your platform (e.g., `mor-launch-darwin-arm64.zip`)
3. Extract to `~/morpheus/`
4. On macOS: `xattr -cr ~/morpheus/`

### Required Files

After installation, `~/morpheus/` should contain:

| File | Purpose |
|------|---------|
| `proxy-router` | The main binary |
| `.env` | Configuration (RPC, contracts, ports) |
| `models-config.json` | Maps blockchain model IDs to API types |
| `.cookie` | Auto-generated auth credentials |

---

## 2. Configuration

### .env File

The `.env` file configures the proxy-router for consumer mode on Base mainnet. Critical variables:

```bash
# RPC endpoint — MUST be set or router silently fails
ETH_NODE_ADDRESS=https://base-mainnet.public.blastapi.io
ETH_NODE_CHAIN_ID=8453

# Contract addresses (Base mainnet)
DIAMOND_CONTRACT_ADDRESS=0x6aBE1d282f72B474E54527D93b979A4f64d3030a
MOR_TOKEN_ADDRESS=0x7431aDa8a591C955a994a21710752EF9b882b8e3

# Wallet key — leave blank, inject at runtime via 1Password
WALLET_PRIVATE_KEY=

# Proxy settings
PROXY_ADDRESS=0.0.0.0:3333
PROXY_STORAGE_PATH=./data/badger/
PROXY_STORE_CHAT_CONTEXT=true
PROXY_FORWARD_CHAT_CONTEXT=true
MODELS_CONFIG_PATH=./models-config.json

# Web API
WEB_ADDRESS=0.0.0.0:8082
WEB_PUBLIC_URL=http://localhost:8082

# Auth
AUTH_CONFIG_FILE_PATH=./proxy.conf
COOKIE_FILE_PATH=./.cookie

# Logging
LOG_COLOR=true
LOG_LEVEL_APP=info
LOG_FOLDER_PATH=./data/logs
ENVIRONMENT=production
```

⚠️ **`ETH_NODE_ADDRESS` MUST be set.** The router silently connects to an empty string without it and all blockchain operations fail. Also **`MODELS_CONFIG_PATH`** must point to your models-config.json.

### models-config.json

⚠️ **This file is required.** Without it, chat completions fail with `"api adapter not found"`.

```json
{
  "$schema": "./internal/config/models-config-schema.json",
  "models": [
    {
      "modelId": "0xb487ee62516981f533d9164a0a3dcca836b06144506ad47a5c024a7a2a33fc58",
      "modelName": "kimi-k2.5:web",
      "apiType": "openai",
      "apiUrl": ""
    },
    {
      "modelId": "0xbb9e920d94ad3fa2861e1e209d0a969dbe9e1af1cf1ad95c49f76d7b63d32d93",
      "modelName": "kimi-k2.5",
      "apiType": "openai",
      "apiUrl": ""
    }
  ]
}
```

⚠️ **Note the format:** The JSON uses a `"models"` array with `"modelId"` / `"modelName"` / `"apiType"` / `"apiUrl"` fields. The `apiUrl` is left empty — the router resolves provider endpoints from the blockchain. Add entries for every model you want to use. See `references/models.md` for the full list.

---

## 3. Starting the Router

### Secure Launch (1Password)

The proxy-router needs your wallet private key. **Never store it on disk.** Inject it at runtime from 1Password:

```bash
bash skills/everclaw/scripts/start.sh
```

Or manually:

```bash
cd ~/morpheus
source .env

# Retrieve private key from 1Password (never touches disk)
export WALLET_PRIVATE_KEY=$(
  OP_SERVICE_ACCOUNT_TOKEN=$(security find-generic-password -a "YOUR_KEYCHAIN_ACCOUNT" -s "op-service-account-token" -w) \
  op item get "YOUR_ITEM_NAME" --vault "YOUR_VAULT_NAME" --fields "Private Key" --reveal
)

export ETH_NODE_ADDRESS
nohup ./proxy-router > ./data/logs/router-stdout.log 2>&1 &
```

### Health Check

Wait a few seconds, then verify:

```bash
COOKIE_PASS=$(cat ~/morpheus/.cookie | cut -d: -f2)
curl -s -u "admin:$COOKIE_PASS" http://localhost:8082/healthcheck
```

Expected: HTTP 200.

### Stopping

```bash
bash skills/everclaw/scripts/stop.sh
```

Or: `pkill -f proxy-router`

---

## 4. MOR Allowance

Before opening sessions, approve the Diamond contract to transfer MOR on your behalf:

```bash
COOKIE_PASS=$(cat ~/morpheus/.cookie | cut -d: -f2)

curl -s -u "admin:$COOKIE_PASS" -X POST \
  "http://localhost:8082/blockchain/approve?spender=0x6aBE1d282f72B474E54527D93b979A4f64d3030a&amount=1000000000000000000000"
```

⚠️ **The `/blockchain/approve` endpoint uses query parameters**, not a JSON body. The `amount` is in wei (1000000000000000000 = 1 MOR). Approve a large amount so you don't need to re-approve frequently.

---

## 5. Opening Sessions

Open a session by **model ID** (not bid ID):

```bash
MODEL_ID="0xb487ee62516981f533d9164a0a3dcca836b06144506ad47a5c024a7a2a33fc58"

curl -s -u "admin:$COOKIE_PASS" -X POST \
  "http://localhost:8082/blockchain/models/${MODEL_ID}/session" \
  -H "Content-Type: application/json" \
  -d '{"sessionDuration": 3600}'
```

⚠️ **Always use the model ID endpoint**, not the bid ID. Using a bid ID results in `"dial tcp: missing address"`.

### Session Duration

- Duration is in **seconds**: 3600 = 1 hour, 86400 = 1 day
- **Two blockchain transactions** occur: approve transfer + open session
- MOR is **staked** (locked) for the session duration
- When the session closes, MOR is **returned** to your wallet

### Response

The response includes a `sessionId` (hex string). Save this — you need it for inference.

### Using the Script

```bash
# Open a 1-hour session for kimi-k2.5:web
bash skills/everclaw/scripts/session.sh open kimi-k2.5:web 3600

# List active sessions
bash skills/everclaw/scripts/session.sh list

# Close a session
bash skills/everclaw/scripts/session.sh close 0xSESSION_ID_HERE
```

---

## 6. Sending Inference

### ⚠️ THE #1 GOTCHA: Headers, Not Body

`session_id` and `model_id` are **HTTP headers**, not JSON body fields. This is the single most common mistake.

**CORRECT:**

```bash
curl -s -u "admin:$COOKIE_PASS" "http://localhost:8082/v1/chat/completions" \
  -H "Content-Type: application/json" \
  -H "session_id: 0xYOUR_SESSION_ID" \
  -H "model_id: 0xYOUR_MODEL_ID" \
  -d '{
    "model": "kimi-k2.5:web",
    "messages": [{"role": "user", "content": "Hello, world!"}],
    "stream": false
  }'
```

**WRONG (will fail with "session not found"):**

```bash
# DON'T DO THIS
curl -s ... -d '{
  "model": "kimi-k2.5:web",
  "session_id": "0x...",   # WRONG — not a body field
  "model_id": "0x...",     # WRONG — not a body field
  "messages": [...]
}'
```

### Using the Chat Script

```bash
bash skills/everclaw/scripts/chat.sh kimi-k2.5:web "What is the meaning of life?"
```

### Streaming

Set `"stream": true` in the request body. The response will be Server-Sent Events (SSE).

---

## 7. Closing Sessions

Close a session to reclaim your staked MOR:

```bash
curl -s -u "admin:$COOKIE_PASS" -X POST \
  "http://localhost:8082/blockchain/sessions/0xSESSION_ID/close"
```

Or use the script:

```bash
bash skills/everclaw/scripts/session.sh close 0xSESSION_ID
```

⚠️ MOR staked in a session is returned when the session closes. Close sessions you're not using to free up MOR for new sessions.

---

## 8. Session Management

### Sessions Are Ephemeral

⚠️ **Sessions are NOT persisted across router restarts.** If you restart the proxy-router, you must re-open sessions. The blockchain still has the session, but the router's in-memory state is lost.

### Monitoring

```bash
# Check balance (MOR + ETH)
bash skills/everclaw/scripts/balance.sh

# List sessions
bash skills/everclaw/scripts/session.sh list
```

### Session Lifecycle

1. **Open** → MOR is staked, session is active
2. **Active** → Send inference requests using session_id header
3. **Expired** → Session duration elapsed; MOR returned automatically
4. **Closed** → Manually closed; MOR returned immediately

### Re-opening After Restart

After restarting the router:

```bash
# Wait for health check
sleep 5

# Re-open sessions for models you need
bash skills/everclaw/scripts/session.sh open kimi-k2.5:web 3600
```

---

## 9. Checking Balances

```bash
COOKIE_PASS=$(cat ~/morpheus/.cookie | cut -d: -f2)

# MOR and ETH balance
curl -s -u "admin:$COOKIE_PASS" http://localhost:8082/blockchain/balance | jq .

# Active sessions
curl -s -u "admin:$COOKIE_PASS" http://localhost:8082/blockchain/sessions | jq .

# Available models
curl -s -u "admin:$COOKIE_PASS" http://localhost:8082/blockchain/models | jq .
```

---

## 10. Troubleshooting

See `references/troubleshooting.md` for a complete guide. Quick hits:

| Error | Fix |
|-------|-----|
| `session not found` | Use session_id/model_id as HTTP **headers**, not body fields |
| `dial tcp: missing address` | Open session by **model ID**, not bid ID |
| `api adapter not found` | Add the model to `models-config.json` |
| `ERC20: transfer amount exceeds balance` | Close old sessions to free staked MOR |
| Sessions gone after restart | Normal — re-open sessions after restart |
| MorpheusUI conflicts | Don't run MorpheusUI and headless router simultaneously |

---

## Key Contract Addresses (Base Mainnet)

| Contract | Address |
|----------|---------|
| Diamond | `0x6aBE1d282f72B474E54527D93b979A4f64d3030a` |
| MOR Token | `0x7431aDa8a591C955a994a21710752EF9b882b8e3` |

## Quick Reference

| Action | Command |
|--------|---------|
| Install | `bash skills/everclaw/scripts/install.sh` |
| Start | `bash skills/everclaw/scripts/start.sh` |
| Stop | `bash skills/everclaw/scripts/stop.sh` |
| Swap ETH→MOR | `bash skills/everclaw/scripts/swap.sh eth 0.01` |
| Swap USDC→MOR | `bash skills/everclaw/scripts/swap.sh usdc 50` |
| Open session | `bash skills/everclaw/scripts/session.sh open <model> [duration]` |
| Close session | `bash skills/everclaw/scripts/session.sh close <session_id>` |
| List sessions | `bash skills/everclaw/scripts/session.sh list` |
| Send prompt | `bash skills/everclaw/scripts/chat.sh <model> "prompt"` |
| Check balance | `bash skills/everclaw/scripts/balance.sh` |
| **Diagnose** | `bash skills/everclaw/scripts/diagnose.sh` |
| Diagnose (config only) | `bash skills/everclaw/scripts/diagnose.sh --config` |
| Diagnose (quick) | `bash skills/everclaw/scripts/diagnose.sh --quick` |

---

## 11. Wallet Management (v0.4)

Everclaw v0.4 includes a self-contained wallet manager that eliminates all external account dependencies. No 1Password, no Foundry, no Safe Wallet — just macOS Keychain and Node.js (already bundled with OpenClaw).

### Setup (One Command)

```bash
node skills/everclaw/scripts/everclaw-wallet.mjs setup
```

This generates a new Ethereum wallet and stores the private key in your macOS Keychain (encrypted at rest, protected by your login password / Touch ID).

### Import Existing Key

```bash
node skills/everclaw/scripts/everclaw-wallet.mjs import-key 0xYOUR_PRIVATE_KEY
```

### Check Balances

```bash
node skills/everclaw/scripts/everclaw-wallet.mjs balance
```

Shows ETH, MOR, USDC balances and MOR allowance for the Diamond contract.

### Swap ETH/USDC for MOR

```bash
# Swap 0.05 ETH for MOR
node skills/everclaw/scripts/everclaw-wallet.mjs swap eth 0.05

# Swap 50 USDC for MOR
node skills/everclaw/scripts/everclaw-wallet.mjs swap usdc 50
```

Executes onchain swaps via Uniswap V3 on Base. No external tools required — uses viem (bundled with OpenClaw).

### Approve MOR for Staking

```bash
node skills/everclaw/scripts/everclaw-wallet.mjs approve
```

Approves the Morpheus Diamond contract to use your MOR for session staking.

### Security Model

- Private key stored in **macOS Keychain** (encrypted at rest)
- Protected by your **login password / Touch ID**
- Key is **injected at runtime** and immediately unset from environment
- Key is **never written to disk** as a plaintext file
- For advanced users: 1Password is supported as a fallback (backward compatible)

### Full Command Reference

| Command | Description |
|---------|-------------|
| `setup` | Generate wallet, store in Keychain |
| `address` | Show wallet address |
| `balance` | Show ETH, MOR, USDC balances |
| `swap eth <amount>` | Swap ETH → MOR via Uniswap V3 |
| `swap usdc <amount>` | Swap USDC → MOR via Uniswap V3 |
| `approve [amount]` | Approve MOR for Morpheus staking |
| `export-key` | Print private key (use with caution) |
| `import-key <0xkey>` | Import existing private key |

---

## 12. OpenAI-Compatible Proxy (v0.2)

The Morpheus proxy-router requires custom auth (Basic auth via `.cookie`) and custom HTTP headers (`session_id`, `model_id`) that standard OpenAI clients don't support. Everclaw includes a lightweight proxy that bridges this gap.

### What It Does

```
OpenClaw/any client → morpheus-proxy (port 8083) → proxy-router (port 8082) → Morpheus P2P → Provider
```

- Accepts standard OpenAI `/v1/chat/completions` requests
- **Auto-opens** blockchain sessions on demand (no manual session management)
- **Auto-renews** sessions before expiry (default: 1 hour before)
- Injects Basic auth + `session_id`/`model_id` headers automatically
- Exposes `/health`, `/v1/models`, `/v1/chat/completions`

### Installation

```bash
bash skills/everclaw/scripts/install-proxy.sh
```

This installs:
- `morpheus-proxy.mjs` → `~/morpheus/proxy/`
- `gateway-guardian.sh` → `~/.openclaw/workspace/scripts/`
- launchd plists for both (macOS, auto-start on boot)

### Configuration

Environment variables (all optional, sane defaults):

| Variable | Default | Description |
|----------|---------|-------------|
| `MORPHEUS_PROXY_PORT` | `8083` | Port the proxy listens on |
| `MORPHEUS_ROUTER_URL` | `http://localhost:8082` | Proxy-router URL |
| `MORPHEUS_COOKIE_PATH` | `~/morpheus/.cookie` | Path to auth cookie |
| `MORPHEUS_SESSION_DURATION` | `604800` (7 days) | Session duration in seconds |
| `MORPHEUS_RENEW_BEFORE` | `3600` (1 hour) | Renew session this many seconds before expiry |
| `MORPHEUS_PROXY_API_KEY` | `morpheus-local` | Bearer token for proxy auth |

### Session Duration

Sessions stake MOR tokens for their duration. Longer sessions = more MOR locked but fewer blockchain transactions:

| Duration | MOR Staked (approx) | Transactions |
|----------|--------------------:|:-------------|
| 1 hour | ~11 MOR | Every hour |
| 1 day | ~274 MOR | Daily |
| 7 days | ~1,915 MOR | Weekly |

MOR is **returned** when the session closes or expires. The proxy auto-renews before expiry, so you get continuous inference with minimal staking overhead.

### Health Check

```bash
curl http://127.0.0.1:8083/health
```

### Available Models

```bash
curl http://127.0.0.1:8083/v1/models
```

### Direct Usage (without OpenClaw)

```bash
curl http://127.0.0.1:8083/v1/chat/completions \
  -H "Content-Type: application/json" \
  -H "Authorization: Bearer morpheus-local" \
  -d '{
    "model": "kimi-k2.5",
    "messages": [{"role": "user", "content": "Hello!"}],
    "stream": false
  }'
```

### Reliability Notes

- **`kimi-k2.5`** (non-web) is the most reliable model — recommended as primary fallback
- **`kimi-k2.5:web`** (web search variant) tends to timeout on P2P routing — avoid for fallback use
- Provider connection resets are transient — retries usually succeed
- The proxy itself runs as a KeepAlive launchd service — auto-restarts if it crashes

### Proxy Resilience (v0.5)

v0.5 adds three critical improvements to the proxy that prevent prolonged outages caused by **cooldown cascades** — where both primary and fallback providers become unavailable simultaneously.

#### Problem: Cooldown Cascades

When a primary provider (e.g., Venice) returns a billing error, OpenClaw's failover engine marks that provider as "in cooldown." If the Morpheus proxy also returns errors that OpenClaw misclassifies as billing errors, **both providers enter cooldown** and the agent goes completely offline — sometimes for 6+ hours.

#### Fix 1: OpenAI-Compatible Error Classification

The proxy now returns errors in the exact format OpenAI uses, with proper `type` and `code` fields:

```json
{
  "error": {
    "message": "Morpheus session unavailable: ...",
    "type": "server_error",
    "code": "morpheus_session_error",
    "param": null
  }
}
```

**Key distinction:** All Morpheus infrastructure errors are typed as `"server_error"` — never `"billing"` or `"rate_limit_error"`. This ensures OpenClaw treats them as transient failures and retries appropriately, instead of putting the provider into extended cooldown.

Error codes returned by the proxy:

| Code | Meaning |
|------|---------|
| `morpheus_session_error` | Failed to open or refresh a blockchain session |
| `morpheus_inference_error` | Provider returned an error during inference |
| `morpheus_upstream_error` | Connection error to the proxy-router |
| `timeout` | Inference request exceeded the time limit |
| `model_not_found` | Requested model not in MODEL_MAP |

#### Fix 2: Automatic Session Retry

When the proxy-router returns a session-related error (expired, invalid, not found, closed), the proxy now:

1. **Invalidates** the cached session
2. **Opens a fresh** blockchain session
3. **Retries** the inference request once

This handles the common case where the proxy-router restarts and loses its in-memory session state, or when a long-running session expires mid-request.

#### Fix 3: Multi-Tier Fallback Chain

Configure OpenClaw with multiple fallback models across providers:

```json5
{
  "agents": {
    "defaults": {
      "model": {
        "primary": "venice/claude-opus-4-6",
        "fallbacks": [
          "venice/claude-opus-45",    // Try different Venice model first
          "venice/kimi-k2-5",         // Try yet another Venice model
          "morpheus/kimi-k2.5"        // Last resort: decentralized inference
        ]
      }
    }
  }
}
```

This way, if the primary model has billing issues, OpenClaw tries other models on the same provider (which may have separate rate limits) before falling back to Morpheus. The cascade is:

1. **venice/claude-opus-4-6** (primary) → billing error
2. **venice/claude-opus-45** (fallback 1) → tries a different model on Venice
3. **venice/kimi-k2-5** (fallback 2) → tries open-source model on Venice
4. **morpheus/kimi-k2.5** (fallback 3) → decentralized inference, always available if MOR is staked

---

## 13. OpenClaw Integration (v0.2)

Configure OpenClaw to use Morpheus as a **fallback provider** so your agent keeps running when primary API credits run out.

### Step 1: Add Morpheus Provider

Add to your `openclaw.json` via config patch or manual edit:

```json5
{
  "models": {
    "providers": {
      "morpheus": {
        "baseUrl": "http://127.0.0.1:8083/v1",
        "apiKey": "morpheus-local",
        "api": "openai-completions",
        "models": [
          {
            "id": "kimi-k2.5",
            "name": "Kimi K2.5 (via Morpheus)",
            "reasoning": true,
            "input": ["text"],
            "cost": { "input": 0, "output": 0, "cacheRead": 0, "cacheWrite": 0 },
            "contextWindow": 131072,
            "maxTokens": 8192
          },
          {
            "id": "kimi-k2-thinking",
            "name": "Kimi K2 Thinking (via Morpheus)",
            "reasoning": true,
            "input": ["text"],
            "cost": { "input": 0, "output": 0, "cacheRead": 0, "cacheWrite": 0 },
            "contextWindow": 131072,
            "maxTokens": 8192
          },
          {
            "id": "glm-4.7-flash",
            "name": "GLM 4.7 Flash (via Morpheus)",
            "reasoning": false,
            "input": ["text"],
            "cost": { "input": 0, "output": 0, "cacheRead": 0, "cacheWrite": 0 },
            "contextWindow": 131072,
            "maxTokens": 8192
          }
        ]
      }
    }
  }
}
```

### Step 2: Set as Fallback

Configure a multi-tier fallback chain (recommended since v0.5):

```json5
{
  "agents": {
    "defaults": {
      "model": {
        "primary": "venice/claude-opus-4-6",
        "fallbacks": [
          "venice/claude-opus-45",   // Different model, same provider
          "venice/kimi-k2-5",        // Open-source model, same provider
          "morpheus/kimi-k2.5"       // Decentralized fallback
        ]
      },
      "models": {
        "venice/claude-opus-45": { "alias": "Claude Opus 4.5" },
        "venice/kimi-k2-5": { "alias": "Kimi K2.5" },
        "morpheus/kimi-k2.5": { "alias": "Kimi K2.5 (Morpheus)" },
        "morpheus/kimi-k2-thinking": { "alias": "Kimi K2 Thinking (Morpheus)" },
        "morpheus/glm-4.7-flash": { "alias": "GLM 4.7 Flash (Morpheus)" }
      }
    }
  }
}
```

⚠️ **Why multi-tier?** A single fallback creates a single point of failure. If both the primary provider and the single fallback enter cooldown simultaneously (e.g., billing error triggers cooldown on both), your agent goes offline. Multiple fallback tiers across different models and providers ensure at least one path remains available.

### Step 3: Add Auth Profiles

OpenClaw supports **multiple API keys per provider** with automatic rotation. When one key's credits run out (billing error), OpenClaw disables *that key only* and rotates to the next one — same model, fresh credits. This is the single most effective way to prevent downtime.

#### Single Key (Minimum Setup)

Add to `~/.openclaw/agents/main/agent/auth-profiles.json`:

```json
{
  "venice:default": {
    "type": "api_key",
    "provider": "venice",
    "key": "VENICE-INFERENCE-KEY-YOUR_KEY_HERE"
  },
  "morpheus:default": {
    "type": "api_key",
    "provider": "morpheus",
    "key": "morpheus-local"
  }
}
```

#### Multiple Keys (Recommended — v0.9.1)

If you have multiple Venice API keys (e.g., from different accounts or plans), add them all as separate profiles. Order them from most credits to least:

**auth-profiles.json:**

```json
{
  "version": 1,
  "profiles": {
    "venice:key1": {
      "type": "api_key",
      "provider": "venice",
      "key": "VENICE-INFERENCE-KEY-YOUR_PRIMARY_KEY"
    },
    "venice:key2": {
      "type": "api_key",
      "provider": "venice",
      "key": "VENICE-INFERENCE-KEY-YOUR_SECOND_KEY"
    },
    "venice:key3": {
      "type": "api_key",
      "provider": "venice",
      "key": "VENICE-INFERENCE-KEY-YOUR_THIRD_KEY"
    },
    "morpheus:default": {
      "type": "api_key",
      "provider": "morpheus",
      "key": "morpheus-local"
    }
  }
}
```

**openclaw.json** — register the profiles and set explicit rotation order:

```json5
{
  "auth": {
    "profiles": {
      "venice:key1": { "provider": "venice", "mode": "api_key" },
      "venice:key2": { "provider": "venice", "mode": "api_key" },
      "venice:key3": { "provider": "venice", "mode": "api_key" },
      "morpheus:default": { "provider": "morpheus", "mode": "api_key" }
    },
    "order": {
      "venice": ["venice:key1", "venice:key2", "venice:key3"]
    }
  }
}
```

⚠️ **`auth.order`** is critical. Without it, OpenClaw uses round-robin (oldest-used first), which may not match your credit balances. With an explicit order, keys are tried in the exact sequence you specify — highest credits first.

#### How Multi-Key Rotation Works

OpenClaw's auth engine handles rotation automatically:

1. **Session stickiness:** A key is pinned per session to keep provider caches warm. It won't flip-flop mid-conversation.
2. **Billing disable:** When a key returns a billing/credit error, that *profile* is disabled with exponential backoff (starts at 5 hours). Other profiles for the same provider remain active.
3. **Rotation on failure:** After disabling a profile, OpenClaw immediately tries the next key in `auth.order`. Same model, same provider — just fresh credits.
4. **Model fallback:** Only after ALL profiles for Venice are disabled does OpenClaw move to the next model in the fallback chain (e.g., Morpheus).
5. **Auto-recovery:** Disabled profiles auto-recover after backoff expires. If credits refill (e.g., daily reset), the profile becomes available again.

#### Venice DIEM Credits

Venice uses "DIEM" as its internal credit unit (1 DIEM ≈ $1 USD). Each API key has its own DIEM balance. Credits appear to reset daily. Expensive models drain credits faster:

| Model | Input Cost | Output Cost | ~Messages per 10 DIEM |
|-------|-----------|-------------|----------------------|
| Claude Opus 4.6 | 6 DIEM/M tokens | 30 DIEM/M tokens | ~5-10 |
| Claude Opus 4.5 | 6 DIEM/M tokens | 30 DIEM/M tokens | ~5-10 |
| Kimi K2.5 | 0.75 DIEM/M tokens | 3.75 DIEM/M tokens | ~50-100 |
| GLM 4.7 Flash | 0.125 DIEM/M tokens | 0.5 DIEM/M tokens | ~500+ |

**Tip:** With multiple keys, the agent can stay on Claude Opus across key rotations. Without multi-key, it would fall to cheaper models or Morpheus after one key's credits run out.

### Failover Behavior (v0.9.1)

The complete failover chain with multi-key rotation:

1. **Key rotation within Venice** — Key 1 credits exhausted → billing disable on *that profile only* → immediately rotates to Key 2 → Key 3 → etc. Same model, fresh credits.
2. **Model fallback** — Only after ALL Venice keys are disabled → tries `venice/claude-opus-45` (all keys again) → `venice/kimi-k2-5` (all keys) → `morpheus/kimi-k2.5`
3. **Morpheus fallback** — The proxy auto-opens a 7-day Morpheus session (if none exists). Inference routes through the Morpheus P2P network.
4. **Gateway Guardian v4** — If all providers enter cooldown despite multi-key rotation → classifies error (billing vs transient) → billing: backs off + notifies owner (restart is useless for empty credits) → transient: restarts gateway (clears cooldowns) → nuclear reinstall if needed. Proactively monitors Venice DIEM balance.
5. **Auto-recovery** — When credits refill (daily reset) or backoff expires, OpenClaw switches back to Venice automatically.

**Example with 6 keys (246 DIEM total):**

```
venice:key1 (98 DIEM) → venice:key2 (50 DIEM) → venice:key3 (40 DIEM) →
venice:key4 (26 DIEM) → venice:key5 (20 DIEM) → venice:key6 (12 DIEM) →
morpheus/kimi-k2.5 (owned, staked MOR) → mor-gateway/kimi-k2.5 (community gateway)
```

**v0.5 improvement:** The Morpheus proxy returns `"server_error"` type errors (not billing errors), so OpenClaw won't put the Morpheus provider into extended cooldown due to transient infrastructure issues. If a Morpheus session expires mid-request, the proxy automatically opens a fresh session and retries once.

---

## 14. Gateway Guardian v4 (v0.9.3)

A self-healing, billing-aware watchdog that monitors the OpenClaw gateway and its ability to run inference. Runs every 2 minutes via launchd.

### Evolution

| Version | What it checked | Fatal flaw |
|---------|----------------|------------|
| v1 | HTTP dashboard alive | Providers in cooldown = brain-dead but HTTP 200 |
| v2 | Raw provider URLs | Provider APIs always return 200 regardless of internal state |
| v3 | Through-OpenClaw inference probe | Billing exhaustion → restart → instant re-disable = dead loop. Also: `set -e` + pkill self-kill = silent no-op restarts |
| **v4** | Through-OpenClaw + **billing classification** + **credit monitoring** | Current version |

### What v4 Fixes Over v3

1. **Billing-aware escalation** — Classifies inference errors as `billing` vs `transient` vs `timeout`. Billing errors trigger backoff + notification instead of useless restarts.
2. **Silent restart bug** — Replaced `set -euo pipefail` with `set -uo pipefail` + explicit ERR trap. Restart failures are now logged instead of silently exiting.
3. **pkill self-kill** — Hard restart now iterates PIDs and excludes the Guardian's own PID. No more accidentally killing the watchdog.
4. **Proactive credit monitoring** — Checks Venice DIEM balance via `x-venice-balance-diem` response header every 10 min. Warns when balance drops below threshold.
5. **DIEM reset awareness** — Calculates hours to midnight UTC (when Venice DIEM resets daily). When billing-dead, enters 30-min backoff instead of hammering every 2 min. Auto-clears when UTC day rolls over.
6. **Signal notifications** — Notifies owner on: billing exhaustion (with ETA to reset), billing recovery, nuclear restart, and total failure.

### How It Works

1. **Billing backoff gate** — If in billing-dead state, check if midnight UTC has passed. If yes, re-probe. If no, skip this run (30-min intervals).
2. **Credit monitoring** — Every 10 min, makes a cheap Kimi K2.5 call to Venice and reads the `x-venice-balance-diem` response header. Warns below 15 DIEM.
3. **Circuit breaker** — Kills sub-agents stuck >30 min with repeated timeouts.
4. **HTTP probe** — Is the gateway process running?
5. **Inference probe** — Can the agent run inference through the full stack?
6. **Error classification** — Parses probe output:
   - `billing` → 402, Insufficient DIEM/USD/balance → **don't restart**, enter billing backoff, notify owner
   - `transient` → auth cooldown without billing keywords → restart (clears cooldown)
   - `timeout` → probe timed out → restart
   - `unknown` → restart (safe default)
7. **Four-stage restart escalation** (for non-billing errors only):
   - `openclaw gateway restart` (graceful — resets cooldown state)
   - Hard kill (excludes own PID) → launchd KeepAlive
   - `launchctl kickstart -k`
   - **🔴 NUCLEAR:** `curl -fsSL https://clawd.bot/install.sh | bash`

### Recommended Config

Pair with reduced billing backoff in `openclaw.json` to minimize downtime:

```json
{
  "auth": {
    "cooldowns": {
      "billingBackoffHoursByProvider": { "venice": 1 },
      "billingMaxHours": 6,
      "failureWindowHours": 12
    }
  }
}
```

### Installation

Included in `install-proxy.sh`, or manually:

```bash
cp skills/everclaw/scripts/gateway-guardian.sh ~/.openclaw/workspace/scripts/
chmod +x ~/.openclaw/workspace/scripts/gateway-guardian.sh

# Install launchd plist (macOS)
# See templates/ai.openclaw.guardian.plist
```

⚠️ **Important:** The launchd plist should include `OPENCLAW_GATEWAY_TOKEN` in its environment variables.

### Manual Test

```bash
bash ~/.openclaw/workspace/scripts/gateway-guardian.sh --verbose
```

### Logs

```bash
tail -f ~/.openclaw/logs/guardian.log
```

### Configuration

| Variable | Default | Description |
|----------|---------|-------------|
| `GATEWAY_PORT` | `18789` | Gateway port to probe |
| `PROBE_TIMEOUT` | `8` | HTTP timeout in seconds |
| `INFERENCE_TIMEOUT` | `45` | Agent probe timeout |
| `FAIL_THRESHOLD` | `2` | HTTP failures before restart |
| `INFERENCE_FAIL_THRESHOLD` | `3` | Inference failures before escalation (~6 min) |
| `BILLING_BACKOFF_INTERVAL` | `1800` | Seconds between probes when billing-dead (30 min) |
| `CREDIT_CHECK_INTERVAL` | `600` | Seconds between Venice DIEM balance checks (10 min) |
| `CREDIT_WARN_THRESHOLD` | `15` | DIEM balance warning threshold |
| `MAX_STUCK_DURATION_SEC` | `1800` | Circuit breaker: kill sub-agents stuck >30 min |
| `STUCK_CHECK_INTERVAL` | `300` | Circuit breaker check interval (5 min) |
| `OWNER_SIGNAL` | `+14432859111` | Signal number for notifications |
| `SIGNAL_ACCOUNT` | `+15129488566` | Signal sender account |

### State Files

| File | Purpose |
|------|---------|
| `~/.openclaw/logs/guardian.state` | HTTP failure counter |
| `~/.openclaw/logs/guardian-inference.state` | Inference failure counter |
| `~/.openclaw/logs/guardian-circuit-breaker.state` | Circuit breaker timestamp |
| `~/.openclaw/logs/guardian-billing.state` | Billing exhaustion start timestamp (0 = healthy) |
| `~/.openclaw/logs/guardian-billing-notified.state` | Whether owner was notified (0/1) |
| `~/.openclaw/logs/guardian-credit-check.state` | Last credit check timestamp |
| `~/.openclaw/logs/guardian.log` | Guardian activity log |

---

## 15. Smart Session Archiver (v0.9.4)

OpenClaw stores every conversation as a `.jsonl` file in `~/.openclaw/agents/main/sessions/`. Over time, these accumulate — and when the dashboard loads, it parses **all** session history into the DOM. At ~17MB (134+ sessions), browsers hit "Page Unresponsive" because the renderer chokes on thousands of chat message elements.

### The Problem

The bottleneck isn't raw memory — Chrome gives each tab 1.4-4GB of V8 heap. The real limit is **DOM rendering performance**. Chrome Lighthouse warns at 800 DOM nodes and errors at 1,400. A hundred sessions with tool calls, code blocks, and long conversations easily generate 5,000+ DOM elements. The browser's layout engine can't keep up.

| Sessions Dir Size | Dashboard Behavior |
|------------------|--------------------|
| < 5 MB | ✅ Loads instantly |
| 5-10 MB | ⚡ Slight delay, usable |
| 10-15 MB | ⚠️ Sluggish, noticeable lag |
| 15-20 MB | 🔴 "Page Unresponsive" likely |
| 20+ MB | 💀 Dashboard won't load |

### Solution: Size-Triggered Archiving

Instead of archiving on a fixed schedule (which may fire too early or too late depending on usage), the session archiver monitors the **actual size** of the sessions directory and only moves files when they exceed a threshold.

**Default threshold: 10MB** — provides good headroom before hitting the ~15MB danger zone, without firing unnecessarily on light usage days.

### Usage

```bash
# Archive if over threshold (default 10MB)
bash skills/everclaw/scripts/session-archive.sh

# Check size without archiving
bash skills/everclaw/scripts/session-archive.sh --check

# Force archive regardless of size
bash skills/everclaw/scripts/session-archive.sh --force

# Detailed output
bash skills/everclaw/scripts/session-archive.sh --verbose
```

### What It Protects

The archiver never moves:
- **Active sessions** — referenced in `sessions.json` (the index file)
- **Guardian health probe** — `guardian-health-probe.jsonl`
- **Recent sessions** — keeps the 5 most recent by modification time (configurable via `KEEP_RECENT`)

Everything else gets moved to `sessions/archive/` — not deleted. You can always move files back if needed.

### Configuration

| Variable | Default | Description |
|----------|---------|-------------|
| `ARCHIVE_THRESHOLD_MB` | `10` | Trigger threshold in MB |
| `SESSIONS_DIR` | `~/.openclaw/agents/main/sessions` | Sessions directory path |
| `KEEP_RECENT` | `5` | Number of recent sessions to always keep |

### Cron Integration

Set up a cron job that runs the archiver periodically. The script is a no-op when under threshold, so it's safe to run frequently:

```json5
{
  "name": "Smart session archiver",
  "schedule": { "kind": "cron", "expr": "0 */6 * * *", "tz": "America/Chicago" },
  "sessionTarget": "isolated",
  "payload": {
    "kind": "agentTurn",
    "model": "morpheus/kimi-k2.5",
    "message": "Run the smart session archiver: bash skills/everclaw/scripts/session-archive.sh --verbose. Report the results. If sessions were archived, mention the before/after size.",
    "timeoutSeconds": 60
  }
}
```

**Recommended: every 6 hours.** Frequent enough to catch growth spurts, cheap enough to run on the LIGHT tier since it's a no-op most of the time.

### Output

The script outputs a JSON summary for programmatic consumption:

```json
{"archived":42,"freedMB":8.2,"beforeMB":12.4,"afterMB":4.2,"threshold":10}
```

### Why 10MB?

Based on real-world testing: 134 sessions totaling 17MB caused "Page Unresponsive" in Chrome, Safari, and Brave on macOS. The dashboard uses a standard web renderer that parses all session JSONL into DOM elements — there's no virtualization or lazy loading. 10MB gives ~50% headroom before the ~15-20MB danger zone where most browsers start struggling.

---

## 17. x402 Payment Client (v0.7)

Everclaw v0.7 includes an x402 payment client that lets your agent make USDC payments to any x402-enabled endpoint. The [x402 protocol](https://x402.org) is an HTTP-native payment standard: when a server returns HTTP 402, your agent automatically signs a USDC payment and retries.

### How x402 Works

```
Agent → request → Server returns 402 + PAYMENT-REQUIRED header
Agent → parse requirements → sign EIP-712 payment → retry with PAYMENT-SIGNATURE header
Server → verify signature via facilitator → settle USDC → return resource
```

### CLI Usage

```bash
# Make a request to an x402-protected endpoint
node scripts/x402-client.mjs GET https://api.example.com/data

# Dry-run: see what would be paid without signing
node scripts/x402-client.mjs --dry-run GET https://api.example.com/data

# Set max payment per request
node scripts/x402-client.mjs --max-amount 0.50 GET https://api.example.com/data

# POST with body
node scripts/x402-client.mjs POST https://api.example.com/task '{"prompt":"hello"}'

# Check daily spending
node scripts/x402-client.mjs --budget
```

### Programmatic Usage

```javascript
import { makePayableRequest, createX402Client } from './scripts/x402-client.mjs';

// One-shot request
const result = await makePayableRequest("https://api.example.com/data");
// result.paid → true if 402 was handled
// result.amount → "$0.010000" (USDC)
// result.body → response content

// Reusable client with budget limits
const client = createX402Client({
  maxPerRequest: 0.50,  // $0.50 USDC max per request
  dailyLimit: 5.00,     // $5.00 USDC per day
  dryRun: false,
});

const res = await client.get("https://agent-api.example.com/query?q=weather");
const data = await client.post("https://agent-api.example.com/task", { prompt: "hello" });

// Check spending
console.log(client.budget());
// { date: "2026-02-11", spent: "$0.520000", remaining: "$4.480000", limit: "$5.000000", transactions: 3 }
```

### Payment Flow Details

1. **Request** — Standard HTTP request to any URL
2. **402 Detection** — Server returns `HTTP 402` with `PAYMENT-REQUIRED` header containing JSON payment requirements
3. **Budget Check** — Verifies amount against per-request max ($1.00 default) and daily limit ($10.00 default)
4. **EIP-712 Signing** — Signs a `TransferWithAuthorization` (EIP-3009) for USDC on Base using the agent's wallet
5. **Retry** — Resends the request with `PAYMENT-SIGNATURE` header containing the signed payment payload
6. **Settlement** — The Coinbase facilitator verifies the signature and settles the USDC transfer
7. **Response** — Server returns the requested resource

### Security

- **Private key from 1Password** at runtime (never on disk) — follows Bagman patterns
- **Budget controls** prevent runaway spending: $1/request max, $10/day by default
- **Dry-run mode** for testing without signing or spending
- **USDC on Base only** — no other chains or tokens (EIP-3009 TransferWithAuthorization)
- **Daily budget tracking** persisted to `.x402-budget.json` (amounts only, no keys)

### Key Addresses

| Item | Address |
|------|---------|
| USDC (Base) | `0x833589fCD6eDb6E08f4c7C32D4f71b54bdA02913` |
| Coinbase Facilitator | `https://api.cdp.coinbase.com/platform/v2/x402` |
| Base Chain ID | `8453` (CAIP-2: `eip155:8453`) |

---

## 18. ERC-8004 Agent Registry (v0.7)

The [ERC-8004](https://eips.ethereum.org/EIPS/eip-8004) protocol provides on-chain registries for agent discovery and trust. Everclaw v0.7 includes a reader that queries the Identity and Reputation registries on Base mainnet.

### What Is ERC-8004?

ERC-8004 defines three registries:

- **Identity Registry** (ERC-721): Each agent is an NFT with a `tokenURI` pointing to a registration file containing name, description, services/endpoints, x402 support, and trust signals
- **Reputation Registry**: Clients give structured feedback (value + tags) to agents. Summary scores aggregate across all clients
- **Validation Registry**: Stake-secured re-execution and zkML verification (read-only in Everclaw)

Agents are discoverable, portable (transferable NFTs), and verifiable across organizational boundaries.

### CLI Usage

```bash
# Look up an agent by ID
node scripts/agent-registry.mjs lookup 1

# Get reputation data
node scripts/agent-registry.mjs reputation 1

# Full discovery (identity + registration file + reputation)
node scripts/agent-registry.mjs discover 1

# List agents in a range
node scripts/agent-registry.mjs list 1 10

# Get total registered agents
node scripts/agent-registry.mjs total
```

### Programmatic Usage

```javascript
import { lookupAgent, getReputation, discoverAgent, totalAgents, listAgents } from './scripts/agent-registry.mjs';

// Look up identity
const agent = await lookupAgent(1);
// {
//   agentId: 1,
//   owner: "0x89E9...",
//   uri: "data:application/json;base64,...",
//   wallet: "0x89E9...",
//   registration: {
//     name: "ClawNews",
//     description: "Hacker News for AI agents...",
//     services: [{ name: "web", endpoint: "https://clawnews.io" }, ...],
//     x402Support: false,
//     active: true,
//     supportedTrust: ["reputation"]
//   }
// }

// Get reputation
const rep = await getReputation(1);
// {
//   agentId: 1,
//   clients: ["0x3975...", "0x718B..."],
//   feedbackCount: 2,
//   summary: { count: 2, value: "100", decimals: 0 },
//   feedback: [{ client: "0x3975...", value: "100", tag1: "tip", tag2: "agent" }, ...]
// }

// Full discovery
const full = await discoverAgent(1);
// Combines identity, registration file, services, and reputation into one object
```

### Registration File Format

Agent registration files (resolved from `tokenURI`) follow the ERC-8004 standard:

```json
{
  "type": "https://eips.ethereum.org/EIPS/eip-8004#registration-v1",
  "name": "MyAgent",
  "description": "What the agent does",
  "image": "https://example.com/logo.png",
  "services": [
    { "name": "web", "endpoint": "https://myagent.com" },
    { "name": "A2A", "endpoint": "https://agent.example/.well-known/agent-card.json", "version": "0.3.0" },
    { "name": "MCP", "endpoint": "https://mcp.agent.eth/", "version": "2025-06-18" }
  ],
  "x402Support": true,
  "active": true,
  "supportedTrust": ["reputation", "crypto-economic"]
}
```

The reader handles all URI types: `data:` URIs (base64-encoded JSON stored on-chain), `ipfs://` URIs (via public IPFS gateway), and `https://` URIs.

### Contract Addresses (Base Mainnet)

| Registry | Address |
|----------|---------|
| Identity | `0x8004A169FB4a3325136EB29fA0ceB6D2e539a432` |
| Reputation | `0x8004BAa17C55a88189AE136b182e5fdA19dE9b63` |

⚠️ **Same addresses on all EVM chains** — Ethereum, Base, Arbitrum, Polygon, Optimism, Linea, Avalanche, etc. The Identity Registry does NOT implement `totalSupply()`, so `totalAgents()` uses a binary search via `ownerOf()`.

### Combining x402 + Agent Registry

The x402 client and agent registry work together for agent-to-agent payments:

```javascript
import { discoverAgent } from './scripts/agent-registry.mjs';
import { makePayableRequest } from './scripts/x402-client.mjs';

// 1. Discover an agent and find its x402-enabled endpoint
const agent = await discoverAgent(42);
const apiEndpoint = agent.services.find(s => s.name === "A2A")?.endpoint;

// 2. Make a paid request — x402 handling is automatic
if (agent.x402Support && apiEndpoint) {
  const result = await makePayableRequest(apiEndpoint, {
    method: "POST",
    headers: { "Content-Type": "application/json" },
    body: JSON.stringify({ task: "Analyze this data..." }),
    maxAmount: 500000n, // $0.50 USDC
  });
  console.log(result.body); // Agent's response
}
```

---

## Quick Reference (v0.9.4)

| Action | Command |
|--------|---------|
| Install Everclaw | `bash skills/everclaw/scripts/install-everclaw.sh` |
| Check for updates | `bash skills/everclaw/scripts/install-everclaw.sh --check` |
| Update (git pull) | `cd skills/everclaw && git pull` |
| Install router | `bash skills/everclaw/scripts/install.sh` |
| Install proxy + guardian | `bash skills/everclaw/scripts/install-proxy.sh` |
| Start router | `bash skills/everclaw/scripts/start.sh` |
| Stop router | `bash skills/everclaw/scripts/stop.sh` |
| Swap ETH→MOR | `bash skills/everclaw/scripts/swap.sh eth 0.01` |
| Swap USDC→MOR | `bash skills/everclaw/scripts/swap.sh usdc 50` |
| Open session | `bash skills/everclaw/scripts/session.sh open <model> [duration]` |
| Close session | `bash skills/everclaw/scripts/session.sh close <session_id>` |
| List sessions | `bash skills/everclaw/scripts/session.sh list` |
| Send prompt | `bash skills/everclaw/scripts/chat.sh <model> "prompt"` |
| Check balance | `bash skills/everclaw/scripts/balance.sh` |
| Proxy health | `curl http://127.0.0.1:8083/health` |
| Guardian test | `bash scripts/gateway-guardian.sh --verbose` |
| Guardian logs | `tail -f ~/.openclaw/logs/guardian.log` |
| Archive sessions | `bash skills/everclaw/scripts/session-archive.sh` |
| Check session size | `bash skills/everclaw/scripts/session-archive.sh --check` |
| Force archive | `bash skills/everclaw/scripts/session-archive.sh --force` |
| x402 request | `node scripts/x402-client.mjs GET <url>` |
| x402 dry-run | `node scripts/x402-client.mjs --dry-run GET <url>` |
| x402 budget | `node scripts/x402-client.mjs --budget` |
| Lookup agent | `node scripts/agent-registry.mjs lookup <id>` |
| Agent reputation | `node scripts/agent-registry.mjs reputation <id>` |
| Discover agent | `node scripts/agent-registry.mjs discover <id>` |
| List agents | `node scripts/agent-registry.mjs list <start> [count]` |
| Total agents | `node scripts/agent-registry.mjs total` |
| Scan a skill | `node security/skillguard/src/cli.js scan <path>` |
| Batch scan | `node security/skillguard/src/cli.js batch <dir>` |
| Security audit | `bash security/clawdstrike/scripts/collect_verified.sh` |
| Detect injection | `python3 security/prompt-guard/scripts/detect.py "text"` |

---

## 15. Security Skills (v0.3)

Everclaw agents handle MOR tokens and private keys — making them high-value targets. v0.3 bundles four security skills to defend against supply chain attacks, prompt injection, credential theft, and configuration exposure.

### 🔍 SkillGuard — Pre-Install Skill Scanner

Scans AgentSkill packages for malicious patterns before you install them. Detects credential theft, code injection, prompt manipulation, data exfiltration, and evasion techniques.

```bash
# Scan a skill directory
node security/skillguard/src/cli.js scan <path>

# Batch scan all installed skills
node security/skillguard/src/cli.js batch <directory>

# Scan a ClawHub skill by slug
node security/skillguard/src/cli.js scan-hub <slug>
```

**Score interpretation:**
- 80-100 ✅ LOW risk — safe to install
- 50-79 ⚠️ MEDIUM — review before installing
- 20-49 🟠 HIGH — significant concerns
- 0-19 🔴 CRITICAL — do NOT install

**When to use:** Before installing any skill from ClawHub or untrusted sources. Run batch scans periodically to audit all installed skills.

Full docs: `security/skillguard/SKILL.md`

### 🔒 ClawdStrike — Config & Exposure Audits

Security audit and threat model for OpenClaw gateway hosts. Verifies configuration, network exposure, installed skills/plugins, and filesystem hygiene. Produces an OK/VULNERABLE report with evidence and remediation steps.

```bash
# Run a full audit
cd security/clawdstrike && \
  OPENCLAW_WORKSPACE_DIR=$HOME/.openclaw/workspace \
  bash scripts/collect_verified.sh
```

**What it checks:**
- Gateway bind address and auth configuration
- Channel exposure (Signal, Telegram, Discord, etc.)
- Installed skills and plugins for known vulnerabilities
- Filesystem permissions and sensitive file access
- Network exposure and firewall rules
- OpenClaw version and known CVEs

**When to use:** After initial setup, after installing new skills, and periodically (weekly recommended).

Full docs: `security/clawdstrike/SKILL.md`

### 🧱 PromptGuard — Prompt Injection Defense

Advanced prompt injection defense system with multi-language detection (EN/KO/JA/ZH), severity scoring, automatic logging, and configurable security policies. Connects to the HiveFence distributed threat intelligence network.

```bash
# Analyze a message for injection attempts
python3 security/prompt-guard/scripts/detect.py "suspicious message here"

# Run audit on prompt injection logs
python3 security/prompt-guard/scripts/audit.py

# Analyze historical logs
python3 security/prompt-guard/scripts/analyze_log.py
```

**Detection categories:**
- Direct injection (instruction overrides, role manipulation)
- Indirect injection (data exfiltration, hidden instructions)
- Jailbreak attempts (DAN mode, filter bypasses)
- Multi-language attacks (cross-language injection)

**When to use:** In group chats, when processing untrusted input, when agents interact with external data sources.

Full docs: `security/prompt-guard/SKILL.md`

### 💰 Bagman — Secure Key Management

Secure key management for AI agents handling private keys, API secrets, and wallet credentials. Covers secure storage patterns, session keys, leak prevention, prompt injection defense specific to financial operations, and MetaMask Delegation Framework (EIP-7710) integration.

**Key principles:**
- **Never store keys on disk** — use 1Password `op run` for runtime injection
- **Session keys** — generate ephemeral keys with limited permissions
- **Delegation Framework** — grant agents scoped authority without exposing master keys
- **Leak prevention** — patterns to detect and block secret exposure

**Reference docs:**
- `security/bagman/references/secure-storage.md` — Storage patterns
- `security/bagman/references/session-keys.md` — Session key architecture
- `security/bagman/references/delegation-framework.md` — EIP-7710 integration
- `security/bagman/references/leak-prevention.md` — Leak detection rules
- `security/bagman/references/prompt-injection-defense.md` — Financial-specific injection defense

**When to use:** Whenever an agent handles private keys, wallet credentials, or API secrets — which Everclaw agents always do.

Full docs: `security/bagman/SKILL.md`

### Security Recommendations

For Everclaw agents handling MOR tokens:

1. **Before installing any new skill:** Run SkillGuard scan
2. **After setup and periodically:** Run ClawdStrike audit
3. **In group chats or with untrusted input:** Enable PromptGuard detection
4. **Always:** Follow Bagman patterns for key management (1Password, session keys, no keys on disk)

---

## 16. Model Router (v0.6)

A lightweight, local prompt classifier that routes requests to the cheapest capable model. Runs in <1ms with zero external API calls.

### Tiers

| Tier | Primary Model | Fallback | Use Case |
|------|--------------|----------|----------|
| **LIGHT** | `morpheus/glm-4.7-flash` | `morpheus/kimi-k2.5` | Cron jobs, heartbeats, simple Q&A, status checks |
| **STANDARD** | `morpheus/kimi-k2.5` | `venice/kimi-k2-5` | Research, drafting, summaries, most sub-agent tasks |
| **HEAVY** | `venice/claude-opus-4-6` | `venice/claude-opus-45` | Complex reasoning, architecture, formal proofs, strategy |

All LIGHT and STANDARD tier models run through Morpheus (inference you own via staked MOR). Only HEAVY tier uses Venice (premium).

### How Scoring Works

The router scores prompts across 13 weighted dimensions:

| Dimension | Weight | What It Detects |
|-----------|--------|----------------|
| `reasoningMarkers` | 0.20 | "prove", "theorem", "step by step", "chain of thought" |
| `codePresence` | 0.14 | `function`, `class`, `import`, backticks, "refactor" |
| `synthesis` | 0.11 | "summarize", "compare", "draft", "analyze", "review" |
| `technicalTerms` | 0.10 | "algorithm", "architecture", "smart contract", "consensus" |
| `multiStepPatterns` | 0.10 | "first...then", "step 1", numbered lists |
| `simpleIndicators` | 0.08 | "what is", "hello", "weather" (negative score → pushes toward LIGHT) |
| `agenticTask` | 0.06 | "edit", "deploy", "install", "debug", "fix" |
| `creativeMarkers` | 0.04 | "story", "poem", "brainstorm" |
| `questionComplexity` | 0.04 | Multiple question marks |
| `tokenCount` | 0.04 | Short prompts skew LIGHT, long prompts skew HEAVY |
| `constraintCount` | 0.04 | "at most", "at least", "maximum", "budget" |
| `domainSpecificity` | 0.04 | "quantum", "zero-knowledge", "genomics" |
| `outputFormat` | 0.03 | "json", "yaml", "table", "csv" |

**Special override:** 2+ reasoning keywords in the user prompt → force HEAVY at 88%+ confidence. This prevents accidental cheap routing of genuinely hard problems.

**Ambiguous prompts** (low confidence) default to STANDARD — the safe middle ground.

### CLI Usage

```bash
# Test routing for a prompt
node scripts/router.mjs "What is 2+2?"
# → LIGHT (morpheus/glm-4.7-flash)

node scripts/router.mjs "Summarize the meeting notes and draft a follow-up"
# → STANDARD (morpheus/kimi-k2.5)

node scripts/router.mjs "Design a distributed consensus algorithm and prove its correctness"
# → HEAVY (venice/claude-opus-4-6)

# JSON output for programmatic use
node scripts/router.mjs --json "Build a React component"

# Pipe from stdin
echo '{"prompt":"hello","system":"You are helpful"}' | node scripts/router.mjs --stdin
```

### Programmatic Usage

```javascript
import { route, classify } from './scripts/router.mjs';

const decision = route("Check the weather in Austin");
// {
//   tier: "LIGHT",
//   model: "morpheus/glm-4.7-flash",
//   fallback: "morpheus/kimi-k2.5",
//   confidence: 0.87,
//   score: -0.10,
//   signals: ["short (7 tok)", "simple (weather)"],
//   reasoning: "score=-0.100 → LIGHT"
// }
```

### Applying to Cron Jobs

Set the `model` field on cron job payloads to route to cheaper models:

```json5
{
  "payload": {
    "kind": "agentTurn",
    "model": "morpheus/kimi-k2.5",   // STANDARD tier — owned via Morpheus
    "message": "Compile a morning briefing...",
    "timeoutSeconds": 300
  }
}
```

For truly simple cron jobs (health checks, pings, status queries):

```json5
{
  "payload": {
    "kind": "agentTurn",
    "model": "morpheus/glm-4.7-flash",  // LIGHT tier — fastest, owned
    "message": "Check proxy health and report any issues",
    "timeoutSeconds": 60
  }
}
```

### Applying to Sub-Agent Spawns

```javascript
// Simple research task → STANDARD
sessions_spawn({ task: "Search for X news", model: "morpheus/kimi-k2.5" });

// Quick lookup → LIGHT
sessions_spawn({ task: "What's the weather?", model: "morpheus/glm-4.7-flash" });

// Complex analysis → let it use the default (HEAVY / Claude 4.6)
sessions_spawn({ task: "Design the x402 payment integration..." });
```

### Cost Impact

With the router in place, only complex reasoning tasks in the main session use premium models. All background work (cron jobs, sub-agents, heartbeats) runs on Morpheus inference you own:

| Before | After |
|--------|-------|
| All cron jobs → Claude 4.6 (premium) | Cron jobs → Kimi K2.5 / GLM Flash (owned) |
| All sub-agents → Claude 4.6 (premium) | Sub-agents → Kimi K2.5 (owned) unless complex |
| Main session → Claude 4.6 | Main session → Claude 4.6 (unchanged) |

---

## 19. Morpheus API Gateway Bootstrap (v0.8)

The Morpheus API Gateway (`api.mor.org`) provides community-powered, OpenAI-compatible inference — no node, no staking, no wallet required. Everclaw v0.8 includes a bootstrap script that configures this as an OpenClaw provider, giving new users **instant access to AI from the first launch**.

### Why This Matters

New OpenClaw users face a cold-start problem: they need an API key (Claude, OpenAI, etc.) before their agent can do anything. Everclaw v0.8 solves this by bundling a community API key for the Morpheus inference marketplace, which is currently in open beta.

**The bootstrap flow:**
1. New user installs OpenClaw + Everclaw
2. Run `node scripts/bootstrap-gateway.mjs` — agent gets inference immediately
3. Agent's first task: guide user to get their own key at `app.mor.org`
4. User upgrades to their own key → can then progress to full Morpheus node + MOR staking

### Quick Start

```bash
# One command — tests the gateway and patches OpenClaw config
node skills/everclaw/scripts/bootstrap-gateway.mjs

# Or with your own API key from app.mor.org
node skills/everclaw/scripts/bootstrap-gateway.mjs --key sk-YOUR_KEY_HERE

# Test the gateway connection
node skills/everclaw/scripts/bootstrap-gateway.mjs --test

# Check current gateway status
node skills/everclaw/scripts/bootstrap-gateway.mjs --status
```

### What It Does

The bootstrap script:

1. **Tests** the Morpheus API Gateway connection with a live inference call
2. **Patches** `openclaw.json` to add `mor-gateway` as a new provider
3. **Adds** `mor-gateway/kimi-k2.5` to the fallback chain
4. **Reports** available models and next steps

### API Gateway Details

| Setting | Value |
|---------|-------|
| Base URL | `https://api.mor.org/api/v1` |
| API format | OpenAI-compatible |
| Auth | Bearer token (`sk-...`) |
| Open beta | Until March 1, 2026 |
| Models | 34 (LLMs, TTS, STT, embeddings) |
| Provider name | `mor-gateway` |

### Available Models (via Gateway)

The gateway exposes all models on the Morpheus inference marketplace:

| Model | Type | Notes |
|-------|------|-------|
| `kimi-k2.5` | LLM | Primary bootstrap model — strong coding + reasoning |
| `glm-4.7-flash` | LLM | Fast, good for simple tasks |
| `llama-3.3-70b` | LLM | General purpose |
| `qwen3-235b` | LLM | Large, strong reasoning |
| `gpt-oss-120b` | LLM | OpenAI-compatible OSS model |
| `hermes-4-14b` | LLM | Lightweight |
| `tts-kokoro` | TTS | Text-to-speech |
| `whisper-v3-large-turbo` | STT | Speech-to-text |
| `text-embedding-bge-m3` | Embedding | Text embeddings |

All models also have `:web` variants with web search capability.

### OpenClaw Config (generated by bootstrap)

```json5
{
  "models": {
    "providers": {
      "mor-gateway": {
        "baseUrl": "https://api.mor.org/api/v1",
        "apiKey": "sk-...",
        "api": "openai-completions",
        "models": [
          { "id": "kimi-k2.5", "name": "Kimi K2.5 (via Morpheus Gateway)", "reasoning": false },
          { "id": "glm-4.7-flash", "name": "GLM 4.7 Flash (via Morpheus Gateway)", "reasoning": false },
          { "id": "llama-3.3-70b", "name": "Llama 3.3 70B (via Morpheus Gateway)", "reasoning": false }
        ]
      }
    }
  }
}
```

**Important:** All gateway models must have `"reasoning": false` — the upstream litellm rejects the `reasoning_effort` parameter.

### Community Bootstrap Key

The bootstrap script includes a community API key (base64-obfuscated) for the SmartAgentProtocol account. This provides open access during the beta period.

**Getting your own key (recommended):**
1. Go to [app.mor.org](https://app.mor.org)
2. Create an account and sign in
3. Click "Create API Key"
4. **Enable "session automation"** in account settings (required for API access)
5. Run: `node scripts/bootstrap-gateway.mjs --key YOUR_KEY`

### Gateway vs Local Proxy vs P2P Node

| Feature | API Gateway (v0.8) | Local Proxy (v0.2) | P2P Node (v0.1) |
|---------|-------------------|-------------------|-----------------|
| Setup | One command | Install proxy + config | Full node install |
| Cost | Open (beta) | Own (MOR staking) | Own (MOR staking) |
| Requires MOR | No | Yes | Yes |
| Requires wallet | No | Yes | Yes |
| Decentralized | Gateway → providers | Direct P2P | Direct P2P |
| Best for | New users, quick start | Daily use, reliability | Full sovereignty |

The recommended progression: **Gateway → Local Proxy → P2P Node** as users gain confidence with the Morpheus ecosystem.

### Fallback Chain with Gateway

With the gateway added, the recommended fallback chain becomes:

```
venice/claude-opus-4-6      # Primary (premium)
  → venice/claude-opus-45   # Venice fallback
  → venice/kimi-k2-5        # Venice open tier
  → morpheus/kimi-k2.5      # Local proxy (MOR staking)
  → mor-gateway/kimi-k2.5   # API Gateway (open beta)
```

For new users without Venice or a local proxy, the gateway is the **first and only** provider — making it the critical bootstrap path.

---

## References

- `references/acquiring-mor.md` — How to get MOR tokens (exchanges, bridges, swaps)
- `references/models.md` — Available models and their blockchain IDs
- `references/api.md` — Complete proxy-router API reference
- `references/economics.md` — How MOR staking economics work
- `references/troubleshooting.md` — Common errors and solutions
- `security/skillguard/SKILL.md` — SkillGuard full documentation
- `security/clawdstrike/SKILL.md` — ClawdStrike full documentation
- `security/prompt-guard/SKILL.md` — PromptGuard full documentation
- `security/bagman/SKILL.md` — Bagman full documentation
- [x402 Protocol](https://x402.org) — HTTP-native payment protocol specification
- [ERC-8004](https://eips.ethereum.org/EIPS/eip-8004) — Trustless Agents EIP specification
- [8004scan](https://www.8004scan.io) — Agent registry explorer
