---
name: eventbrite
description: |
  Eventbrite API integration with managed OAuth. Manage events, venues, ticket classes, orders, and attendees.
  Use this skill when users want to create and manage events, check orders, view attendees, or access event categories.
  For other third party apps, use the api-gateway skill (https://clawhub.ai/byungkyu/api-gateway).
  Requires network access and valid Maton API key.
metadata:
  author: maton
  version: "1.0"
  clawdbot:
    emoji: 🧠
    requires:
      env:
        - MATON_API_KEY
---

# Eventbrite

Access the Eventbrite API with managed OAuth authentication. Manage events, venues, ticket classes, orders, attendees, and more.

## Quick Start

```bash
# Get current user
python <<'EOF'
import urllib.request, os, json
req = urllib.request.Request('https://gateway.maton.ai/eventbrite/v3/users/me/')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

## Base URL

```
https://gateway.maton.ai/eventbrite/{native-api-path}
```

Replace `{native-api-path}` with the actual Eventbrite API endpoint path. The gateway proxies requests to `www.eventbriteapi.com` and automatically injects your OAuth token.

## Authentication

All requests require the Maton API key in the Authorization header:

```
Authorization: Bearer $MATON_API_KEY
```

**Environment Variable:** Set your API key as `MATON_API_KEY`:

```bash
export MATON_API_KEY="YOUR_API_KEY"
```

### Getting Your API Key

1. Sign in or create an account at [maton.ai](https://maton.ai)
2. Go to [maton.ai/settings](https://maton.ai/settings)
3. Copy your API key

## Connection Management

Manage your Eventbrite OAuth connections at `https://ctrl.maton.ai`.

### List Connections

```bash
python <<'EOF'
import urllib.request, os, json
req = urllib.request.Request('https://ctrl.maton.ai/connections?app=eventbrite&status=ACTIVE')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

### Create Connection

```bash
python <<'EOF'
import urllib.request, os, json
data = json.dumps({'app': 'eventbrite'}).encode()
req = urllib.request.Request('https://ctrl.maton.ai/connections', data=data, method='POST')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
req.add_header('Content-Type', 'application/json')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

### Get Connection

```bash
python <<'EOF'
import urllib.request, os, json
req = urllib.request.Request('https://ctrl.maton.ai/connections/{connection_id}')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

**Response:**
```json
{
  "connection": {
    "connection_id": "a2dd9063-64b4-4fe2-b4c5-8dd711648244",
    "status": "ACTIVE",
    "creation_time": "2026-02-07T09:11:20.516013Z",
    "last_updated_time": "2026-02-07T09:14:35.273822Z",
    "url": "https://connect.maton.ai/?session_token=...",
    "app": "eventbrite",
    "metadata": {}
  }
}
```

Open the returned `url` in a browser to complete OAuth authorization.

### Delete Connection

```bash
python <<'EOF'
import urllib.request, os, json
req = urllib.request.Request('https://ctrl.maton.ai/connections/{connection_id}', method='DELETE')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

### Specifying Connection

If you have multiple Eventbrite connections, specify which one to use with the `Maton-Connection` header:

```bash
python <<'EOF'
import urllib.request, os, json
req = urllib.request.Request('https://gateway.maton.ai/eventbrite/v3/users/me/')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
req.add_header('Maton-Connection', 'a2dd9063-64b4-4fe2-b4c5-8dd711648244')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

If omitted, the gateway uses the default (oldest) active connection.

## API Reference

### User Operations

#### Get Current User

```bash
GET /eventbrite/v3/users/me/
```

**Response:**
```json
{
  "emails": [{"email": "user@example.com", "verified": true, "primary": true}],
  "id": "1234567890",
  "name": "John Doe",
  "first_name": "John",
  "last_name": "Doe",
  "is_public": false,
  "image_id": null
}
```

#### List User Organizations

```bash
GET /eventbrite/v3/users/me/organizations/
```

#### List User Orders

```bash
GET /eventbrite/v3/users/me/orders/
```

### Organization Operations

#### List Organization Events

```bash
GET /eventbrite/v3/organizations/{organization_id}/events/
```

Query parameters:
- `status` - Filter by status: `draft`, `live`, `started`, `ended`, `completed`, `canceled`
- `order_by` - Sort order: `start_asc`, `start_desc`, `created_asc`, `created_desc`
- `time_filter` - Filter by time: `current_future`, `past`

#### List Organization Venues

```bash
GET /eventbrite/v3/organizations/{organization_id}/venues/
```

#### Create Venue

```bash
POST /eventbrite/v3/organizations/{organization_id}/venues/
Content-Type: application/json

{
  "venue": {
    "name": "Conference Center",
    "address": {
      "address_1": "123 Main St",
      "city": "San Francisco",
      "region": "CA",
      "postal_code": "94105",
      "country": "US"
    }
  }
}
```

### Event Operations

#### Get Event

```bash
GET /eventbrite/v3/events/{event_id}/
```

#### Create Event

Events must be created under an organization:

```bash
POST /eventbrite/v3/organizations/{organization_id}/events/
Content-Type: application/json

{
  "event": {
    "name": {"html": "My Event"},
    "description": {"html": "<p>Event description</p>"},
    "start": {
      "timezone": "America/Los_Angeles",
      "utc": "2026-03-01T19:00:00Z"
    },
    "end": {
      "timezone": "America/Los_Angeles",
      "utc": "2026-03-01T22:00:00Z"
    },
    "currency": "USD",
    "online_event": false,
    "listed": true,
    "shareable": true,
    "capacity": 100,
    "category_id": "103",
    "format_id": "1"
  }
}
```

#### Update Event

```bash
POST /eventbrite/v3/events/{event_id}/
Content-Type: application/json

{
  "event": {
    "name": {"html": "Updated Event Name"},
    "capacity": 200
  }
}
```

#### Publish Event

```bash
POST /eventbrite/v3/events/{event_id}/publish/
```

#### Unpublish Event

```bash
POST /eventbrite/v3/events/{event_id}/unpublish/
```

#### Cancel Event

```bash
POST /eventbrite/v3/events/{event_id}/cancel/
```

#### Delete Event

```bash
DELETE /eventbrite/v3/events/{event_id}/
```

### Ticket Class Operations

#### List Ticket Classes

```bash
GET /eventbrite/v3/events/{event_id}/ticket_classes/
```

#### Create Ticket Class

```bash
POST /eventbrite/v3/events/{event_id}/ticket_classes/
Content-Type: application/json

{
  "ticket_class": {
    "name": "General Admission",
    "description": "Standard entry ticket",
    "quantity_total": 100,
    "cost": "USD,2500",
    "sales_start": "2026-01-01T00:00:00Z",
    "sales_end": "2026-02-28T23:59:59Z",
    "minimum_quantity": 1,
    "maximum_quantity": 10
  }
}
```

For free tickets, omit the `cost` field or set `free: true`.

#### Update Ticket Class

```bash
POST /eventbrite/v3/events/{event_id}/ticket_classes/{ticket_class_id}/
Content-Type: application/json

{
  "ticket_class": {
    "quantity_total": 150
  }
}
```

#### Delete Ticket Class

```bash
DELETE /eventbrite/v3/events/{event_id}/ticket_classes/{ticket_class_id}/
```

### Attendee Operations

#### List Event Attendees

```bash
GET /eventbrite/v3/events/{event_id}/attendees/
```

Query parameters:
- `status` - Filter by status: `attending`, `not_attending`, `unpaid`
- `changed_since` - ISO 8601 timestamp to get attendees changed after

#### Get Attendee

```bash
GET /eventbrite/v3/events/{event_id}/attendees/{attendee_id}/
```

### Order Operations

#### List Event Orders

```bash
GET /eventbrite/v3/events/{event_id}/orders/
```

Query parameters:
- `status` - Filter by status: `active`, `inactive`, `all`
- `changed_since` - ISO 8601 timestamp

#### Get Order

```bash
GET /eventbrite/v3/orders/{order_id}/
```

### Venue Operations

#### Get Venue

```bash
GET /eventbrite/v3/venues/{venue_id}/
```

#### Update Venue

```bash
POST /eventbrite/v3/venues/{venue_id}/
Content-Type: application/json

{
  "venue": {
    "name": "Updated Venue Name"
  }
}
```

### Reference Data

#### List Categories

```bash
GET /eventbrite/v3/categories/
```

**Response:**
```json
{
  "locale": "en_US",
  "pagination": {"object_count": 21, "page_number": 1, "page_size": 50},
  "categories": [
    {"id": "103", "name": "Music", "short_name": "Music"},
    {"id": "101", "name": "Business & Professional", "short_name": "Business"},
    {"id": "110", "name": "Food & Drink", "short_name": "Food & Drink"}
  ]
}
```

#### Get Category

```bash
GET /eventbrite/v3/categories/{category_id}/
```

#### List Subcategories

```bash
GET /eventbrite/v3/subcategories/
```

#### List Formats

```bash
GET /eventbrite/v3/formats/
```

**Common formats:**
- `1` - Conference
- `2` - Seminar or Talk
- `5` - Festival or Fair
- `6` - Concert or Performance
- `9` - Class, Training, or Workshop
- `10` - Meeting or Networking Event
- `11` - Party or Social Gathering

#### List Countries

```bash
GET /eventbrite/v3/system/countries/
```

#### List Regions

```bash
GET /eventbrite/v3/system/regions/
```

## Pagination

Eventbrite uses a combination of page-based and continuation-based pagination:

```bash
GET /eventbrite/v3/organizations/{org_id}/events/?page_size=50
```

**Response:**
```json
{
  "pagination": {
    "object_count": 150,
    "page_number": 1,
    "page_size": 50,
    "page_count": 3,
    "has_more_items": true,
    "continuation": "eyJwYWdlIjogMn0"
  },
  "events": [...]
}
```

For subsequent pages, use the `continuation` token:

```bash
GET /eventbrite/v3/organizations/{org_id}/events/?continuation=eyJwYWdlIjogMn0
```

## Expansions

Include related data by using the `expand` parameter:

```bash
GET /eventbrite/v3/events/{event_id}/?expand=venue,ticket_classes,category
```

Common expansions:
- `venue` - Include venue details
- `ticket_classes` - Include ticket information
- `category` - Include category details
- `subcategory` - Include subcategory details
- `format` - Include format details
- `organizer` - Include organizer information

## Code Examples

### JavaScript

```javascript
const response = await fetch(
  'https://gateway.maton.ai/eventbrite/v3/users/me/',
  {
    headers: {
      'Authorization': `Bearer ${process.env.MATON_API_KEY}`
    }
  }
);
const user = await response.json();
```

### Python

```python
import os
import requests

response = requests.get(
    'https://gateway.maton.ai/eventbrite/v3/users/me/',
    headers={'Authorization': f'Bearer {os.environ["MATON_API_KEY"]}'}
)
user = response.json()
```

## Notes

- All endpoint paths should end with a trailing slash (`/`)
- Event creation requires an organization - use organization-based endpoints
- Legacy user-based event endpoints are deprecated; use organization equivalents
- Timestamps are in ISO 8601 format (UTC)
- Currency amounts are in minor units (cents) - e.g., "USD,2500" = $25.00
- Rate limit: 1,000 calls per hour, 48,000 calls per day
- Event Search API is no longer publicly available (deprecated February 2020)
- IMPORTANT: When using curl commands, use `curl -g` when URLs contain brackets to disable glob parsing
- IMPORTANT: When piping curl output to `jq` or other commands, environment variables like `$MATON_API_KEY` may not expand correctly in some shell environments

## Error Handling

| Status | Meaning |
|--------|---------|
| 400 | Missing Eventbrite connection or invalid arguments |
| 401 | Invalid or missing Maton API key |
| 403 | Not authorized (check scopes or use organization endpoints) |
| 404 | Resource not found |
| 429 | Rate limited |
| 4xx/5xx | Passthrough error from Eventbrite API |

### Common Errors

**NOT_AUTHORIZED with legacy user endpoints:**
```json
{"status_code": 403, "error": "NOT_AUTHORIZED", "error_description": "This user is not able to use legacy user endpoints, please use the organization equivalent."}
```
Solution: Use `/organizations/{org_id}/events/` instead of `/users/me/owned_events/`

### Troubleshooting: API Key Issues

1. Check that the `MATON_API_KEY` environment variable is set:

```bash
echo $MATON_API_KEY
```

2. Verify the API key is valid by listing connections:

```bash
python <<'EOF'
import urllib.request, os, json
req = urllib.request.Request('https://ctrl.maton.ai/connections')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

### Troubleshooting: Invalid App Name

1. Ensure your URL path starts with `eventbrite`. For example:

- Correct: `https://gateway.maton.ai/eventbrite/v3/users/me/`
- Incorrect: `https://gateway.maton.ai/v3/users/me/`

## Resources

- [Eventbrite API Documentation](https://www.eventbrite.com/platform/api)
- [API Basics](https://www.eventbrite.com/platform/docs/api-basics)
- [API Explorer](https://www.eventbrite.com/platform/docs/api-explorer)
- [Maton Community](https://discord.com/invite/dBfFAcefs2)
- [Maton Support](mailto:support@maton.ai)
