"""
Telegram notification system for trading alerts
"""
import logging
import json
import os
from datetime import datetime
import requests

logger = logging.getLogger(__name__)

class TelegramNotifier:
    """Sends Telegram notifications for trading activities"""
    
    def __init__(self, config):
        self.config = config.get('telegram', {})
        self.enabled = self.config.get('enabled', False)
        self.bot_token = self.config.get('botToken', '')
        self.chat_id = self.config.get('chatId', '')
        
        if self.enabled and (not self.bot_token or not self.chat_id):
            logger.warning("Telegram enabled but token or chat ID missing")
            self.enabled = False
        
        logger.info(f"Telegram notifier initialized: {'ENABLED' if self.enabled else 'DISABLED'}")
    
    def send_message(self, message, parse_mode='Markdown'):
        """Send a message to Telegram"""
        if not self.enabled:
            return False
        
        try:
            url = f"https://api.telegram.org/bot{self.bot_token}/sendMessage"
            payload = {
                "chat_id": self.chat_id,
                "text": message,
                "parse_mode": parse_mode,
                "disable_web_page_preview": True
            }
            
            response = requests.post(url, json=payload, timeout=10)
            
            if response.status_code == 200:
                logger.debug("Telegram message sent successfully")
                return True
            else:
                logger.error(f"Telegram API error: {response.status_code} - {response.text}")
                return False
                
        except Exception as e:
            logger.error(f"Error sending Telegram message: {e}")
            return False
    
    def send_trade_alert(self, trade_action, details):
        """Send trade execution alert"""
        if not self.config.get('sendTradeAlerts', True):
            return False
        
        symbol = details.get('symbol', 'Unknown')
        action = trade_action.upper()
        quantity = details.get('quantity', 0)
        price = details.get('price', 0)
        total = details.get('total', 0)
        reason = details.get('reason', '')
        
        message = f"""
🚀 *TRADE EXECUTED* 🚀

*Action:* {action} {symbol}
*Quantity:* {quantity} shares
*Price:* ${price:.2f}
*Total:* ${total:.2f}
*Reason:* {reason}

*Time:* {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}
"""
        
        return self.send_message(message.strip())
    
    def send_congressional_alert(self, trade):
        """Send congressional trade alert"""
        politician = trade.get('politician', 'Unknown')
        symbol = trade.get('ticker', 'Unknown')
        action = trade.get('transaction_type', '').upper()
        amount = trade.get('amount', 0)
        date = trade.get('transaction_date', '')
        
        message = f"""
📊 *CONGRESSIONAL TRADE ALERT* 📊

*Politician:* {politician}
*Action:* {action} {symbol}
*Amount:* ${amount:,.2f}
*Date:* {date}

*Time Detected:* {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}
"""
        
        return self.send_message(message.strip())
    
    def send_error_alert(self, error_type, error_message, context=""):
        """Send error alert"""
        if not self.config.get('sendErrorAlerts', True):
            return False
        
        message = f"""
⚠️ *SYSTEM ERROR* ⚠️

*Type:* {error_type}
*Error:* {error_message}
*Context:* {context}
*Time:* {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}
"""
        
        return self.send_message(message.strip())
    
    def send_daily_summary(self, summary_data):
        """Send daily trading summary"""
        if not self.config.get('sendDailySummary', True):
            return False
        
        date = summary_data.get('date', datetime.now().strftime('%Y-%m-%d'))
        trades_made = summary_data.get('trades_made', 0)
        total_volume = summary_data.get('total_volume', 0)
        pnl = summary_data.get('pnl', 0)
        account_balance = summary_data.get('account_balance', 0)
        
        pnl_emoji = "📈" if pnl >= 0 else "📉"
        
        message = f"""
📋 *DAILY TRADING SUMMARY* 📋

*Date:* {date}
*Trades Made:* {trades_made}
*Total Volume:* ${total_volume:,.2f}
*Daily P&L:* {pnl_emoji} ${pnl:,.2f}
*Account Balance:* ${account_balance:,.2f}

*Summary Time:* {datetime.now().strftime('%H:%M:%S')}
"""
        
        return self.send_message(message.strip())
    
    def send_test_message(self):
        """Send test message to verify setup"""
        message = f"""
✅ *TRADING SYSTEM TEST* ✅

*System:* E*TRADE Congressional Trading Bot
*Status:* Online and Ready
*Account:* $50,000 Brokerage
*Time:* {datetime.now().strftime('%Y-%m-%d %H:%M:%S EST')}

Test message successful! The system is ready to trade.
"""
        
        success = self.send_message(message.strip())
        if success:
            logger.info("Telegram test message sent successfully")
        return success