"""
ClawBack - Configuration Loader
Reads configuration with support for environment variables and secrets management
"""
import json
import os
import re
import logging
from pathlib import Path
from typing import Any, Dict, Optional

logger = logging.getLogger(__name__)

# Environment variable pattern: ${VAR_NAME}
ENV_PATTERN = re.compile(r'\$\{([^}]+)\}')


def substitute_env_vars(value: Any) -> Any:
    """Recursively substitute environment variables in config values"""
    if isinstance(value, str):
        # Check for ${VAR} pattern
        match = ENV_PATTERN.fullmatch(value)
        if match:
            var_name = match.group(1)
            env_value = os.environ.get(var_name)
            if env_value:
                return env_value
            else:
                logger.warning(f"Environment variable {var_name} not set")
                return value
        # Also handle partial substitution
        def replace_var(m):
            var_name = m.group(1)
            return os.environ.get(var_name, m.group(0))
        return ENV_PATTERN.sub(replace_var, value)
    elif isinstance(value, dict):
        return {k: substitute_env_vars(v) for k, v in value.items()}
    elif isinstance(value, list):
        return [substitute_env_vars(item) for item in value]
    return value


def load_secrets(secrets_path: str = None) -> Dict[str, str]:
    """Load secrets from a JSON file"""
    if secrets_path is None:
        # Try common locations
        locations = [
            'config/secrets.json',
            '.secrets.json',
            os.path.expanduser('~/.clawback/secrets.json'),
            os.path.expanduser('~/.config/clawback/secrets.json')
        ]
        for loc in locations:
            if os.path.exists(loc):
                secrets_path = loc
                break

    if secrets_path and os.path.exists(secrets_path):
        try:
            with open(secrets_path) as f:
                secrets = json.load(f)
            logger.info(f"Loaded secrets from {secrets_path}")
            return secrets
        except Exception as e:
            logger.warning(f"Could not load secrets from {secrets_path}: {e}")

    return {}


def load_config(config_path: str = 'config/config.json') -> Dict[str, Any]:
    """
    Load configuration with environment variable substitution

    Priority order:
    1. Environment variables (highest)
    2. Secrets file (config/secrets.json or ~/.clawback/secrets.json)
    3. Config file values (lowest)
    """
    # Load base config
    if not os.path.exists(config_path):
        # Try template
        template_path = config_path.replace('.json', '.template.json')
        if os.path.exists(template_path):
            logger.warning(f"Config not found, using template: {template_path}")
            config_path = template_path
        else:
            raise FileNotFoundError(f"Configuration file not found: {config_path}")

    with open(config_path) as f:
        config = json.load(f)

    # Load secrets and set as environment variables (if not already set)
    secrets = load_secrets()
    for key, value in secrets.items():
        if key not in os.environ:
            os.environ[key] = str(value)

    # Substitute environment variables
    config = substitute_env_vars(config)

    # Remove comment fields
    def remove_comments(obj):
        if isinstance(obj, dict):
            return {k: remove_comments(v) for k, v in obj.items()
                    if not k.startswith('_')}
        elif isinstance(obj, list):
            return [remove_comments(item) for item in obj]
        return obj

    config = remove_comments(config)

    return config


def create_secrets_template():
    """Create a template secrets file"""
    template = {
        "_instructions": "Fill in your API keys and secrets. This file should NEVER be committed to git.",
        "BROKER_API_KEY": "your-broker-api-key",
        "BROKER_API_SECRET": "your-broker-api-secret",
        "BROKER_ACCOUNT_ID": "your-account-id",
        "TELEGRAM_BOT_TOKEN": "your-telegram-bot-token",
        "TELEGRAM_CHAT_ID": "your-telegram-chat-id"
    }
    return template


def setup_config_interactive():
    """Interactive setup for configuration"""
    print("\n" + "="*60)
    print("  ClawBack - Configuration Setup")
    print("="*60)

    secrets = {}

    print("\nBroker API Credentials")
    print("   For E*TRADE, get these from https://developer.etrade.com/")
    secrets['BROKER_API_KEY'] = input("   API Key: ").strip()
    secrets['BROKER_API_SECRET'] = input("   API Secret: ").strip()
    secrets['BROKER_ACCOUNT_ID'] = input("   Account ID: ").strip()

    print("\nOptional: Telegram Notifications")
    telegram = input("   Enable Telegram? (y/n): ").strip().lower()
    if telegram == 'y':
        secrets['TELEGRAM_BOT_TOKEN'] = input("   Bot Token: ").strip()
        secrets['TELEGRAM_CHAT_ID'] = input("   Chat ID: ").strip()

    # Save secrets
    secrets_path = 'config/secrets.json'
    os.makedirs('config', exist_ok=True)
    with open(secrets_path, 'w') as f:
        json.dump(secrets, f, indent=2)

    print(f"\nSecrets saved to {secrets_path}")
    print("   This file is in .gitignore and won't be committed.")

    # Copy template to config if needed
    config_path = 'config/config.json'
    template_path = 'config/config.template.json'
    if not os.path.exists(config_path) and os.path.exists(template_path):
        import shutil
        shutil.copy(template_path, config_path)
        print(f"Created {config_path} from template")

    print("\nSetup complete! Run 'python3 src/main.py interactive' to start.")


if __name__ == "__main__":
    import sys
    if len(sys.argv) > 1 and sys.argv[1] == 'setup':
        setup_config_interactive()
    else:
        # Test loading config
        try:
            config = load_config()
            print("Configuration loaded successfully!")
            broker = config.get('broker', {})
            print(f"  Broker adapter: {broker.get('adapter', 'not set')}")
            env = broker.get('environment', 'production')
            print(f"  Environment: {env}")
        except Exception as e:
            print(f"Error loading config: {e}")
