---
name: clawback
description: Mirror congressional stock trades with automated broker execution and risk management
version: 1.0.0
author: dayne
metadata:
  openclaw:
    requires:
      - python3
      - pdfplumber
      - selenium
      - yfinance
    config:
      - BROKER_API_KEY
      - BROKER_API_SECRET
      - BROKER_ACCOUNT_ID
    optional_config:
      - TELEGRAM_BOT_TOKEN
      - TELEGRAM_CHAT_ID
---

# ClawBack

**Mirror congressional stock trades with automated broker execution**

ClawBack tracks stock trades disclosed by members of Congress (House and Senate) and executes scaled positions in your brokerage account. Built on the premise that congressional leaders consistently outperform the market due to informational advantages.

## Features

- **Real-time disclosure tracking** from official House Clerk and Senate eFD sources
- **Automated trade execution** via broker API (E*TRADE adapter included)
- **Smart position sizing** - scales trades to your account size
- **Trailing stop-losses** - lock in profits, limit losses
- **Risk management** - drawdown limits, consecutive loss protection
- **Telegram notifications** - get alerts for new trades and stop-losses
- **Backtesting engine** - test strategies on historical data

## Performance (Backtest Results)

| Strategy | Win Rate | Return | Sharpe |
|----------|----------|--------|--------|
| 3-day delay, 30-day hold | 42.9% | +6.2% | 0.39 |
| 9-day delay, 90-day hold | 57.1% | +4.7% | 0.22 |

Congressional leaders have outperformed the S&P 500 by 47% annually according to NBER research.

## Quick Start

```bash
# Clone and setup
git clone https://github.com/openclaw/clawback
cd clawback
python3 -m venv venv && source venv/bin/activate
pip install -r requirements.txt

# Configure secrets
python3 src/config_loader.py setup

# Authenticate with broker
python3 src/main.py interactive
# Select option 1 to authenticate

# Set up automation
./scripts/setup_cron.sh
```

## Configuration

ClawBack reads secrets from environment variables or `config/secrets.json`:

```json
{
  "BROKER_API_KEY": "your-broker-api-key",
  "BROKER_API_SECRET": "your-broker-api-secret",
  "BROKER_ACCOUNT_ID": "your-account-id",
  "TELEGRAM_BOT_TOKEN": "optional-for-notifications",
  "TELEGRAM_CHAT_ID": "optional-for-notifications"
}
```

### Supported Brokers

ClawBack uses an adapter pattern for broker integration. Each broker implements a common interface defined in `broker_adapter.py`.

| Broker | Adapter | Status |
|--------|---------|--------|
| E*TRADE | `etrade_adapter.py` | Supported |
| Schwab | `schwab_adapter.py` | Planned |
| Fidelity | `fidelity_adapter.py` | Planned |

To specify which broker to use, set `broker.adapter` in your config:

```json
{
  "broker": {
    "adapter": "etrade",
    "credentials": {
      "apiKey": "${BROKER_API_KEY}",
      "apiSecret": "${BROKER_API_SECRET}"
    }
  }
}
```

## Data Sources

All data is scraped directly from official government sources:

| Source | Data | Method |
|--------|------|--------|
| House Clerk | House PTR filings | PDF parsing |
| Senate eFD | Senate PTR filings | Selenium scraping |

No third-party APIs required for congressional data.

## Strategy Settings

Edit `config/config.json` to customize:

```json
{
  "strategy": {
    "entryDelayDays": 3,
    "holdingPeriodDays": 30,
    "purchasesOnly": true,
    "minimumTradeSize": 50000
  },
  "riskManagement": {
    "positionStopLoss": 0.08,
    "trailingStopActivation": 0.10,
    "trailingStopPercent": 0.05,
    "maxDrawdown": 0.15
  }
}
```

## Commands

```bash
# Interactive mode
python3 src/main.py interactive

# Single check cycle
python3 src/main.py run

# Scheduled trading
python3 src/main.py schedule 24

# Run backtest
python3 src/backtester.py
```

## Cron Automation

```bash
# Install cron jobs
./scripts/setup_cron.sh

# Manual runs
./scripts/run_bot.sh check    # Check for new trades
./scripts/run_bot.sh monitor  # Check stop-losses
./scripts/run_bot.sh full     # Both
```

## Architecture

```
clawback/
├── src/
│   ├── main.py              # Main entry point
│   ├── congress_tracker.py  # Congressional data collection
│   ├── trade_engine.py      # Trade execution & risk management
│   ├── broker_adapter.py    # Abstract broker interface
│   ├── etrade_adapter.py    # E*TRADE broker implementation
│   ├── database.py          # SQLite state management
│   └── config_loader.py     # Configuration handling
├── config/
│   ├── config.json          # Main configuration
│   └── secrets.json         # API keys (git-ignored)
├── scripts/
│   ├── run_bot.sh           # Cron runner
│   └── setup_cron.sh        # Cron installer
└── data/
    └── trading.db           # SQLite database
```

## Risk Disclaimer

This software is for educational purposes only. Trading stocks involves substantial risk of loss. Past performance of congressional trades does not guarantee future results. The authors are not financial advisors. Use at your own risk.

## License

MIT License - See LICENSE file

---

*Built with ClawBack for the OpenClaw community*
