---
name: esimpal-api-agent
description: Use when building or debugging an agent (e.g. Telegram/WhatsApp bot, AI assistant) that integrates with the eSIMPal API to buy eSIMs for end-users, create orders, and deliver activation links, QR codes, or manual-install details.
required_credentials:
  - ESIMPAL_API_KEY
primary_credential_env: ESIMPAL_API_KEY
---

# eSIMPal API - agent integration skill

Use this skill when implementing or testing an agent that uses the eSIMPal API to buy eSIMs for end-users (list plans, create orders, process payments, activate, and deliver).

## Safety and approval rules

- This skill is for integration guidance and controlled runtime calls. It must **not** initiate purchases autonomously.
- Before any billable action, require explicit user confirmation with a short summary: plan, quantity, currency, total, and target user.
- Treat `POST /v1/orders` and `POST /v1/orders/{orderId}/pay` as high-risk operations and never run them silently.
- Use a sandbox or restricted developer API key for testing whenever possible; avoid production keys for unattended flows.
- Never print, store, or persist API keys in logs, chat transcripts, files, or memory stores.
- Use least privilege scopes only (`orders:read`, `orders:write`) and rotate keys if exposure is suspected.

## Base URL and auth

- **Base URL**: `https://getesimpal.com/api` (or the env override the user provides, always ending in `/api`).
- **Full example**: `GET https://getesimpal.com/api/v1/plans?country=TR&min_data_gb=1`
- **Auth**: Every request must include header `Authorization: Bearer <API_KEY>`.
- **API key**: Provide at runtime from `ESIMPAL_API_KEY`; do not hardcode. Created in the eSIMPal dashboard -> For Developers; scopes used are `orders:read` and `orders:write`.

## Idempotency-Key (create order and start payment)

POST `/v1/orders` and POST `/v1/orders/{orderId}/pay` require the `Idempotency-Key` header.

- **Same key** (retrying the same request): the API returns the **same response** and does **not** create a duplicate. Use when retrying after a timeout or 5xx.
- **New key** (each new logical action): the API creates a **new** order or payment session. Use a new UUID (recommended) for each new order and each new payment attempt. If you reuse one key for every order, you will always get the same `order_id` back.
- Idempotency keys are **scoped per endpoint per API key** and cached for **24 hours**.

**Examples:**

```
Idempotency-Key: 550e8400-e29b-41d4-a716-446655440000   # new order
Idempotency-Key: 550e8400-e29b-41d4-a716-446655440000   # retry -> same order returned
Idempotency-Key: 7c9e6679-7425-40de-944b-e07fc1f90ae7   # different order
```

## Typical flow (buy -> pay -> deliver)

1. **List plans** - `GET /v1/plans?country={ISO2}&min_data_gb={number}`
   Returns `{ plans: [...] }`. Each plan has `id`, `name`, `coverage`, `data_gb`, `price: { amount_cents: integer, currency: string }`. Use `plan_id` when creating the order.

2. **Create order** - `POST /v1/orders`
   - Headers: `Content-Type: application/json`, `Idempotency-Key: <uuid>` (required).
   - Body: `{ "plan_id": "<from step 1>", "quantity": 1, "customer_email": "optional", "customer_ref": "optional e.g. telegram:123" }`.
   - Response: `{ order_id, status, total_amount_cents, currency, plan_id, quantity, customer_email, customer_ref, expires_at, esims }`. Store `order_id`.
   - **Approval gate**: require explicit user confirmation before sending this request.

3. **Start payment** - `POST /v1/orders/{orderId}/pay`
   - Headers: `Idempotency-Key: <uuid>` (required). Use a **new** UUID per new payment attempt.
   - Response: `{ status, checkout_url, expires_at }`. Send `checkout_url` to the user so they can pay in the browser.
   - **Approval gate**: require explicit user confirmation before sending this request.

4. **Poll until ready** - `GET /v1/orders/{orderId}`
   Poll until `status` is `ready`, `failed`, or `expired`. When `status === "ready"`, the order is paid and provisioned; `esims` is populated.

   **Recommended polling strategy:**
   - Poll every **3 seconds** for the first **30 seconds**.
   - Then back off to every **10 seconds** for the next **2 minutes**.
   - After **3 minutes** total, stop polling and tell the user to check back later.
   - If the response includes `Retry-After`, respect that value instead.

5. **Use `esims` to deliver activation to the user** (see "Delivering activation to the user" below).

## Activation (when order is ready)

- **Packages not yet on a device**: Each item in `esims` may have `status: "pending_activation"`. Then either:
  - **New device**: `POST /v1/orders/{orderId}/packages/{packageId}/activate/new` (no body). Creates a new eSIM profile and assigns the package. Response includes activation links and `qr_code_url`.
  - **Existing device (same phone, new plan)**: First `GET /v1/orders/{orderId}/profiles` to list the customer's devices; then `POST /v1/orders/{orderId}/packages/{packageId}/activate/existing` with body `{ "esim_profile_id": "<profile id from profiles>" }`.

- **Already activated**: If `esims[i].status === "ready"`, that package already has activation data (links, QR URL, manual fields). Use it directly.

## Delivering activation to the user (Telegram/WhatsApp etc.)

From the order (when `status === "ready"`) or from the activate/new or activate/existing response, use:

| Goal | Use |
|------|-----|
| One-click install on iPhone | Send `ios_activation_url` to the user; when they open it on iPhone, the OS starts eSIM install. |
| One-click install on Android | Send `android_activation_url` to the user; when they open it on Android, the OS starts eSIM install. |
| Send a QR image | `qr_code_url` is a URL. GET it with `Authorization: Bearer <API_KEY>`; response is `image/png`. Upload/send that image (e.g. Telegram `sendPhoto`). **Do not cache** - QR URLs are short-lived. |
| Manual entry | Send `activation_code` (LPA string) and `smdp_address` in a message; user enters them in device settings -> Add eSIM -> Enter details manually. |
| Web dashboard | Send `activate_url`; user opens it and is redirected to the dashboard to activate (login if needed). |

All of the above are optional and nullable; prefer one method (e.g. one link per platform or QR) and fall back to manual if needed.

## Endpoints quick reference

All paths are relative to the base URL (`https://getesimpal.com/api`).

| Method | Path | Scope | Notes |
|--------|------|--------|--------|
| GET | `/v1/plans?country=&min_data_gb=` | (read) | List plans. Price is `{ amount_cents, currency }`. |
| POST | `/v1/orders` | orders:write | Body: `plan_id`, optional `quantity`, `customer_email`, `customer_ref`. Header: `Idempotency-Key` (UUID). |
| GET | `/v1/orders/{orderId}` | orders:read | Poll until `status` is ready/failed/expired. When ready, `esims` has packages. |
| POST | `/v1/orders/{orderId}/pay` | orders:write | Header: `Idempotency-Key` (UUID). Returns `checkout_url`. |
| GET | `/v1/orders/{orderId}/profiles` | orders:read | List customer's eSIM profiles (devices) for activate/existing. |
| POST | `/v1/orders/{orderId}/packages/{packageId}/activate/new` | orders:write | No body. Creates a new eSIM profile for this package. |
| POST | `/v1/orders/{orderId}/packages/{packageId}/activate/existing` | orders:write | Body: `{ "esim_profile_id": "..." }`. Adds package to existing device. |
| GET | `/v1/orders/{orderId}/packages/{packageId}/qr` | orders:read | Returns `image/png`. Do not cache (`Cache-Control: no-store`). |

## Errors and retries

- **401**: Invalid or missing API key.
- **403**: API key cannot access this resource (e.g. another client's order).
- **404**: Order or package not found, or QR not available (e.g. package not yet activated).
- **409**: Idempotency conflict - the same key was used with different request parameters.
- **429**: Rate limited. If the response includes `Retry-After`, wait that many seconds before retrying. Otherwise back off exponentially (2s -> 4s -> 8s).
- **5xx**: Server error. Response body may include `{ code, message, retryable }`. If `retryable === true`, retry with exponential backoff up to 3 attempts.

Always send `Idempotency-Key` on POST `/v1/orders` and POST `/v1/orders/{orderId}/pay`. Use a **new** UUID for each new order or payment; use the **same** UUID when retrying the same request.

## Response shapes

### Order (when status is ready)

```json
{
  "order_id": "uuid",
  "status": "ready",
  "total_amount_cents": 800,
  "currency": "USD",
  "plan_id": "uuid",
  "quantity": 1,
  "customer_email": "user@example.com",
  "customer_ref": "telegram:123",
  "expires_at": "2026-03-08T12:00:00Z",
  "esims": [...]
}
```

### Each item in `esims`

```json
{
  "package_id": "uuid",
  "status": "ready",
  "activation_code": "LPA:1$smdp.example.com$MATCHING-ID",
  "smdp_address": "smdp.example.com",
  "ios_activation_url": "https://...",
  "android_activation_url": "https://...",
  "activate_url": "https://...",
  "qr_code_url": "https://..."
}
```

All activation fields are **nullable**. `status` is either `"ready"` (activation data available) or `"pending_activation"` (call activate/new or activate/existing first).
