#!/usr/bin/env python3
"""ERPClaw Selling Skill — db_query.py

Order-to-cash cycle: customers, quotations, sales orders, delivery notes,
sales invoices, credit notes, sales partners, recurring invoices.
Draft->Submit->Cancel lifecycle with GL, SLE, and PLE postings.

Usage: python3 db_query.py --action <action-name> [--flags ...]
Output: JSON to stdout, exit 0 on success, exit 1 on error.
"""
import argparse
import json
import os
import sqlite3
import sys
import uuid
import calendar
from datetime import datetime, timezone, timedelta, date as date_type
from decimal import Decimal, InvalidOperation

# Add shared lib to path
try:
    sys.path.insert(0, os.path.expanduser("~/.openclaw/erpclaw/lib"))
    from erpclaw_lib.db import get_connection, ensure_db_exists, DEFAULT_DB_PATH
    from erpclaw_lib.decimal_utils import to_decimal, round_currency
    from erpclaw_lib.validation import check_input_lengths
    from erpclaw_lib.naming import get_next_name
    from erpclaw_lib.stock_posting import (
        insert_sle_entries,
        reverse_sle_entries,
        get_stock_balance,
        get_valuation_rate,
        create_perpetual_inventory_gl,
    )
    from erpclaw_lib.gl_posting import insert_gl_entries, reverse_gl_entries
    from erpclaw_lib.response import ok, err, row_to_dict
    from erpclaw_lib.audit import audit
    from erpclaw_lib.dependencies import check_required_tables
except ImportError:
    import json as _json
    print(_json.dumps({"status": "error", "error": "ERPClaw foundation not installed. Install erpclaw-setup first: clawhub install erpclaw-setup", "suggestion": "clawhub install erpclaw-setup"}))
    sys.exit(1)

REQUIRED_TABLES = ["company", "account", "item"]

VALID_CUSTOMER_TYPES = ("company", "individual")
VALID_FREQUENCIES = ("weekly", "monthly", "quarterly", "semi_annually", "annually")



# ---------------------------------------------------------------------------
# Helpers
# ---------------------------------------------------------------------------

def _parse_json_arg(value, name):
    if value is None:
        return None
    try:
        return json.loads(value)
    except (json.JSONDecodeError, TypeError):
        err(f"Invalid JSON for --{name}: {value}")


def _get_fiscal_year(conn, posting_date: str) -> str | None:
    """Return the fiscal year name for a posting date, or None."""
    fy = conn.execute(
        "SELECT name FROM fiscal_year WHERE start_date <= ? AND end_date >= ? AND is_closed = 0",
        (posting_date, posting_date),
    ).fetchone()
    return fy["name"] if fy else None


def _get_cost_center(conn, company_id: str) -> str | None:
    """Return the first non-group cost center for a company, or None."""
    cc = conn.execute(
        "SELECT id FROM cost_center WHERE company_id = ? AND is_group = 0 LIMIT 1",
        (company_id,),
    ).fetchone()
    return cc["id"] if cc else None


def _get_receivable_account(conn, company_id: str) -> str | None:
    """Return the default receivable account for a company."""
    company = conn.execute(
        "SELECT default_receivable_account_id FROM company WHERE id = ?",
        (company_id,),
    ).fetchone()
    if company and company["default_receivable_account_id"]:
        return company["default_receivable_account_id"]
    acct = conn.execute(
        "SELECT id FROM account WHERE account_type = 'receivable' "
        "AND company_id = ? AND is_group = 0 LIMIT 1",
        (company_id,),
    ).fetchone()
    return acct["id"] if acct else None


def _get_income_account(conn, company_id: str) -> str | None:
    """Return the default income account for a company."""
    company = conn.execute(
        "SELECT default_income_account_id FROM company WHERE id = ?",
        (company_id,),
    ).fetchone()
    if company and company["default_income_account_id"]:
        return company["default_income_account_id"]
    acct = conn.execute(
        "SELECT id FROM account WHERE root_type = 'income' "
        "AND company_id = ? AND is_group = 0 LIMIT 1",
        (company_id,),
    ).fetchone()
    return acct["id"] if acct else None


def _get_cogs_account(conn, company_id: str) -> str | None:
    """Return the COGS account for a company."""
    acct = conn.execute(
        "SELECT id FROM account WHERE account_type = 'cost_of_goods_sold' "
        "AND company_id = ? AND is_group = 0 LIMIT 1",
        (company_id,),
    ).fetchone()
    return acct["id"] if acct else None


def _get_stock_in_hand_account(conn, company_id: str, warehouse_id: str = None) -> str | None:
    """Return the stock-in-hand account for a warehouse or company default."""
    if warehouse_id:
        wh = conn.execute(
            "SELECT account_id FROM warehouse WHERE id = ?", (warehouse_id,)
        ).fetchone()
        if wh and wh["account_id"]:
            return wh["account_id"]
    acct = conn.execute(
        "SELECT id FROM account WHERE account_type = 'stock' "
        "AND company_id = ? AND is_group = 0 LIMIT 1",
        (company_id,),
    ).fetchone()
    return acct["id"] if acct else None


def _calculate_tax(conn, subtotal: Decimal, tax_template_id: str) -> tuple:
    """Calculate tax from a tax template.

    Returns (tax_amount, list_of_tax_lines) where each tax line is a dict
    with account_id, rate, amount.
    """
    if not tax_template_id:
        return Decimal("0"), []

    lines = conn.execute(
        """SELECT ttl.*, a.name as account_name
           FROM tax_template_line ttl
           LEFT JOIN account a ON a.id = ttl.tax_account_id
           WHERE ttl.tax_template_id = ?
           ORDER BY ttl.row_order""",
        (tax_template_id,),
    ).fetchall()

    total_tax = Decimal("0")
    tax_lines = []
    cumulative_amount = subtotal

    for line in lines:
        rate = to_decimal(line["rate"])
        charge_type = line["charge_type"]
        add_deduct = line["add_deduct"]

        if charge_type == "on_net_total":
            tax_amount = round_currency(subtotal * rate / Decimal("100"))
        elif charge_type == "actual":
            tax_amount = round_currency(rate)
        elif charge_type == "on_previous_row_total":
            tax_amount = round_currency(cumulative_amount * rate / Decimal("100"))
        elif charge_type == "on_previous_row_amount":
            prev_tax = tax_lines[-1]["amount"] if tax_lines else Decimal("0")
            tax_amount = round_currency(prev_tax * rate / Decimal("100"))
        elif charge_type == "on_item_quantity":
            tax_amount = round_currency(rate)
        else:
            tax_amount = Decimal("0")

        if add_deduct == "deduct":
            tax_amount = -tax_amount

        total_tax += tax_amount
        cumulative_amount += tax_amount
        tax_lines.append({
            "account_id": line["tax_account_id"],
            "account_name": line["account_name"] if line["account_name"] else "",
            "rate": str(rate),
            "amount": tax_amount,
        })

    return round_currency(total_tax), tax_lines


def _apply_pricing_rule(conn, item_id: str, customer_id: str,
                        qty: Decimal, posting_date: str,
                        company_id: str) -> dict | None:
    """Find and return the best applicable pricing rule for an item."""
    row = conn.execute(
        """SELECT * FROM pricing_rule
           WHERE active = 1 AND company_id = ?
             AND (
                 (applies_to = 'item' AND entity_id = ?)
                 OR (applies_to = 'customer' AND entity_id = ?)
                 OR applies_to = 'item_group'
                 OR applies_to = 'customer_group'
             )
             AND (min_qty IS NULL OR min_qty + 0 <= ? + 0)
             AND (max_qty IS NULL OR max_qty + 0 >= ? + 0)
             AND (valid_from IS NULL OR valid_from <= ?)
             AND (valid_to IS NULL OR valid_to >= ?)
           ORDER BY priority DESC, COALESCE(min_qty, '0') + 0 DESC
           LIMIT 1""",
        (company_id, item_id, customer_id, str(qty), str(qty),
         posting_date, posting_date),
    ).fetchone()
    return row_to_dict(row) if row else None


def _calculate_line_items(conn, items_json, company_id: str,
                          customer_id: str = None, posting_date: str = None,
                          apply_pricing: bool = False):
    """Validate and calculate line items from JSON.

    Returns (total_amount, item_rows_list) where each item_row is a tuple
    ready for insertion.
    """
    if not items_json or not isinstance(items_json, list):
        err("--items must be a non-empty JSON array")

    total = Decimal("0")
    rows = []
    for i, item in enumerate(items_json):
        item_id = item.get("item_id")
        if not item_id:
            err(f"Item {i}: item_id is required")

        item_row = conn.execute("SELECT id, item_name, stock_uom FROM item WHERE id = ?",
                                (item_id,)).fetchone()
        if not item_row:
            err(f"Item {i}: item {item_id} not found")

        qty = to_decimal(item.get("qty", "0"))
        if qty <= 0:
            err(f"Item {i}: qty must be > 0")

        rate = to_decimal(item.get("rate", "0"))
        discount_pct = to_decimal(item.get("discount_percentage", "0"))
        warehouse_id = item.get("warehouse_id")
        pricing_rule_id = None

        # Apply pricing rule if requested
        if apply_pricing and customer_id and posting_date and rate > 0:
            rule = _apply_pricing_rule(conn, item_id, customer_id, qty,
                                       posting_date, company_id)
            if rule:
                pricing_rule_id = rule.get("id")
                if rule.get("discount_percentage"):
                    discount_pct = to_decimal(rule["discount_percentage"])
                elif rule.get("rate"):
                    rate = to_decimal(rule["rate"])

        if discount_pct > 0:
            effective_rate = round_currency(rate * (Decimal("1") - discount_pct / Decimal("100")))
        else:
            effective_rate = rate

        amount = round_currency(qty * rate)
        net_amount = round_currency(qty * effective_rate)
        total += net_amount

        rows.append({
            "item_id": item_id,
            "qty": str(round_currency(qty)),
            "uom": item.get("uom") or item_row["stock_uom"],
            "rate": str(round_currency(rate)),
            "amount": str(amount),
            "discount_percentage": str(round_currency(discount_pct)),
            "net_amount": str(net_amount),
            "warehouse_id": warehouse_id,
            "pricing_rule_id": pricing_rule_id,
            "description": item.get("description"),
        })

    return round_currency(total), rows


def _add_months(d: date_type, months: int) -> date_type:
    """Add months to a date, clamping to last day of month if needed."""
    month = d.month - 1 + months
    year = d.year + month // 12
    month = month % 12 + 1
    day = min(d.day, calendar.monthrange(year, month)[1])
    return date_type(year, month, day)


def _next_invoice_date(current_date_str: str, frequency: str) -> str:
    """Calculate next invoice date based on frequency."""
    parts = current_date_str.split("-")
    d = date_type(int(parts[0]), int(parts[1]), int(parts[2]))

    if frequency == "weekly":
        d += timedelta(days=7)
    elif frequency == "monthly":
        d = _add_months(d, 1)
    elif frequency == "quarterly":
        d = _add_months(d, 3)
    elif frequency == "semi_annually":
        d = _add_months(d, 6)
    elif frequency == "annually":
        d = _add_months(d, 12)
    else:
        d = _add_months(d, 1)

    return d.isoformat()


# ---------------------------------------------------------------------------
# 1. add-customer
# ---------------------------------------------------------------------------

def add_customer(conn, args):
    """Create a customer record."""
    if not args.name:
        err("--name is required")
    if not args.company_id:
        err("--company-id is required")

    if not conn.execute("SELECT id FROM company WHERE id = ?",
                        (args.company_id,)).fetchone():
        err(f"Company {args.company_id} not found")

    customer_type = args.customer_type or "company"
    if customer_type not in VALID_CUSTOMER_TYPES:
        err(f"--customer-type must be one of: {', '.join(VALID_CUSTOMER_TYPES)}")

    if args.payment_terms_id:
        if not conn.execute("SELECT id FROM payment_terms WHERE id = ?",
                            (args.payment_terms_id,)).fetchone():
            err(f"Payment terms {args.payment_terms_id} not found")

    credit_limit = str(round_currency(to_decimal(args.credit_limit or "0")))
    exempt = int(args.exempt_from_sales_tax) if args.exempt_from_sales_tax else 0

    primary_address = args.primary_address
    primary_contact = args.primary_contact

    cust_id = str(uuid.uuid4())
    try:
        conn.execute(
            """INSERT INTO customer
               (id, name, customer_type, customer_group, payment_terms_id,
                credit_limit, tax_id, exempt_from_sales_tax,
                primary_address, primary_contact, status, company_id)
               VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'active', ?)""",
            (cust_id, args.name, customer_type, args.customer_group,
             args.payment_terms_id, credit_limit, args.tax_id, exempt,
             primary_address, primary_contact, args.company_id),
        )
    except sqlite3.IntegrityError as e:
        sys.stderr.write(f"[erpclaw-selling] {e}\n")
        err("Customer creation failed — check for duplicates or invalid data")

    audit(conn, "erpclaw-selling", "add-customer", "customer", cust_id,
           new_values={"name": args.name, "customer_type": customer_type})
    conn.commit()
    ok({"customer_id": cust_id, "name": args.name,
         "customer_type": customer_type})


# ---------------------------------------------------------------------------
# 2. update-customer
# ---------------------------------------------------------------------------

def update_customer(conn, args):
    """Update an existing customer."""
    if not args.customer_id:
        err("--customer-id is required")

    cust = conn.execute("SELECT * FROM customer WHERE id = ? OR name = ?",
                        (args.customer_id, args.customer_id)).fetchone()
    if not cust:
        err(f"Customer {args.customer_id} not found",
             suggestion="Use 'list customers' to see available customers.")
    args.customer_id = cust["id"]  # normalize to id

    updates, params, updated_fields = [], [], []

    if args.name is not None:
        updates.append("name = ?")
        params.append(args.name)
        updated_fields.append("name")
    if args.credit_limit is not None:
        updates.append("credit_limit = ?")
        params.append(str(round_currency(to_decimal(args.credit_limit))))
        updated_fields.append("credit_limit")
    if args.payment_terms_id is not None:
        updates.append("payment_terms_id = ?")
        params.append(args.payment_terms_id)
        updated_fields.append("payment_terms_id")
    if args.customer_group is not None:
        updates.append("customer_group = ?")
        params.append(args.customer_group)
        updated_fields.append("customer_group")
    if args.customer_type is not None:
        if args.customer_type not in VALID_CUSTOMER_TYPES:
            err(f"--customer-type must be one of: {', '.join(VALID_CUSTOMER_TYPES)}")
        updates.append("customer_type = ?")
        params.append(args.customer_type)
        updated_fields.append("customer_type")

    if not updated_fields:
        err("No fields to update")

    updates.append("updated_at = datetime('now')")
    params.append(args.customer_id)
    conn.execute(f"UPDATE customer SET {', '.join(updates)} WHERE id = ?", params)

    audit(conn, "erpclaw-selling", "update-customer", "customer", args.customer_id,
           new_values={"updated_fields": updated_fields})
    conn.commit()
    ok({"customer_id": args.customer_id, "updated_fields": updated_fields})


# ---------------------------------------------------------------------------
# 3. get-customer
# ---------------------------------------------------------------------------

def get_customer(conn, args):
    """Get customer with outstanding summary."""
    if not args.customer_id:
        err("--customer-id is required")

    cust = conn.execute("SELECT * FROM customer WHERE id = ? OR name = ?",
                        (args.customer_id, args.customer_id)).fetchone()
    if not cust:
        err(f"Customer {args.customer_id} not found",
             suggestion="Use 'list customers' to see available customers.")

    data = row_to_dict(cust)

    # Outstanding summary from sales invoices
    outstanding_row = conn.execute(
        """SELECT COALESCE(decimal_sum(outstanding_amount), '0') as total_outstanding,
                  COUNT(*) as invoice_count
           FROM sales_invoice
           WHERE customer_id = ? AND status IN ('submitted', 'overdue', 'partially_paid')
             AND outstanding_amount + 0 > 0""",
        (args.customer_id,),
    ).fetchone()

    data["total_outstanding"] = str(round_currency(
        to_decimal(str(outstanding_row["total_outstanding"]))))
    data["outstanding_invoice_count"] = outstanding_row["invoice_count"]

    ok(data)


# ---------------------------------------------------------------------------
# 4. list-customers
# ---------------------------------------------------------------------------

def list_customers(conn, args):
    """Query customers with filtering."""
    conditions = ["1=1"]
    params = []

    if args.company_id:
        conditions.append("c.company_id = ?")
        params.append(args.company_id)
    if args.customer_group:
        conditions.append("c.customer_group = ?")
        params.append(args.customer_group)
    if args.search:
        conditions.append("(c.name LIKE ? OR c.tax_id LIKE ?)")
        params.extend([f"%{args.search}%", f"%{args.search}%"])

    where = " AND ".join(conditions)

    count_row = conn.execute(
        f"SELECT COUNT(*) FROM customer c WHERE {where}", params
    ).fetchone()
    total_count = count_row[0]

    limit = int(args.limit) if args.limit else 20
    offset = int(args.offset) if args.offset else 0
    params.extend([limit, offset])

    rows = conn.execute(
        f"""SELECT c.id, c.name, c.customer_type, c.customer_group,
               c.credit_limit, c.status, c.company_id
           FROM customer c WHERE {where}
           ORDER BY c.name
           LIMIT ? OFFSET ?""",
        params,
    ).fetchall()

    ok({"customers": [row_to_dict(r) for r in rows], "total_count": total_count,
         "limit": limit, "offset": offset, "has_more": offset + limit < total_count})


# ---------------------------------------------------------------------------
# 5. add-quotation
# ---------------------------------------------------------------------------

def add_quotation(conn, args):
    """Create a quotation in draft."""
    if not args.customer_id:
        err("--customer-id is required")
    if not args.posting_date:
        err("--posting-date is required")
    if not args.items:
        err("--items is required (JSON array)")
    if not args.company_id:
        err("--company-id is required")

    cust_row = conn.execute(
        "SELECT id FROM customer WHERE (id = ? OR name = ?) AND status = 'active'",
        (args.customer_id, args.customer_id)).fetchone()
    if not cust_row:
        err(f"Active customer {args.customer_id} not found")
    args.customer_id = cust_row["id"]  # normalize to id
    if not conn.execute("SELECT id FROM company WHERE id = ?",
                        (args.company_id,)).fetchone():
        err(f"Company {args.company_id} not found")

    items = _parse_json_arg(args.items, "items")
    total_amount, item_rows = _calculate_line_items(
        conn, items, args.company_id, args.customer_id, args.posting_date)

    tax_amount, tax_lines = _calculate_tax(conn, total_amount, args.tax_template_id)
    grand_total = round_currency(total_amount + tax_amount)

    q_id = str(uuid.uuid4())

    # Insert parent quotation first
    conn.execute(
        """INSERT INTO quotation
           (id, customer_id, quotation_date, valid_until, total_amount,
            tax_amount, grand_total, tax_template_id, status, company_id)
           VALUES (?, ?, ?, ?, ?, ?, ?, ?, 'draft', ?)""",
        (q_id, args.customer_id, args.posting_date, args.valid_till,
         str(total_amount), str(tax_amount), str(grand_total),
         args.tax_template_id, args.company_id),
    )

    # Insert child quotation_item rows
    for row in item_rows:
        conn.execute(
            """INSERT INTO quotation_item
               (id, quotation_id, item_id, quantity, uom, rate, amount,
                discount_percentage, net_amount, description)
               VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)""",
            (str(uuid.uuid4()), q_id, row["item_id"], row["qty"],
             row["uom"], row["rate"], row["amount"],
             row["discount_percentage"], row["net_amount"],
             row["description"]),
        )

    audit(conn, "erpclaw-selling", "add-quotation", "quotation", q_id,
           new_values={"customer_id": args.customer_id, "grand_total": str(grand_total)})
    conn.commit()
    ok({"quotation_id": q_id, "total_amount": str(total_amount),
         "tax_amount": str(tax_amount), "grand_total": str(grand_total)})


# ---------------------------------------------------------------------------
# 6. update-quotation
# ---------------------------------------------------------------------------

def update_quotation(conn, args):
    """Update a draft quotation."""
    if not args.quotation_id:
        err("--quotation-id is required")

    q = conn.execute("SELECT * FROM quotation WHERE id = ?",
                     (args.quotation_id,)).fetchone()
    if not q:
        err(f"Quotation {args.quotation_id} not found")
    if q["status"] != "draft":
        err(f"Cannot update: quotation is '{q['status']}' (must be 'draft')",
             suggestion="Cancel the document first, then make changes.")

    updated_fields = []

    if args.valid_till is not None:
        conn.execute("UPDATE quotation SET valid_until = ?, updated_at = datetime('now') WHERE id = ?",
                     (args.valid_till, args.quotation_id))
        updated_fields.append("valid_until")

    if args.items:
        items = _parse_json_arg(args.items, "items")
        total_amount, item_rows = _calculate_line_items(
            conn, items, q["company_id"], q["customer_id"], q["quotation_date"])

        tax_amount, _ = _calculate_tax(conn, total_amount, q["tax_template_id"])
        grand_total = round_currency(total_amount + tax_amount)

        # Delete old items, insert new
        conn.execute("DELETE FROM quotation_item WHERE quotation_id = ?",
                     (args.quotation_id,))
        for row in item_rows:
            conn.execute(
                """INSERT INTO quotation_item
                   (id, quotation_id, item_id, quantity, uom, rate, amount,
                    discount_percentage, net_amount, description)
                   VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)""",
                (str(uuid.uuid4()), args.quotation_id, row["item_id"],
                 row["qty"], row["uom"], row["rate"], row["amount"],
                 row["discount_percentage"], row["net_amount"],
                 row["description"]),
            )

        conn.execute(
            """UPDATE quotation SET total_amount = ?, tax_amount = ?,
               grand_total = ?, updated_at = datetime('now') WHERE id = ?""",
            (str(total_amount), str(tax_amount), str(grand_total),
             args.quotation_id),
        )
        updated_fields.extend(["items", "total_amount", "tax_amount", "grand_total"])

    if not updated_fields:
        err("No fields to update")

    audit(conn, "erpclaw-selling", "update-quotation", "quotation", args.quotation_id,
           new_values={"updated_fields": updated_fields})
    conn.commit()
    ok({"quotation_id": args.quotation_id, "updated_fields": updated_fields})


# ---------------------------------------------------------------------------
# 7. get-quotation
# ---------------------------------------------------------------------------

def get_quotation(conn, args):
    """Get quotation with items."""
    if not args.quotation_id:
        err("--quotation-id is required")

    q = conn.execute("SELECT * FROM quotation WHERE id = ?",
                     (args.quotation_id,)).fetchone()
    if not q:
        err(f"Quotation {args.quotation_id} not found")

    data = row_to_dict(q)
    items = conn.execute(
        """SELECT qi.*, i.item_name
           FROM quotation_item qi
           LEFT JOIN item i ON i.id = qi.item_id
           WHERE qi.quotation_id = ?
           ORDER BY qi.rowid""",
        (args.quotation_id,),
    ).fetchall()
    data["items"] = [row_to_dict(r) for r in items]
    ok(data)


# ---------------------------------------------------------------------------
# 8. list-quotations
# ---------------------------------------------------------------------------

def list_quotations(conn, args):
    """Query quotations with filtering."""
    conditions = ["1=1"]
    params = []

    if args.company_id:
        conditions.append("q.company_id = ?")
        params.append(args.company_id)
    if args.customer_id:
        conditions.append("q.customer_id = ?")
        params.append(args.customer_id)
    if args.doc_status:
        conditions.append("q.status = ?")
        params.append(args.doc_status)
    if args.from_date:
        conditions.append("q.quotation_date >= ?")
        params.append(args.from_date)
    if args.to_date:
        conditions.append("q.quotation_date <= ?")
        params.append(args.to_date)

    where = " AND ".join(conditions)

    count_row = conn.execute(
        f"SELECT COUNT(*) FROM quotation q WHERE {where}", params
    ).fetchone()
    total_count = count_row[0]

    limit = int(args.limit) if args.limit else 20
    offset = int(args.offset) if args.offset else 0
    params.extend([limit, offset])

    rows = conn.execute(
        f"""SELECT q.id, q.naming_series, q.customer_id, q.quotation_date,
               q.grand_total, q.status, q.company_id
           FROM quotation q WHERE {where}
           ORDER BY q.quotation_date DESC, q.created_at DESC
           LIMIT ? OFFSET ?""",
        params,
    ).fetchall()

    ok({"quotations": [row_to_dict(r) for r in rows], "total_count": total_count,
         "limit": limit, "offset": offset, "has_more": offset + limit < total_count})


# ---------------------------------------------------------------------------
# 9. submit-quotation
# ---------------------------------------------------------------------------

def submit_quotation(conn, args):
    """Submit a quotation: draft -> open."""
    if not args.quotation_id:
        err("--quotation-id is required")

    q = conn.execute("SELECT * FROM quotation WHERE id = ?",
                     (args.quotation_id,)).fetchone()
    if not q:
        err(f"Quotation {args.quotation_id} not found")
    if q["status"] != "draft":
        err(f"Cannot submit: quotation is '{q['status']}' (must be 'draft')")

    # Generate naming series
    naming = get_next_name(conn, "quotation", company_id=q["company_id"])

    conn.execute(
        """UPDATE quotation SET status = 'open', naming_series = ?,
           updated_at = datetime('now') WHERE id = ?""",
        (naming, args.quotation_id),
    )

    audit(conn, "erpclaw-selling", "submit-quotation", "quotation", args.quotation_id,
           new_values={"status": "open", "naming_series": naming})
    conn.commit()
    ok({"quotation_id": args.quotation_id, "naming_series": naming,
         "status": "open"})


# ---------------------------------------------------------------------------
# 10. convert-quotation-to-so
# ---------------------------------------------------------------------------

def convert_quotation_to_so(conn, args):
    """Create a sales order from a quotation."""
    if not args.quotation_id:
        err("--quotation-id is required")

    q = conn.execute("SELECT * FROM quotation WHERE id = ?",
                     (args.quotation_id,)).fetchone()
    if not q:
        err(f"Quotation {args.quotation_id} not found")
    if q["status"] not in ("open", "draft"):
        err(f"Cannot convert: quotation is '{q['status']}' (must be 'open' or 'draft')")

    q_dict = row_to_dict(q)

    # Fetch quotation items
    q_items = conn.execute(
        "SELECT * FROM quotation_item WHERE quotation_id = ? ORDER BY rowid",
        (args.quotation_id,),
    ).fetchall()
    if not q_items:
        err("Quotation has no items")

    so_id = str(uuid.uuid4())
    today = datetime.now(timezone.utc).strftime("%Y-%m-%d")

    # Insert parent sales_order first
    conn.execute(
        """INSERT INTO sales_order
           (id, customer_id, order_date, delivery_date, currency, exchange_rate,
            total_amount, tax_amount, grand_total, tax_template_id,
            payment_terms_id, status, quotation_id, company_id)
           VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'draft', ?, ?)""",
        (so_id, q_dict["customer_id"], today, args.delivery_date,
         q_dict["currency"], q_dict["exchange_rate"],
         q_dict["total_amount"], q_dict["tax_amount"], q_dict["grand_total"],
         q_dict["tax_template_id"], q_dict["payment_terms_id"],
         args.quotation_id, q_dict["company_id"]),
    )

    # Insert child sales_order_item rows from quotation items
    for qi in q_items:
        qi_dict = row_to_dict(qi)
        conn.execute(
            """INSERT INTO sales_order_item
               (id, sales_order_id, item_id, quantity, uom, rate, amount,
                discount_percentage, net_amount, warehouse_id)
               VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)""",
            (str(uuid.uuid4()), so_id, qi_dict["item_id"],
             qi_dict["quantity"], qi_dict["uom"], qi_dict["rate"],
             qi_dict["amount"], qi_dict["discount_percentage"],
             qi_dict["net_amount"], None),
        )

    # Update quotation status to ordered
    conn.execute(
        """UPDATE quotation SET status = 'ordered', converted_to = ?,
           updated_at = datetime('now') WHERE id = ?""",
        (so_id, args.quotation_id),
    )

    audit(conn, "erpclaw-selling", "convert-quotation-to-so", "quotation", args.quotation_id,
           new_values={"sales_order_id": so_id})
    conn.commit()
    ok({"quotation_id": args.quotation_id, "sales_order_id": so_id,
         "status": "ordered"})


# ---------------------------------------------------------------------------
# 11. add-sales-order
# ---------------------------------------------------------------------------

def add_sales_order(conn, args):
    """Create a sales order in draft."""
    if not args.customer_id:
        err("--customer-id is required")
    if not args.posting_date:
        err("--posting-date is required")
    if not args.items:
        err("--items is required (JSON array)")
    if not args.company_id:
        err("--company-id is required")

    cust = conn.execute("SELECT * FROM customer WHERE (id = ? OR name = ?) AND status = 'active'",
                        (args.customer_id, args.customer_id)).fetchone()
    if not cust:
        err(f"Active customer {args.customer_id} not found")
    args.customer_id = cust["id"]  # normalize to id
    if not conn.execute("SELECT id FROM company WHERE id = ?",
                        (args.company_id,)).fetchone():
        err(f"Company {args.company_id} not found")

    items = _parse_json_arg(args.items, "items")
    total_amount, item_rows = _calculate_line_items(
        conn, items, args.company_id, args.customer_id, args.posting_date,
        apply_pricing=True)

    tax_amount, _ = _calculate_tax(conn, total_amount, args.tax_template_id)
    grand_total = round_currency(total_amount + tax_amount)

    so_id = str(uuid.uuid4())

    # Insert parent sales_order first
    conn.execute(
        """INSERT INTO sales_order
           (id, customer_id, order_date, delivery_date, total_amount,
            tax_amount, grand_total, tax_template_id, status, company_id)
           VALUES (?, ?, ?, ?, ?, ?, ?, ?, 'draft', ?)""",
        (so_id, args.customer_id, args.posting_date, args.delivery_date,
         str(total_amount), str(tax_amount), str(grand_total),
         args.tax_template_id, args.company_id),
    )

    # Insert child sales_order_item rows
    for row in item_rows:
        conn.execute(
            """INSERT INTO sales_order_item
               (id, sales_order_id, item_id, quantity, uom, rate, amount,
                discount_percentage, net_amount, warehouse_id)
               VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)""",
            (str(uuid.uuid4()), so_id, row["item_id"], row["qty"],
             row["uom"], row["rate"], row["amount"],
             row["discount_percentage"], row["net_amount"],
             row["warehouse_id"]),
        )

    audit(conn, "erpclaw-selling", "add-sales-order", "sales_order", so_id,
           new_values={"customer_id": args.customer_id, "grand_total": str(grand_total)})
    conn.commit()
    ok({"sales_order_id": so_id, "total_amount": str(total_amount),
         "tax_amount": str(tax_amount), "grand_total": str(grand_total)})


# ---------------------------------------------------------------------------
# 12. update-sales-order
# ---------------------------------------------------------------------------

def update_sales_order(conn, args):
    """Update a draft sales order."""
    if not args.sales_order_id:
        err("--sales-order-id is required")

    so = conn.execute("SELECT * FROM sales_order WHERE id = ?",
                      (args.sales_order_id,)).fetchone()
    if not so:
        err(f"Sales order {args.sales_order_id} not found")
    if so["status"] != "draft":
        err(f"Cannot update: sales order is '{so['status']}' (must be 'draft')",
             suggestion="Cancel the document first, then make changes.")

    updated_fields = []

    if args.delivery_date is not None:
        conn.execute(
            "UPDATE sales_order SET delivery_date = ?, updated_at = datetime('now') WHERE id = ?",
            (args.delivery_date, args.sales_order_id))
        updated_fields.append("delivery_date")

    if args.items:
        items = _parse_json_arg(args.items, "items")
        total_amount, item_rows = _calculate_line_items(
            conn, items, so["company_id"], so["customer_id"], so["order_date"],
            apply_pricing=True)

        tax_amount, _ = _calculate_tax(conn, total_amount, so["tax_template_id"])
        grand_total = round_currency(total_amount + tax_amount)

        # Delete old items, insert new
        conn.execute("DELETE FROM sales_order_item WHERE sales_order_id = ?",
                     (args.sales_order_id,))
        for row in item_rows:
            conn.execute(
                """INSERT INTO sales_order_item
                   (id, sales_order_id, item_id, quantity, uom, rate, amount,
                    discount_percentage, net_amount, warehouse_id)
                   VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)""",
                (str(uuid.uuid4()), args.sales_order_id, row["item_id"],
                 row["qty"], row["uom"], row["rate"], row["amount"],
                 row["discount_percentage"], row["net_amount"],
                 row["warehouse_id"]),
            )

        conn.execute(
            """UPDATE sales_order SET total_amount = ?, tax_amount = ?,
               grand_total = ?, updated_at = datetime('now') WHERE id = ?""",
            (str(total_amount), str(tax_amount), str(grand_total),
             args.sales_order_id),
        )
        updated_fields.extend(["items", "total_amount", "tax_amount", "grand_total"])

    if not updated_fields:
        err("No fields to update")

    audit(conn, "erpclaw-selling", "update-sales-order", "sales_order", args.sales_order_id,
           new_values={"updated_fields": updated_fields})
    conn.commit()
    ok({"sales_order_id": args.sales_order_id, "updated_fields": updated_fields})


# ---------------------------------------------------------------------------
# 13. get-sales-order
# ---------------------------------------------------------------------------

def get_sales_order(conn, args):
    """Get sales order with items and fulfillment/billing status."""
    if not args.sales_order_id:
        err("--sales-order-id is required")

    so = conn.execute("SELECT * FROM sales_order WHERE id = ?",
                      (args.sales_order_id,)).fetchone()
    if not so:
        err(f"Sales order {args.sales_order_id} not found")

    data = row_to_dict(so)

    items = conn.execute(
        """SELECT soi.*, i.item_name, i.item_code
           FROM sales_order_item soi
           LEFT JOIN item i ON i.id = soi.item_id
           WHERE soi.sales_order_id = ?
           ORDER BY soi.rowid""",
        (args.sales_order_id,),
    ).fetchall()
    data["items"] = [row_to_dict(r) for r in items]

    # Delivery note summary
    dn_rows = conn.execute(
        """SELECT id, naming_series, status, posting_date
           FROM delivery_note WHERE sales_order_id = ? AND status != 'cancelled'""",
        (args.sales_order_id,),
    ).fetchall()
    data["delivery_notes"] = [row_to_dict(r) for r in dn_rows]

    # Invoice summary
    si_rows = conn.execute(
        """SELECT id, naming_series, status, posting_date, grand_total, outstanding_amount
           FROM sales_invoice WHERE sales_order_id = ? AND status != 'cancelled'""",
        (args.sales_order_id,),
    ).fetchall()
    data["sales_invoices"] = [row_to_dict(r) for r in si_rows]

    ok(data)


# ---------------------------------------------------------------------------
# 14. list-sales-orders
# ---------------------------------------------------------------------------

def list_sales_orders(conn, args):
    """Query sales orders with filtering."""
    conditions = ["1=1"]
    params = []

    if args.company_id:
        conditions.append("so.company_id = ?")
        params.append(args.company_id)
    if args.customer_id:
        conditions.append("so.customer_id = ?")
        params.append(args.customer_id)
    if args.doc_status:
        conditions.append("so.status = ?")
        params.append(args.doc_status)
    if args.from_date:
        conditions.append("so.order_date >= ?")
        params.append(args.from_date)
    if args.to_date:
        conditions.append("so.order_date <= ?")
        params.append(args.to_date)

    where = " AND ".join(conditions)

    count_row = conn.execute(
        f"SELECT COUNT(*) FROM sales_order so WHERE {where}", params
    ).fetchone()
    total_count = count_row[0]

    limit = int(args.limit) if args.limit else 20
    offset = int(args.offset) if args.offset else 0
    params.extend([limit, offset])

    rows = conn.execute(
        f"""SELECT so.id, so.naming_series, so.customer_id, so.order_date,
               so.delivery_date, so.grand_total, so.status, so.per_delivered,
               so.per_invoiced, so.company_id
           FROM sales_order so WHERE {where}
           ORDER BY so.order_date DESC, so.created_at DESC
           LIMIT ? OFFSET ?""",
        params,
    ).fetchall()

    ok({"sales_orders": [row_to_dict(r) for r in rows], "total_count": total_count,
         "limit": limit, "offset": offset, "has_more": offset + limit < total_count})


# ---------------------------------------------------------------------------
# 15. submit-sales-order
# ---------------------------------------------------------------------------

def submit_sales_order(conn, args):
    """Confirm a sales order with credit limit check."""
    if not args.sales_order_id:
        err("--sales-order-id is required")

    so = conn.execute("SELECT * FROM sales_order WHERE id = ?",
                      (args.sales_order_id,)).fetchone()
    if not so:
        err(f"Sales order {args.sales_order_id} not found")
    if so["status"] != "draft":
        err(f"Cannot submit: sales order is '{so['status']}' (must be 'draft')")

    so_dict = row_to_dict(so)
    customer_id = so_dict["customer_id"]
    company_id = so_dict["company_id"]
    this_grand_total = to_decimal(so_dict["grand_total"])

    # Verify customer is active
    cust = conn.execute("SELECT * FROM customer WHERE id = ?",
                        (customer_id,)).fetchone()
    if not cust:
        err(f"Customer {customer_id} not found")
    if cust["status"] != "active":
        err(f"Customer is '{cust['status']}', cannot confirm order")

    # Verify items have qty > 0 and rate > 0
    so_items = conn.execute(
        "SELECT * FROM sales_order_item WHERE sales_order_id = ?",
        (args.sales_order_id,),
    ).fetchall()
    if not so_items:
        err("Sales order has no items")
    for item in so_items:
        if to_decimal(item["quantity"]) <= 0:
            err(f"Item {item['item_id']}: quantity must be > 0")
        if to_decimal(item["rate"]) <= 0:
            err(f"Item {item['item_id']}: rate must be > 0")

    # Credit limit check
    credit_limit = to_decimal(cust["credit_limit"] or "0")
    if credit_limit > 0:
        # Outstanding invoices
        outstanding_row = conn.execute(
            """SELECT COALESCE(decimal_sum(outstanding_amount), '0') as total
               FROM sales_invoice
               WHERE customer_id = ? AND status IN ('submitted', 'overdue', 'partially_paid')
                 AND outstanding_amount + 0 > 0""",
            (customer_id,),
        ).fetchone()
        outstanding = to_decimal(str(outstanding_row["total"]))

        # Unbilled confirmed orders (excluding this one)
        unbilled_row = conn.execute(
            """SELECT COALESCE(decimal_sum(grand_total), '0') as total
               FROM sales_order
               WHERE customer_id = ? AND status = 'confirmed'
                 AND id != ?""",
            (customer_id, args.sales_order_id),
        ).fetchone()
        unconfirmed_orders = to_decimal(str(unbilled_row["total"]))

        total_exposure = outstanding + unconfirmed_orders + this_grand_total
        if total_exposure > credit_limit:
            err(
                f"Customer credit limit of {credit_limit} exceeded. "
                f"Current outstanding: {outstanding}. "
                f"Unbilled orders: {unconfirmed_orders}. "
                f"This order: {this_grand_total}. "
                f"Total exposure: {total_exposure}."
            )

    # Generate naming series
    naming = get_next_name(conn, "sales_order", company_id=company_id)

    conn.execute(
        """UPDATE sales_order SET status = 'confirmed', naming_series = ?,
           updated_at = datetime('now') WHERE id = ?""",
        (naming, args.sales_order_id),
    )

    audit(conn, "erpclaw-selling", "submit-sales-order", "sales_order", args.sales_order_id,
           new_values={"status": "confirmed", "naming_series": naming})
    conn.commit()
    ok({"sales_order_id": args.sales_order_id, "naming_series": naming,
         "status": "confirmed"})


# ---------------------------------------------------------------------------
# 16. cancel-sales-order
# ---------------------------------------------------------------------------

def cancel_sales_order(conn, args):
    """Cancel a sales order (only if no linked deliveries or invoices)."""
    if not args.sales_order_id:
        err("--sales-order-id is required")

    so = conn.execute("SELECT * FROM sales_order WHERE id = ?",
                      (args.sales_order_id,)).fetchone()
    if not so:
        err(f"Sales order {args.sales_order_id} not found")
    if so["status"] == "cancelled":
        err("Sales order is already cancelled")

    # Check for linked delivery notes
    dn_count = conn.execute(
        """SELECT COUNT(*) as cnt FROM delivery_note
           WHERE sales_order_id = ? AND status != 'cancelled'""",
        (args.sales_order_id,),
    ).fetchone()["cnt"]
    if dn_count > 0:
        err(f"Cannot cancel: {dn_count} active delivery note(s) linked to this order")

    # Check for linked invoices
    si_count = conn.execute(
        """SELECT COUNT(*) as cnt FROM sales_invoice
           WHERE sales_order_id = ? AND status != 'cancelled'""",
        (args.sales_order_id,),
    ).fetchone()["cnt"]
    if si_count > 0:
        err(f"Cannot cancel: {si_count} active sales invoice(s) linked to this order")

    conn.execute(
        """UPDATE sales_order SET status = 'cancelled',
           updated_at = datetime('now') WHERE id = ?""",
        (args.sales_order_id,),
    )

    audit(conn, "erpclaw-selling", "cancel-sales-order", "sales_order", args.sales_order_id,
           new_values={"status": "cancelled"})
    conn.commit()
    ok({"sales_order_id": args.sales_order_id, "status": "cancelled"})


# ---------------------------------------------------------------------------
# 17. create-delivery-note
# ---------------------------------------------------------------------------

def create_delivery_note(conn, args):
    """Create a delivery note from a sales order."""
    if not args.sales_order_id:
        err("--sales-order-id is required")

    so = conn.execute("SELECT * FROM sales_order WHERE id = ?",
                      (args.sales_order_id,)).fetchone()
    if not so:
        err(f"Sales order {args.sales_order_id} not found")
    if so["status"] not in ("confirmed", "partially_delivered"):
        err(f"Cannot create DN: sales order is '{so['status']}' (must be 'confirmed' or 'partially_delivered')")

    so_dict = row_to_dict(so)
    posting_date = args.posting_date or datetime.now(timezone.utc).strftime("%Y-%m-%d")

    # Fetch SO items
    so_items = conn.execute(
        "SELECT * FROM sales_order_item WHERE sales_order_id = ? ORDER BY rowid",
        (args.sales_order_id,),
    ).fetchall()
    if not so_items:
        err("Sales order has no items")

    # Determine items to deliver
    items_to_deliver = []
    if args.items:
        # Partial delivery: user specifies which items and how many
        partial_items = _parse_json_arg(args.items, "items")
        if not partial_items or not isinstance(partial_items, list):
            err("--items must be a non-empty JSON array")

        # Build a lookup of SO items
        so_item_map = {}
        for soi in so_items:
            soi_dict = row_to_dict(soi)
            so_item_map[soi_dict["item_id"]] = soi_dict
            so_item_map[soi_dict["id"]] = soi_dict  # also index by SO item ID

        for pi in partial_items:
            item_id = pi.get("item_id")
            so_item_id = pi.get("sales_order_item_id")
            qty = to_decimal(pi.get("qty", "0"))

            if so_item_id and so_item_id in so_item_map:
                soi_dict = so_item_map[so_item_id]
            elif item_id and item_id in so_item_map:
                soi_dict = so_item_map[item_id]
            else:
                err(f"Item {item_id or so_item_id} not found in sales order")

            remaining = to_decimal(soi_dict["quantity"]) - to_decimal(soi_dict["delivered_qty"])
            if qty <= 0:
                err(f"Item {soi_dict['item_id']}: qty must be > 0")
            if qty > remaining:
                err(f"Item {soi_dict['item_id']}: qty {qty} exceeds remaining {remaining}")

            items_to_deliver.append({
                "so_item": soi_dict,
                "qty": qty,
                "warehouse_id": pi.get("warehouse_id") or soi_dict.get("warehouse_id"),
                "batch_id": pi.get("batch_id"),
                "serial_numbers": pi.get("serial_numbers"),
            })
    else:
        # Full delivery: copy all undelivered items
        for soi in so_items:
            soi_dict = row_to_dict(soi)
            remaining = to_decimal(soi_dict["quantity"]) - to_decimal(soi_dict["delivered_qty"])
            if remaining > 0:
                items_to_deliver.append({
                    "so_item": soi_dict,
                    "qty": remaining,
                    "warehouse_id": soi_dict.get("warehouse_id"),
                    "batch_id": None,
                    "serial_numbers": None,
                })

    if not items_to_deliver:
        err("No items to deliver (all items already fully delivered)")

    total_qty = sum(item["qty"] for item in items_to_deliver)
    dn_id = str(uuid.uuid4())

    # Insert parent delivery_note first
    conn.execute(
        """INSERT INTO delivery_note
           (id, customer_id, posting_date, sales_order_id, status,
            total_qty, company_id)
           VALUES (?, ?, ?, ?, 'draft', ?, ?)""",
        (dn_id, so_dict["customer_id"], posting_date,
         args.sales_order_id, str(round_currency(total_qty)),
         so_dict["company_id"]),
    )

    # Insert child delivery_note_item rows
    for item in items_to_deliver:
        soi = item["so_item"]
        conn.execute(
            """INSERT INTO delivery_note_item
               (id, delivery_note_id, item_id, quantity, uom,
                sales_order_item_id, warehouse_id, batch_id,
                serial_numbers, rate, amount)
               VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)""",
            (str(uuid.uuid4()), dn_id, soi["item_id"],
             str(round_currency(item["qty"])), soi.get("uom"),
             soi["id"], item["warehouse_id"], item["batch_id"],
             item["serial_numbers"], soi["rate"],
             str(round_currency(item["qty"] * to_decimal(soi["rate"])))),
        )

    audit(conn, "erpclaw-selling", "create-delivery-note", "delivery_note", dn_id,
           new_values={"sales_order_id": args.sales_order_id,
                       "item_count": len(items_to_deliver)})
    conn.commit()
    ok({"delivery_note_id": dn_id, "sales_order_id": args.sales_order_id,
         "total_qty": str(round_currency(total_qty)),
         "item_count": len(items_to_deliver)})


# ---------------------------------------------------------------------------
# 18. get-delivery-note
# ---------------------------------------------------------------------------

def get_delivery_note(conn, args):
    """Get delivery note with items."""
    if not args.delivery_note_id:
        err("--delivery-note-id is required")

    dn = conn.execute("SELECT * FROM delivery_note WHERE id = ?",
                      (args.delivery_note_id,)).fetchone()
    if not dn:
        err(f"Delivery note {args.delivery_note_id} not found")

    data = row_to_dict(dn)

    items = conn.execute(
        """SELECT dni.*, i.item_name, i.item_code
           FROM delivery_note_item dni
           LEFT JOIN item i ON i.id = dni.item_id
           WHERE dni.delivery_note_id = ?
           ORDER BY dni.rowid""",
        (args.delivery_note_id,),
    ).fetchall()
    data["items"] = [row_to_dict(r) for r in items]
    ok(data)


# ---------------------------------------------------------------------------
# 19. list-delivery-notes
# ---------------------------------------------------------------------------

def list_delivery_notes(conn, args):
    """Query delivery notes with filtering."""
    conditions = ["1=1"]
    params = []

    if args.company_id:
        conditions.append("dn.company_id = ?")
        params.append(args.company_id)
    if args.customer_id:
        conditions.append("dn.customer_id = ?")
        params.append(args.customer_id)
    if args.doc_status:
        conditions.append("dn.status = ?")
        params.append(args.doc_status)
    if args.from_date:
        conditions.append("dn.posting_date >= ?")
        params.append(args.from_date)
    if args.to_date:
        conditions.append("dn.posting_date <= ?")
        params.append(args.to_date)

    where = " AND ".join(conditions)

    count_row = conn.execute(
        f"SELECT COUNT(*) FROM delivery_note dn WHERE {where}", params
    ).fetchone()
    total_count = count_row[0]

    limit = int(args.limit) if args.limit else 20
    offset = int(args.offset) if args.offset else 0
    params.extend([limit, offset])

    rows = conn.execute(
        f"""SELECT dn.id, dn.naming_series, dn.customer_id, dn.posting_date,
               dn.sales_order_id, dn.status, dn.total_qty, dn.company_id
           FROM delivery_note dn WHERE {where}
           ORDER BY dn.posting_date DESC, dn.created_at DESC
           LIMIT ? OFFSET ?""",
        params,
    ).fetchall()

    ok({"delivery_notes": [row_to_dict(r) for r in rows], "total_count": total_count,
         "limit": limit, "offset": offset, "has_more": offset + limit < total_count})


# ---------------------------------------------------------------------------
# 20. submit-delivery-note
# ---------------------------------------------------------------------------

def submit_delivery_note(conn, args):
    """Submit a delivery note: creates SLE + COGS GL entries."""
    if not args.delivery_note_id:
        err("--delivery-note-id is required")

    dn = conn.execute("SELECT * FROM delivery_note WHERE id = ?",
                      (args.delivery_note_id,)).fetchone()
    if not dn:
        err(f"Delivery note {args.delivery_note_id} not found")
    if dn["status"] != "draft":
        err(f"Cannot submit: delivery note is '{dn['status']}' (must be 'draft')")

    dn_dict = row_to_dict(dn)
    company_id = dn_dict["company_id"]
    posting_date = dn_dict["posting_date"]

    # Verify linked SO is not cancelled
    if dn_dict.get("sales_order_id"):
        so = conn.execute("SELECT status FROM sales_order WHERE id = ?",
                          (dn_dict["sales_order_id"],)).fetchone()
        if so and so["status"] == "cancelled":
            err("Cannot submit: linked sales order is cancelled")

    # Fetch DN items
    dn_items = conn.execute(
        "SELECT * FROM delivery_note_item WHERE delivery_note_id = ? ORDER BY rowid",
        (args.delivery_note_id,),
    ).fetchall()
    if not dn_items:
        err("Delivery note has no items")

    fiscal_year = _get_fiscal_year(conn, posting_date)
    cost_center_id = _get_cost_center(conn, company_id)
    cogs_account_id = _get_cogs_account(conn, company_id)

    # Build SLE entries (negative qty from warehouse)
    sle_entries = []
    for dni in dn_items:
        dni_dict = row_to_dict(dni)
        qty = to_decimal(dni_dict["quantity"])
        warehouse_id = dni_dict.get("warehouse_id")
        if not warehouse_id:
            # Try to get default warehouse from company
            comp = conn.execute("SELECT default_warehouse_id FROM company WHERE id = ?",
                                (company_id,)).fetchone()
            warehouse_id = comp["default_warehouse_id"] if comp else None
        if not warehouse_id:
            err(f"No warehouse specified for item {dni_dict['item_id']} and no default warehouse")

        sle_entries.append({
            "item_id": dni_dict["item_id"],
            "warehouse_id": warehouse_id,
            "actual_qty": str(round_currency(-qty)),
            "incoming_rate": "0",
            "batch_id": dni_dict.get("batch_id"),
            "serial_number": dni_dict.get("serial_numbers"),
            "fiscal_year": fiscal_year,
        })

    # Insert SLE entries
    try:
        sle_ids = insert_sle_entries(
            conn, sle_entries,
            voucher_type="delivery_note",
            voucher_id=args.delivery_note_id,
            posting_date=posting_date,
            company_id=company_id,
        )
    except ValueError as e:
        sys.stderr.write(f"[erpclaw-selling] {e}\n")
        err(f"SLE posting failed: {e}")

    # Build COGS GL entries from SLE data
    sle_rows = conn.execute(
        """SELECT * FROM stock_ledger_entry
           WHERE voucher_type = 'delivery_note' AND voucher_id = ?
             AND is_cancelled = 0""",
        (args.delivery_note_id,),
    ).fetchall()
    sle_dicts = [row_to_dict(r) for r in sle_rows]

    gl_entries = create_perpetual_inventory_gl(
        conn, sle_dicts,
        voucher_type="delivery_note",
        voucher_id=args.delivery_note_id,
        posting_date=posting_date,
        company_id=company_id,
        expense_account_id=cogs_account_id,
        cost_center_id=cost_center_id,
    )

    gl_ids = []
    if gl_entries:
        for gle in gl_entries:
            gle["fiscal_year"] = fiscal_year
        try:
            gl_ids = insert_gl_entries(
                conn, gl_entries,
                voucher_type="delivery_note",
                voucher_id=args.delivery_note_id,
                posting_date=posting_date,
                company_id=company_id,
                remarks=f"Delivery Note {dn_dict.get('naming_series', args.delivery_note_id)}",
            )
        except ValueError as e:
            sys.stderr.write(f"[erpclaw-selling] {e}\n")
            err(f"GL posting failed: {e}")

    # Generate naming series
    naming = get_next_name(conn, "delivery_note", company_id=company_id)

    # Update DN status
    conn.execute(
        """UPDATE delivery_note SET status = 'submitted', naming_series = ?,
           updated_at = datetime('now') WHERE id = ?""",
        (naming, args.delivery_note_id),
    )

    # Update SO delivered_qty
    if dn_dict.get("sales_order_id"):
        for dni in dn_items:
            dni_dict = row_to_dict(dni)
            if dni_dict.get("sales_order_item_id"):
                conn.execute(
                    """UPDATE sales_order_item
                       SET delivered_qty = CAST(
                           delivered_qty + 0 + ?
                       AS TEXT)
                       WHERE id = ?""",
                    (dni_dict["quantity"], dni_dict["sales_order_item_id"]),
                )

        # Recalculate SO per_delivered and status
        _update_so_delivery_status(conn, dn_dict["sales_order_id"])

    audit(conn, "erpclaw-selling", "submit-delivery-note", "delivery_note", args.delivery_note_id,
           new_values={"sle_count": len(sle_ids), "gl_count": len(gl_ids),
                       "naming_series": naming})
    conn.commit()
    ok({"delivery_note_id": args.delivery_note_id, "naming_series": naming,
         "status": "submitted", "sle_entries_created": len(sle_ids),
         "gl_entries_created": len(gl_ids)})


def _update_so_delivery_status(conn, sales_order_id: str):
    """Recalculate SO per_delivered and update status accordingly."""
    items = conn.execute(
        "SELECT quantity, delivered_qty FROM sales_order_item WHERE sales_order_id = ?",
        (sales_order_id,),
    ).fetchall()

    total_qty = Decimal("0")
    total_delivered = Decimal("0")
    for item in items:
        total_qty += to_decimal(item["quantity"])
        total_delivered += to_decimal(item["delivered_qty"])

    if total_qty > 0:
        per_delivered = round_currency(total_delivered / total_qty * Decimal("100"))
    else:
        per_delivered = Decimal("0")

    if per_delivered >= Decimal("100"):
        new_status = "fully_delivered"
    elif per_delivered > 0:
        new_status = "partially_delivered"
    else:
        return  # No change needed

    conn.execute(
        "UPDATE sales_order SET per_delivered = ?, status = ?, updated_at = datetime('now') WHERE id = ?",
        (str(per_delivered), new_status, sales_order_id),
    )


# ---------------------------------------------------------------------------
# 21. cancel-delivery-note
# ---------------------------------------------------------------------------

def cancel_delivery_note(conn, args):
    """Cancel a delivery note: reverse SLE + GL, update SO delivered_qty."""
    if not args.delivery_note_id:
        err("--delivery-note-id is required")

    dn = conn.execute("SELECT * FROM delivery_note WHERE id = ?",
                      (args.delivery_note_id,)).fetchone()
    if not dn:
        err(f"Delivery note {args.delivery_note_id} not found")
    if dn["status"] != "submitted":
        err(f"Cannot cancel: delivery note is '{dn['status']}' (must be 'submitted')")

    dn_dict = row_to_dict(dn)
    posting_date = dn_dict["posting_date"]

    # Check no invoices reference this DN
    si_count = conn.execute(
        """SELECT COUNT(*) as cnt FROM sales_invoice
           WHERE delivery_note_id = ? AND status != 'cancelled'""",
        (args.delivery_note_id,),
    ).fetchone()["cnt"]
    if si_count > 0:
        err(f"Cannot cancel: {si_count} active invoice(s) reference this delivery note")

    # Reverse SLE entries
    try:
        reversal_sle_ids = reverse_sle_entries(
            conn,
            voucher_type="delivery_note",
            voucher_id=args.delivery_note_id,
            posting_date=posting_date,
        )
    except ValueError as e:
        sys.stderr.write(f"[erpclaw-selling] {e}\n")
        err(f"SLE reversal failed: {e}")

    # Reverse GL entries
    reversal_gl_ids = []
    try:
        reversal_gl_ids = reverse_gl_entries(
            conn,
            voucher_type="delivery_note",
            voucher_id=args.delivery_note_id,
            posting_date=posting_date,
        )
    except ValueError:
        reversal_gl_ids = []

    # Update DN status
    conn.execute(
        """UPDATE delivery_note SET status = 'cancelled',
           updated_at = datetime('now') WHERE id = ?""",
        (args.delivery_note_id,),
    )

    # Reverse SO delivered_qty
    if dn_dict.get("sales_order_id"):
        dn_items = conn.execute(
            "SELECT * FROM delivery_note_item WHERE delivery_note_id = ?",
            (args.delivery_note_id,),
        ).fetchall()
        for dni in dn_items:
            dni_dict = row_to_dict(dni)
            if dni_dict.get("sales_order_item_id"):
                conn.execute(
                    """UPDATE sales_order_item
                       SET delivered_qty = CAST(
                           MAX(delivered_qty + 0 - ?, 0)
                       AS TEXT)
                       WHERE id = ?""",
                    (dni_dict["quantity"], dni_dict["sales_order_item_id"]),
                )

        # Recalculate SO delivery status
        _update_so_delivery_status_after_cancel(conn, dn_dict["sales_order_id"])

    audit(conn, "erpclaw-selling", "cancel-delivery-note", "delivery_note", args.delivery_note_id,
           new_values={"reversed": True})
    conn.commit()
    ok({"delivery_note_id": args.delivery_note_id, "status": "cancelled",
         "sle_reversals": len(reversal_sle_ids),
         "gl_reversals": len(reversal_gl_ids)})


def _update_so_delivery_status_after_cancel(conn, sales_order_id: str):
    """Recalculate SO delivery status after a DN cancellation."""
    items = conn.execute(
        "SELECT quantity, delivered_qty FROM sales_order_item WHERE sales_order_id = ?",
        (sales_order_id,),
    ).fetchall()

    total_qty = Decimal("0")
    total_delivered = Decimal("0")
    for item in items:
        total_qty += to_decimal(item["quantity"])
        total_delivered += to_decimal(item["delivered_qty"])

    if total_qty > 0:
        per_delivered = round_currency(total_delivered / total_qty * Decimal("100"))
    else:
        per_delivered = Decimal("0")

    if per_delivered >= Decimal("100"):
        new_status = "fully_delivered"
    elif per_delivered > 0:
        new_status = "partially_delivered"
    else:
        new_status = "confirmed"

    conn.execute(
        "UPDATE sales_order SET per_delivered = ?, status = ?, updated_at = datetime('now') WHERE id = ?",
        (str(per_delivered), new_status, sales_order_id),
    )


# ---------------------------------------------------------------------------
# 22. create-sales-invoice
# ---------------------------------------------------------------------------

def create_sales_invoice(conn, args):
    """Create a sales invoice from SO, DN, or standalone."""
    company_id = args.company_id
    customer_id = args.customer_id
    tax_template_id = args.tax_template_id
    sales_order_id = args.sales_order_id
    delivery_note_id = args.delivery_note_id
    posting_date = args.posting_date or datetime.now(timezone.utc).strftime("%Y-%m-%d")
    update_stock = 1  # Default for US perpetual inventory

    if sales_order_id:
        # Create from Sales Order
        so = conn.execute("SELECT * FROM sales_order WHERE id = ?",
                          (sales_order_id,)).fetchone()
        if not so:
            err(f"Sales order {sales_order_id} not found")
        if so["status"] not in ("confirmed", "partially_delivered", "fully_delivered",
                                 "partially_invoiced"):
            err(f"Cannot invoice: sales order is '{so['status']}'")

        so_dict = row_to_dict(so)
        customer_id = so_dict["customer_id"]
        company_id = so_dict["company_id"]
        tax_template_id = tax_template_id or so_dict.get("tax_template_id")
        payment_terms_id = so_dict.get("payment_terms_id")

        # If SO has delivery notes, set update_stock=0 (stock already moved)
        dn_count = conn.execute(
            "SELECT COUNT(*) as cnt FROM delivery_note WHERE sales_order_id = ? AND status = 'submitted'",
            (sales_order_id,),
        ).fetchone()["cnt"]
        if dn_count > 0:
            update_stock = 0

        so_items = conn.execute(
            "SELECT * FROM sales_order_item WHERE sales_order_id = ? ORDER BY rowid",
            (sales_order_id,),
        ).fetchall()

        si_items_data = []
        total_amount = Decimal("0")
        for soi in so_items:
            soi_dict = row_to_dict(soi)
            remaining = to_decimal(soi_dict["quantity"]) - to_decimal(soi_dict.get("invoiced_qty", "0"))
            if remaining <= 0:
                continue
            net = round_currency(remaining * to_decimal(soi_dict["rate"]))
            total_amount += net
            si_items_data.append({
                "item_id": soi_dict["item_id"],
                "qty": str(round_currency(remaining)),
                "uom": soi_dict.get("uom"),
                "rate": soi_dict["rate"],
                "amount": str(round_currency(remaining * to_decimal(soi_dict["rate"]))),
                "discount_percentage": soi_dict.get("discount_percentage", "0"),
                "net_amount": str(net),
                "sales_order_item_id": soi_dict["id"],
                "warehouse_id": soi_dict.get("warehouse_id"),
            })

        if not si_items_data:
            err("No uninvoiced items in sales order")

    elif delivery_note_id:
        # Create from Delivery Note
        dn = conn.execute("SELECT * FROM delivery_note WHERE id = ?",
                          (delivery_note_id,)).fetchone()
        if not dn:
            err(f"Delivery note {delivery_note_id} not found")
        if dn["status"] != "submitted":
            err(f"Cannot invoice: delivery note is '{dn['status']}'")

        dn_dict = row_to_dict(dn)
        customer_id = dn_dict["customer_id"]
        company_id = dn_dict["company_id"]
        sales_order_id = dn_dict.get("sales_order_id")
        update_stock = 0  # Stock already moved by DN

        dn_items = conn.execute(
            "SELECT * FROM delivery_note_item WHERE delivery_note_id = ? ORDER BY rowid",
            (delivery_note_id,),
        ).fetchall()

        si_items_data = []
        total_amount = Decimal("0")
        for dni in dn_items:
            dni_dict = row_to_dict(dni)
            qty = to_decimal(dni_dict["quantity"])
            rate = to_decimal(dni_dict["rate"])
            net = round_currency(qty * rate)
            total_amount += net
            si_items_data.append({
                "item_id": dni_dict["item_id"],
                "qty": str(round_currency(qty)),
                "uom": dni_dict.get("uom"),
                "rate": str(round_currency(rate)),
                "amount": str(round_currency(qty * rate)),
                "discount_percentage": "0",
                "net_amount": str(net),
                "sales_order_item_id": dni_dict.get("sales_order_item_id"),
                "delivery_note_item_id": dni_dict["id"],
                "warehouse_id": dni_dict.get("warehouse_id"),
            })

    else:
        # Standalone invoice
        if not customer_id:
            err("--customer-id is required for standalone invoices")
        if not args.items:
            err("--items is required for standalone invoices")
        if not company_id:
            err("--company-id is required for standalone invoices")

        cust_chk = conn.execute(
            "SELECT id FROM customer WHERE (id = ? OR name = ?) AND status = 'active'",
            (customer_id, customer_id)).fetchone()
        if not cust_chk:
            err(f"Active customer {customer_id} not found")
        customer_id = cust_chk["id"]  # normalize to id

        items = _parse_json_arg(args.items, "items")
        total_amount, item_rows = _calculate_line_items(
            conn, items, company_id, customer_id, posting_date)

        si_items_data = []
        for row in item_rows:
            si_items_data.append({
                "item_id": row["item_id"],
                "qty": row["qty"],
                "uom": row["uom"],
                "rate": row["rate"],
                "amount": row["amount"],
                "discount_percentage": row["discount_percentage"],
                "net_amount": row["net_amount"],
                "sales_order_item_id": None,
                "delivery_note_item_id": None,
                "warehouse_id": row["warehouse_id"],
            })

    # Calculate tax
    total_amount = round_currency(total_amount)
    tax_amount, tax_lines = _calculate_tax(conn, total_amount, tax_template_id)
    grand_total = round_currency(total_amount + tax_amount)

    # Resolve payment_terms_id: SO sets it above; fallback to customer default
    try:
        payment_terms_id  # set in SO code path
    except NameError:
        payment_terms_id = getattr(args, 'payment_terms_id', None)
        if not payment_terms_id and customer_id:
            cust = conn.execute(
                "SELECT payment_terms_id FROM customer WHERE id = ?",
                (customer_id,),
            ).fetchone()
            if cust:
                payment_terms_id = cust["payment_terms_id"]

    # Calculate due date from payment terms or default +30 days
    due_date = args.due_date
    if not due_date:
        due_days = 30
        if payment_terms_id:
            pt = conn.execute(
                "SELECT due_days FROM payment_terms WHERE id = ?",
                (payment_terms_id,),
            ).fetchone()
            if pt and pt["due_days"]:
                due_days = pt["due_days"]
        parts = posting_date.split("-")
        d = date_type(int(parts[0]), int(parts[1]), int(parts[2]))
        due_date = (d + timedelta(days=due_days)).isoformat()

    si_id = str(uuid.uuid4())

    # Insert parent sales_invoice first
    conn.execute(
        """INSERT INTO sales_invoice
           (id, customer_id, posting_date, due_date, total_amount, tax_amount,
            grand_total, outstanding_amount, tax_template_id, payment_terms_id,
            status, sales_order_id, delivery_note_id, update_stock, company_id)
           VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'draft', ?, ?, ?, ?)""",
        (si_id, customer_id, posting_date, due_date,
         str(total_amount), str(tax_amount), str(grand_total), str(grand_total),
         tax_template_id, payment_terms_id, sales_order_id, delivery_note_id,
         update_stock, company_id),
    )

    # Insert child sales_invoice_item rows
    for row in si_items_data:
        conn.execute(
            """INSERT INTO sales_invoice_item
               (id, sales_invoice_id, item_id, quantity, uom, rate, amount,
                discount_percentage, net_amount, sales_order_item_id,
                delivery_note_item_id)
               VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)""",
            (str(uuid.uuid4()), si_id, row["item_id"], row["qty"],
             row["uom"], row["rate"], row["amount"],
             row["discount_percentage"], row["net_amount"],
             row.get("sales_order_item_id"), row.get("delivery_note_item_id")),
        )

    audit(conn, "erpclaw-selling", "create-sales-invoice", "sales_invoice", si_id,
           new_values={"customer_id": customer_id, "grand_total": str(grand_total),
                       "update_stock": update_stock})
    conn.commit()
    ok({"sales_invoice_id": si_id, "total_amount": str(total_amount),
         "tax_amount": str(tax_amount), "grand_total": str(grand_total),
         "update_stock": update_stock})


# ---------------------------------------------------------------------------
# 23. update-sales-invoice
# ---------------------------------------------------------------------------

def update_sales_invoice(conn, args):
    """Update a draft sales invoice."""
    if not args.sales_invoice_id:
        err("--sales-invoice-id is required")

    si = conn.execute("SELECT * FROM sales_invoice WHERE id = ?",
                      (args.sales_invoice_id,)).fetchone()
    if not si:
        err(f"Sales invoice {args.sales_invoice_id} not found")
    if si["status"] != "draft":
        err(f"Cannot update: sales invoice is '{si['status']}' (must be 'draft')",
             suggestion="Cancel the document first, then make changes.")

    updated_fields = []

    if args.due_date is not None:
        conn.execute(
            "UPDATE sales_invoice SET due_date = ?, updated_at = datetime('now') WHERE id = ?",
            (args.due_date, args.sales_invoice_id))
        updated_fields.append("due_date")

    if args.items:
        items = _parse_json_arg(args.items, "items")
        total_amount, item_rows = _calculate_line_items(
            conn, items, si["company_id"])

        tax_amount, _ = _calculate_tax(conn, total_amount, si["tax_template_id"])
        grand_total = round_currency(total_amount + tax_amount)

        conn.execute("DELETE FROM sales_invoice_item WHERE sales_invoice_id = ?",
                     (args.sales_invoice_id,))
        for row in item_rows:
            conn.execute(
                """INSERT INTO sales_invoice_item
                   (id, sales_invoice_id, item_id, quantity, uom, rate, amount,
                    discount_percentage, net_amount)
                   VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)""",
                (str(uuid.uuid4()), args.sales_invoice_id, row["item_id"],
                 row["qty"], row["uom"], row["rate"], row["amount"],
                 row["discount_percentage"], row["net_amount"]),
            )

        conn.execute(
            """UPDATE sales_invoice SET total_amount = ?, tax_amount = ?,
               grand_total = ?, outstanding_amount = ?,
               updated_at = datetime('now') WHERE id = ?""",
            (str(total_amount), str(tax_amount), str(grand_total),
             str(grand_total), args.sales_invoice_id),
        )
        updated_fields.extend(["items", "total_amount", "tax_amount", "grand_total"])

    if not updated_fields:
        err("No fields to update")

    audit(conn, "erpclaw-selling", "update-sales-invoice", "sales_invoice", args.sales_invoice_id,
           new_values={"updated_fields": updated_fields})
    conn.commit()
    ok({"sales_invoice_id": args.sales_invoice_id,
         "updated_fields": updated_fields})


# ---------------------------------------------------------------------------
# 24. get-sales-invoice
# ---------------------------------------------------------------------------

def get_sales_invoice(conn, args):
    """Get sales invoice with items and payment info."""
    if not args.sales_invoice_id:
        err("--sales-invoice-id is required")

    si = conn.execute("SELECT * FROM sales_invoice WHERE id = ?",
                      (args.sales_invoice_id,)).fetchone()
    if not si:
        err(f"Sales invoice {args.sales_invoice_id} not found")

    data = row_to_dict(si)

    items = conn.execute(
        """SELECT sii.*, i.item_name, i.item_code
           FROM sales_invoice_item sii
           LEFT JOIN item i ON i.id = sii.item_id
           WHERE sii.sales_invoice_id = ?
           ORDER BY sii.rowid""",
        (args.sales_invoice_id,),
    ).fetchall()
    data["items"] = [row_to_dict(r) for r in items]

    # Payment ledger entries (match both sales_invoice and credit_note voucher types)
    ple_rows = conn.execute(
        """SELECT * FROM payment_ledger_entry
           WHERE against_voucher_type IN ('sales_invoice', 'credit_note')
             AND against_voucher_id = ?""",
        (args.sales_invoice_id,),
    ).fetchall()
    data["payments"] = [row_to_dict(r) for r in ple_rows]

    ok(data)


# ---------------------------------------------------------------------------
# 25. list-sales-invoices
# ---------------------------------------------------------------------------

def list_sales_invoices(conn, args):
    """Query sales invoices with filtering."""
    conditions = ["1=1"]
    params = []

    if args.company_id:
        conditions.append("si.company_id = ?")
        params.append(args.company_id)
    if args.customer_id:
        conditions.append("si.customer_id = ?")
        params.append(args.customer_id)
    if args.doc_status:
        conditions.append("si.status = ?")
        params.append(args.doc_status)
    if args.from_date:
        conditions.append("si.posting_date >= ?")
        params.append(args.from_date)
    if args.to_date:
        conditions.append("si.posting_date <= ?")
        params.append(args.to_date)

    where = " AND ".join(conditions)

    count_row = conn.execute(
        f"SELECT COUNT(*) FROM sales_invoice si WHERE {where}", params
    ).fetchone()
    total_count = count_row[0]

    limit = int(args.limit) if args.limit else 20
    offset = int(args.offset) if args.offset else 0
    params.extend([limit, offset])

    rows = conn.execute(
        f"""SELECT si.id, si.naming_series, si.customer_id, si.posting_date,
               si.due_date, si.grand_total, si.outstanding_amount,
               si.status, si.is_return, si.company_id
           FROM sales_invoice si WHERE {where}
           ORDER BY si.posting_date DESC, si.created_at DESC
           LIMIT ? OFFSET ?""",
        params,
    ).fetchall()

    ok({"sales_invoices": [row_to_dict(r) for r in rows],
         "total_count": total_count, "limit": limit, "offset": offset,
         "has_more": offset + limit < total_count})


# ---------------------------------------------------------------------------
# 26. submit-sales-invoice
# ---------------------------------------------------------------------------

def submit_sales_invoice(conn, args):
    """Submit a sales invoice: revenue GL + PLE (+ optional SLE)."""
    if not args.sales_invoice_id:
        err("--sales-invoice-id is required")

    si = conn.execute("SELECT * FROM sales_invoice WHERE id = ?",
                      (args.sales_invoice_id,)).fetchone()
    if not si:
        err(f"Sales invoice {args.sales_invoice_id} not found")
    if si["status"] != "draft":
        err(f"Cannot submit: sales invoice is '{si['status']}' (must be 'draft')")

    si_dict = row_to_dict(si)
    company_id = si_dict["company_id"]
    customer_id = si_dict["customer_id"]
    posting_date = si_dict["posting_date"]
    grand_total = to_decimal(si_dict["grand_total"])
    total_amount = to_decimal(si_dict["total_amount"])
    tax_amount = to_decimal(si_dict["tax_amount"])
    update_stock = si_dict.get("update_stock", 1)

    # Verify customer is active
    cust = conn.execute("SELECT * FROM customer WHERE id = ?",
                        (customer_id,)).fetchone()
    if not cust:
        err(f"Customer {customer_id} not found")
    if cust["status"] != "active":
        err(f"Customer is '{cust['status']}', cannot submit invoice")

    # Verify items
    si_items = conn.execute(
        "SELECT * FROM sales_invoice_item WHERE sales_invoice_id = ?",
        (args.sales_invoice_id,),
    ).fetchall()
    if not si_items:
        err("Sales invoice has no items")

    fiscal_year = _get_fiscal_year(conn, posting_date)
    cost_center_id = _get_cost_center(conn, company_id)

    # --- Revenue GL entries ---
    receivable_account_id = _get_receivable_account(conn, company_id)
    if not receivable_account_id:
        err("No receivable account found for company")
    income_account_id = _get_income_account(conn, company_id)
    if not income_account_id:
        err("No income account found for company")

    gl_entries = []

    # DR: Accounts Receivable (with party)
    is_return = bool(si_dict.get("is_return", 0))
    voucher_type = "credit_note" if is_return else "sales_invoice"

    # For credit notes, amounts are stored as negatives -- use abs() and swap DR/CR
    abs_grand_total = abs(grand_total)
    abs_total_amount = abs(total_amount)
    abs_tax_amount = abs(tax_amount)

    if is_return:
        # Credit note: CR Receivable (reverse the original DR)
        gl_entries.append({
            "account_id": receivable_account_id,
            "debit": "0",
            "credit": str(round_currency(abs_grand_total)),
            "party_type": "customer",
            "party_id": customer_id,
        })
    else:
        gl_entries.append({
            "account_id": receivable_account_id,
            "debit": str(round_currency(abs_grand_total)),
            "credit": "0",
            "party_type": "customer",
            "party_id": customer_id,
        })

    # CR: Revenue (for credit notes: DR Revenue to reverse)
    if is_return:
        gl_entries.append({
            "account_id": income_account_id,
            "debit": str(round_currency(abs_total_amount)),
            "credit": "0",
            "cost_center_id": cost_center_id,
        })
    else:
        gl_entries.append({
            "account_id": income_account_id,
            "debit": "0",
            "credit": str(round_currency(abs_total_amount)),
            "cost_center_id": cost_center_id,
        })

    # CR: Tax Payable (if tax exists) — for returns, use abs() and DR (reverse)
    if abs_tax_amount > 0:
        _, tax_lines = _calculate_tax(conn, abs_total_amount, si_dict.get("tax_template_id"))
        for tl in tax_lines:
            amt = abs(tl["amount"])
            if amt > 0:
                if is_return:
                    gl_entries.append({
                        "account_id": tl["account_id"],
                        "debit": str(round_currency(amt)),
                        "credit": "0",
                        "cost_center_id": cost_center_id,
                    })
                else:
                    gl_entries.append({
                        "account_id": tl["account_id"],
                        "debit": "0",
                        "credit": str(round_currency(amt)),
                        "cost_center_id": cost_center_id,
                    })

    # Generate naming series early so GL remarks include the human-readable name
    if is_return:
        naming = get_next_name(conn, "credit_note", company_id=company_id)
    else:
        naming = get_next_name(conn, "sales_invoice", company_id=company_id)

    # Add fiscal_year to all GL entries
    for gle in gl_entries:
        gle["fiscal_year"] = fiscal_year

    try:
        gl_ids = insert_gl_entries(
            conn, gl_entries,
            voucher_type=voucher_type,
            voucher_id=args.sales_invoice_id,
            posting_date=posting_date,
            company_id=company_id,
            remarks=f"{'Credit Note' if is_return else 'Sales Invoice'} {naming or args.sales_invoice_id}",
        )
    except ValueError as e:
        sys.stderr.write(f"[erpclaw-selling] {e}\n")
        err(f"GL posting failed: {e}")

    # --- Optional SLE + COGS GL (if update_stock = 1) ---
    sle_ids = []
    cogs_gl_ids = []
    if update_stock:
        cogs_account_id = _get_cogs_account(conn, company_id)
        sle_entries = []
        for sii in si_items:
            sii_dict = row_to_dict(sii)
            qty = abs(to_decimal(sii_dict["quantity"]))
            if qty <= 0:
                continue

            # Determine warehouse
            warehouse_id = None
            if sii_dict.get("sales_order_item_id"):
                soi = conn.execute(
                    "SELECT warehouse_id FROM sales_order_item WHERE id = ?",
                    (sii_dict["sales_order_item_id"],),
                ).fetchone()
                if soi and soi["warehouse_id"]:
                    warehouse_id = soi["warehouse_id"]
            if not warehouse_id:
                comp = conn.execute(
                    "SELECT default_warehouse_id FROM company WHERE id = ?",
                    (company_id,),
                ).fetchone()
                warehouse_id = comp["default_warehouse_id"] if comp else None

            if not warehouse_id:
                continue

            # Check if item is a stock item
            item_row = conn.execute(
                "SELECT is_stock_item FROM item WHERE id = ?",
                (sii_dict["item_id"],),
            ).fetchone()
            if not item_row or not item_row["is_stock_item"]:
                continue

            sle_entries.append({
                "item_id": sii_dict["item_id"],
                "warehouse_id": warehouse_id,
                "actual_qty": str(round_currency(-qty if not is_return else qty)),
                "incoming_rate": "0" if not is_return else str(round_currency(to_decimal(sii_dict["rate"]))),
                "fiscal_year": fiscal_year,
            })

        if sle_entries:
            try:
                sle_ids = insert_sle_entries(
                    conn, sle_entries,
                    voucher_type=voucher_type,
                    voucher_id=args.sales_invoice_id,
                    posting_date=posting_date,
                    company_id=company_id,
                )
            except ValueError as e:
                sys.stderr.write(f"[erpclaw-selling] {e}\n")
                err(f"SLE posting failed: {e}")

            # COGS GL from SLE
            sle_rows = conn.execute(
                """SELECT * FROM stock_ledger_entry
                   WHERE voucher_type = ? AND voucher_id = ?
                     AND is_cancelled = 0""",
                (voucher_type, args.sales_invoice_id),
            ).fetchall()
            sle_dicts = [row_to_dict(r) for r in sle_rows]

            cogs_gl_entries = create_perpetual_inventory_gl(
                conn, sle_dicts,
                voucher_type=voucher_type,
                voucher_id=args.sales_invoice_id,
                posting_date=posting_date,
                company_id=company_id,
                expense_account_id=cogs_account_id,
                cost_center_id=cost_center_id,
            )

            if cogs_gl_entries:
                for gle in cogs_gl_entries:
                    gle["fiscal_year"] = fiscal_year
                # Insert COGS GL entries via shared lib (entry_set="cogs"
                # allows multiple GL sets per voucher without idempotency conflict)
                try:
                    cogs_gl_ids = insert_gl_entries(
                        conn, cogs_gl_entries,
                        voucher_type=voucher_type,
                        voucher_id=args.sales_invoice_id,
                        posting_date=posting_date,
                        company_id=company_id,
                        remarks=f"COGS - {'Credit Note' if is_return else 'Sales Invoice'} {args.sales_invoice_id}",
                        entry_set="cogs",
                    )
                except ValueError as e:
                    sys.stderr.write(f"[erpclaw-selling] {e}\n")
                    err(f"COGS GL posting failed: {e}")

    # --- Payment Ledger Entry (PLE) ---
    # For credit notes: PLE amount is negative (reduces receivable)
    ple_amount = str(round_currency(abs_grand_total if not is_return else -abs_grand_total))
    ple_id = str(uuid.uuid4())
    # For credit notes, against_voucher points to the original invoice
    against_vtype = voucher_type
    against_vid = args.sales_invoice_id
    if is_return and si_dict.get("return_against"):
        against_vtype = "sales_invoice"
        against_vid = si_dict["return_against"]
    conn.execute(
        """INSERT INTO payment_ledger_entry
           (id, posting_date, account_id, party_type, party_id,
            voucher_type, voucher_id, against_voucher_type, against_voucher_id,
            amount, amount_in_account_currency, currency, remarks)
           VALUES (?, ?, ?, 'customer', ?, ?, ?, ?, ?,
                   ?, ?, 'USD', ?)""",
        (ple_id, posting_date, receivable_account_id, customer_id,
         voucher_type, args.sales_invoice_id, against_vtype, against_vid,
         ple_amount, ple_amount,
         f"{'Credit Note' if is_return else 'Sales Invoice'} {args.sales_invoice_id}"),
    )

    # Update invoice status (naming already generated above before GL posting)
    conn.execute(
        """UPDATE sales_invoice SET status = 'submitted', naming_series = ?,
           updated_at = datetime('now') WHERE id = ?""",
        (naming, args.sales_invoice_id),
    )

    # Update SO invoiced_qty if linked
    if si_dict.get("sales_order_id"):
        for sii in si_items:
            sii_dict = row_to_dict(sii)
            if sii_dict.get("sales_order_item_id"):
                conn.execute(
                    """UPDATE sales_order_item
                       SET invoiced_qty = CAST(
                           invoiced_qty + 0 + ?
                       AS TEXT)
                       WHERE id = ?""",
                    (sii_dict["quantity"], sii_dict["sales_order_item_id"]),
                )
        _update_so_invoice_status(conn, si_dict["sales_order_id"])

    audit(conn, "erpclaw-selling", "submit-sales-invoice", "sales_invoice", args.sales_invoice_id,
           new_values={"naming_series": naming, "gl_count": len(gl_ids),
                       "sle_count": len(sle_ids), "update_stock": update_stock})
    conn.commit()
    ok({"sales_invoice_id": args.sales_invoice_id, "naming_series": naming,
         "status": "submitted",
         "gl_entries_created": len(gl_ids) + len(cogs_gl_ids),
         "sle_entries_created": len(sle_ids),
         "update_stock": bool(update_stock)})


def _update_so_invoice_status(conn, sales_order_id: str):
    """Recalculate SO per_invoiced and update status."""
    items = conn.execute(
        "SELECT quantity, invoiced_qty FROM sales_order_item WHERE sales_order_id = ?",
        (sales_order_id,),
    ).fetchall()

    total_qty = Decimal("0")
    total_invoiced = Decimal("0")
    for item in items:
        total_qty += to_decimal(item["quantity"])
        total_invoiced += to_decimal(item["invoiced_qty"])

    if total_qty > 0:
        per_invoiced = round_currency(total_invoiced / total_qty * Decimal("100"))
    else:
        per_invoiced = Decimal("0")

    so = conn.execute("SELECT status FROM sales_order WHERE id = ?",
                      (sales_order_id,)).fetchone()
    if so and so["status"] not in ("cancelled",):
        if per_invoiced >= Decimal("100"):
            new_status = "fully_invoiced"
        elif per_invoiced > 0:
            new_status = "partially_invoiced"
        else:
            return

        conn.execute(
            "UPDATE sales_order SET per_invoiced = ?, status = ?, updated_at = datetime('now') WHERE id = ?",
            (str(per_invoiced), new_status, sales_order_id),
        )


# ---------------------------------------------------------------------------
# 27. cancel-sales-invoice
# ---------------------------------------------------------------------------

def cancel_sales_invoice(conn, args):
    """Cancel a sales invoice: reverse GL + PLE (+ optional SLE)."""
    if not args.sales_invoice_id:
        err("--sales-invoice-id is required")

    si = conn.execute("SELECT * FROM sales_invoice WHERE id = ?",
                      (args.sales_invoice_id,)).fetchone()
    if not si:
        err(f"Sales invoice {args.sales_invoice_id} not found")
    if si["status"] not in ("submitted", "overdue", "partially_paid"):
        err(f"Cannot cancel: sales invoice is '{si['status']}' (must be 'submitted', 'overdue', or 'partially_paid')")

    si_dict = row_to_dict(si)
    posting_date = si_dict["posting_date"]
    update_stock = si_dict.get("update_stock", 1)
    is_return = bool(si_dict.get("is_return", 0))
    cancel_voucher_type = "credit_note" if is_return else "sales_invoice"

    # Reverse GL entries
    try:
        reversal_gl_ids = reverse_gl_entries(
            conn,
            voucher_type=cancel_voucher_type,
            voucher_id=args.sales_invoice_id,
            posting_date=posting_date,
        )
    except ValueError as e:
        sys.stderr.write(f"[erpclaw-selling] {e}\n")
        err(f"GL reversal failed: {e}")

    # COGS GL entries (entry_set="cogs") are reversed by the same call above
    # since reverse_gl_entries finds ALL entries for (voucher_type, voucher_id)

    # Reverse SLE if update_stock
    reversal_sle_ids = []
    if update_stock:
        try:
            reversal_sle_ids = reverse_sle_entries(
                conn,
                voucher_type=cancel_voucher_type,
                voucher_id=args.sales_invoice_id,
                posting_date=posting_date,
            )
        except ValueError:
            reversal_sle_ids = []

    # Mark PLE as delinked
    conn.execute(
        """UPDATE payment_ledger_entry SET delinked = 1, updated_at = datetime('now')
           WHERE voucher_type = ? AND voucher_id = ?""",
        (cancel_voucher_type, args.sales_invoice_id),
    )

    # Update invoice status
    conn.execute(
        """UPDATE sales_invoice SET status = 'cancelled', outstanding_amount = '0',
           updated_at = datetime('now') WHERE id = ?""",
        (args.sales_invoice_id,),
    )

    # Reverse SO invoiced_qty if linked
    if si_dict.get("sales_order_id"):
        si_items = conn.execute(
            "SELECT * FROM sales_invoice_item WHERE sales_invoice_id = ?",
            (args.sales_invoice_id,),
        ).fetchall()
        for sii in si_items:
            sii_dict = row_to_dict(sii)
            if sii_dict.get("sales_order_item_id"):
                conn.execute(
                    """UPDATE sales_order_item
                       SET invoiced_qty = CAST(
                           MAX(invoiced_qty + 0 - ?, 0)
                       AS TEXT)
                       WHERE id = ?""",
                    (sii_dict["quantity"], sii_dict["sales_order_item_id"]),
                )
        _update_so_invoice_status(conn, si_dict["sales_order_id"])

    audit(conn, "erpclaw-selling", "cancel-sales-invoice", "sales_invoice", args.sales_invoice_id,
           new_values={"reversed": True})
    conn.commit()
    ok({"sales_invoice_id": args.sales_invoice_id, "status": "cancelled",
         "gl_reversals": len(reversal_gl_ids),
         "sle_reversals": len(reversal_sle_ids)})


# ---------------------------------------------------------------------------
# 28. create-credit-note
# ---------------------------------------------------------------------------

def create_credit_note(conn, args):
    """Create a credit note (return invoice with negative amounts)."""
    if not args.against_invoice_id:
        err("--against-invoice-id is required")
    if not args.items:
        err("--items is required (JSON array)")

    orig = conn.execute("SELECT * FROM sales_invoice WHERE id = ?",
                        (args.against_invoice_id,)).fetchone()
    if not orig:
        err(f"Original invoice {args.against_invoice_id} not found")
    if orig["status"] not in ("submitted", "overdue", "partially_paid", "paid"):
        err(f"Cannot create credit note: original invoice is '{orig['status']}'")

    orig_dict = row_to_dict(orig)
    company_id = orig_dict["company_id"]
    customer_id = orig_dict["customer_id"]
    posting_date = args.posting_date or datetime.now(timezone.utc).strftime("%Y-%m-%d")

    items = _parse_json_arg(args.items, "items")
    if not items or not isinstance(items, list):
        err("--items must be a non-empty JSON array")

    # Validate return items against original invoice items
    orig_items = conn.execute(
        "SELECT * FROM sales_invoice_item WHERE sales_invoice_id = ?",
        (args.against_invoice_id,),
    ).fetchall()
    orig_item_map = {row_to_dict(oi)["item_id"]: row_to_dict(oi) for oi in orig_items}

    total_amount = Decimal("0")
    si_items_data = []
    for i, item in enumerate(items):
        item_id = item.get("item_id")
        if not item_id:
            err(f"Item {i}: item_id is required")
        if item_id not in orig_item_map:
            err(f"Item {i}: item {item_id} not found in original invoice")

        qty = to_decimal(item.get("qty", "0"))
        if qty <= 0:
            err(f"Item {i}: qty must be > 0")

        orig_item = orig_item_map[item_id]
        if qty > to_decimal(orig_item["quantity"]):
            err(f"Item {i}: return qty {qty} exceeds original qty {orig_item['quantity']}")

        rate = to_decimal(item.get("rate", orig_item["rate"]))
        net = round_currency(qty * rate)
        total_amount += net

        si_items_data.append({
            "item_id": item_id,
            "qty": str(round_currency(-qty)),  # Negative for returns
            "uom": orig_item.get("uom"),
            "rate": str(round_currency(rate)),
            "amount": str(round_currency(-qty * rate)),
            "discount_percentage": "0",
            "net_amount": str(round_currency(-net)),
            "sales_order_item_id": orig_item.get("sales_order_item_id"),
        })

    # Negate totals for credit note
    total_amount = round_currency(-total_amount)
    tax_amount, _ = _calculate_tax(conn, abs(total_amount), orig_dict.get("tax_template_id"))
    tax_amount = round_currency(-tax_amount)
    grand_total = round_currency(total_amount + tax_amount)

    si_id = str(uuid.uuid4())

    # Insert parent sales_invoice (is_return=1)
    conn.execute(
        """INSERT INTO sales_invoice
           (id, customer_id, posting_date, due_date, total_amount, tax_amount,
            grand_total, outstanding_amount, tax_template_id, status,
            sales_order_id, is_return, return_against, update_stock, company_id)
           VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, 'draft', ?, 1, ?, ?, ?)""",
        (si_id, customer_id, posting_date, posting_date,
         str(total_amount), str(tax_amount), str(grand_total), str(grand_total),
         orig_dict.get("tax_template_id"), orig_dict.get("sales_order_id"),
         args.against_invoice_id, orig_dict.get("update_stock", 0),
         company_id),
    )

    # Insert child items
    for row in si_items_data:
        conn.execute(
            """INSERT INTO sales_invoice_item
               (id, sales_invoice_id, item_id, quantity, uom, rate, amount,
                discount_percentage, net_amount, sales_order_item_id)
               VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)""",
            (str(uuid.uuid4()), si_id, row["item_id"], row["qty"],
             row["uom"], row["rate"], row["amount"],
             row["discount_percentage"], row["net_amount"],
             row.get("sales_order_item_id")),
        )

    audit(conn, "erpclaw-selling", "create-credit-note", "sales_invoice", si_id,
           new_values={"against_invoice_id": args.against_invoice_id,
                       "reason": args.reason, "grand_total": str(grand_total)})
    conn.commit()
    ok({"credit_note_id": si_id, "against_invoice_id": args.against_invoice_id,
         "grand_total": str(grand_total), "is_return": True})


# ---------------------------------------------------------------------------
# 29. update-invoice-outstanding (cross-skill)
# ---------------------------------------------------------------------------

def update_invoice_outstanding(conn, args):
    """Called by erpclaw-payments when payment is allocated."""
    if not args.sales_invoice_id:
        err("--sales-invoice-id is required")
    if not args.amount:
        err("--amount is required")

    si = conn.execute("SELECT * FROM sales_invoice WHERE id = ?",
                      (args.sales_invoice_id,)).fetchone()
    if not si:
        err(f"Sales invoice {args.sales_invoice_id} not found")
    if si["status"] not in ("submitted", "overdue", "partially_paid"):
        err(f"Cannot update outstanding: invoice is '{si['status']}'")

    current_outstanding = to_decimal(si["outstanding_amount"])
    payment_amount = to_decimal(args.amount)

    if payment_amount <= 0:
        err("--amount must be > 0")
    if payment_amount > current_outstanding:
        err(f"Payment amount {payment_amount} exceeds outstanding {current_outstanding}")

    new_outstanding = round_currency(current_outstanding - payment_amount)

    if new_outstanding <= Decimal("0"):
        new_status = "paid"
        new_outstanding = Decimal("0")
    else:
        new_status = "partially_paid"

    conn.execute(
        """UPDATE sales_invoice SET outstanding_amount = ?, status = ?,
           updated_at = datetime('now') WHERE id = ?""",
        (str(new_outstanding), new_status, args.sales_invoice_id),
    )

    audit(conn, "erpclaw-selling", "update-invoice-outstanding", "sales_invoice",
           args.sales_invoice_id,
           new_values={"payment_amount": str(payment_amount),
                       "new_outstanding": str(new_outstanding),
                       "new_status": new_status})
    conn.commit()
    ok({"sales_invoice_id": args.sales_invoice_id,
         "outstanding_amount": str(new_outstanding),
         "status": new_status})


# ---------------------------------------------------------------------------
# 30. add-sales-partner
# ---------------------------------------------------------------------------

def add_sales_partner(conn, args):
    """Create a sales partner."""
    if not args.name:
        err("--name is required")
    if not args.commission_rate:
        err("--commission-rate is required")

    rate = round_currency(to_decimal(args.commission_rate))
    sp_id = str(uuid.uuid4())

    try:
        conn.execute(
            "INSERT INTO sales_partner (id, name, commission_rate) VALUES (?, ?, ?)",
            (sp_id, args.name, str(rate)),
        )
    except sqlite3.IntegrityError as e:
        sys.stderr.write(f"[erpclaw-selling] {e}\n")
        err("Sales partner creation failed — check for duplicates or invalid data")

    audit(conn, "erpclaw-selling", "add-sales-partner", "sales_partner", sp_id,
           new_values={"name": args.name, "commission_rate": str(rate)})
    conn.commit()
    ok({"sales_partner_id": sp_id, "name": args.name,
         "commission_rate": str(rate)})


# ---------------------------------------------------------------------------
# 31. list-sales-partners
# ---------------------------------------------------------------------------

def list_sales_partners(conn, args):
    """List sales partners."""
    limit = int(args.limit) if args.limit else 20
    offset = int(args.offset) if args.offset else 0

    rows = conn.execute(
        "SELECT * FROM sales_partner ORDER BY name LIMIT ? OFFSET ?",
        (limit, offset),
    ).fetchall()

    count_row = conn.execute("SELECT COUNT(*) FROM sales_partner").fetchone()
    total_count = count_row[0]
    ok({"sales_partners": [row_to_dict(r) for r in rows],
         "total_count": total_count, "limit": limit, "offset": offset,
         "has_more": offset + limit < total_count})


# ---------------------------------------------------------------------------
# 32. add-recurring-template
# ---------------------------------------------------------------------------

def add_recurring_template(conn, args):
    """Create a recurring invoice template."""
    if not args.customer_id:
        err("--customer-id is required")
    if not args.items:
        err("--items is required (JSON array)")
    if not args.frequency:
        err("--frequency is required")
    if not args.start_date:
        err("--start-date is required")
    if not args.company_id:
        err("--company-id is required")

    if args.frequency not in VALID_FREQUENCIES:
        err(f"--frequency must be one of: {', '.join(VALID_FREQUENCIES)}")

    cust_sub = conn.execute(
        "SELECT id FROM customer WHERE (id = ? OR name = ?) AND status = 'active'",
        (args.customer_id, args.customer_id)).fetchone()
    if not cust_sub:
        err(f"Active customer {args.customer_id} not found")
    args.customer_id = cust_sub["id"]  # normalize to id
    if not conn.execute("SELECT id FROM company WHERE id = ?",
                        (args.company_id,)).fetchone():
        err(f"Company {args.company_id} not found")

    items = _parse_json_arg(args.items, "items")
    if not items or not isinstance(items, list):
        err("--items must be a non-empty JSON array")

    rt_id = str(uuid.uuid4())

    # Insert parent template first
    conn.execute(
        """INSERT INTO recurring_invoice_template
           (id, customer_id, frequency, start_date, end_date,
            next_invoice_date, tax_template_id, payment_terms_id,
            status, company_id)
           VALUES (?, ?, ?, ?, ?, ?, ?, ?, 'draft', ?)""",
        (rt_id, args.customer_id, args.frequency, args.start_date,
         args.end_date, args.start_date, args.tax_template_id,
         args.payment_terms_id, args.company_id),
    )

    # Insert child items
    for i, item in enumerate(items):
        item_id = item.get("item_id")
        if not item_id:
            err(f"Item {i}: item_id is required")
        qty = to_decimal(item.get("qty", "0"))
        rate = to_decimal(item.get("rate", "0"))
        amount = round_currency(qty * rate)

        conn.execute(
            """INSERT INTO recurring_invoice_template_item
               (id, template_id, item_id, quantity, uom, rate, amount)
               VALUES (?, ?, ?, ?, ?, ?, ?)""",
            (str(uuid.uuid4()), rt_id, item_id, str(round_currency(qty)),
             item.get("uom"), str(round_currency(rate)), str(amount)),
        )

    audit(conn, "erpclaw-selling", "add-recurring-template", "recurring_invoice_template", rt_id,
           new_values={"customer_id": args.customer_id, "frequency": args.frequency})
    conn.commit()
    ok({"template_id": rt_id, "frequency": args.frequency,
         "start_date": args.start_date, "next_invoice_date": args.start_date})


# ---------------------------------------------------------------------------
# 33. update-recurring-template
# ---------------------------------------------------------------------------

def update_recurring_template(conn, args):
    """Update a recurring invoice template."""
    if not args.template_id:
        err("--template-id is required")

    rt = conn.execute("SELECT * FROM recurring_invoice_template WHERE id = ?",
                      (args.template_id,)).fetchone()
    if not rt:
        err(f"Template {args.template_id} not found")

    updated_fields = []

    if args.frequency is not None:
        if args.frequency not in VALID_FREQUENCIES:
            err(f"--frequency must be one of: {', '.join(VALID_FREQUENCIES)}")
        conn.execute(
            "UPDATE recurring_invoice_template SET frequency = ?, updated_at = datetime('now') WHERE id = ?",
            (args.frequency, args.template_id))
        updated_fields.append("frequency")

    if args.template_status is not None:
        if args.template_status not in ("active", "paused", "cancelled"):
            err("--status must be 'active', 'paused', or 'cancelled'")
        conn.execute(
            "UPDATE recurring_invoice_template SET status = ?, updated_at = datetime('now') WHERE id = ?",
            (args.template_status, args.template_id))
        updated_fields.append("status")

    if args.items:
        items = _parse_json_arg(args.items, "items")
        if not items or not isinstance(items, list):
            err("--items must be a non-empty JSON array")

        conn.execute(
            "DELETE FROM recurring_invoice_template_item WHERE template_id = ?",
            (args.template_id,))

        for i, item in enumerate(items):
            item_id = item.get("item_id")
            if not item_id:
                err(f"Item {i}: item_id is required")
            qty = to_decimal(item.get("qty", "0"))
            rate = to_decimal(item.get("rate", "0"))
            amount = round_currency(qty * rate)

            conn.execute(
                """INSERT INTO recurring_invoice_template_item
                   (id, template_id, item_id, quantity, uom, rate, amount)
                   VALUES (?, ?, ?, ?, ?, ?, ?)""",
                (str(uuid.uuid4()), args.template_id, item_id,
                 str(round_currency(qty)), item.get("uom"),
                 str(round_currency(rate)), str(amount)),
            )
        updated_fields.append("items")

    if not updated_fields:
        err("No fields to update")

    audit(conn, "erpclaw-selling", "update-recurring-template", "recurring_invoice_template",
           args.template_id, new_values={"updated_fields": updated_fields})
    conn.commit()
    ok({"template_id": args.template_id, "updated_fields": updated_fields})


# ---------------------------------------------------------------------------
# 34. list-recurring-templates
# ---------------------------------------------------------------------------

def list_recurring_templates(conn, args):
    """List recurring invoice templates."""
    conditions = ["1=1"]
    params = []

    if args.company_id:
        conditions.append("rt.company_id = ?")
        params.append(args.company_id)
    if args.customer_id:
        conditions.append("rt.customer_id = ?")
        params.append(args.customer_id)
    if args.template_status:
        conditions.append("rt.status = ?")
        params.append(args.template_status)

    where = " AND ".join(conditions)

    count_row = conn.execute(
        f"SELECT COUNT(*) FROM recurring_invoice_template rt WHERE {where}", params
    ).fetchone()
    total_count = count_row[0]

    limit = int(args.limit) if args.limit else 20
    offset = int(args.offset) if args.offset else 0
    params.extend([limit, offset])

    rows = conn.execute(
        f"""SELECT rt.*
           FROM recurring_invoice_template rt WHERE {where}
           ORDER BY rt.next_invoice_date ASC
           LIMIT ? OFFSET ?""",
        params,
    ).fetchall()

    ok({"recurring_templates": [row_to_dict(r) for r in rows],
         "total_count": total_count, "limit": limit, "offset": offset,
         "has_more": offset + limit < total_count})


# ---------------------------------------------------------------------------
# 35. generate-recurring-invoices
# ---------------------------------------------------------------------------

def generate_recurring_invoices(conn, args):
    """Cron: auto-generate invoices from due templates."""
    if not args.company_id:
        err("--company-id is required")

    as_of_date = args.as_of_date or datetime.now(timezone.utc).strftime("%Y-%m-%d")

    templates = conn.execute(
        """SELECT * FROM recurring_invoice_template
           WHERE status = 'active'
             AND next_invoice_date <= ?
             AND company_id = ?
             AND (end_date IS NULL OR end_date >= ?)""",
        (as_of_date, args.company_id, as_of_date),
    ).fetchall()

    invoices_generated = []
    templates_completed = 0
    errors = []

    for tmpl in templates:
        tmpl_dict = row_to_dict(tmpl)
        template_id = tmpl_dict["id"]
        customer_id = tmpl_dict["customer_id"]
        company_id = tmpl_dict["company_id"]
        next_date = tmpl_dict["next_invoice_date"]
        frequency = tmpl_dict["frequency"]

        try:
            # Fetch template items
            tmpl_items = conn.execute(
                "SELECT * FROM recurring_invoice_template_item WHERE template_id = ?",
                (template_id,),
            ).fetchall()
            if not tmpl_items:
                errors.append({"template_id": template_id,
                               "error": "Template has no items"})
                continue

            # Build invoice items
            total_amount = Decimal("0")
            si_items_data = []
            for ti in tmpl_items:
                ti_dict = row_to_dict(ti)
                qty = to_decimal(ti_dict["quantity"])
                rate = to_decimal(ti_dict["rate"])
                net = round_currency(qty * rate)
                total_amount += net
                si_items_data.append({
                    "item_id": ti_dict["item_id"],
                    "qty": str(round_currency(qty)),
                    "uom": ti_dict.get("uom"),
                    "rate": str(round_currency(rate)),
                    "amount": str(round_currency(qty * rate)),
                    "net_amount": str(net),
                })

            total_amount = round_currency(total_amount)
            tax_amount, tax_lines = _calculate_tax(
                conn, total_amount, tmpl_dict.get("tax_template_id"))
            grand_total = round_currency(total_amount + tax_amount)

            # Calculate due date
            parts = next_date.split("-")
            d = date_type(int(parts[0]), int(parts[1]), int(parts[2]))
            due_date = (d + timedelta(days=30)).isoformat()

            si_id = str(uuid.uuid4())

            # Create draft invoice
            conn.execute(
                """INSERT INTO sales_invoice
                   (id, customer_id, posting_date, due_date, total_amount,
                    tax_amount, grand_total, outstanding_amount,
                    tax_template_id, status, update_stock, company_id)
                   VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, 'draft', 0, ?)""",
                (si_id, customer_id, next_date, due_date,
                 str(total_amount), str(tax_amount), str(grand_total),
                 str(grand_total), tmpl_dict.get("tax_template_id"),
                 company_id),
            )

            for row in si_items_data:
                conn.execute(
                    """INSERT INTO sales_invoice_item
                       (id, sales_invoice_id, item_id, quantity, uom, rate,
                        amount, discount_percentage, net_amount)
                       VALUES (?, ?, ?, ?, ?, ?, ?, '0', ?)""",
                    (str(uuid.uuid4()), si_id, row["item_id"], row["qty"],
                     row["uom"], row["rate"], row["amount"], row["net_amount"]),
                )

            # Auto-submit the invoice
            # We need to do this inline since we cannot call argparse again
            fiscal_year = _get_fiscal_year(conn, next_date)
            cost_center_id = _get_cost_center(conn, company_id)
            receivable_account_id = _get_receivable_account(conn, company_id)
            income_account_id = _get_income_account(conn, company_id)

            if receivable_account_id and income_account_id:
                gl_entries = [
                    {
                        "account_id": receivable_account_id,
                        "debit": str(round_currency(grand_total)),
                        "credit": "0",
                        "party_type": "customer",
                        "party_id": customer_id,
                        "fiscal_year": fiscal_year,
                    },
                    {
                        "account_id": income_account_id,
                        "debit": "0",
                        "credit": str(round_currency(total_amount)),
                        "cost_center_id": cost_center_id,
                        "fiscal_year": fiscal_year,
                    },
                ]

                # Add tax GL entries
                if tax_amount > 0 and tax_lines:
                    for tl in tax_lines:
                        amt = abs(tl["amount"])
                        if amt > 0:
                            gl_entries.append({
                                "account_id": tl["account_id"],
                                "debit": "0",
                                "credit": str(round_currency(amt)),
                                "cost_center_id": cost_center_id,
                                "fiscal_year": fiscal_year,
                            })

                try:
                    insert_gl_entries(
                        conn, gl_entries,
                        voucher_type="sales_invoice",
                        voucher_id=si_id,
                        posting_date=next_date,
                        company_id=company_id,
                        remarks=f"Recurring Invoice from template {template_id}",
                    )
                except ValueError as e:
                    sys.stderr.write(f"[erpclaw-selling] {e}\n")
                    errors.append({"template_id": template_id,
                                   "error": "GL posting failed"})
                    conn.rollback()
                    continue

                # PLE
                ple_id = str(uuid.uuid4())
                conn.execute(
                    """INSERT INTO payment_ledger_entry
                       (id, posting_date, account_id, party_type, party_id,
                        voucher_type, voucher_id, against_voucher_type,
                        against_voucher_id, amount, amount_in_account_currency,
                        currency, remarks)
                       VALUES (?, ?, ?, 'customer', ?, 'sales_invoice', ?,
                               'sales_invoice', ?, ?, ?, 'USD', ?)""",
                    (ple_id, next_date, receivable_account_id, customer_id,
                     si_id, si_id, str(round_currency(grand_total)),
                     str(round_currency(grand_total)),
                     f"Recurring Invoice {si_id}"),
                )

            # Generate naming and mark as submitted
            naming = get_next_name(conn, "sales_invoice", company_id=company_id)
            conn.execute(
                """UPDATE sales_invoice SET status = 'submitted',
                   naming_series = ?, updated_at = datetime('now') WHERE id = ?""",
                (naming, si_id),
            )

            # Update template dates
            new_next = _next_invoice_date(next_date, frequency)
            conn.execute(
                """UPDATE recurring_invoice_template
                   SET last_invoice_date = ?, next_invoice_date = ?,
                       updated_at = datetime('now') WHERE id = ?""",
                (next_date, new_next, template_id),
            )

            # Check if template is completed
            if tmpl_dict.get("end_date") and new_next > tmpl_dict["end_date"]:
                conn.execute(
                    """UPDATE recurring_invoice_template SET status = 'completed',
                       updated_at = datetime('now') WHERE id = ?""",
                    (template_id,),
                )
                templates_completed += 1

            invoices_generated.append({
                "template_id": template_id,
                "invoice_id": si_id,
                "naming_series": naming,
                "customer_id": customer_id,
                "amount": str(grand_total),
            })

        except Exception as e:
            errors.append({"template_id": template_id,
                           "error": str(e)})
            continue

    conn.commit()
    ok({"invoices_generated": len(invoices_generated),
         "templates_processed": len(templates),
         "templates_completed": templates_completed,
         "invoices": invoices_generated,
         "errors": errors})


# ---------------------------------------------------------------------------
# 36. status
# ---------------------------------------------------------------------------

def status_action(conn, args):
    """Selling summary for a company."""
    company_id = args.company_id
    if not company_id:
        row = conn.execute("SELECT id FROM company LIMIT 1").fetchone()
        if not row:
            err("No company found. Create one with erpclaw-setup first.",
                 suggestion="Run 'tutorial' to create a demo company, or 'setup company' to create your own.")
        company_id = row["id"]

    # Customer count
    cust_count = conn.execute(
        "SELECT COUNT(*) as cnt FROM customer WHERE company_id = ?",
        (company_id,),
    ).fetchone()["cnt"]

    # Quotations by status
    q_rows = conn.execute(
        "SELECT status, COUNT(*) as cnt FROM quotation WHERE company_id = ? GROUP BY status",
        (company_id,),
    ).fetchall()
    q_counts = {}
    for row in q_rows:
        q_counts[row["status"]] = row["cnt"]

    # Sales orders by status
    so_rows = conn.execute(
        "SELECT status, COUNT(*) as cnt FROM sales_order WHERE company_id = ? GROUP BY status",
        (company_id,),
    ).fetchall()
    so_counts = {}
    for row in so_rows:
        so_counts[row["status"]] = row["cnt"]

    # Sales invoices by status
    si_rows = conn.execute(
        "SELECT status, COUNT(*) as cnt FROM sales_invoice WHERE company_id = ? GROUP BY status",
        (company_id,),
    ).fetchall()
    si_counts = {}
    for row in si_rows:
        si_counts[row["status"]] = row["cnt"]

    # Total outstanding
    outstanding_row = conn.execute(
        """SELECT COALESCE(decimal_sum(outstanding_amount), '0') as total
           FROM sales_invoice
           WHERE company_id = ? AND status IN ('submitted', 'overdue', 'partially_paid')
             AND outstanding_amount + 0 > 0""",
        (company_id,),
    ).fetchone()
    total_outstanding = str(round_currency(
        to_decimal(str(outstanding_row["total"]))))

    ok({
        "customers": cust_count,
        "quotations": q_counts,
        "sales_orders": so_counts,
        "sales_invoices": si_counts,
        "total_outstanding": total_outstanding,
    })


# ---------------------------------------------------------------------------
# import-customers
# ---------------------------------------------------------------------------

def import_customers(conn, args):
    """Bulk import customers from a CSV file.

    CSV columns: name, customer_type (optional), territory (optional),
    default_currency (optional), email (optional), phone (optional).
    """
    csv_path = args.csv_path
    company_id = args.company_id
    if not csv_path:
        err("--csv-path is required")
    if not company_id:
        err("--company-id is required")

    # Path safety: resolve symlinks, require .csv extension, must be a regular file
    csv_real = os.path.realpath(csv_path)
    if not csv_real.lower().endswith(".csv"):
        err("--csv-path must point to a .csv file")
    if not os.path.isfile(csv_real):
        err(f"File not found: {csv_path}")

    from erpclaw_lib.csv_import import validate_csv, parse_csv_rows

    errors = validate_csv(csv_real, "customer")
    if errors:
        err(f"CSV validation failed: {'; '.join(errors)}")

    rows = parse_csv_rows(csv_real, "customer")
    if not rows:
        err("CSV file is empty")

    imported = 0
    skipped = 0
    for row in rows:
        name = row.get("name", "")

        # Check for duplicate
        existing = conn.execute(
            "SELECT id FROM customer WHERE name = ? AND company_id = ?",
            (name, company_id),
        ).fetchone()
        if existing:
            skipped += 1
            continue

        customer_id = str(uuid.uuid4())
        naming = get_next_name(conn, "customer")
        conn.execute(
            """INSERT INTO customer (id, name, naming_series, customer_type,
               territory, default_currency, email, phone, tax_id, company_id)
               VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)""",
            (customer_id, name, naming,
             row.get("customer_type", "Company"),
             row.get("territory"),
             row.get("default_currency", "USD"),
             row.get("email"), row.get("phone"), row.get("tax_id"),
             company_id),
        )
        imported += 1

    conn.commit()
    ok({"imported": imported, "skipped": skipped, "total_rows": len(rows)})


# ---------------------------------------------------------------------------
# Intercompany Account Map
# ---------------------------------------------------------------------------


def add_intercompany_account_map(conn, args):
    """Add a mapping from source company account to target company account."""
    source_company_id = args.company_id
    target_company_id = args.target_company_id
    source_account_id = args.source_account_id
    target_account_id = args.target_account_id

    if not source_company_id:
        err("--company-id (source company) is required")
    if not target_company_id:
        err("--target-company-id is required")
    if not source_account_id:
        err("--source-account-id is required")
    if not target_account_id:
        err("--target-account-id is required")
    if source_company_id == target_company_id:
        err("Source and target company must be different")

    # Validate companies exist
    for cid, label in [(source_company_id, "Source"), (target_company_id, "Target")]:
        if not conn.execute("SELECT 1 FROM company WHERE id = ?", (cid,)).fetchone():
            err(f"{label} company not found: {cid}")

    # Validate accounts exist and belong to correct companies
    src_acct = conn.execute(
        "SELECT id, company_id FROM account WHERE id = ?", (source_account_id,)
    ).fetchone()
    if not src_acct:
        err(f"Source account not found: {source_account_id}")
    if src_acct["company_id"] != source_company_id:
        err("Source account does not belong to source company")

    tgt_acct = conn.execute(
        "SELECT id, company_id FROM account WHERE id = ?", (target_account_id,)
    ).fetchone()
    if not tgt_acct:
        err(f"Target account not found: {target_account_id}")
    if tgt_acct["company_id"] != target_company_id:
        err("Target account does not belong to target company")

    # Check for duplicate
    existing = conn.execute(
        """SELECT id FROM intercompany_account_map
           WHERE source_company_id = ? AND target_company_id = ?
             AND source_account_id = ?""",
        (source_company_id, target_company_id, source_account_id),
    ).fetchone()
    if existing:
        err("Mapping already exists for this source account and company pair")

    map_id = str(uuid.uuid4())
    conn.execute(
        """INSERT INTO intercompany_account_map
           (id, source_company_id, target_company_id, source_account_id, target_account_id)
           VALUES (?, ?, ?, ?, ?)""",
        (map_id, source_company_id, target_company_id, source_account_id, target_account_id),
    )
    conn.commit()
    ok({"map_id": map_id})


def list_intercompany_account_maps(conn, args):
    """List intercompany account mappings for a company pair."""
    source_company_id = args.company_id
    target_company_id = args.target_company_id

    if not source_company_id:
        err("--company-id (source company) is required")

    query = """SELECT m.id, m.source_company_id, m.target_company_id,
                      m.source_account_id, sa.name as source_account_name,
                      m.target_account_id, ta.name as target_account_name
               FROM intercompany_account_map m
               JOIN account sa ON sa.id = m.source_account_id
               JOIN account ta ON ta.id = m.target_account_id
               WHERE m.source_company_id = ?"""
    params = [source_company_id]
    if target_company_id:
        query += " AND m.target_company_id = ?"
        params.append(target_company_id)

    rows = conn.execute(query, params).fetchall()
    ok({"mappings": [dict(r) for r in rows], "total": len(rows)})


# ---------------------------------------------------------------------------
# Intercompany Invoice Mirroring
# ---------------------------------------------------------------------------


def create_intercompany_invoice(conn, args):
    """Create a mirror purchase invoice in the target company from a sales invoice.

    Requires:
      --sales-invoice-id  (the submitted SI in source company)
      --target-company-id (the company where the mirror PI will be created)
      --supplier-id       (the supplier in target company representing source company)
    """
    si_id = args.sales_invoice_id
    target_company_id = args.target_company_id
    supplier_id = args.supplier_id

    if not si_id:
        err("--sales-invoice-id is required")
    if not target_company_id:
        err("--target-company-id is required")
    if not supplier_id:
        err("--supplier-id is required (supplier in target company representing source)")

    # Fetch the sales invoice
    si = conn.execute(
        "SELECT * FROM sales_invoice WHERE id = ?", (si_id,)
    ).fetchone()
    if not si:
        err(f"Sales invoice not found: {si_id}")

    source_company_id = si["company_id"]

    # Validations
    if si["status"] not in ("submitted", "partially_paid", "paid", "overdue"):
        err(f"Sales invoice must be submitted (current: {si['status']})")
    if source_company_id == target_company_id:
        err("Source and target company must be different")
    if si["is_intercompany"] == 1:
        err("Sales invoice is already an intercompany invoice")

    # Validate target company exists
    target_company = conn.execute(
        "SELECT id, name FROM company WHERE id = ?", (target_company_id,)
    ).fetchone()
    if not target_company:
        err(f"Target company not found: {target_company_id}")

    # Validate supplier exists in target company
    supplier = conn.execute(
        "SELECT id, company_id FROM supplier WHERE id = ?", (supplier_id,)
    ).fetchone()
    if not supplier:
        err(f"Supplier not found: {supplier_id}")
    if supplier["company_id"] != target_company_id:
        err("Supplier does not belong to target company")

    # Fetch SI items
    si_items = conn.execute(
        "SELECT * FROM sales_invoice_item WHERE sales_invoice_id = ?", (si_id,)
    ).fetchall()
    if not si_items:
        err("Sales invoice has no items")

    # Look up account mappings (source income → target expense)
    mappings = {}
    map_rows = conn.execute(
        """SELECT source_account_id, target_account_id
           FROM intercompany_account_map
           WHERE source_company_id = ? AND target_company_id = ?""",
        (source_company_id, target_company_id),
    ).fetchall()
    for m in map_rows:
        mappings[m["source_account_id"]] = m["target_account_id"]

    # Find a default expense account in target company if no mappings
    default_expense = conn.execute(
        """SELECT id FROM account WHERE root_type = 'expense'
           AND company_id = ? AND is_group = 0 LIMIT 1""",
        (target_company_id,),
    ).fetchone()
    default_expense_id = default_expense["id"] if default_expense else None

    # Create mirror purchase invoice (draft)
    pi_id = str(uuid.uuid4())
    posting_date = si["posting_date"]
    due_date = si["due_date"]
    total_amount = si["total_amount"]
    tax_amount = si["tax_amount"]
    grand_total = si["grand_total"]
    currency = si["currency"]
    exchange_rate = si["exchange_rate"]

    conn.execute(
        """INSERT INTO purchase_invoice
           (id, supplier_id, posting_date, due_date, currency, exchange_rate,
            total_amount, tax_amount, grand_total, outstanding_amount,
            tax_template_id, status, update_stock,
            is_intercompany, intercompany_reference_id, company_id)
           VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'draft', 0, 1, ?, ?)""",
        (pi_id, supplier_id, posting_date, due_date, currency, exchange_rate,
         total_amount, tax_amount, grand_total, grand_total,
         si["tax_template_id"],
         si_id, target_company_id),
    )

    # Create mirror PI items
    for item in si_items:
        pi_item_id = str(uuid.uuid4())
        # Find expense account for this item via mappings
        expense_acct = default_expense_id
        if mappings:
            # Try to find source income account from SI's company
            income_acct = conn.execute(
                """SELECT id FROM account WHERE root_type = 'income'
                   AND company_id = ? AND is_group = 0 LIMIT 1""",
                (source_company_id,),
            ).fetchone()
            if income_acct and income_acct["id"] in mappings:
                expense_acct = mappings[income_acct["id"]]

        conn.execute(
            """INSERT INTO purchase_invoice_item
               (id, purchase_invoice_id, item_id, quantity, uom, rate, amount,
                expense_account_id)
               VALUES (?, ?, ?, ?, ?, ?, ?, ?)""",
            (pi_item_id, pi_id, item["item_id"], item["quantity"],
             item["uom"], item["rate"], item["amount"], expense_acct),
        )

    # Mark SI as intercompany
    conn.execute(
        """UPDATE sales_invoice SET is_intercompany = 1,
           intercompany_reference_id = ? WHERE id = ?""",
        (pi_id, si_id),
    )

    conn.commit()
    ok({
        "purchase_invoice_id": pi_id,
        "sales_invoice_id": si_id,
        "source_company_id": source_company_id,
        "target_company_id": target_company_id,
        "total_amount": total_amount,
        "grand_total": grand_total,
        "items_mirrored": len(si_items),
    })


def list_intercompany_invoices(conn, args):
    """List all intercompany invoices for a company."""
    company_id = args.company_id
    if not company_id:
        err("--company-id is required")

    limit = int(args.limit or "20")
    offset = int(args.offset or "0")

    # Sales invoices that are intercompany (this company is seller)
    si_rows = conn.execute(
        """SELECT si.id, si.naming_series, si.posting_date, si.grand_total,
                  si.status, si.intercompany_reference_id,
                  c.name as customer_name, 'sales' as direction
           FROM sales_invoice si
           JOIN customer c ON c.id = si.customer_id
           WHERE si.company_id = ? AND si.is_intercompany = 1
           ORDER BY si.posting_date DESC
           LIMIT ? OFFSET ?""",
        (company_id, limit, offset),
    ).fetchall()

    # Purchase invoices that are intercompany (this company is buyer)
    pi_rows = conn.execute(
        """SELECT pi.id, pi.naming_series, pi.posting_date, pi.grand_total,
                  pi.status, pi.intercompany_reference_id,
                  s.name as supplier_name, 'purchase' as direction
           FROM purchase_invoice pi
           JOIN supplier s ON s.id = pi.supplier_id
           WHERE pi.company_id = ? AND pi.is_intercompany = 1
           ORDER BY pi.posting_date DESC
           LIMIT ? OFFSET ?""",
        (company_id, limit, offset),
    ).fetchall()

    invoices = [dict(r) for r in si_rows] + [dict(r) for r in pi_rows]
    ok({"invoices": invoices, "total": len(invoices)})


def cancel_intercompany_invoice(conn, args):
    """Cancel an intercompany sales invoice and cascade to the mirror purchase invoice.

    Requires: --sales-invoice-id
    """
    si_id = args.sales_invoice_id
    if not si_id:
        err("--sales-invoice-id is required")

    si = conn.execute(
        "SELECT * FROM sales_invoice WHERE id = ?", (si_id,)
    ).fetchone()
    if not si:
        err(f"Sales invoice not found: {si_id}")
    if si["is_intercompany"] != 1:
        err("Sales invoice is not an intercompany invoice")

    pi_id = si["intercompany_reference_id"]
    posting_date = si["posting_date"]
    source_company_id = si["company_id"]

    # --- Cancel the Sales Invoice (source company) ---
    si_status = si["status"]
    si_gl_reversals = 0
    si_sle_reversals = 0
    if si_status in ("submitted", "overdue", "partially_paid"):
        voucher_type = "credit_note" if si["is_return"] else "sales_invoice"

        # Reverse GL (includes COGS entries — reverses all for the voucher)
        try:
            rev_gl = reverse_gl_entries(conn, voucher_type, si_id, posting_date)
            si_gl_reversals = len(rev_gl)
        except ValueError:
            pass  # No GL entries to reverse

        # Reverse SLE if applicable
        if si["update_stock"]:
            try:
                rev_sle = reverse_sle_entries(conn, voucher_type, si_id, posting_date)
                si_sle_reversals = len(rev_sle)
            except ValueError:
                pass

        # Mark PLE as delinked
        conn.execute(
            "UPDATE payment_ledger_entry SET delinked = 1 WHERE voucher_type = ? AND voucher_id = ?",
            (voucher_type, si_id),
        )

        # Update SI status
        conn.execute(
            "UPDATE sales_invoice SET status = 'cancelled', outstanding_amount = '0' WHERE id = ?",
            (si_id,),
        )
    elif si_status == "cancelled":
        pass  # Already cancelled
    else:
        err(f"Cannot cancel sales invoice in status: {si_status}")

    # --- Cancel the mirror Purchase Invoice (target company) ---
    pi_gl_reversals = 0
    pi_sle_reversals = 0
    if pi_id:
        pi = conn.execute(
            "SELECT * FROM purchase_invoice WHERE id = ?", (pi_id,)
        ).fetchone()
        if pi and pi["status"] in ("submitted", "overdue", "partially_paid"):
            pi_voucher = "debit_note" if pi["is_return"] else "purchase_invoice"
            pi_posting = pi["posting_date"]

            # Reverse PI GL
            rev_pi_gl = reverse_gl_entries(conn, pi_voucher, pi_id, pi_posting)
            pi_gl_reversals = len(rev_pi_gl)

            # Reverse PI SLE if applicable
            if pi["update_stock"]:
                rev_pi_sle = reverse_sle_entries(conn, pi_voucher, pi_id, pi_posting)
                pi_sle_reversals = len(rev_pi_sle)

            # Mark PI PLE as delinked
            conn.execute(
                "UPDATE payment_ledger_entry SET delinked = 1 WHERE voucher_type = ? AND voucher_id = ?",
                (pi_voucher, pi_id),
            )

            # Update PI status
            conn.execute(
                "UPDATE purchase_invoice SET status = 'cancelled', outstanding_amount = '0' WHERE id = ?",
                (pi_id,),
            )
        elif pi and pi["status"] == "draft":
            # Just delete the draft PI and its items
            conn.execute("DELETE FROM purchase_invoice_item WHERE purchase_invoice_id = ?", (pi_id,))
            conn.execute("DELETE FROM purchase_invoice WHERE id = ?", (pi_id,))

    conn.commit()
    ok({
        "sales_invoice_id": si_id,
        "purchase_invoice_id": pi_id,
        "si_status": "cancelled",
        "si_gl_reversals": si_gl_reversals,
        "si_sle_reversals": si_sle_reversals,
        "pi_gl_reversals": pi_gl_reversals,
        "pi_sle_reversals": pi_sle_reversals,
    })


# ---------------------------------------------------------------------------
# Action dispatch
# ---------------------------------------------------------------------------

ACTIONS = {
    "add-customer": add_customer,
    "update-customer": update_customer,
    "get-customer": get_customer,
    "list-customers": list_customers,
    "add-quotation": add_quotation,
    "update-quotation": update_quotation,
    "get-quotation": get_quotation,
    "list-quotations": list_quotations,
    "submit-quotation": submit_quotation,
    "convert-quotation-to-so": convert_quotation_to_so,
    "add-sales-order": add_sales_order,
    "update-sales-order": update_sales_order,
    "get-sales-order": get_sales_order,
    "list-sales-orders": list_sales_orders,
    "submit-sales-order": submit_sales_order,
    "cancel-sales-order": cancel_sales_order,
    "create-delivery-note": create_delivery_note,
    "get-delivery-note": get_delivery_note,
    "list-delivery-notes": list_delivery_notes,
    "submit-delivery-note": submit_delivery_note,
    "cancel-delivery-note": cancel_delivery_note,
    "create-sales-invoice": create_sales_invoice,
    "update-sales-invoice": update_sales_invoice,
    "get-sales-invoice": get_sales_invoice,
    "list-sales-invoices": list_sales_invoices,
    "submit-sales-invoice": submit_sales_invoice,
    "cancel-sales-invoice": cancel_sales_invoice,
    "create-credit-note": create_credit_note,
    "update-invoice-outstanding": update_invoice_outstanding,
    "add-sales-partner": add_sales_partner,
    "list-sales-partners": list_sales_partners,
    "add-recurring-template": add_recurring_template,
    "update-recurring-template": update_recurring_template,
    "list-recurring-templates": list_recurring_templates,
    "generate-recurring-invoices": generate_recurring_invoices,
    "import-customers": import_customers,
    "add-intercompany-account-map": add_intercompany_account_map,
    "list-intercompany-account-maps": list_intercompany_account_maps,
    "create-intercompany-invoice": create_intercompany_invoice,
    "list-intercompany-invoices": list_intercompany_invoices,
    "cancel-intercompany-invoice": cancel_intercompany_invoice,
    "status": status_action,
}


def main():
    parser = argparse.ArgumentParser(description="ERPClaw Selling Skill")
    parser.add_argument("--action", required=True, choices=sorted(ACTIONS.keys()))
    parser.add_argument("--db-path", default=None)

    # Customer fields
    parser.add_argument("--customer-id")
    parser.add_argument("--customer-type")
    parser.add_argument("--customer-group")
    parser.add_argument("--payment-terms-id")
    parser.add_argument("--credit-limit")
    parser.add_argument("--tax-id")
    parser.add_argument("--exempt-from-sales-tax")
    parser.add_argument("--primary-address")
    parser.add_argument("--primary-contact")

    # Common fields
    parser.add_argument("--name")
    parser.add_argument("--company-id")
    parser.add_argument("--csv-path")
    parser.add_argument("--posting-date")
    parser.add_argument("--items")  # JSON
    parser.add_argument("--tax-template-id")

    # Quotation fields
    parser.add_argument("--quotation-id")
    parser.add_argument("--valid-till")

    # Sales order fields
    parser.add_argument("--sales-order-id")
    parser.add_argument("--delivery-date")

    # Delivery note fields
    parser.add_argument("--delivery-note-id")

    # Sales invoice fields
    parser.add_argument("--sales-invoice-id")
    parser.add_argument("--due-date")

    # Credit note fields
    parser.add_argument("--against-invoice-id")
    parser.add_argument("--reason")

    # Payment / outstanding
    parser.add_argument("--amount")

    # Sales partner
    parser.add_argument("--commission-rate")

    # Recurring template fields
    parser.add_argument("--template-id")
    parser.add_argument("--frequency")
    parser.add_argument("--start-date")
    parser.add_argument("--end-date")
    parser.add_argument("--as-of-date")
    parser.add_argument("--template-status", dest="template_status")

    # Intercompany fields
    parser.add_argument("--target-company-id")
    parser.add_argument("--supplier-id")
    parser.add_argument("--source-account-id")
    parser.add_argument("--target-account-id")

    # Status filter (for list queries)
    parser.add_argument("--status", dest="doc_status")

    # Search / pagination
    parser.add_argument("--search")
    parser.add_argument("--from-date")
    parser.add_argument("--to-date")
    parser.add_argument("--limit", default="20")
    parser.add_argument("--offset", default="0")

    args, _unknown = parser.parse_known_args()
    check_input_lengths(args)

    db_path = args.db_path or DEFAULT_DB_PATH
    ensure_db_exists(db_path)
    conn = get_connection(db_path)

    # Dependency check
    _dep = check_required_tables(conn, REQUIRED_TABLES)
    if _dep:
        _dep["suggestion"] = "clawhub install " + " ".join(_dep.get("missing_skills", []))
        print(json.dumps(_dep, indent=2))
        conn.close()
        sys.exit(1)

    try:
        ACTIONS[args.action](conn, args)
    except Exception as e:
        conn.rollback()
        sys.stderr.write(f"[erpclaw-selling] {e}\n")
        err("An unexpected error occurred")
    finally:
        conn.close()


if __name__ == "__main__":
    main()
