---
name: env-setup
description: Scan codebase for environment variables, generate .env.example, validate .env, and ensure .gitignore safety
version: 1.0.0
author: Sovereign Skills
tags: [openclaw, agent-skills, automation, productivity, free, env, environment, configuration, security]
triggers:
  - scan env vars
  - generate env example
  - check env
  - environment setup
  - env audit
---

# env-setup — Environment Variable Manager

Scan your codebase for all referenced environment variables, generate `.env.example`, validate your current `.env`, and ensure secrets aren't committed.

## Steps

### 1. Scan Codebase for Environment Variables

Search for env var references across all common patterns:

```bash
# Node.js / JavaScript / TypeScript
grep -rn "process\.env\.\w\+" --include="*.js" --include="*.ts" --include="*.jsx" --include="*.tsx" . | grep -v node_modules | grep -v dist

# Python
grep -rn "os\.environ\|os\.getenv\|environ\.get" --include="*.py" . | grep -v __pycache__ | grep -v .venv

# Rust
grep -rn "env::var\|env::var_os\|dotenv" --include="*.rs" . | grep -v target

# Go
grep -rn "os\.Getenv\|os\.LookupEnv\|viper\." --include="*.go" . | grep -v vendor

# Docker / docker-compose
grep -rn "\${.*}" --include="*.yml" --include="*.yaml" docker-compose* 2>/dev/null

# General .env references in config files
grep -rn "env\." --include="*.toml" --include="*.yaml" --include="*.yml" . 2>/dev/null
```

**Windows PowerShell alternative:**
```powershell
Get-ChildItem -Recurse -Include *.js,*.ts,*.jsx,*.tsx -Exclude node_modules,dist | Select-String "process\.env\.\w+"
Get-ChildItem -Recurse -Include *.py -Exclude __pycache__,.venv | Select-String "os\.environ|os\.getenv"
```

### 2. Extract Variable Names

Parse grep output to extract unique variable names:

- `process.env.DATABASE_URL` → `DATABASE_URL`
- `os.environ.get("SECRET_KEY", "default")` → `SECRET_KEY` (default: `default`)
- `os.getenv("API_KEY")` → `API_KEY`
- `env::var("RUST_LOG")` → `RUST_LOG`

Deduplicate and sort alphabetically. Note which file and line each var is referenced in.

### 3. Classify Variables

Categorize each variable:

| Category | Pattern | Examples |
|----------|---------|---------|
| 🔴 Secrets | `*KEY*`, `*SECRET*`, `*TOKEN*`, `*PASSWORD*`, `*CREDENTIAL*` | `API_KEY`, `JWT_SECRET` |
| 🟡 Service URLs | `*URL*`, `*HOST*`, `*ENDPOINT*`, `*URI*` | `DATABASE_URL`, `REDIS_HOST` |
| 🟢 Configuration | `*PORT*`, `*ENV*`, `*MODE*`, `*LEVEL*`, `*DEBUG*` | `PORT`, `NODE_ENV`, `LOG_LEVEL` |
| ⚪ Other | Everything else | `APP_NAME`, `MAX_RETRIES` |

### 4. Generate .env.example

Create `.env.example` with descriptions, categories, and safe defaults:

```env
# ============================================
# Environment Configuration
# Generated by env-setup skill
# ============================================

# --- App Configuration ---
NODE_ENV=development
PORT=3000
LOG_LEVEL=info

# --- Database ---
DATABASE_URL=postgresql://user:password@localhost:5432/dbname

# --- Authentication (🔴 SECRET — never commit real values) ---
JWT_SECRET=change-me-in-production
API_KEY=your-api-key-here

# --- External Services ---
REDIS_URL=redis://localhost:6379
```

Rules:
- Secrets get placeholder values (`change-me`, `your-xxx-here`)
- Config vars get sensible defaults
- Group by category with comment headers
- Add `🔴 SECRET` warning on sensitive vars

### 5. Validate Current .env

If `.env` exists, compare against discovered variables:

```markdown
## .env Validation Report

### ❌ Missing (required by code but not in .env)
- `STRIPE_SECRET_KEY` — referenced in src/billing.ts:14
- `SMTP_PASSWORD` — referenced in src/email.ts:8

### ⚠️ Unused (in .env but not referenced in code)
- `OLD_API_ENDPOINT` — may be safe to remove

### ✅ Present and referenced
- `DATABASE_URL` ✓
- `PORT` ✓
- `NODE_ENV` ✓
```

### 6. Ensure .gitignore Safety

Check that `.env` is in `.gitignore`:

```bash
grep -q "^\.env$\|^\.env\.\*" .gitignore 2>/dev/null
```

If not found, offer to add:
```
# Environment files
.env
.env.local
.env.*.local
```

Also check git history for accidentally committed `.env` files:
```bash
git log --all --diff-filter=A -- .env .env.local .env.production 2>/dev/null
```

If found, warn the user that secrets may be in git history and suggest `git filter-branch` or `BFG Repo-Cleaner`.

### 7. Output Summary

```markdown
# Environment Variable Report
| Metric | Count |
|--------|-------|
| Total vars found | 15 |
| 🔴 Secrets | 4 |
| ❌ Missing from .env | 2 |
| ⚠️ Unused in .env | 1 |
| ✅ Properly configured | 12 |
| .gitignore protection | ✅ |
```

## Edge Cases

- **Framework-specific env**: Next.js uses `NEXT_PUBLIC_*` (client-exposed); flag these distinctly
- **Docker env**: Check `docker-compose.yml` `environment:` section too
- **Multiple .env files**: `.env.development`, `.env.production`, `.env.test` — validate all
- **No .env exists**: Generate both `.env.example` and a starter `.env`
- **Interpolated vars**: `${VAR:-default}` in shell scripts — extract `VAR`

## Error Handling

| Error | Resolution |
|-------|-----------|
| No env vars found | Project may not use env vars — confirm with user |
| .env has syntax errors | Flag lines that don't match `KEY=value` pattern |
| Binary files in scan | Exclude with `--binary-files=without-match` |
| Permission denied on .env | Check file permissions; may need elevated access |

---
*Built by Clawb (SOVEREIGN) — more skills at [coming soon]*
