import fs from "node:fs/promises";
import path from "node:path";
import os from "node:os";
import crypto from "node:crypto";

const DEFAULTS = {
  historySize: 100,
  halfLifeHours: 12,
  trendWindowHours: 24,
  maxUserEntries: 3,
  maxAgentEntries: 2,
  maxOtherAgents: 3,
  confidenceMin: 0.35,
  model: "gpt-4o-mini",
  stateFileName: "emotion-state.json",
  fetchTimeoutMs: 5000,
  maxUsers: 50,
  lockStaleMs: 10_000,
};

const DEFAULT_LABELS = [
  "neutral",
  "calm",
  "happy",
  "excited",
  "sad",
  "anxious",
  "frustrated",
  "angry",
  "confused",
  "focused",
  "relieved",
  "optimistic",
];

const INTENSITY_WORDS: Record<string, string> = {
  low: "mildly",
  medium: "moderately",
  high: "strongly",
};

type EmotionEntry = {
  timestamp: string;
  label: string;
  intensity: "low" | "medium" | "high";
  reason: string;
  confidence: number;
  source_hash?: string;
  source_role?: string;
};

type EmotionState = {
  version: number;
  users: Record<string, { latest?: EmotionEntry; history: EmotionEntry[] }>;
  agents: Record<string, { latest?: EmotionEntry; history: EmotionEntry[] }>;
};

type SessionMessage = {
  role: "user" | "assistant" | "system" | "tool";
  content: string;
  timestamp?: string;
};

function envNumber(name: string, fallback: number) {
  const raw = process.env[name];
  if (!raw) return fallback;
  const value = Number(raw);
  return Number.isFinite(value) ? value : fallback;
}

function envString(name: string, fallback?: string) {
  const value = process.env[name];
  return value && value.trim() ? value.trim() : fallback;
}

function envLabels() {
  const raw = envString("EMOTION_LABELS");
  if (!raw) return DEFAULT_LABELS;
  const labels = raw.split(",").map((label) => label.trim()).filter(Boolean);
  return labels.length > 0 ? labels : DEFAULT_LABELS;
}

function normalizeLabel(label: string, labels: string[]) {
  const normalized = label.trim().toLowerCase();
  return labels.includes(normalized) ? normalized : "neutral";
}

function normalizeIntensity(intensity: string) {
  const normalized = intensity.trim().toLowerCase();
  if (normalized === "low" || normalized === "medium" || normalized === "high") return normalized;
  return "low";
}

function ensureSentence(text: string) {
  const trimmed = text.trim();
  if (!trimmed) return "unsure";
  if (/[.!?]$/.test(trimmed)) return trimmed;
  return `${trimmed}.`;
}

function formatTimestamp(timestamp: string, timeZone?: string) {
  const date = new Date(timestamp);
  if (Number.isNaN(date.getTime())) return timestamp;
  const formatter = new Intl.DateTimeFormat("en-CA", {
    timeZone,
    year: "numeric",
    month: "2-digit",
    day: "2-digit",
    hour: "2-digit",
    minute: "2-digit",
    hour12: false,
  });
  const parts = formatter.formatToParts(date);
  const lookup = (type: string) => parts.find((part) => part.type === type)?.value ?? "";
  return `${lookup("year")}-${lookup("month")}-${lookup("day")} ${lookup("hour")}:${lookup("minute")}`;
}

function hashText(text: string) {
  return crypto.createHash("sha256").update(text).digest("hex");
}

async function readJsonFile<T>(filePath: string): Promise<T | null> {
  try {
    const raw = await fs.readFile(filePath, "utf8");
    return JSON.parse(raw) as T;
  } catch {
    return null;
  }
}

async function writeJsonFile(filePath: string, data: unknown) {
  await fs.mkdir(path.dirname(filePath), { recursive: true });
  const tmp = `${filePath}.tmp`;
  await fs.writeFile(tmp, JSON.stringify(data, null, 2), "utf8");
  await fs.rename(tmp, filePath);
}

function buildEmptyState(): EmotionState {
  return { version: 1, users: {}, agents: {} };
}

function getRole(entry: Record<string, any>): SessionMessage["role"] | null {
  const role = entry.role || entry.author || entry.sender || entry.type;
  if (!role) return null;
  const normalized = String(role).toLowerCase();
  if (normalized.includes("user")) return "user";
  if (normalized.includes("assistant") || normalized.includes("agent")) return "assistant";
  if (normalized.includes("system")) return "system";
  if (normalized.includes("tool")) return "tool";
  return null;
}

function extractContent(entry: Record<string, any>) {
  if (typeof entry.content === "string") return entry.content;
  if (typeof entry.text === "string") return entry.text;
  if (typeof entry.message === "string") return entry.message;
  if (typeof entry.value === "string") return entry.value;
  if (Array.isArray(entry.content)) {
    const textPart = entry.content.find((part: any) => part?.type === "text");
    if (textPart?.text) return String(textPart.text);
  }
  return "";
}

function extractMessagesFromContainer(container: any): SessionMessage[] {
  if (!container) return [];
  const messages: SessionMessage[] = [];
  const list = container.messages || container.entries || container.events || container.items;
  if (Array.isArray(list)) {
    for (const entry of list) {
      const role = getRole(entry);
      const content = extractContent(entry);
      if (!role || !content) continue;
      const timestamp = entry.timestamp || entry.time || entry.created_at || entry.createdAt;
      messages.push({ role, content, timestamp });
    }
  }
  return messages;
}

async function readJsonlMessages(filePath: string): Promise<SessionMessage[]> {
  try {
    const raw = await fs.readFile(filePath, "utf8");
    const messages: SessionMessage[] = [];
    for (const line of raw.split("\n")) {
      if (!line.trim()) continue;
      try {
        const entry = JSON.parse(line);
        if (entry.type !== "message" || !entry.message) continue;
        const msg = entry.message;
        const role = getRole(msg);
        const content = extractContent(msg);
        if (!role || !content) continue;
        const timestamp = entry.timestamp || msg.timestamp;
        messages.push({ role, content, timestamp });
      } catch {
        continue;
      }
    }
    return messages;
  } catch {
    return [];
  }
}

async function extractMessages(sessionEntry: any, sessionFile?: string) {
  const fromEntry = extractMessagesFromContainer(sessionEntry);
  if (fromEntry.length > 0) return fromEntry;
  if (!sessionFile) return [];
  const fromJsonl = await readJsonlMessages(sessionFile);
  if (fromJsonl.length > 0) return fromJsonl;
  const fromFile = await readJsonFile<any>(sessionFile);
  if (!fromFile) return [];
  return extractMessagesFromContainer(fromFile);
}

function pickLatest(messages: SessionMessage[], role: "user" | "assistant") {
  for (let i = messages.length - 1; i >= 0; i -= 1) {
    if (messages[i].role === role && messages[i].content.trim()) return messages[i];
  }
  return null;
}

function resolveAgentId(sessionKey?: string, sessionFile?: string) {
  if (sessionKey && sessionKey.includes(":")) {
    const parts = sessionKey.split(":");
    if (parts.length >= 2 && parts[0] === "agent" && parts[1]) return parts[1];
  }
  if (sessionFile) {
    const parts = sessionFile.split(path.sep);
    const agentIndex = parts.lastIndexOf("agents");
    if (agentIndex >= 0 && parts[agentIndex + 1]) return parts[agentIndex + 1];
  }
  return "main";
}

function resolveUserKey(senderId?: string, sessionKey?: string) {
  if (senderId) return senderId;
  if (sessionKey) return sessionKey;
  return "unknown-user";
}

function resolveAgentDir(sessionFile: string | undefined, agentId: string) {
  if (sessionFile) {
    const parts = sessionFile.split(path.sep);
    const agentsIndex = parts.lastIndexOf("agents");
    if (agentsIndex >= 0 && parts[agentsIndex + 1]) {
      const stateRoot = parts.slice(0, agentsIndex + 1).join(path.sep);
      return path.join(stateRoot, parts[agentsIndex + 1], "agent");
    }
  }
  const stateRoot = envString("OPENCLAW_STATE_DIR") || path.join(os.homedir(), ".openclaw");
  return path.join(stateRoot, "agents", agentId, "agent");
}

function getOtherAgentsRoot(agentDir: string) {
  return path.resolve(agentDir, "..", "..");
}

async function readState(statePath: string): Promise<EmotionState> {
  const existing = await readJsonFile<EmotionState>(statePath);
  if (!existing) return buildEmptyState();
  if (!existing.users) existing.users = {} as any;
  if (!existing.agents) existing.agents = {} as any;
  if (!existing.version) existing.version = 1;
  return existing;
}

async function classifyWithEndpoint(url: string, payload: { text: string; role: string }) {
  const timeoutMs = envNumber("EMOTION_FETCH_TIMEOUT_MS", DEFAULTS.fetchTimeoutMs);
  const response = await fetch(url, {
    method: "POST",
    headers: { "content-type": "application/json" },
    body: JSON.stringify(payload),
    signal: AbortSignal.timeout(timeoutMs),
  });
  if (!response.ok) throw new Error(`Classifier returned ${response.status}`);
  return response.json();
}

async function classifyWithOpenAI(text: string, role: string, model: string) {
  const apiKey = envString("OPENAI_API_KEY");
  if (!apiKey) throw new Error("Missing OPENAI_API_KEY");
  const baseUrl = envString("OPENAI_BASE_URL", "https://api.openai.com/v1");
  const timeoutMs = envNumber("EMOTION_FETCH_TIMEOUT_MS", DEFAULTS.fetchTimeoutMs);
  const systemPrompt =
    "You are an emotion classifier. Return only JSON with keys: label, intensity, reason, confidence. " +
    "label is a short emotion word, intensity is low|medium|high, reason is a short clause, confidence is 0..1.";
  const response = await fetch(`${baseUrl.replace(/\/$/, "")}/chat/completions`, {
    method: "POST",
    headers: {
      "content-type": "application/json",
      authorization: `Bearer ${apiKey}`,
    },
    body: JSON.stringify({
      model,
      messages: [
        { role: "system", content: systemPrompt },
        { role: "user", content: `Classify the emotion in this ${role} message:\n\n${text}` },
      ],
      temperature: 0.2,
      response_format: { type: "json_object" },
    }),
    signal: AbortSignal.timeout(timeoutMs),
  });
  if (!response.ok) {
    throw new Error(`OpenAI returned ${response.status}`);
  }
  const data = await response.json();
  const content = data?.choices?.[0]?.message?.content || "";
  const match = content.match(/\{[\s\S]*?\}/);
  if (!match) throw new Error("No JSON in OpenAI response");
  return JSON.parse(match[0]);
}

async function classifyEmotion(text: string, role: string) {
  const classifierUrl = envString("EMOTION_CLASSIFIER_URL");
  const model = envString("EMOTION_MODEL", DEFAULTS.model) as string;
  if (classifierUrl) {
    return classifyWithEndpoint(classifierUrl, { text, role });
  }
  return classifyWithOpenAI(text, role, model);
}

function coerceEntry(raw: any, labels: string[], confidenceMin: number): EmotionEntry {
  const label = normalizeLabel(String(raw?.label || "neutral"), labels);
  const intensity = normalizeIntensity(String(raw?.intensity || "low")) as EmotionEntry["intensity"];
  const reason = ensureSentence(String(raw?.reason || "unsure"));
  const confidence = Number(raw?.confidence ?? 0);
  const safeConfidence = Number.isFinite(confidence) ? confidence : 0;
  if (safeConfidence < confidenceMin) {
    return {
      timestamp: new Date().toISOString(),
      label: "neutral",
      intensity: "low",
      reason: "unsure",
      confidence: safeConfidence,
    };
  }
  return {
    timestamp: new Date().toISOString(),
    label,
    intensity,
    reason,
    confidence: safeConfidence,
  };
}

function updateBucket(bucket: { latest?: EmotionEntry; history: EmotionEntry[] }, entry: EmotionEntry, historySize: number) {
  bucket.latest = entry;
  bucket.history.unshift(entry);
  if (bucket.history.length > historySize) bucket.history.length = historySize;
}

function pruneUsers(state: EmotionState, maxUsers: number) {
  const keys = Object.keys(state.users);
  if (keys.length <= maxUsers) return;
  const sorted = keys
    .map((key) => ({ key, ts: state.users[key].latest?.timestamp || "" }))
    .sort((a, b) => (a.ts < b.ts ? -1 : a.ts > b.ts ? 1 : 0));
  const toRemove = sorted.slice(0, keys.length - maxUsers);
  for (const { key } of toRemove) {
    delete state.users[key];
  }
}

async function maybeUpdate(
  bucket: { latest?: EmotionEntry; history: EmotionEntry[] },
  message: SessionMessage,
  role: string,
  labels: string[],
  confidenceMin: number,
  historySize: number,
) {
  const sourceHash = hashText(message.content);
  if (bucket.latest?.source_hash === sourceHash) return false;
  try {
    const raw = await classifyEmotion(message.content, role);
    const entry = coerceEntry(raw, labels, confidenceMin);
    entry.source_hash = sourceHash;
    entry.source_role = role;
    updateBucket(bucket, entry, historySize);
    return true;
  } catch (err) {
    console.error("[emotion-state] Classification failed, falling back to neutral:", err);
    const entry: EmotionEntry = {
      timestamp: new Date().toISOString(),
      label: "neutral",
      intensity: "low",
      reason: "unsure",
      confidence: 0,
      source_hash: sourceHash,
      source_role: role,
    };
    updateBucket(bucket, entry, historySize);
    return true;
  }
}

function computeDominantLabel(entries: EmotionEntry[], now: Date, halfLifeHours: number, windowHours: number) {
  const weights: Record<string, number> = {};
  const nowMs = now.getTime();
  for (const entry of entries) {
    const ts = new Date(entry.timestamp).getTime();
    if (Number.isNaN(ts)) continue;
    const ageHours = (nowMs - ts) / 3_600_000;
    if (ageHours < 0 || ageHours > windowHours) continue;
    const weight = Math.pow(0.5, ageHours / halfLifeHours);
    weights[entry.label] = (weights[entry.label] || 0) + weight;
  }
  let topLabel = "neutral";
  let topWeight = 0;
  for (const [label, weight] of Object.entries(weights)) {
    if (weight > topWeight) {
      topWeight = weight;
      topLabel = label;
    }
  }
  return topWeight > 0 ? topLabel : "neutral";
}

function formatEntry(entry: EmotionEntry, timeZone?: string) {
  const ts = formatTimestamp(entry.timestamp, timeZone);
  const intensityWord = INTENSITY_WORDS[entry.intensity] || "mildly";
  const reason = ensureSentence(entry.reason);
  return `${ts}: Felt ${intensityWord} ${entry.label} because ${reason}`;
}

async function loadOtherAgents(
  rootDir: string,
  currentAgentId: string,
  stateFileName: string,
  maxAgents: number,
) {
  const results: { id: string; latest: EmotionEntry }[] = [];
  try {
    const entries = await fs.readdir(rootDir, { withFileTypes: true });
    for (const entry of entries) {
      if (!entry.isDirectory()) continue;
      if (entry.name === currentAgentId) continue;
      const filePath = path.join(rootDir, entry.name, "agent", stateFileName);
      const state = await readJsonFile<EmotionState>(filePath);
      const latest = state?.agents?.[entry.name]?.latest || state?.users?.[entry.name]?.latest;
      if (latest) results.push({ id: entry.name, latest });
      if (results.length >= maxAgents) break;
    }
  } catch (err) {
    console.error("[emotion-state] Failed to load other agents:", err);
    return [];
  }
  return results;
}

function buildEmotionBlock(
  state: EmotionState,
  userKey: string,
  agentId: string,
  options: {
    maxUserEntries: number;
    maxAgentEntries: number;
    halfLifeHours: number;
    trendWindowHours: number;
    timeZone?: string;
    otherAgents: { id: string; latest: EmotionEntry }[];
  },
) {
  const now = new Date();
  const userBucket = state.users[userKey];
  const agentBucket = state.agents[agentId];
  const userEntries = userBucket?.history?.slice(0, options.maxUserEntries) || [];
  const agentEntries = agentBucket?.history?.slice(0, options.maxAgentEntries) || [];
  if (userEntries.length === 0 && agentEntries.length === 0 && options.otherAgents.length === 0) return "";

  const userTrend = userBucket?.history?.length
    ? computeDominantLabel(userBucket.history, now, options.halfLifeHours, options.trendWindowHours)
    : null;
  const agentTrend = agentBucket?.history?.length
    ? computeDominantLabel(agentBucket.history, now, options.halfLifeHours, options.trendWindowHours)
    : null;

  const lines: string[] = ["<emotion_state>", "  <user>"];
  for (const entry of userEntries) {
    lines.push(`    ${formatEntry(entry, options.timeZone)}`);
  }
  if (userTrend) lines.push(`    Trend (last ${options.trendWindowHours}h): mostly ${userTrend}.`);
  lines.push("  </user>");
  lines.push("  <agent>");
  for (const entry of agentEntries) {
    lines.push(`    ${formatEntry(entry, options.timeZone)}`);
  }
  if (agentTrend) lines.push(`    Trend (last ${options.trendWindowHours}h): mostly ${agentTrend}.`);
  lines.push("  </agent>");
  if (options.otherAgents.length > 0) {
    lines.push("  <others>");
    for (const other of options.otherAgents) {
      lines.push(
        `    ${other.id} — ${formatEntry(other.latest, options.timeZone)}`,
      );
    }
    lines.push("  </others>");
  }
  lines.push("</emotion_state>");
  return lines.join("\n");
}

async function acquireLock(lockPath: string, staleMs: number): Promise<boolean> {
  try {
    const handle = await fs.open(lockPath, "wx");
    await handle.close();
    return true;
  } catch (err: any) {
    if (err?.code !== "EEXIST") return false;
    try {
      const stat = await fs.stat(lockPath);
      if (Date.now() - stat.mtimeMs > staleMs) {
        await fs.unlink(lockPath).catch(() => {});
        const handle = await fs.open(lockPath, "wx");
        await handle.close();
        return true;
      }
    } catch {
      /* lock contention, give up */
    }
    return false;
  }
}

async function releaseLock(lockPath: string) {
  await fs.unlink(lockPath).catch(() => {});
}

function injectBootstrap(context: any, content: string) {
  if (!content) return;
  if (!context.bootstrapFiles) context.bootstrapFiles = [];
  const existing = context.bootstrapFiles.find((file: any) => file?.path === "EMOTIONS.md");
  if (existing) {
    existing.content = content;
    existing.text = content;
    return;
  }
  context.bootstrapFiles.push({
    path: "EMOTIONS.md",
    content,
    text: content,
  });
}

export default async function handler(event: any) {
  try {
    if (!event || event.type !== "agent" || event.action !== "bootstrap") return;

    const context = event.context || {};
    const sessionEntry = context.sessionEntry;
    const sessionFile = context.sessionFile;

    const labels = envLabels();
    const confidenceMin = envNumber("EMOTION_CONFIDENCE_MIN", DEFAULTS.confidenceMin);
    const historySize = envNumber("EMOTION_HISTORY_SIZE", DEFAULTS.historySize);
    const halfLifeHours = envNumber("EMOTION_HALF_LIFE_HOURS", DEFAULTS.halfLifeHours);
    const trendWindowHours = envNumber("EMOTION_TREND_WINDOW_HOURS", DEFAULTS.trendWindowHours);
    const maxUserEntries = envNumber("EMOTION_MAX_USER_ENTRIES", DEFAULTS.maxUserEntries);
    const maxAgentEntries = envNumber("EMOTION_MAX_AGENT_ENTRIES", DEFAULTS.maxAgentEntries);
    const maxOtherAgents = envNumber("EMOTION_MAX_OTHER_AGENTS", DEFAULTS.maxOtherAgents);
    const maxUsers = envNumber("EMOTION_MAX_USERS", DEFAULTS.maxUsers);
    const staleMs = DEFAULTS.lockStaleMs;
    const timeZone = envString("EMOTION_TIMEZONE");

    const messages = await extractMessages(sessionEntry, sessionFile);
    const latestUser = pickLatest(messages, "user");
    const latestAssistant = pickLatest(messages, "assistant");

    const userKey = resolveUserKey(context.senderId, context.sessionKey);
    const agentId = resolveAgentId(context.sessionKey, sessionFile);
    const agentDir = resolveAgentDir(sessionFile, agentId);
    const statePath = path.join(agentDir, DEFAULTS.stateFileName);
    const lockPath = `${statePath}.lock`;

    const locked = await acquireLock(lockPath, staleMs);
    if (!locked) {
      console.error("[emotion-state] Could not acquire lock, skipping state update");
    }

    let state: EmotionState;
    try {
      state = await readState(statePath);
      if (!state.users[userKey]) state.users[userKey] = { history: [] };
      if (!state.agents[agentId]) state.agents[agentId] = { history: [] };

      let updated = false;
      if (latestUser?.content) {
        updated =
          (await maybeUpdate(state.users[userKey], latestUser, "user", labels, confidenceMin, historySize)) ||
          updated;
      }
      if (latestAssistant?.content) {
        updated =
          (await maybeUpdate(state.agents[agentId], latestAssistant, "assistant", labels, confidenceMin, historySize)) ||
          updated;
      }

      if (updated) {
        pruneUsers(state, maxUsers);
        if (locked) {
          await writeJsonFile(statePath, state);
        } else {
          console.error("[emotion-state] Skipping write — no lock held");
        }
      }
    } finally {
      if (locked) await releaseLock(lockPath);
    }

    const otherAgents = await loadOtherAgents(
      getOtherAgentsRoot(agentDir),
      agentId,
      DEFAULTS.stateFileName,
      maxOtherAgents,
    );

    const block = buildEmotionBlock(state, userKey, agentId, {
      maxUserEntries,
      maxAgentEntries,
      halfLifeHours,
      trendWindowHours,
      timeZone,
      otherAgents,
    });

    injectBootstrap(context, block);
  } catch (err) {
    console.error("[emotion-state] Unhandled error in hook handler:", err);
  }
}
