#!/usr/bin/env python3
"""Email triage system — IMAP scanner with AI classification.

Scans unread emails via IMAP, categorizes them using a local LLM (Ollama)
or heuristic fallback, tracks state to avoid re-processing, and surfaces
important emails for consumption by agents or heartbeat checks.

Categories:
  🔴 urgent:         Needs immediate attention (outages, security, legal, time-sensitive)
  🟡 needs-response: Requires a reply (business inquiries, questions, action items)
  🔵 informational:  FYI only (billing, receipts, confirmations, newsletters)
  ⚫ spam:           Junk, marketing, irrelevant

Configuration (environment variables):
  IMAP_HOST           IMAP server host (required)
  IMAP_PORT           IMAP port (default: 993)
  IMAP_USER           IMAP username/email (required)
  IMAP_PASS           IMAP password (required)
  EMAIL_TRIAGE_STATE  State file path (default: ./data/email-triage.json)
  OLLAMA_URL          Ollama endpoint (default: http://127.0.0.1:11434)
  OLLAMA_MODEL        Model name (default: qwen2.5:7b)

Usage:
    python3 email-triage.py scan            # Scan + categorize new emails
    python3 email-triage.py report          # Show unsurfaced important emails
    python3 email-triage.py mark-surfaced   # Mark reported emails as surfaced
    python3 email-triage.py stats           # Show triage statistics
    python3 email-triage.py scan --dry-run  # Scan without saving state
"""

import argparse
import email
import email.header
import email.message
import email.utils
import hashlib
import imaplib
import json
import os
import sys
from datetime import datetime, timezone
from pathlib import Path

# ---------------------------------------------------------------------------
# Configuration — all from environment variables
# ---------------------------------------------------------------------------
IMAP_HOST = os.environ.get("IMAP_HOST", "")
IMAP_PORT = int(os.environ.get("IMAP_PORT", "993"))
IMAP_USER = os.environ.get("IMAP_USER", "")
IMAP_PASS = os.environ.get("IMAP_PASS", "")
STATE_FILE = Path(os.environ.get("EMAIL_TRIAGE_STATE", "./data/email-triage.json"))
OLLAMA_URL = os.environ.get("OLLAMA_URL", "http://127.0.0.1:11434")
OLLAMA_MODEL = os.environ.get("OLLAMA_MODEL", "qwen2.5:7b")

MAX_EMAILS_PER_SCAN = 20
CLASSIFICATION_TIMEOUT = 30  # seconds per email


# ---------------------------------------------------------------------------
# Helpers
# ---------------------------------------------------------------------------

def _require_imap_config():
    """Validate that required IMAP env vars are set."""
    missing = []
    if not IMAP_HOST:
        missing.append("IMAP_HOST")
    if not IMAP_USER:
        missing.append("IMAP_USER")
    if not IMAP_PASS:
        missing.append("IMAP_PASS")
    if missing:
        print(
            f"ERROR: Missing required environment variable(s): {', '.join(missing)}",
            file=sys.stderr,
        )
        sys.exit(1)


def decode_header(raw: str) -> str:
    """Decode a MIME-encoded email header."""
    if not raw:
        return ""
    parts = email.header.decode_header(raw)
    decoded = []
    for part, charset in parts:
        if isinstance(part, bytes):
            decoded.append(part.decode(charset or "utf-8", errors="replace"))
        else:
            decoded.append(part)
    return " ".join(decoded)


def get_body_preview(msg: email.message.Message, max_chars: int = 500) -> str:
    """Extract a plain-text preview from the email body."""
    body = None
    if msg.is_multipart():
        for part in msg.walk():
            if part.get_content_type() == "text/plain":
                payload = part.get_payload(decode=True)
                if payload:
                    charset = part.get_content_charset() or "utf-8"
                    body = payload.decode(charset, errors="replace")
                    break
    else:
        if msg.get_content_type() == "text/plain":
            payload = msg.get_payload(decode=True)
            if payload:
                charset = msg.get_content_charset() or "utf-8"
                body = payload.decode(charset, errors="replace")

    if not body:
        return "(no plain-text body)"

    preview = " ".join(body.split())
    if len(preview) > max_chars:
        preview = preview[:max_chars].rstrip() + "…"
    return preview


def make_email_key(msg_id: str, subject: str, sender: str) -> str:
    """Create a stable key for deduplication. Prefers Message-ID, falls back to hash."""
    if msg_id:
        return msg_id.strip().strip("<>")
    combo = f"{subject}|{sender}"
    return hashlib.sha256(combo.encode()).hexdigest()[:16]


# ---------------------------------------------------------------------------
# State management
# ---------------------------------------------------------------------------

def load_state() -> dict:
    """Load triage state from disk."""
    if STATE_FILE.exists():
        try:
            with open(STATE_FILE) as f:
                return json.load(f)
        except (json.JSONDecodeError, OSError):
            pass
    return {"last_check": None, "emails": {}}


def save_state(state: dict):
    """Write triage state to disk."""
    STATE_FILE.parent.mkdir(parents=True, exist_ok=True)
    state["last_check"] = datetime.now(timezone.utc).isoformat()
    with open(STATE_FILE, "w") as f:
        json.dump(state, f, indent=2)


# ---------------------------------------------------------------------------
# Classification
# ---------------------------------------------------------------------------

def classify_with_ollama(sender: str, subject: str, preview: str) -> tuple[str, str]:
    """Classify an email using the local Ollama LLM.

    Returns (category, reason) tuple. Falls back to heuristics on failure.
    """
    import urllib.request

    prompt = f"""Classify this email into exactly one category. Reply with ONLY a JSON object, no other text.

Categories:
- "urgent": Server outages, security alerts, legal notices, payment failures, time-critical action needed
- "needs-response": Business inquiries, questions requiring answers, partnership proposals, support requests
- "informational": Billing statements, receipts, confirmations, newsletters, status updates, automated notifications
- "spam": Marketing, promotions, unsolicited sales, irrelevant

Email:
From: {sender}
Subject: {subject}
Preview: {preview[:300]}

Reply format: {{"category": "<category>", "reason": "<brief reason>"}}"""

    payload = json.dumps({
        "model": OLLAMA_MODEL,
        "prompt": prompt,
        "stream": False,
        "options": {
            "temperature": 0.1,
            "num_predict": 100,
        },
    }).encode()

    try:
        req = urllib.request.Request(
            f"{OLLAMA_URL}/api/generate",
            data=payload,
            headers={"Content-Type": "application/json"},
        )
        with urllib.request.urlopen(req, timeout=CLASSIFICATION_TIMEOUT) as resp:
            result = json.loads(resp.read())
            response_text = result.get("response", "").strip()

            # Parse JSON from response (handle markdown fences)
            if "```" in response_text:
                response_text = response_text.split("```")[1]
                if response_text.startswith("json"):
                    response_text = response_text[4:]
                response_text = response_text.strip()

            parsed = json.loads(response_text)
            category = parsed.get("category", "informational").lower()
            reason = parsed.get("reason", "LLM classification")

            valid = {"urgent", "needs-response", "informational", "spam"}
            if category not in valid:
                category = "informational"

            return category, reason

    except Exception:
        # Ollama unavailable or returned garbage — fall back to heuristics
        return classify_heuristic(sender, subject, preview)


def classify_heuristic(sender: str, subject: str, preview: str) -> tuple[str, str]:
    """Rule-based fallback classification when Ollama is unavailable."""
    sender_lower = sender.lower()
    subject_lower = subject.lower()
    combined = f"{subject_lower} {preview.lower()}"

    # Urgent patterns
    urgent_keywords = [
        "outage", "down", "critical", "security alert", "breach",
        "suspended", "terminated", "legal notice", "court",
        "payment failed", "overdue", "final notice",
    ]
    if any(kw in combined for kw in urgent_keywords):
        return "urgent", "Matched urgent keywords"

    # Spam patterns
    spam_patterns = [
        "unsubscribe", "opt out", "special offer", "limited time",
        "click here", "act now", "congratulations", "you've won",
        "free trial", "exclusive deal",
    ]
    spam_senders = ["noreply@", "marketing@", "promo@", "newsletter@"]
    if any(p in combined for p in spam_patterns) and any(s in sender_lower for s in spam_senders):
        return "spam", "Marketing/promotional pattern"

    # Informational patterns (automated notifications)
    info_patterns = [
        "billing statement", "invoice", "receipt", "confirmation",
        "your order", "shipping", "newsletter", "weekly digest",
        "monthly report", "notification", "automated", "no-reply",
        "noreply", "do not reply",
    ]
    info_senders = ["no-reply", "noreply", "notifications@", "alerts@", "billing@"]
    if any(p in combined for p in info_patterns) or any(s in sender_lower for s in info_senders):
        return "informational", "Automated notification pattern"

    # Needs-response patterns
    response_patterns = [
        "question", "inquiry", "proposal", "partnership",
        "following up", "request", "can you", "would you",
        "please review", "feedback", "meeting",
    ]
    if any(p in combined for p in response_patterns):
        return "needs-response", "Appears to need a reply"

    return "informational", "Default classification (no strong signals)"


# ---------------------------------------------------------------------------
# Commands
# ---------------------------------------------------------------------------

def scan_emails(dry_run: bool = False, verbose: bool = False) -> dict:
    """Scan IMAP inbox for unread emails and classify them."""
    _require_imap_config()
    state = load_state()

    mail = imaplib.IMAP4_SSL(IMAP_HOST, IMAP_PORT)
    mail.login(IMAP_USER, IMAP_PASS)
    mail.select("INBOX", readonly=True)

    status, data = mail.search(None, "UNSEEN")
    if status != "OK" or not data[0]:
        mail.logout()
        if verbose:
            print("No unread emails.")
        return {"new": 0, "total_unread": 0}

    msg_ids = data[0].split()
    total_unread = len(msg_ids)
    msg_ids = list(reversed(msg_ids))[:MAX_EMAILS_PER_SCAN]

    new_count = 0
    for mid in msg_ids:
        status, msg_data = mail.fetch(mid, "(BODY.PEEK[])")
        if status != "OK":
            continue

        raw = msg_data[0][1]
        msg = email.message_from_bytes(raw)

        message_id = msg.get("Message-ID", "")
        sender = decode_header(msg.get("From", ""))
        subject = decode_header(msg.get("Subject", "(no subject)"))
        date_raw = msg.get("Date", "")
        date_parsed = email.utils.parsedate_to_datetime(date_raw) if date_raw else None
        date_str = (
            date_parsed.astimezone(timezone.utc).isoformat()
            if date_parsed else datetime.now(timezone.utc).isoformat()
        )
        preview = get_body_preview(msg)

        key = make_email_key(message_id, subject, sender)

        # Skip if already triaged
        if key in state["emails"]:
            if verbose:
                print(f"  [skip] {subject[:60]} (already triaged)")
            continue

        # Classify
        category, reason = classify_with_ollama(sender, subject, preview)
        new_count += 1

        entry = {
            "subject": subject,
            "from": sender,
            "date": date_str,
            "preview": preview[:200],
            "category": category,
            "reason": reason,
            "surfaced": False,
            "triaged_at": datetime.now(timezone.utc).isoformat(),
        }

        if verbose:
            icon = {"urgent": "🔴", "needs-response": "🟡", "informational": "🔵", "spam": "⚫"}.get(category, "⚪")
            print(f"  {icon} [{category}] {subject[:60]}")
            print(f"     From: {sender}")
            print(f"     Reason: {reason}")

        if not dry_run:
            state["emails"][key] = entry

    mail.logout()

    if not dry_run:
        # Prune old entries (keep last 200)
        if len(state["emails"]) > 200:
            sorted_keys = sorted(
                state["emails"].keys(),
                key=lambda k: state["emails"][k].get("triaged_at", ""),
                reverse=True,
            )
            state["emails"] = {k: state["emails"][k] for k in sorted_keys[:200]}
        save_state(state)

    result = {
        "new": new_count,
        "total_unread": total_unread,
    }

    if verbose:
        print(f"\nScanned {len(msg_ids)} emails, {new_count} newly triaged, {total_unread} total unread.")

    return result


def report(as_json: bool = False) -> list[dict]:
    """Report unsurfaced important emails (urgent + needs-response)."""
    state = load_state()
    important = []

    for key, entry in state["emails"].items():
        if entry.get("surfaced"):
            continue
        if entry["category"] in ("urgent", "needs-response"):
            important.append({"key": key, **entry})

    # Sort by priority (urgent first), then by date
    priority_order = {"urgent": 0, "needs-response": 1}
    important.sort(key=lambda e: (priority_order.get(e["category"], 9), e.get("date", "")))

    if as_json:
        print(json.dumps({"count": len(important), "emails": important}, indent=2))
    else:
        if not important:
            print("No important unsurfaced emails.")
        else:
            print(f"📬 {len(important)} email(s) needing attention:\n")
            for e in important:
                icon = "🔴" if e["category"] == "urgent" else "🟡"
                print(f"  {icon} {e['subject']}")
                print(f"     From: {e['from']}")
                print(f"     Date: {e['date']}")
                print(f"     Category: {e['category']} — {e['reason']}")
                print()

    return important


def mark_surfaced():
    """Mark all important emails as surfaced after they've been reported."""
    state = load_state()
    count = 0
    for key, entry in state["emails"].items():
        if not entry.get("surfaced") and entry["category"] in ("urgent", "needs-response"):
            entry["surfaced"] = True
            count += 1
    save_state(state)
    print(f"Marked {count} email(s) as surfaced.")


def stats():
    """Show triage statistics."""
    state = load_state()
    categories = {"urgent": 0, "needs-response": 0, "informational": 0, "spam": 0}
    unsurfaced = {"urgent": 0, "needs-response": 0}

    for entry in state["emails"].values():
        cat = entry.get("category", "informational")
        categories[cat] = categories.get(cat, 0) + 1
        if not entry.get("surfaced") and cat in unsurfaced:
            unsurfaced[cat] += 1

    print("Email Triage Stats")
    print(f"  Last check: {state.get('last_check', 'never')}")
    print(f"  Total triaged: {len(state['emails'])}")
    print("  Breakdown:")
    for cat, count in categories.items():
        icon = {"urgent": "🔴", "needs-response": "🟡", "informational": "🔵", "spam": "⚫"}.get(cat, "⚪")
        print(f"    {icon} {cat}: {count}")
    print("  Unsurfaced important:")
    print(f"    🔴 urgent: {unsurfaced['urgent']}")
    print(f"    🟡 needs-response: {unsurfaced['needs-response']}")


# ---------------------------------------------------------------------------
# CLI entry point
# ---------------------------------------------------------------------------

def main():
    parser = argparse.ArgumentParser(description="Email triage — IMAP scanner with AI classification")
    parser.add_argument(
        "command",
        choices=["scan", "report", "mark-surfaced", "stats"],
        help="Command to run",
    )
    parser.add_argument("--dry-run", action="store_true", help="Scan without saving state")
    parser.add_argument("--json", action="store_true", help="Output as JSON")
    parser.add_argument("--verbose", "-v", action="store_true", help="Verbose output")
    args = parser.parse_args()

    if args.command == "scan":
        result = scan_emails(dry_run=args.dry_run, verbose=args.verbose or args.dry_run)
        if args.json:
            print(json.dumps(result, indent=2))
    elif args.command == "report":
        report(as_json=args.json)
    elif args.command == "mark-surfaced":
        mark_surfaced()
    elif args.command == "stats":
        stats()


if __name__ == "__main__":
    main()
