#!/usr/bin/env python3
"""
Fetch emails from Gmail using the Gmail API.
Outputs JSON format for easy parsing by the OpenClaw agent.
"""

import os
import sys
import json
import argparse
import base64
from pathlib import Path

try:
    from google.auth.transport.requests import Request
    from google.oauth2.credentials import Credentials
    from google_auth_oauthlib.flow import InstalledAppFlow
    from googleapiclient.discovery import build
    from googleapiclient.errors import HttpError
except ImportError:
    print(json.dumps({
        "error": "Required packages not installed",
        "message": "Please install: pip install google-auth google-auth-oauthlib google-auth-httplib2 google-api-python-client"
    }))
    sys.exit(1)

# Gmail API scopes
SCOPES = ['https://www.googleapis.com/auth/gmail.readonly']


def get_credentials():
    """Authenticate and return Gmail API credentials."""
    creds = None
    credentials_path = os.getenv('GMAIL_CREDENTIALS_PATH')

    if not credentials_path:
        return None, "Environment variable GMAIL_CREDENTIALS_PATH not set"

    credentials_path = Path(credentials_path).expanduser()
    token_path = credentials_path.parent / 'token.json'

    # Token file stores user's access and refresh tokens
    if token_path.exists():
        creds = Credentials.from_authorized_user_file(str(token_path), SCOPES)

    # If there are no valid credentials, let the user log in
    if not creds or not creds.valid:
        if creds and creds.expired and creds.refresh_token:
            creds.refresh(Request())
        else:
            if not credentials_path.exists():
                return None, f"Credentials file not found at {credentials_path}"

            flow = InstalledAppFlow.from_client_secrets_file(
                str(credentials_path), SCOPES)
            creds = flow.run_local_server(port=0)

        # Save the credentials for the next run
        with open(token_path, 'w') as token:
            token.write(creds.to_json())

    return creds, None


def decode_email_body(payload):
    """Extract and decode email body from message payload."""
    body = ""

    if 'parts' in payload:
        for part in payload['parts']:
            if part['mimeType'] == 'text/plain':
                if 'data' in part['body']:
                    body = base64.urlsafe_b64decode(part['body']['data']).decode('utf-8')
                    break
            elif part['mimeType'] == 'text/html' and not body:
                if 'data' in part['body']:
                    body = base64.urlsafe_b64decode(part['body']['data']).decode('utf-8')
    else:
        if 'body' in payload and 'data' in payload['body']:
            body = base64.urlsafe_b64decode(payload['body']['data']).decode('utf-8')

    return body


def get_header_value(headers, name):
    """Get header value by name."""
    for header in headers:
        if header['name'].lower() == name.lower():
            return header['value']
    return ''


def fetch_emails(count=10, fetch_all=False):
    """Fetch emails from Gmail."""
    creds, error = get_credentials()
    if error:
        return {"error": error}

    try:
        service = build('gmail', 'v1', credentials=creds)

        # Fetch unread messages
        query = 'is:unread'
        if fetch_all:
            results = service.users().messages().list(
                userId='me', q=query).execute()
        else:
            results = service.users().messages().list(
                userId='me', q=query, maxResults=count).execute()

        messages = results.get('messages', [])

        if not messages:
            return {"emails": [], "message": "No unread emails found"}

        emails = []
        for msg in messages:
            # Get full message details
            message = service.users().messages().get(
                userId='me', id=msg['id'], format='full').execute()

            headers = message['payload']['headers']
            subject = get_header_value(headers, 'Subject')
            sender = get_header_value(headers, 'From')
            date = get_header_value(headers, 'Date')

            body = decode_email_body(message['payload'])

            # Truncate body to reasonable length (first 1000 chars)
            if len(body) > 1000:
                body = body[:1000] + "..."

            emails.append({
                "id": msg['id'],
                "subject": subject,
                "from": sender,
                "date": date,
                "body": body,
                "snippet": message.get('snippet', '')
            })

        return {"emails": emails, "count": len(emails)}

    except HttpError as error:
        return {"error": f"Gmail API error: {error}"}
    except Exception as e:
        return {"error": f"Unexpected error: {str(e)}"}


def main():
    parser = argparse.ArgumentParser(description='Fetch emails from Gmail')
    parser.add_argument('--count', type=int, default=10,
                        help='Number of emails to fetch (default: 10)')
    parser.add_argument('--all', action='store_true',
                        help='Fetch all unread emails')

    args = parser.parse_args()

    result = fetch_emails(count=args.count, fetch_all=args.all)
    print(json.dumps(result, indent=2, ensure_ascii=False))


if __name__ == '__main__':
    main()
