#!/usr/bin/env node

const nodemailer = require('nodemailer');
const imap = require('imap-simple');
const { simpleParser } = require('mailparser');

const EMAIL_USER = process.env.EMAIL_USER;
const EMAIL_PASS = process.env.EMAIL_PASS;

const args = process.argv.slice(2);
const command = args[0];

// Check credentials only if command requires them
function requireCredentials() {
  if (!EMAIL_USER || !EMAIL_PASS) {
    console.error('Error: EMAIL_USER and EMAIL_PASS environment variables required.');
    console.error('Set them before running email commands.');
    process.exit(1);
  }
}

// Zoho EU Configuration (default)
const smtpConfig = {
  host: 'smtp.zoho.eu',
  port: 465,
  secure: true,
  auth: {
    user: EMAIL_USER,
    pass: EMAIL_PASS
  }
};

const imapConfig = {
  imap: {
    user: EMAIL_USER,
    password: EMAIL_PASS,
    host: 'imap.zoho.eu',
    port: 993,
    tls: true,
    authTimeout: 20000,
    tlsOptions: { rejectUnauthorized: false }
  }
};

// Helper: Format file size
function formatBytes(bytes) {
  if (bytes === 0) return '0 B';
  const k = 1024;
  const sizes = ['B', 'KB', 'MB', 'GB'];
  const i = Math.floor(Math.log(bytes) / Math.log(k));
  return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
}

// Helper: Format attachment info
function formatAttachments(attachments) {
  if (!attachments || attachments.length === 0) {
    return 'No attachments';
  }
  
  return attachments.map((att, idx) => {
    const size = formatBytes(att.size || 0);
    const type = att.contentType || 'unknown';
    return `  [${idx + 1}] ${att.filename || 'unnamed'} (${type}, ${size})`;
  }).join('\n');
}

// Helper: Build IMAP search criteria
function buildSearchCriteria(options) {
  const criteria = [];
  
  if (options.from) {
    criteria.push(['FROM', options.from]);
  }
  
  if (options.subject) {
    criteria.push(['SUBJECT', options.subject]);
  }
  
  if (options.since) {
    criteria.push(['SINCE', options.since]);
  }
  
  if (options.before) {
    criteria.push(['BEFORE', options.before]);
  }
  
  if (options.unseen) {
    criteria.push('UNSEEN');
  } else if (options.seen) {
    criteria.push('SEEN');
  }
  
  if (options.body) {
    criteria.push(['BODY', options.body]);
  }
  
  // If no criteria, return ALL
  return criteria.length > 0 ? criteria : ['ALL'];
}

// Helper: Parse command-line arguments for search
function parseSearchArgs(args) {
  const options = {};
  
  for (let i = 0; i < args.length; i++) {
    const arg = args[i];
    
    if (arg === '--from' && args[i + 1]) {
      options.from = args[++i];
    } else if (arg === '--subject' && args[i + 1]) {
      options.subject = args[++i];
    } else if (arg === '--since' && args[i + 1]) {
      options.since = args[++i];
    } else if (arg === '--before' && args[i + 1]) {
      options.before = args[++i];
    } else if (arg === '--unseen') {
      options.unseen = true;
    } else if (arg === '--seen') {
      options.seen = true;
    } else if (arg === '--body' && args[i + 1]) {
      options.body = args[++i];
    } else if (arg === '--limit' && args[i + 1]) {
      options.limit = parseInt(args[++i]);
    } else if (!arg.startsWith('--')) {
      // First non-flag argument is the limit (for backward compatibility)
      if (!options.limit) {
        options.limit = parseInt(arg);
      }
    }
  }
  
  options.limit = options.limit || 5;
  return options;
}

async function sendEmail(to, subject, text) {
  requireCredentials();
  console.log(`Sending email to ${to}...`);
  const transporter = nodemailer.createTransport(smtpConfig);
  const info = await transporter.sendMail({
    from: EMAIL_USER,
    to,
    subject,
    text
  });
  console.log(`✓ Email sent: ${info.messageId}`);
}

async function readEmails(options = {}) {
  requireCredentials();
  const limit = options.limit || 5;
  console.log(`Searching emails with criteria...`);
  
  let connection;
  try {
    connection = await imap.connect(imapConfig);
    await connection.openBox('INBOX');
    
    const searchCriteria = buildSearchCriteria(options);
    console.log(`Search criteria: ${JSON.stringify(searchCriteria)}`);
    
    const fetchOptions = {
      bodies: ['HEADER', 'TEXT', ''],
      markSeen: false // Don't mark as seen by default
    };
    
    const messages = await connection.search(searchCriteria, fetchOptions);
    console.log(`Found ${messages.length} matching messages.\n`);
    
    if (messages.length === 0) {
      console.log('No emails found matching criteria.');
      return;
    }

    const recentMessages = messages
      .sort((a, b) => new Date(b.attributes.date) - new Date(a.attributes.date))
      .slice(0, limit);

    for (const item of recentMessages) {
      const all = item.parts.find(part => part.which === '');
      const uid = item.attributes.uid;
      const idHeader = `Imap-Id: ${uid}\r\n`;
      
      const mail = await simpleParser(idHeader + all.body);
      
      console.log(`━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━`);
      console.log(`📧 UID: ${uid}`);
      console.log(`From: ${mail.from.text}`);
      console.log(`To: ${mail.to ? mail.to.text : EMAIL_USER}`);
      console.log(`Subject: ${mail.subject}`);
      console.log(`Date: ${mail.date}`);
      console.log(`Attachments: ${mail.attachments.length > 0 ? mail.attachments.length : 'None'}`);
      
      if (mail.attachments.length > 0) {
        console.log(`\n📎 Attachment Details:`);
        console.log(formatAttachments(mail.attachments));
      }
      
      console.log(`\n📝 Body:\n${mail.text.substring(0, 500)}${mail.text.length > 500 ? '...' : ''}`);
      console.log(`━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n`);
    }
  } catch (err) {
    console.error("❌ IMAP Operation Error:", err.message);
  } finally {
    if (connection) {
      connection.end();
    }
  }
}

async function listFolders() {
  requireCredentials();
  console.log('Listing IMAP folders...\n');
  
  let connection;
  try {
    connection = await imap.connect(imapConfig);
    const boxes = await connection.getBoxes();
    
    function printBox(box, name, indent = '') {
      const flags = box.attribs || [];
      const delimiter = box.delimiter || '/';
      const hasChildren = box.children && Object.keys(box.children).length > 0;
      
      console.log(`${indent}📁 ${name}${hasChildren ? delimiter : ''} ${flags.join(', ')}`);
      
      if (box.children) {
        Object.keys(box.children).forEach(childName => {
          printBox(box.children[childName], childName, indent + '  ');
        });
      }
    }
    
    Object.keys(boxes).forEach(name => {
      printBox(boxes[name], name);
    });
    
  } catch (err) {
    console.error("❌ Error listing folders:", err.message);
  } finally {
    if (connection) {
      connection.end();
    }
  }
}

async function moveEmail(uid, targetFolder) {
  requireCredentials();
  console.log(`Moving email UID ${uid} to folder "${targetFolder}"...`);
  
  let connection;
  try {
    connection = await imap.connect(imapConfig);
    
    // First, verify the target folder exists
    const boxes = await connection.getBoxes();
    const folderExists = checkFolderExists(boxes, targetFolder);
    
    if (!folderExists) {
      console.error(`❌ Error: Folder "${targetFolder}" does not exist.`);
      console.log('\nAvailable folders:');
      Object.keys(boxes).forEach(name => console.log(`  - ${name}`));
      return;
    }
    
    await connection.openBox('INBOX');
    
    // Move the message
    await connection.moveMessage(uid, targetFolder);
    
    console.log(`✓ Email UID ${uid} moved to "${targetFolder}"`);
    
  } catch (err) {
    console.error("❌ Error moving email:", err.message);
  } finally {
    if (connection) {
      connection.end();
    }
  }
}

// Helper: Check if folder exists recursively
function checkFolderExists(boxes, targetFolder) {
  function search(box, name) {
    if (name === targetFolder) return true;
    
    if (box.children) {
      for (const childName of Object.keys(box.children)) {
        if (search(box.children[childName], childName)) {
          return true;
        }
      }
    }
    return false;
  }
  
  for (const name of Object.keys(boxes)) {
    if (search(boxes[name], name)) return true;
  }
  
  return false;
}

async function searchEmails(searchOptions) {
  console.log('🔍 Advanced search...\n');
  await readEmails(searchOptions);
}

function showHelp() {
  console.log(`
Email Manager Lite v0.2.0
=========================

USAGE:
  email.js <command> [options]

COMMANDS:
  send <to> <subject> <body>    Send an email
  read [limit]                   Read recent emails (default: 5)
  search [options]               Advanced search with filters
  folders                        List all IMAP folders
  move <uid> <folder>            Move email to specific folder

SEARCH OPTIONS:
  --from <email>                 Filter by sender email
  --subject <text>               Filter by subject keywords
  --since <date>                 Filter emails since date (format: Jan 1, 2025)
  --before <date>                Filter emails before date
  --unseen                       Show only unread emails
  --seen                         Show only read emails
  --body <text>                  Search in email body (slow)
  --limit <n>                    Limit number of results (default: 5)

EXAMPLES:
  # Send email
  email.js send "user@example.com" "Hello" "This is the body"
  
  # Read 10 latest emails
  email.js read 10
  
  # Search emails from specific sender
  email.js search --from "boss@company.com" --limit 3
  
  # Search by subject
  email.js search --subject "invoice" --unseen
  
  # Search by date range
  email.js search --since "Jan 1, 2026" --before "Feb 1, 2026"
  
  # Search in body (slow)
  email.js search --body "meeting tomorrow"
  
  # List all folders
  email.js folders
  
  # Move email to Archive
  email.js move 12345 "Archive"

ENVIRONMENT VARIABLES:
  EMAIL_USER    Your email address
  EMAIL_PASS    Your email password or app password
`);
}

async function main() {
  if (!command || command === 'help' || command === '--help' || command === '-h') {
    showHelp();
    return;
  }

  switch (command) {
    case 'send':
      if (args.length < 4) {
        console.error('Error: send requires <to> <subject> <body>');
        process.exit(1);
      }
      await sendEmail(args[1], args[2], args[3]);
      break;
      
    case 'read':
      const limit = parseInt(args[1]) || 5;
      await readEmails({ limit });
      break;
      
    case 'search':
      const searchOptions = parseSearchArgs(args.slice(1));
      await searchEmails(searchOptions);
      break;
      
    case 'folders':
      await listFolders();
      break;
      
    case 'move':
      if (args.length < 3) {
        console.error('Error: move requires <uid> <folder>');
        process.exit(1);
      }
      await moveEmail(args[1], args[2]);
      break;
      
    default:
      console.error(`Unknown command: ${command}`);
      console.log('Run "email.js help" for usage information.');
      process.exit(1);
  }
}

main().catch(err => {
  console.error('Fatal error:', err);
  process.exit(1);
});
