# Mix Commands Reference

## Project Management

```bash
mix new my_app --sup            # New project with supervision tree
mix phx.new my_app --no-html    # Phoenix API-only project
mix deps.get                    # Fetch dependencies
mix deps.update --all           # Update all deps
mix deps.tree                   # Show dependency tree
mix deps.unlock --unused        # Clean unused locks
mix compile                     # Compile project
mix compile --force             # Force recompile everything
mix clean                       # Remove build artifacts
```

## Testing

```bash
mix test                                    # Run all
mix test test/my_app/accounts_test.exs      # Single file
mix test test/my_app/accounts_test.exs:42   # Single test (line)
mix test --only integration                 # By tag
mix test --exclude integration              # Exclude tag
mix test --failed                           # Re-run failures
mix test --cover                            # With coverage
mix test --trace                            # Verbose output
mix test --seed 0                           # Deterministic order
mix test --max-failures 3                   # Stop after N failures
mix test --stale                            # Only changed modules
MIX_ENV=test mix test                       # Explicit env (usually automatic)
```

### Test Tags

```elixir
# In test file
@tag :integration
test "sends email" do ... end

@tag timeout: 120_000
test "slow operation" do ... end

# In test_helper.exs
ExUnit.configure(exclude: [:integration])
```

## Code Quality

```bash
mix format                      # Format all files
mix format --check-formatted    # Check without modifying (CI)
mix format --dry-run            # Show what would change

mix credo                       # Default checks
mix credo --strict              # All checks
mix credo suggest               # Suggestions only
mix credo list                  # List all issues
mix credo explain Credo.Check.Readability.ModuleDoc  # Explain a check
mix credo --format json         # JSON output

mix dialyzer                    # Type checking
mix dialyzer --format short     # Compact output
mix dialyzer --format dialyxir  # Detailed format
```

## Database (Ecto)

```bash
mix ecto.create                 # Create database
mix ecto.drop                   # Drop database
mix ecto.reset                  # Drop + create + migrate + seed

mix ecto.gen.migration create_users  # Generate migration
mix ecto.migrate                     # Run pending migrations
mix ecto.rollback                    # Rollback last migration
mix ecto.rollback --step 3           # Rollback 3 migrations
mix ecto.rollback --to 20240101000000  # Rollback to specific version
mix ecto.migrations                  # List migration status

mix run priv/repo/seeds.exs     # Run seeds
```

## Phoenix

```bash
mix phx.server                  # Start server
mix phx.routes                  # List all routes
mix phx.gen.context Accounts User users email:string name:string
mix phx.gen.json Accounts User users email:string name:string  # JSON resource
mix phx.gen.schema Accounts.User users email:string            # Schema only
mix phx.gen.auth Accounts User users                           # Full auth system
mix phx.digest                  # Digest static assets
```

## Release

```bash
mix release                     # Build release
mix release --overwrite         # Overwrite existing
MIX_ENV=prod mix release        # Production release

# Runtime migrations (in release)
bin/my_app eval "MyApp.Release.migrate()"
```

## IEx

```bash
iex -S mix                      # IEx with project loaded
iex -S mix phx.server           # IEx + Phoenix server

# Inside IEx
recompile()                     # Recompile changed modules
Mix.Task.rerun("ecto.migrate")  # Run mix task
MyApp.Repo.all(MyApp.Accounts.User)  # Query directly
```

## Environment Variables

```bash
MIX_ENV=test mix test           # Set mix env
MIX_ENV=prod mix compile        # Compile for prod
DATABASE_URL=... mix ecto.migrate  # Runtime DB config
```
