import requests
import os
import base64
from pathlib import Path
from dotenv import load_dotenv

# Load environment variables from workspace .env
load_dotenv(dotenv_path=os.path.join(os.path.dirname(__file__), '..', '..', '..', '.env'))

class ElevenLabsClient:
    """Client for ElevenLabs Text-to-Speech API"""
    
    def __init__(self, api_key=None):
        self.api_key = api_key or os.getenv('ELEVENLABS_API_KEY')
        self.base_url = "https://api.elevenlabs.io/v1"
        
    def text_to_speech(self, text, voice_id="21m00Tcm4TlvDq8ikWAM", model_id="eleven_turbo_v2_5", 
                       output_path="output.mp3", stability=0.5, similarity_boost=0.75):
        """
        Convert text to speech using ElevenLabs API
        
        Default voice: Rachel (21m00Tcm4TlvDq8ikWAM) - natural, versatile
        """
        
        url = f"{self.base_url}/text-to-speech/{voice_id}"
        
        headers = {
            "Accept": "audio/mpeg",
            "Content-Type": "application/json",
            "xi-api-key": self.api_key
        }
        
        payload = {
            "text": text,
            "model_id": model_id,
            "voice_settings": {
                "stability": stability,
                "similarity_boost": similarity_boost
            }
        }
        
        try:
            response = requests.post(url, headers=headers, json=payload, timeout=60)
            response.raise_for_status()
            
            # Save audio file
            with open(output_path, 'wb') as f:
                f.write(response.content)
            
            return {
                "success": True,
                "file_path": output_path,
                "size_bytes": len(response.content)
            }
                
        except requests.exceptions.RequestException as e:
            return {"success": False, "error": f"Request failed: {str(e)}"}
        except Exception as e:
            return {"success": False, "error": str(e)}
    
    def get_voices(self):
        """Get list of available voices"""
        
        url = f"{self.base_url}/voices"
        
        headers = {
            "Accept": "application/json",
            "xi-api-key": self.api_key
        }
        
        try:
            response = requests.get(url, headers=headers, timeout=30)
            response.raise_for_status()
            
            data = response.json()
            voices = []
            for voice in data.get('voices', []):
                voices.append({
                    'voice_id': voice['voice_id'],
                    'name': voice['name'],
                    'category': voice.get('category', 'standard'),
                    'preview_url': voice.get('preview_url', '')
                })
            
            return {"success": True, "voices": voices}
                
        except requests.exceptions.RequestException as e:
            return {"success": False, "error": f"Request failed: {str(e)}"}
        except Exception as e:
            return {"success": False, "error": str(e)}


def main():
    """CLI interface for ElevenLabs Speech"""
    import argparse
    
    parser = argparse.ArgumentParser(description='ElevenLabs Speech Client')
    parser.add_argument('action', choices=['tts', 'voices'], help='Action: tts or voices')
    parser.add_argument('--text', '-t', help='Text to convert to speech')
    parser.add_argument('--output', '-o', default='output.mp3', help='Output file path')
    parser.add_argument('--voice', '-v', default='21m00Tcm4TlvDq8ikWAM', help='Voice ID')
    
    args = parser.parse_args()
    
    client = ElevenLabsClient()
    
    if args.action == 'tts':
        if not args.text:
            print("Error: --text required for TTS")
            return
        result = client.text_to_speech(args.text, voice_id=args.voice, output_path=args.output)
        print(json.dumps(result, indent=2))
    
    elif args.action == 'voices':
        result = client.get_voices()
        print(json.dumps(result, indent=2))


if __name__ == "__main__":
    import json
    main()
