import requests
import os
from dotenv import load_dotenv

load_dotenv()

class ElevenLabsScribe:
    """Speech-to-Text using ElevenLabs Scribe"""
    
    def __init__(self, api_key=None):
        self.api_key = api_key or os.getenv('ELEVENLABS_API_KEY')
        self.base_url = "https://api.elevenlabs.io/v1"
    
    def transcribe(self, audio_file_path, language_code=None, tag_audio_events=True, 
                   num_speakers=None, timestamps_granularity="word"):
        """
        Transcribe audio file to text using ElevenLabs Scribe
        
        Supports: mp3, mp4, mpeg, mpga, m4a, wav, webm
        Max file size: 100 MB
        """
        url = f"{self.base_url}/speech-to-text"
        
        headers = {
            "xi-api-key": self.api_key
        }
        
        data = {
            "model_id": "scribe_v1",
            "tag_audio_events": str(tag_audio_events).lower(),
            "timestamps_granularity": timestamps_granularity
        }
        
        if language_code:
            data["language_code"] = language_code
        if num_speakers:
            data["num_speakers"] = num_speakers
        
        try:
            with open(audio_file_path, 'rb') as audio_file:
                files = {
                    'file': (os.path.basename(audio_file_path), audio_file)
                }
                
                response = requests.post(url, headers=headers, data=data, files=files, timeout=120)
                response.raise_for_status()
                
                result = response.json()
                return {
                    "success": True,
                    "text": result.get('text', ''),
                    "language": result.get('language_code', 'unknown'),
                    "words": result.get('words', []),
                    "speakers": result.get('speakers', [])
                }
                
        except requests.exceptions.RequestException as e:
            return {"success": False, "error": f"Request failed: {str(e)}"}
        except Exception as e:
            return {"success": False, "error": str(e)}


def main():
    import argparse
    
    parser = argparse.ArgumentParser(description='ElevenLabs Scribe STT')
    parser.add_argument('audio_file', help='Path to audio file')
    parser.add_argument('--language', '-l', help='Language code (e.g., ar, en)')
    parser.add_argument('--speakers', '-s', type=int, help='Number of speakers')
    
    args = parser.parse_args()
    
    client = ElevenLabsScribe()
    result = client.transcribe(args.audio_file, language_code=args.language, num_speakers=args.speakers)
    print(result)


if __name__ == "__main__":
    import json
    main()
