# Column Types Reference

## Type Mapping: Ecto → PostgreSQL

| Ecto Type | PostgreSQL | Use When |
|-----------|-----------|----------|
| `:string` | `varchar(255)` | Short text: names, emails, slugs, enum values |
| `:text` | `text` | Long text: descriptions, body content, notes |
| `:integer` | `integer` (32-bit) | Counts, small numbers, money in cents |
| `:bigint` | `bigint` (64-bit) | External IDs, large counters, Snowflake IDs |
| `:float` | `double precision` | Scientific data. **Never for money.** |
| `:decimal` | `numeric` | Money, precise decimals. Specify precision/scale. |
| `:boolean` | `boolean` | True/false flags |
| `:binary_id` | `uuid` | Primary keys, foreign keys |
| `:binary` | `bytea` | Raw binary data, file content |
| `:map` | `jsonb` | Schemaless JSON, metadata, settings |
| `{:map, inner}` | `jsonb` | Typed map values |
| `{:array, :string}` | `varchar[]` | Tags, categories, small lists |
| `{:array, :integer}` | `integer[]` | ID lists, numeric arrays |
| `:date` | `date` | Calendar dates (no time) |
| `:time` | `time` | Time of day (no date) |
| `:utc_datetime` | `timestamptz` | Timestamps (second precision) |
| `:utc_datetime_usec` | `timestamptz` | Timestamps (microsecond precision) — **preferred** |
| `:naive_datetime` | `timestamp` | **Avoid** — no timezone info |
| `Ecto.Enum` | `varchar` | Enumerated values (validated in app) |

## Special PostgreSQL Types

| PostgreSQL Type | Ecto Migration Syntax | Use When |
|----------------|----------------------|----------|
| `citext` | `:citext` | Case-insensitive text (emails, usernames). Requires `citext` extension. |
| `inet` | `:inet` | IP addresses |
| `macaddr` | `:macaddr` | MAC addresses |
| `tstzrange` | Custom Ecto type | Time ranges (booking systems) |
| `tsvector` | `:tsvector` | Full-text search |
| `point` | Custom Ecto type | Geographic coordinates |

## Column Options

```elixir
add :email, :string, null: false                        # Required
add :role, :string, null: false, default: "member"      # With default
add :amount, :decimal, precision: 12, scale: 2          # Money
add :count, :integer, default: 0                        # Counter
add :metadata, :map, default: %{}                       # Empty JSON
add :tags, {:array, :string}, default: []               # Empty array
add :score, :float                                      # Nullable by default
add :name, :string, size: 100                           # Custom varchar length
```

## Decision Guide

### IDs

- **Internal PKs/FKs**: `:binary_id` (UUID v4). Set `primary_key: false` on table.
- **External IDs** (Stripe, GitHub): `:string` — they're opaque strings.
- **Snowflake/Twitter IDs**: `:bigint` — they overflow 32-bit integers.
- **Sequential display IDs**: `:integer` or `:bigserial` — human-readable.

### Money

- **Preferred**: `:integer` storing cents/smallest unit. `1099` = $10.99.
- **Alternative**: `:decimal` with `precision: 12, scale: 2`.
- **Never**: `:float` — floating point rounding errors.

```elixir
add :amount_cents, :integer, null: false, default: 0
# OR
add :amount, :decimal, precision: 12, scale: 2, null: false
```

### Text

- `:string` (varchar 255) — emails, names, slugs, short identifiers.
- `:text` — markdown content, descriptions, HTML, anything potentially long.
- `:citext` — case-insensitive lookups (email, username). Needs extension.

### Dates & Times

- `:utc_datetime_usec` — **always use this** for timestamps. Microsecond precision, timezone aware.
- `:date` — birthdays, expiry dates, calendar events (date only).
- `:time` — recurring times ("daily at 09:00").
- **Never** `:naive_datetime` — loses timezone context.

### JSON (`:map`)

Good for:
- User preferences/settings
- Metadata/tags
- Flexible schema attributes
- API response caching

Bad for:
- Data you need to query frequently (use proper columns)
- Data with strict schema requirements (use embedded schemas + columns)

When using `:map`, consider adding a GIN index for queries:

```elixir
add :metadata, :map, default: %{}
create index(:events, [:metadata], using: :gin)
```

### Arrays

Good for:
- Tags: `{:array, :string}`
- Small ID lists: `{:array, :binary_id}`
- Feature flags: `{:array, :string}`

Bad for:
- Large lists (>100 items) — use a join table instead
- Frequently updated lists — arrays rewrite the whole value

### Boolean

- Use for binary flags: `is_active`, `email_verified`, `archived`.
- Always set `null: false` with a `default`:

```elixir
add :is_active, :boolean, null: false, default: true
add :archived, :boolean, null: false, default: false
```

### Enums

**Prefer `:string` + `Ecto.Enum`** over PostgreSQL native enums:

```elixir
# Migration
add :status, :string, null: false, default: "pending"

# Schema
field :status, Ecto.Enum, values: [:pending, :active, :suspended, :deleted]
```

PostgreSQL enums are hard to modify (can't remove values, adding values requires special handling). String columns are flexible and `Ecto.Enum` provides validation.
