# 价格比较逻辑

本文档详细说明电商价格比较的算法和逻辑。

## 比较维度

### 1. 价格维度（权重：40%）
- **商品售价**：当前显示价格
- **优惠后价格**：使用优惠券、满减后的实际支付价格
- **历史价格**：与历史价格对比，判断是否处于低位
- **平台补贴**：百亿补贴、平台优惠等

### 2. 成本维度（权重：20%）
- **运费**：是否包邮，运费金额
- **税费**：需要缴纳的税费
- **其他费用**：安装费、服务费等

### 3. 信誉维度（权重：25%）
- **店铺评分**：店铺综合评分
- **商品评价**：商品好评率
- **评价数量**：评价基数，数量越多越可信
- **店铺类型**：官方旗舰店 > 品牌专卖店 > 普通店铺

### 4. 服务维度（权重：15%）
- **发货速度**：预计发货时间
- **售后服务**：退换货政策、保修期限
- **客服质量**：客服响应速度和服务态度

## 评分算法

### 综合得分计算
```
综合得分 = 价格得分 × 40% + 成本得分 × 20% + 信誉得分 × 25% + 服务得分 × 15%
```

### 各维度得分计算

#### 价格得分（0-100分）
```
价格得分 = 100 × (最低价格 / 当前价格)
```
- 最低价格：所有平台中该商品的最低价格
- 当前价格：该平台该商品的当前价格

#### 成本得分（0-100分）
```
成本得分 = 100 × (1 - 额外成本比例)
额外成本比例 = 额外成本 / 商品价格
额外成本 = 运费 + 税费 + 其他费用
```

#### 信誉得分（0-100分）
```
信誉得分 = 店铺评分 × 20 + 商品评价 × 30 + 店铺类型系数 × 50
店铺类型系数：官方旗舰店=1.0，品牌专卖店=0.8，普通店铺=0.6
```

#### 服务得分（0-100分）
```
服务得分 = 发货速度得分 × 40 + 售后服务得分 × 40 + 客服质量得分 × 20
```

## 推荐逻辑

### 推荐等级
1. **强烈推荐**（得分 ≥ 85）：价格优势明显，信誉良好
2. **推荐**（得分 70-84）：性价比高，无明显短板
3. **一般**（得分 60-69）：价格适中，可考虑
4. **不推荐**（得分 < 60）：价格偏高或信誉较差

### 特殊情况处理

#### 价格异常低
- 检查是否为二手、翻新、展示机
- 检查是否有隐藏条件（如需要拼团）
- 检查店铺信誉，防止诈骗

#### 评价数量过少
- 新店铺或新商品需要谨慎
- 结合店铺整体信誉判断
- 建议等待更多评价

#### 平台差异
- 京东：正品保障好，物流快
- 淘宝：商品丰富，价格灵活
- 天猫：品牌官方，质量有保障
- 拼多多：价格优势明显，注意辨别

## 高级分析

### 价格趋势分析
```python
def analyze_price_trend(prices):
    """
    分析价格趋势
    prices: 历史价格列表，按时间排序
    返回：趋势（上涨/下跌/平稳），幅度，建议
    """
    if len(prices) < 2:
        return "数据不足", 0, "需要更多价格数据"
    
    latest = prices[-1]
    previous = prices[-2] if len(prices) >= 2 else latest
    change = ((latest - previous) / previous) * 100
    
    if change < -5:
        return "下跌", change, "价格下降，适合购买"
    elif change > 5:
        return "上涨", change, "价格上涨，建议观望"
    else:
        return "平稳", change, "价格稳定，可随时购买"
```

### 性价比分析
```python
def calculate_value_score(price, rating, features):
    """
    计算性价比得分
    price: 价格
    rating: 评分
    features: 功能特性列表
    返回：性价比得分（0-100）
    """
    # 基础价格得分
    price_score = max(0, 100 - (price / 1000) * 10)
    
    # 评分转换
    rating_score = rating * 20  # 5分制转100分制
    
    # 特性加分
    feature_bonus = len(features) * 5
    
    # 综合计算
    value_score = price_score * 0.4 + rating_score * 0.4 + feature_bonus * 0.2
    
    return min(100, value_score)
```

## 自定义配置

### 权重调整
用户可以根据自己的偏好调整各维度权重：

```json
{
  "price_weight": 0.4,
  "cost_weight": 0.2,
  "reputation_weight": 0.25,
  "service_weight": 0.15,
  "min_rating_threshold": 4.0,
  "min_review_count": 100
}
```

### 过滤条件
- **价格范围**：设置可接受的价格区间
- **店铺类型**：只显示特定类型的店铺
- **发货地区**：指定发货地区
- **评价要求**：最低评分和评价数量

## 输出格式

### 简要报告
```
商品：iPhone 15 128GB
搜索时间：2024-01-01 12:00:00

平台对比：
1. 京东 - ¥5999 (推荐) ★★★★☆
   优点：官方正品，物流快
   缺点：价格稍高

2. 拼多多 - ¥5499 (强烈推荐) ★★★★★
   优点：价格最低，百亿补贴
   缺点：发货稍慢

3. 天猫 - ¥5799 (推荐) ★★★★☆
   优点：品牌官方，质量保障
   缺点：价格中等

最佳选择：拼多多，节省¥500
建议：如果追求性价比，选择拼多多；如果看重正品保障，选择京东。
```

### 详细报告
包含价格历史图表、各维度详细评分、店铺信息对比等。

## 更新日志

### 算法优化
- 2024.01.01：初始版本
- 2024.01.15：增加价格趋势分析
- 2024.02.01：优化信誉评分算法

### 平台适配
- 定期更新各平台抓取规则
- 根据平台特点调整权重
- 优化反爬策略