import unittest
from unittest.mock import patch, MagicMock
import json
import sys
import os

# Add current directory to path so we can import the client
sys.path.append(os.path.dirname(os.path.abspath(__file__)))

import echodecks_client

class TestEchoDecksClient(unittest.TestCase):

    def setUp(self):
        # Mock environment variable
        self.env_patcher = patch.dict(os.environ, {"ECHODECKS_API_KEY": "test_key"})
        self.env_patcher.start()

    def tearDown(self):
        self.env_patcher.stop()

    @patch('requests.get')
    def test_list_decks(self, mock_get):
        # Setup mock response
        mock_response = MagicMock()
        mock_response.json.return_value = {"decks": [{"id": "1", "title": "Test Deck"}]}
        mock_response.raise_for_status.return_value = None
        mock_get.return_value = mock_response

        # Call the handler directly
        args = MagicMock()
        args.id = None
        result = echodecks_client.handle_list_decks(args)

        # Assertions
        self.assertEqual(result, {"decks": [{"id": "1", "title": "Test Deck"}]})
        mock_get.assert_called_with(
            "https://echodecks.com/functions/externalApi?resource=decks",
            headers={"X-API-KEY": "test_key", "Content-Type": "application/json"}
        )

    @patch('requests.post')
    def test_submit_review(self, mock_post):
        mock_response = MagicMock()
        mock_response.json.return_value = {"success": True}
        mock_response.raise_for_status.return_value = None
        mock_post.return_value = mock_response

        args = MagicMock()
        args.card_id = "card_123"
        args.quality = 3
        
        result = echodecks_client.handle_submit_review(args)

        self.assertEqual(result, {"success": True})
        mock_post.assert_called_with(
            "https://echodecks.com/functions/externalApi?resource=study&action=submit",
            headers={"X-API-KEY": "test_key", "Content-Type": "application/json"},
            json={"cardId": "card_123", "quality": 3}
        )

    @patch('requests.post')
    def test_generate_podcast(self, mock_post):
        mock_response = MagicMock()
        mock_response.json.return_value = {"jobId": "job_999"}
        mock_post.return_value = mock_response

        args = MagicMock()
        args.deck_id = "deck_55"
        args.voice = "nova"
        args.type = "summary"

        result = echodecks_client.handle_generate_podcast(args)
        
        mock_post.assert_called_with(
            "https://echodecks.com/functions/externalApi?resource=podcasts&action=generate",
            headers={"X-API-KEY": "test_key", "Content-Type": "application/json"},
            json={"deckId": "deck_55", "voice": "nova", "type": "summary"}
        )
        self.assertEqual(result, {"jobId": "job_999"})

if __name__ == '__main__':
    unittest.main()
