#!/usr/bin/env bash
# Register for an API key at the ecap Trust Registry
# Usage: bash scripts/register.sh <agent-name>
# Saves key to config/credentials.json

set -euo pipefail

# Dependencies: curl, jq
for cmd in curl jq; do
  if ! command -v "$cmd" &>/dev/null; then
    echo "❌ Required dependency '$cmd' not found. Install it first." >&2
    exit 1
  fi
done

REGISTRY_URL="${ECAP_REGISTRY_URL:-https://skillaudit-api.vercel.app}"
SCRIPT_DIR="$(cd "$(dirname "$0")" && pwd)"
CRED_FILE="$SCRIPT_DIR/../config/credentials.json"

AGENT_NAME="${1:-}"
if [ -z "$AGENT_NAME" ]; then
  echo "Usage: bash scripts/register.sh <agent-name>" >&2
  echo "  agent-name: alphanumeric, dashes, underscores, dots (2-64 chars)" >&2
  exit 1
fi

# Sanitize agent name: only allow alphanumeric, dashes, underscores, dots
if ! echo "$AGENT_NAME" | grep -qE '^[a-zA-Z0-9._-]{2,64}$'; then
  echo "❌ Invalid agent name. Use only alphanumeric, dashes, underscores, dots (2-64 chars)." >&2
  exit 1
fi

# Check if already registered
if [ -f "$CRED_FILE" ]; then
  EXISTING_KEY=$(jq -r '.api_key // empty' "$CRED_FILE" 2>/dev/null || true)
  if [ -n "$EXISTING_KEY" ]; then
    echo "Already registered. Key found in $CRED_FILE"
    exit 0
  fi
fi

echo "Registering agent '$AGENT_NAME' at $REGISTRY_URL/api/register ..."

# Use jq to safely build JSON payload (prevents injection)
JSON_PAYLOAD=$(jq -n --arg name "$AGENT_NAME" '{agent_name: $name}')

RESPONSE=$(curl -s -w "\n%{http_code}" -X POST "$REGISTRY_URL/api/register" \
  -H "Content-Type: application/json" \
  -d "$JSON_PAYLOAD")

HTTP_CODE=$(echo "$RESPONSE" | tail -1)
BODY=$(echo "$RESPONSE" | sed '$d')

if [ "$HTTP_CODE" -ge 200 ] && [ "$HTTP_CODE" -lt 300 ]; then
  mkdir -p "$(dirname "$CRED_FILE")"
  echo "$BODY" | jq '{api_key: .api_key, agent_name: .agent_name}' > "$CRED_FILE"
  chmod 600 "$CRED_FILE"
  echo "✅ Registered successfully!"
  echo "Credentials saved to: $CRED_FILE"
else
  echo "❌ Registration failed (HTTP $HTTP_CODE):" >&2
  echo "$BODY" >&2
  exit 1
fi
