# MASTER FIX SPEC — Scripts & Prompts

**Erstellt:** 2025-07-17  
**Quellen:** ADVERSARIAL-TEST.md, INTEGRATION-TEST.md, DOCS-REVIEW.md  
**Status:** Ready for implementation

---

## 1. verify.sh

### FIX-V1: Usage-Kommentar falsch (Zeile 3)

**Problem:** Kommentar sagt `Usage: ./scripts/verify.sh [API_URL]`, tatsächlich erwartet das Script `<package-name>` als erstes Argument.  
**Quellen:** DOCS-REVIEW §3, INTEGRATION-TEST §7  
**Severity:** Low (Doku-Fehler, aber irreführend für Agents)

**Aktuell (Zeile 3):**
```bash
# Usage: ./scripts/verify.sh [API_URL]
```

**Fix:**
```bash
# Usage: ./scripts/verify.sh <package-name>
```

> Hinweis: `[api-url]` wird durch FIX-V2 entfernt, daher hier nicht mehr erwähnt.

---

### FIX-V2: API URL Hardcoden — Custom URL entfernen

**Problem:** Zweites Argument erlaubt beliebige API URL. Ein bösartiges SKILL.md könnte `verify.sh "pkg" "https://evil.com/api/integrity"` aufrufen und Fake-Hashes liefern → Integrity-Check komplett ausgehebelt.  
**Quellen:** ADVERSARIAL-TEST §3.4 (Severity: High), DOCS-REVIEW §3  
**Severity:** High

**Empfehlung:** Option (a) — Hardcoden. Kein Override möglich.

**Aktuell (Zeilen 7-8):**
```bash
PACKAGE="${1:?Usage: verify.sh <package-name> [api-url]}"
API_URL="${2:-https://skillaudit-api.vercel.app/api/integrity}"
```

**Fix:**
```bash
PACKAGE="${1:?Usage: verify.sh <package-name>}"
API_URL="https://skillaudit-api.vercel.app/api/integrity"
```

> Begründung: verify.sh ist ein Sicherheits-Tool. Die Integrität der Verifikation darf nicht durch Argumente unterlaufen werden. `ECAP_REGISTRY_URL` bleibt absichtlich NICHT unterstützt hier — im Gegensatz zu upload.sh/register.sh, wo Self-Hosting sinnvoll ist, wäre es bei verify.sh ein Angriffsvektor.

---

### FIX-V3: URL-Encoding für Package-Name

**Problem:** `$PACKAGE` wird direkt in die curl URL interpoliert. Sonderzeichen wie `&`, `#`, Leerzeichen manipulieren die HTTP-Anfrage.  
**Quellen:** ADVERSARIAL-TEST §1.4 (Severity: Medium)  
**Severity:** Medium

**Aktuell (Zeile 30):**
```bash
RESPONSE=$(curl -sf "${API_URL}?package=${PACKAGE}") || { echo "❌ API request failed"; exit 1; }
```

**Fix:**
```bash
# URL-encode package name (handles &, #, spaces, etc.)
PACKAGE_ENCODED=$(printf '%s' "$PACKAGE" | jq -sRr @uri)
RESPONSE=$(curl -sf "${API_URL}?package=${PACKAGE_ENCODED}") || { echo "❌ API request failed"; exit 1; }
```

> Platzierung: Direkt nach der `PACKAGE=` Zeile oder direkt vor dem curl-Aufruf. `jq` ist bereits eine deklarierte Dependency.

---

### FIX-V4: Hardcoded FILES Array dokumentieren

**Problem:** FILES-Array ist hardcoded auf ecap-eigene Dateien. Für andere Packages (npm, pip, MCP) ist verify.sh nutzlos.  
**Quellen:** INTEGRATION-TEST §3, §4 (BUG-002, Severity: High)  
**Severity:** High

**Empfehlung:** Dynamisch aus `/api/integrity` Response laden.

**Aktuell (Zeilen 19-26):**
```bash
FILES=(
  "SKILL.md"
  "scripts/upload.sh"
  "scripts/register.sh"
  "prompts/audit-prompt.md"
  "prompts/review-prompt.md"
  "README.md"
)
```

**Fix:**
```bash
echo "🔍 Fetching official hashes from registry..."
PACKAGE_ENCODED=$(printf '%s' "$PACKAGE" | jq -sRr @uri)
RESPONSE=$(curl -sf "${API_URL}?package=${PACKAGE_ENCODED}") || { echo "❌ API request failed"; exit 1; }

# Dynamisch: Dateiliste aus API-Response extrahieren
mapfile -t FILES < <(echo "$RESPONSE" | jq -r '.files | keys[]')

if [ ${#FILES[@]} -eq 0 ]; then
  echo "⚠️  No tracked files found for package '${PACKAGE}'"
  exit 1
fi
```

> Hinweis: Der curl-Aufruf muss VOR das FILES-Array verschoben werden. Der zweite curl-Aufruf (Zeile 30) entfällt dann — `$RESPONSE` ist bereits vorhanden. Das gesamte Script muss entsprechend umstrukturiert werden.

**Komplettes verify.sh nach allen Fixes (V1-V4):**

```bash
#!/usr/bin/env bash
# verify.sh — Verify local skill files against the ecap Trust Registry
# Usage: ./scripts/verify.sh <package-name>
# Dependencies: curl, jq, sha256sum (or shasum on macOS)
set -euo pipefail

PACKAGE="${1:?Usage: verify.sh <package-name>}"
API_URL="https://skillaudit-api.vercel.app/api/integrity"
SCRIPT_DIR="$(cd "$(dirname "$0")" && pwd)"
ROOT_DIR="$(cd "$SCRIPT_DIR/.." && pwd)"

# Detect sha256 command
if command -v sha256sum &>/dev/null; then
  SHA_CMD="sha256sum"
elif command -v shasum &>/dev/null; then
  SHA_CMD="shasum -a 256"
else
  echo "❌ No sha256sum or shasum found"; exit 1
fi

# URL-encode package name
PACKAGE_ENCODED=$(printf '%s' "$PACKAGE" | jq -sRr @uri)

echo "🔍 Fetching official hashes from registry..."
RESPONSE=$(curl -sf "${API_URL}?package=${PACKAGE_ENCODED}") || { echo "❌ API request failed"; exit 1; }

# Dynamic file list from API response
mapfile -t FILES < <(echo "$RESPONSE" | jq -r '.files | keys[]')

if [ ${#FILES[@]} -eq 0 ]; then
  echo "⚠️  No tracked files found for package '${PACKAGE}'"
  exit 1
fi

MISMATCH=0
CHECKED=0

echo ""
echo "Package: ${PACKAGE}"
echo "Repo:    $(echo "$RESPONSE" | jq -r '.repo')"
echo "Commit:  $(echo "$RESPONSE" | jq -r '.commit' | head -c 12)"
echo "Verified: $(echo "$RESPONSE" | jq -r '.verified_at')"
echo ""

for file in "${FILES[@]}"; do
  LOCAL_PATH="${ROOT_DIR}/${file}"
  REMOTE_HASH=$(echo "$RESPONSE" | jq -r ".files[\"${file}\"].sha256 // empty")

  if [ -z "$REMOTE_HASH" ] || [ "$REMOTE_HASH" = "null" ]; then
    echo "⚠️  ${file} — not tracked by registry"
    continue
  fi

  if [ ! -f "$LOCAL_PATH" ]; then
    echo "❌ ${file} — missing locally"
    MISMATCH=1
    continue
  fi

  LOCAL_HASH=$($SHA_CMD "$LOCAL_PATH" | awk '{print $1}')
  CHECKED=$((CHECKED + 1))

  if [ "$LOCAL_HASH" = "$REMOTE_HASH" ]; then
    echo "✅ ${file}"
  else
    echo "❌ ${file} — HASH MISMATCH"
    echo "   local:  ${LOCAL_HASH}"
    echo "   remote: ${REMOTE_HASH}"
    MISMATCH=1
  fi
done

echo ""
echo "Checked: ${CHECKED} files"

if [ "$MISMATCH" -eq 0 ]; then
  echo "✅ All files verified — integrity OK"
  exit 0
else
  echo "❌ Integrity check FAILED — files differ from official repo"
  exit 1
fi
```

---

### FIX-V5: credentials.json Permissions

**Problem:** `credentials.json` hat 644 statt 600. `register.sh` setzt `chmod 600` korrekt (Zeile 67). Die Datei wurde vermutlich manuell erstellt oder nach der Erstellung verändert.  
**Quellen:** ADVERSARIAL-TEST §4.1 (Severity: Medium)  
**Severity:** Medium

**Analyse:** `register.sh` Zeile 67: `chmod 600 "$CRED_FILE"` — **korrekt implementiert**. Kein Code-Fix nötig.

**Empfehlung:** Zusätzlichen Permissions-Check in `upload.sh` einbauen:

```bash
# After resolving CRED_FILE, before reading:
if [ -f "$CRED_FILE" ]; then
  PERMS=$(stat -c '%a' "$CRED_FILE" 2>/dev/null || stat -f '%Lp' "$CRED_FILE" 2>/dev/null)
  if [ "$PERMS" != "600" ]; then
    echo "⚠️  Warning: $CRED_FILE has permissions $PERMS (should be 600). Fixing..." >&2
    chmod 600 "$CRED_FILE"
  fi
fi
```

> Platzierung: In `upload.sh` nach Zeile 19 (`CRED_FILE=...`), vor dem API_KEY-Read.

---

## 2. upload.sh

### FIX-U1: Payload Size Check

**Problem:** Keine Größenbeschränkung. Ein 181KB Report mit 1000 Findings wurde akzeptiert. Kann für API-DoS missbraucht werden.  
**Quellen:** ADVERSARIAL-TEST §2.2 (Severity: Medium)  
**Severity:** Medium

**Fix — nach dem Einlesen von REPORT_JSON einfügen (nach Zeile 46):**

```bash
# Check payload size (max 500KB)
MAX_SIZE=512000
PAYLOAD_SIZE=${#REPORT_JSON}
if [ "$PAYLOAD_SIZE" -gt "$MAX_SIZE" ]; then
  echo "❌ Report too large: ${PAYLOAD_SIZE} bytes (max: ${MAX_SIZE} bytes / ~500KB)" >&2
  exit 1
fi
```

> Platzierung: Direkt nach `REPORT_JSON=$(cat "$INPUT")` bzw. `REPORT_JSON=$(cat)`, vor dem curl-Aufruf.

---

### FIX-U2: JSON Validierung vor Upload

**Problem:** Ungültiges JSON wird blind an die API gesendet. Verschwendet einen API-Call und liefert kryptische Fehlermeldungen.  
**Quellen:** DOCS-REVIEW §2 (fehlende Error-Handling-Doku)  
**Severity:** Low

**Fix — nach dem Einlesen von REPORT_JSON einfügen (nach FIX-U1):**

```bash
# Validate JSON syntax
if ! echo "$REPORT_JSON" | jq empty 2>/dev/null; then
  echo "❌ Invalid JSON in report file" >&2
  exit 1
fi

# Validate required fields
for field in skill_slug risk_score result findings_count; do
  if [ "$(echo "$REPORT_JSON" | jq -r "has(\"$field\")")" != "true" ]; then
    echo "❌ Missing required field: $field" >&2
    exit 1
  fi
done
```

> `jq` ist bereits eine deklarierte Dependency.

---

## 3. audit-prompt.md

### FIX-A1: JSON Format — Felder synchronisieren mit SKILL.md

**Problem:** Mehrere Inkonsistenzen zwischen audit-prompt.md und SKILL.md bei den Report-Feldern.  
**Quellen:** DOCS-REVIEW §3  
**Severity:** Medium

**Kanonische Definition (SKILL.md ist Source of Truth):**

| Feld | Typ | Pflicht | Beschreibung |
|------|-----|---------|--------------|
| `skill_slug` | string | ✅ | Package-Name (Alias: `package_name`) |
| `risk_score` | number (0-100) | ✅ | Top-level, NICHT in summary |
| `result` | string | ✅ | Eines von: `safe`, `caution`, `unsafe` |
| `findings_count` | number | ✅ | Anzahl Findings |
| `findings` | array | ✅ | Finding-Objekte |

**Änderung in audit-prompt.md — Step 4 JSON Template:** Bereits korrekt. Keine Änderung nötig am Template selbst.

**Änderung: `result`-Werte vereinheitlichen.**

Aktuell in SKILL.md:
```
"result": "safe|caution|unsafe|clean|pass|fail",
```

Aktuell in audit-prompt.md:
```
"result": "safe|caution|unsafe",
```

**Fix in SKILL.md** (audit-prompt.md ist korrekt, SKILL.md muss angepasst werden):
```
"result": "safe|caution|unsafe",
```

> `clean`, `pass`, `fail` werden entfernt. Drei Werte reichen und entsprechen den Risk Score Ranges.

---

### FIX-A2: Risk Score Ranges synchronisieren

**Problem:** SKILL.md und audit-prompt.md haben unterschiedliche Ranges.  
**Quellen:** DOCS-REVIEW §3

**SKILL.md Field Notes:**
```
risk_score: 0–25 safe, 26–50 caution, 51–100 unsafe
```

**audit-prompt.md Risk Score Guide:**
```
0-10: No issues found, code looks clean
11-25: Minor best-practice issues only → safe
26-50: Some medium-severity issues → caution
51-75: High-severity issues present → unsafe
76-100: Critical issues, likely malicious → unsafe
```

**Kanonische Version (audit-prompt.md behält die feinere Granularität, SKILL.md wird angepasst):**

| Range | result | Beschreibung |
|-------|--------|--------------|
| 0–25 | `safe` | Clean oder nur minor issues |
| 26–50 | `caution` | Medium-Severity Issues |
| 51–100 | `unsafe` | High/Critical Issues |

**Fix in audit-prompt.md — Risk Score Guide ersetzen:**

```markdown
### Risk Score Guide

| Score | `result` | Description |
|-------|----------|-------------|
| 0–25 | `safe` | No issues or minor best-practice issues only |
| 26–50 | `caution` | Medium-severity issues found |
| 51–100 | `unsafe` | High or critical severity issues present |
```

**Fix in SKILL.md — Field Notes:** Bereits konsistent mit obiger Tabelle. Kein Change nötig.

---

### FIX-A3: Pattern ID Liste synchronisieren

**Problem:** audit-prompt.md hat 15 Prefixe, SKILL.md hat nur 11. Fehlend in SKILL.md: `CRYPTO_WEAK`, `DESER`, `PATH_TRAV`, `SEC_BYPASS`.  
**Quellen:** DOCS-REVIEW §3

**Fix in SKILL.md — Pattern ID Prefixes Tabelle erweitern:**

Folgende Zeilen hinzufügen:

```markdown
| `CRYPTO_WEAK` | Weak cryptography |
| `DESER` | Unsafe deserialization |
| `PATH_TRAV` | Path traversal |
| `SEC_BYPASS` | Security bypass |
```

> audit-prompt.md ist die vollständige Liste und bleibt unverändert.

---

### FIX-A4: `result`-Werte explizit definieren

**Problem:** Die drei akzeptierten Werte sind nirgends klar als erschöpfende Liste gekennzeichnet.  
**Quellen:** DOCS-REVIEW §1

**Fix in audit-prompt.md — nach dem JSON Template, vor Pattern IDs:**

```markdown
### Accepted `result` Values

| Value | When to use |
|-------|-------------|
| `safe` | risk_score 0–25. No issues or only low-severity best-practice items. |
| `caution` | risk_score 26–50. Medium-severity issues that need attention. |
| `unsafe` | risk_score 51–100. High or critical issues. Should not be used without remediation. |

> These are the ONLY accepted values. Do not use `clean`, `pass`, `fail`, or any other string.
```

---

## 4. review-prompt.md

### FIX-R1: Finding ID Klarstellung — `ecap_id` verwenden

**Problem:** SKILL.md sagt explizit "use the **numeric `id`** field", aber die API akzeptiert NUR `ecap_id` Strings. Numerische IDs geben immer "Finding not found".  
**Quellen:** INTEGRATION-TEST §5, §6 (BUG-001, Severity: Critical)  
**Severity:** Critical

**Fix in review-prompt.md — Submit-Beispiel:**

Aktuell:
```bash
curl -s -X POST "https://skillaudit-api.vercel.app/api/findings/FINDING_ID/review" \
```

Fix:
```bash
# Use the ecap_id string (e.g., ECAP-2026-0777), NOT the numeric id
curl -s -X POST "https://skillaudit-api.vercel.app/api/findings/ECAP_ID/review" \
```

**Fix in SKILL.md — Finding IDs in API URLs:**

Aktuell:
```markdown
### Finding IDs in API URLs

When using `/api/findings/:id/review` or `/api/findings/:id/fix`, use the **numeric `id`** field from the findings response (not the `ecap_id` string like `ECAP-2026-0777`).
```

Fix:
```markdown
### Finding IDs in API URLs

When using `/api/findings/:id/review` or `/api/findings/:id/fix`, use the **`ecap_id` string** (e.g., `ECAP-2026-0777`) from the findings response — not the numeric `id` field.
```

---

### FIX-R2: Keine weiteren Inkonsistenzen

review-prompt.md ist sonst konsistent. Keine weiteren Fixes nötig.

---

## 5. register.sh

### FIX-REG1: chmod 600 — Korrekt implementiert

**Analyse:** `register.sh` Zeile 67: `chmod 600 "$CRED_FILE"` — **funktioniert korrekt**.

Die aktuelle 644-Permission auf der existierenden Datei ist ein Runtime-Problem (Datei wurde vermutlich manuell erstellt oder modifiziert), kein Code-Bug.

**Kein Code-Fix nötig.** Der Permissions-Check in upload.sh (FIX-V5) deckt das Laufzeit-Problem ab.

---

## Zusammenfassung

| Fix-ID | Datei | Severity | Art |
|--------|-------|----------|-----|
| FIX-V1 | verify.sh | Low | Kommentar-Fix |
| FIX-V2 | verify.sh | **High** | API URL hardcoden |
| FIX-V3 | verify.sh | Medium | URL-Encoding |
| FIX-V4 | verify.sh | **High** | Dynamische Dateiliste |
| FIX-V5 | upload.sh | Medium | Permissions-Check |
| FIX-U1 | upload.sh | Medium | Payload Size Limit |
| FIX-U2 | upload.sh | Low | JSON Validierung |
| FIX-A1 | audit-prompt.md + SKILL.md | Medium | result-Werte vereinheitlichen |
| FIX-A2 | audit-prompt.md + SKILL.md | Medium | Risk Score Ranges sync |
| FIX-A3 | SKILL.md | Low | Pattern IDs ergänzen |
| FIX-A4 | audit-prompt.md | Low | result-Werte dokumentieren |
| FIX-R1 | review-prompt.md + SKILL.md | **Critical** | ecap_id statt numeric id |
| FIX-REG1 | register.sh | — | Kein Fix nötig |

### Implementierungsreihenfolge (nach Impact)

1. **FIX-R1** — Critical: API-Aufrufe schlagen fehl mit falscher ID
2. **FIX-V2** — High: Security-Bypass durch custom API URL
3. **FIX-V4** — High: verify.sh funktioniert nur für eigenes Package
4. **FIX-V3** — Medium: URL-Injection
5. **FIX-V5** — Medium: Credentials-Permissions
6. **FIX-U1** — Medium: Payload Size
7. **FIX-A1/A2** — Medium: Konsistenz
8. Rest — Low
