#!/bin/bash
# should-dream.sh — Check if the agent should dream during this quiet hour
# Returns 0 (and prints a dream topic) if yes, 1 if no
# Uses data/dream-state.json to track nightly limits

set -e

# Find workspace root (where data/ and memory/ live)
SCRIPT_DIR="$(cd "$(dirname "$0")" && pwd)"
WORKSPACE="${WORKSPACE:-$(cd "$SCRIPT_DIR/.." && pwd)}"
cd "$WORKSPACE"

# ============================================================================
# CONFIGURATION — Edit these to customize
# ============================================================================

# Quiet hours window (24-hour format)
QUIET_START=23  # 11 PM
QUIET_END=7     # 7 AM

# Dream topics — format: "category:prompt"
# Override by creating data/dream-config.json with a "topics" array
# Example: {"topics": ["future:What could X become?", "creative:Wild ideas"]}
DEFAULT_TOPICS=(
    "future:What could this project become in 5 years if everything goes right?"
    "tangent:An interesting technology or concept worth exploring"
    "strategy:Long-term thinking about goals and direction"
    "creative:A wild idea that might be crazy or brilliant"
    "reflection:Looking back at recent work and what it means"
    "hypothetical:What-if scenarios worth considering"
    "connection:Unexpected links between different domains or concepts"
)

# Load custom topics from config if it exists
CONFIG_FILE="data/dream-config.json"
if [[ -f "$CONFIG_FILE" ]]; then
    CUSTOM_TOPICS=$(jq -r '.topics[]? // empty' "$CONFIG_FILE" 2>/dev/null)
    if [[ -n "$CUSTOM_TOPICS" ]]; then
        TOPICS=()
        while IFS= read -r line; do
            TOPICS+=("$line")
        done <<< "$CUSTOM_TOPICS"
    else
        TOPICS=("${DEFAULT_TOPICS[@]}")
    fi
else
    TOPICS=("${DEFAULT_TOPICS[@]}")
fi

# ============================================================================
# IMPLEMENTATION — Probably don't need to edit below here
# ============================================================================

STATE_FILE="data/dream-state.json"
CURRENT_DATE=$(date +%Y-%m-%d)
CURRENT_HOUR=$(date +%H)

# Check if we're in quiet hours
in_quiet_hours() {
    local hour=$((10#$CURRENT_HOUR))  # Force base-10
    if [[ $QUIET_START -gt $QUIET_END ]]; then
        # Window crosses midnight (e.g., 23:00 - 07:00)
        [[ $hour -ge $QUIET_START || $hour -lt $QUIET_END ]]
    else
        # Window within same day (e.g., 01:00 - 05:00)
        [[ $hour -ge $QUIET_START && $hour -lt $QUIET_END ]]
    fi
}

if ! in_quiet_hours; then
    exit 1
fi

# Ensure data directory exists
mkdir -p "$(dirname "$STATE_FILE")"

# Initialize state file if missing
if [[ ! -f "$STATE_FILE" ]]; then
    echo '{"lastDreamDate":null,"dreamsTonight":0,"maxDreamsPerNight":1,"dreamChance":1.0}' > "$STATE_FILE"
fi

# Read state (requires jq)
if ! command -v jq &>/dev/null; then
    echo "Error: jq required but not installed" >&2
    exit 1
fi

STATE=$(cat "$STATE_FILE")
LAST_DATE=$(echo "$STATE" | jq -r '.lastDreamDate // ""')
DREAMS_TONIGHT=$(echo "$STATE" | jq -r '.dreamsTonight // 0')
MAX_DREAMS=$(echo "$STATE" | jq -r '.maxDreamsPerNight // 1')
DREAM_CHANCE=$(echo "$STATE" | jq -r '.dreamChance // 1.0')

# Reset counter if new night
# "New night" = date changed AND we're in late-night hours (after QUIET_START)
if [[ "$LAST_DATE" != "$CURRENT_DATE" ]]; then
    hour=$((10#$CURRENT_HOUR))
    if [[ $hour -ge $QUIET_START || $hour -lt $QUIET_END ]]; then
        DREAMS_TONIGHT=0
    fi
fi

# Check if we've hit the nightly limit
if [[ "$DREAMS_TONIGHT" -ge "$MAX_DREAMS" ]]; then
    exit 1
fi

# Roll the dice
ROLL=$(python3 -c "import random; print(1 if random.random() < $DREAM_CHANCE else 0)" 2>/dev/null || echo "1")
if [[ "$ROLL" != "1" ]]; then
    exit 1
fi

# Pick a random topic
TOPIC=${TOPICS[$RANDOM % ${#TOPICS[@]}]}

# Update state
NEW_DREAMS=$((DREAMS_TONIGHT + 1))
echo "$STATE" | jq --arg date "$CURRENT_DATE" --argjson dreams "$NEW_DREAMS" \
    '.lastDreamDate = $date | .dreamsTonight = $dreams' > "$STATE_FILE"

# Output the topic
echo "$TOPIC"
exit 0
