/**
 * Doctor Strange — Snapshot Manager
 * Atomic writes, auto-cleanup, diff engine.
 * Zero dependencies.
 */

import { writeFileSync, readFileSync, mkdirSync, readdirSync, unlinkSync, renameSync } from 'node:fs';
import { join, dirname } from 'node:path';
import { fileURLToPath } from 'node:url';
import { randomBytes } from 'node:crypto';
import { get } from './config.js';

const __dirname = dirname(fileURLToPath(import.meta.url));
const DATA_DIR = join(__dirname, '..', 'data');
const SNAPSHOTS_DIR = join(DATA_DIR, 'snapshots');
const REPORTS_DIR = join(DATA_DIR, 'reports');

// ─── Atomic File Write ──────────────────────────────────

/**
 * Write JSON atomically: write to .tmp then rename.
 * Prevents corruption from concurrent writes or crashes.
 * @param {string} filepath
 * @param {any} data
 */
export function atomicWriteJson(filepath, data) {
  const dir = dirname(filepath);
  mkdirSync(dir, { recursive: true });

  const tmp = `${filepath}.${randomBytes(4).toString('hex')}.tmp`;
  writeFileSync(tmp, JSON.stringify(data, null, 2), 'utf-8');

  try {
    // On Windows, renameSync may fail if target is locked or exists
    // unlinkSync first to avoid EPERM on Windows NTFS
    try { unlinkSync(filepath); } catch { /* target may not exist */ }
    renameSync(tmp, filepath);
  } catch {
    // Final fallback: direct write (less atomic but works)
    writeFileSync(filepath, JSON.stringify(data, null, 2), 'utf-8');
    try { unlinkSync(tmp); } catch { /* ignore */ }
  }
}

/**
 * Safe JSON file read — returns fallback on any error.
 * @param {string} filepath
 * @param {any} [fallback=null]
 */
export function safeReadJson(filepath, fallback = null) {
  try {
    return JSON.parse(readFileSync(filepath, 'utf-8'));
  } catch {
    return fallback;
  }
}

// ─── Snapshot Operations ────────────────────────────────

/**
 * Save a scan snapshot + latest.json.
 * @param {object[]} markets - parsed market objects
 * @param {string} timestamp - ISO timestamp
 * @returns {string} snapshot filename
 */
export function saveSnapshot(markets, timestamp) {
  const snapshot = {
    timestamp,
    market_count: markets.length,
    markets,
  };

  // Save timestamped snapshot
  const filename = timestamp.replace(/:/g, '-').replace(/\.\d+Z$/, 'Z') + '.json';
  const snapPath = join(SNAPSHOTS_DIR, filename);
  atomicWriteJson(snapPath, snapshot);

  // Save latest.json
  const latestPath = join(DATA_DIR, 'latest.json');
  atomicWriteJson(latestPath, snapshot);

  return filename;
}

/**
 * Get the N most recent snapshot file paths.
 * @param {number} [n=2]
 * @returns {string[]}
 */
export function getLatestSnapshots(n = 2) {
  mkdirSync(SNAPSHOTS_DIR, { recursive: true });
  try {
    const files = readdirSync(SNAPSHOTS_DIR)
      .filter(f => f.endsWith('.json'))
      .sort()
      .reverse();
    return files.slice(0, n).map(f => join(SNAPSHOTS_DIR, f));
  } catch {
    return [];
  }
}

/**
 * Auto-cleanup old snapshots beyond retention period.
 */
export function cleanupSnapshots() {
  const retentionDays = get('snapshot_retention_days', 30);
  const cutoff = new Date();
  cutoff.setDate(cutoff.getDate() - retentionDays);
  const cutoffStr = cutoff.toISOString().slice(0, 10);

  mkdirSync(SNAPSHOTS_DIR, { recursive: true });
  const ISO_DATE_RE = /^\d{4}-\d{2}-\d{2}/;
  let removed = 0;
  try {
    for (const file of readdirSync(SNAPSHOTS_DIR)) {
      // Only process files matching ISO date pattern (YYYY-MM-DD...)
      if (!ISO_DATE_RE.test(file)) continue;
      const datePrefix = file.slice(0, 10);
      if (datePrefix < cutoffStr) {
        try {
          unlinkSync(join(SNAPSHOTS_DIR, file));
          removed++;
        } catch { /* ignore individual file errors */ }
      }
    }
  } catch { /* ignore */ }

  if (removed > 0) {
    process.stderr.write(`[snapshot] Cleaned up ${removed} old snapshot(s)\n`);
  }
}

// ─── Diff Engine ────────────────────────────────────────

/**
 * Compare two snapshots, find markets with significant probability changes.
 * @param {object} oldSnap - { markets: [...] }
 * @param {object} newSnap - { markets: [...] }
 * @param {number} [thresholdPct=5]
 * @returns {object[]}
 */
export function diffSnapshots(oldSnap, newSnap, thresholdPct = 5) {
  const oldIndex = new Map();
  for (const m of (oldSnap.markets || [])) {
    const key = m.id || m.slug;
    if (key) oldIndex.set(key, m);
  }

  const movers = [];
  for (const m of (newSnap.markets || [])) {
    const key = m.id || m.slug;
    if (!key || !oldIndex.has(key)) continue;

    const old = oldIndex.get(key);
    const change = m.probability_pct - old.probability_pct;

    if (Math.abs(change) >= thresholdPct) {
      movers.push({
        id: key,
        question: m.question || '',
        category: m.category || '',
        old_probability: old.probability_pct,
        new_probability: m.probability_pct,
        change_pct: Math.round(change * 10) / 10,
        volume_24h: m.volume_24h || 0,
        old_timestamp: oldSnap.timestamp || '',
        new_timestamp: newSnap.timestamp || '',
      });
    }
  }

  return movers.sort((a, b) => Math.abs(b.change_pct) - Math.abs(a.change_pct));
}

/**
 * Detect alerts by comparing last 2 snapshots.
 * @param {number} [thresholdPct]
 * @returns {object[]}
 */
export function detectAlerts(thresholdPct) {
  const threshold = thresholdPct || get('alert_threshold_pct', 15);
  const paths = getLatestSnapshots(2);
  if (paths.length < 2) return [];

  const newSnap = safeReadJson(paths[0]);
  const oldSnap = safeReadJson(paths[1]);
  if (!newSnap || !oldSnap) return [];

  return diffSnapshots(oldSnap, newSnap, threshold);
}

// ─── Report Meta ────────────────────────────────────────

/**
 * Save report metadata (timing, counts).
 * @param {object} meta
 * @param {string} [filename]
 */
export function saveReportMeta(meta, filename) {
  const fname = filename || new Date().toISOString().slice(0, 16).replace(/:/g, '-');
  const metaPath = join(REPORTS_DIR, `${fname}.meta.json`);
  atomicWriteJson(metaPath, {
    ...meta,
    saved_at: new Date().toISOString(),
  });
}
