/**
 * Doctor Strange — Gamma API Client
 * Zero-dependency Polymarket data fetcher.
 * Built-in: retry with exponential backoff, safe parsing, multi-outcome detection.
 */

import { get } from './config.js';

const GAMMA_API = 'https://gamma-api.polymarket.com';
const USER_AGENT = 'DoctorStrange/2.0 (OpenClaw Skill)';

// ─── Category Classification ────────────────────────────

const CATEGORY_KEYWORDS = {
  '体育': [
    'nba', 'nfl', 'mlb', 'soccer', 'football', 'basketball', 'baseball',
    'tennis', 'golf', 'olympics', 'world cup', 'championship', 'super bowl',
    'lakers', 'warriors', 'cavaliers', 'nets', 'nuggets', 'bucks', 'bulls',
    'pelicans', 'jazz', 'timberwolves', 'celtics', 'knicks', 'heat', '76ers',
    'arsenal', 'chelsea', 'liverpool', 'fulham', 'tottenham', 'manchester',
    'atletico', 'barcelona', 'real madrid', 'la liga', 'premier league',
    'lol:', 'esports', 't20', 'cricket', ' vs. ', ' vs ',
  ],
  '加密': [
    'bitcoin', 'btc', 'ethereum', 'eth', 'crypto', 'solana', 'sol', 'defi',
    'nft', 'token', 'blockchain', 'binance', 'coinbase',
  ],
  '科技': [
    'apple', 'google', 'microsoft', 'nvidia', 'openai', 'chatgpt', 'tesla',
    'spacex', 'meta', 'amazon', 'chip', 'semiconductor', 'tech', 'robot',
    'quantum', 'artificial intelligence',
  ],
  '宏观经济': [
    'fed', 'rate cut', 'inflation', 'gdp', 'recession', 'unemployment', 'jobs',
    'interest rate', 'treasury', 'debt', 'deficit', 'cpi', 'pce', 'fomc',
    'economic', 'trade', 'tariff',
  ],
  '政治': [
    'president', 'election', 'trump', 'biden', 'congress', 'senate',
    'supreme court', 'governor', 'political', 'impeach', 'vote', 'party',
    'democrat', 'republican', 'parliament', 'minister', 'nato', 'military',
    'iran', 'russia', 'ukraine', 'china', 'israel', 'gaza', 'ceasefire',
    'sanctions', 'khamenei',
  ],
};

// Keywords that need word-boundary matching to avoid false positives
const BOUNDARY_WORDS = new Set(['war', 'ai', 'fed']);

// Priority order: sports first to avoid "warriors" → "war" → 政治
const CATEGORY_ORDER = ['体育', '加密', '科技', '宏观经济', '政治'];

/**
 * Classify a market question into a category.
 * @param {string} question
 * @returns {string}
 */
export function categorize(question) {
  const q = question.toLowerCase();
  for (const cat of CATEGORY_ORDER) {
    for (const kw of CATEGORY_KEYWORDS[cat]) {
      if (BOUNDARY_WORDS.has(kw.trim())) {
        const re = new RegExp(`\\b${kw.trim()}\\b`, 'i');
        if (re.test(q)) return cat;
      } else if (q.includes(kw)) {
        return cat;
      }
    }
  }
  return '其他';
}

// ─── Safe Parsing Helpers ───────────────────────────────

/**
 * Safe float parser — never throws.
 * @param {any} val
 * @param {number} [fallback=0]
 * @returns {number}
 */
export function safeFloat(val, fallback = 0) {
  if (val == null) return fallback;
  const n = Number(val);
  return Number.isFinite(n) ? n : fallback;
}

/**
 * Safe JSON parse — never throws.
 * @param {any} val
 * @param {any} [fallback=[]]
 */
function safeJsonParse(val, fallback = []) {
  if (Array.isArray(val)) return val;
  if (typeof val !== 'string') return fallback;
  try {
    return JSON.parse(val);
  } catch {
    return fallback;
  }
}

// ─── API Fetching with Retry ────────────────────────────

/**
 * Fetch with retry + exponential backoff.
 * @param {string} url
 * @param {object} [options]
 * @param {number} [maxRetries=3]
 * @returns {Promise<any>}
 */
async function fetchWithRetry(url, options = {}, maxRetries = 3) {
  for (let attempt = 0; attempt < maxRetries; attempt++) {
    let timeout;
    try {
      const controller = new AbortController();
      timeout = setTimeout(() => controller.abort(), 30_000);

      const resp = await fetch(url, {
        ...options,
        signal: controller.signal,
        headers: { 'User-Agent': USER_AGENT, ...options.headers },
      });

      if (!resp.ok) {
        throw new Error(`HTTP ${resp.status}: ${resp.statusText}`);
      }
      return await resp.json();
    } catch (err) {
      const isLast = attempt === maxRetries - 1;
      if (isLast) throw err;

      const delay = Math.pow(2, attempt) * 1000; // 1s, 2s, 4s
      process.stderr.write(
        `[gamma] Retry ${attempt + 1}/${maxRetries} after ${delay}ms: ${err.message}\n`
      );
      await new Promise(r => setTimeout(r, delay));
    } finally {
      clearTimeout(timeout);
    }
  }
}

/**
 * Fetch markets from Gamma API.
 * @param {object} opts
 * @param {number} [opts.limit=100]
 * @param {string} [opts.order='volume24hr']
 * @param {boolean} [opts.active=true]
 * @returns {Promise<object[]>}
 */
export async function fetchMarkets({ limit = 100, order = 'volume24hr', active = true } = {}) {
  const cap = get('scan_limit_cap', 500);
  const effectiveLimit = Math.min(limit, cap);

  const params = new URLSearchParams({
    limit: String(effectiveLimit),
    order,
    ascending: 'false',
    active: String(active),
  });

  const data = await fetchWithRetry(`${GAMMA_API}/markets?${params}`);

  if (Array.isArray(data)) return data;
  if (data && typeof data === 'object') return data.value ?? data.data ?? [];
  return [];
}

/**
 * Fetch events from Gamma API.
 */
export async function fetchEvents({ limit = 50 } = {}) {
  const params = new URLSearchParams({
    limit: String(limit),
    order: 'volume24hr',
    ascending: 'false',
    active: 'true',
  });

  const data = await fetchWithRetry(`${GAMMA_API}/events?${params}`);

  if (Array.isArray(data)) return data;
  if (data && typeof data === 'object') return data.value ?? data.data ?? [];
  return [];
}

// ─── Market Parsing ─────────────────────────────────────

/**
 * Parse a raw Gamma API market object into our clean schema.
 * Handles binary AND multi-outcome markets safely.
 * @param {object} raw
 * @returns {object}
 */
export function parseMarket(raw) {
  const prices = safeJsonParse(raw.outcomePrices);
  const outcomes = safeJsonParse(raw.outcomes);
  const isBinary = prices.length === 2;

  const yesPrice = isBinary ? safeFloat(prices[0]) : 0;
  const noPrice = isBinary ? safeFloat(prices[1]) : 0;

  // For multi-outcome: find the leading outcome
  let leadingOutcome = null;
  if (!isBinary && prices.length > 0) {
    let maxIdx = 0;
    let maxPrice = 0;
    for (let i = 0; i < prices.length; i++) {
      const p = safeFloat(prices[i]);
      if (p > maxPrice) { maxPrice = p; maxIdx = i; }
    }
    leadingOutcome = {
      name: outcomes[maxIdx] || `Outcome ${maxIdx}`,
      price: Math.round(maxPrice * 10000) / 10000,
      probability_pct: Math.round(maxPrice * 1000) / 10,
    };
  }

  const question = raw.question || '';
  const change1d = safeFloat(raw.oneDayPriceChange);
  const change1w = safeFloat(raw.oneWeekPriceChange);
  const change1m = safeFloat(raw.oneMonthPriceChange);

  return {
    id: raw.id ?? '',
    question,
    slug: raw.slug ?? '',
    category: categorize(question),
    market_type: isBinary ? 'binary' : 'multi',
    outcome_count: prices.length,
    yes_price: Math.round(yesPrice * 10000) / 10000,
    no_price: Math.round(noPrice * 10000) / 10000,
    probability_pct: isBinary
      ? Math.round(yesPrice * 1000) / 10
      : (leadingOutcome ? leadingOutcome.probability_pct : 0),
    leading_outcome: leadingOutcome,
    volume_24h: Math.round(safeFloat(raw.volume24hr)),
    volume_total: Math.round(safeFloat(raw.volumeNum)),
    liquidity: Math.round(safeFloat(raw.liquidityNum)),
    change_1d: Math.round(change1d * 10000) / 10000,
    change_1d_pct: Math.round(change1d * 1000) / 10,
    change_1w: Math.round(change1w * 10000) / 10000,
    change_1w_pct: Math.round(change1w * 1000) / 10,
    change_1m: Math.round(change1m * 10000) / 10000,
    end_date: (raw.endDateIso || '').slice(0, 10),
    last_trade: safeFloat(raw.lastTradePrice),
  };
}

// ─── Settlement Detection ───────────────────────────────

/**
 * Determine if a market is settled (should be filtered out).
 * Three conditions (any = settled):
 *   1. Probability 100%/0% + liquidity = 0
 *   2. Probability extreme (>=99.5% or <=0.5%) + end_date past
 *   3. Probability extreme + liquidity = 0
 * @param {object} m - parsed market
 * @returns {boolean}
 */
export function isSettled(m) {
  const prob = m.probability_pct;
  const liq = m.liquidity;
  const end = m.end_date;
  const today = new Date().toISOString().slice(0, 10);

  // Fully settled: 100%/0% with no liquidity
  if ((prob >= 99.95 || prob <= 0.05) && liq === 0) return true;

  // Extreme probability + expired
  if ((prob >= 99.5 || prob <= 0.5) && end && end <= today) return true;

  // Extreme probability + no liquidity
  if ((prob >= 99.5 || prob <= 0.5) && liq === 0) return true;

  return false;
}

// ─── High-Level Operations ──────────────────────────────

/**
 * Execute a scan: fetch, parse, filter, classify.
 * @param {object} opts
 * @param {number} [opts.limit] - markets to scan (from config if not provided)
 * @param {string} [opts.keyword] - optional keyword filter (e.g. "iran")
 * @param {boolean} [opts.filterSettled=true]
 * @returns {Promise<object>}
 */
export async function scan({ limit, keyword, filterSettled } = {}) {
  if (filterSettled === undefined) filterSettled = get('filter_settled', true);
  const scanLimit = limit || get('scan_limit', 150);
  const cap = get('scan_limit_cap', 500);
  const fetchLimit = Math.min(keyword ? 500 : scanLimit * 3, cap);

  const rawMarkets = await fetchMarkets({ limit: fetchLimit });
  let markets = rawMarkets.map(parseMarket);

  // Keyword filter — narrows to matching markets before all other logic
  if (keyword) {
    const kw = keyword.toLowerCase();
    markets = markets.filter(m => m.question.toLowerCase().includes(kw));
  }

  if (filterSettled) {
    markets = markets.filter(m => !isSettled(m));
  }
  markets = markets.slice(0, scanLimit);

  const now = new Date();
  const timestamp = now.toISOString();
  const trendingCount = get('trending_count', 15);
  const moversThreshold = get('movers_threshold', 5);
  const certHigh = get('certainty_high', 85);
  const certLow = get('certainty_low', 15);

  return {
    timestamp,
    total_markets: markets.length,
    trending: markets.slice(0, trendingCount),
    movers: markets
      .filter(m => Math.abs(m.change_1d_pct) >= moversThreshold)
      .sort((a, b) => Math.abs(b.change_1d_pct) - Math.abs(a.change_1d_pct)),
    certainties: markets.filter(
      m => m.probability_pct >= certHigh || m.probability_pct <= certLow
    ),
    _allMarkets: markets, // for snapshot saving
  };
}

/**
 * Search markets by keyword (local filter on top N by volume).
 * @param {string} keyword
 * @param {object} opts
 * @param {number} [opts.limit]
 * @returns {Promise<object[]>}
 */
export async function search(keyword, { limit } = {}) {
  const searchLimit = limit || get('search_limit', 20);
  const fetchLimit = get('search_fetch_limit', 200);

  const rawMarkets = await fetchMarkets({ limit: fetchLimit });
  const kw = keyword.toLowerCase();

  return rawMarkets
    .filter(m => (m.question || '').toLowerCase().includes(kw))
    .slice(0, searchLimit)
    .map(parseMarket);
}
