/**
 * Doctor Strange — Report Formatter
 * Terminal UI output: tree diagrams + categorized listings.
 * Supports markdown (Slack/messaging) and ANSI terminal modes.
 * Zero dependencies.
 */

import { get } from './config.js';

// ─── Formatting Helpers ─────────────────────────────────

function fmtVolume(v) {
  if (v >= 1_000_000) return `$${(v / 1_000_000).toFixed(2)}M`;
  if (v >= 1_000) return `$${(v / 1_000).toFixed(1)}K`;
  return `$${Math.round(v)}`;
}

function fmtProb(pct) {
  if (pct >= 99.5) return '99.9%';
  if (pct <= 0.5) return '0.1%';
  return `${pct.toFixed(1)}%`;
}

function fmtChange(pct) {
  if (pct > 0) return `↑+${pct.toFixed(1)}%`;
  if (pct < 0) return `↓${pct.toFixed(1)}%`;
  return '→ 0%';
}

function changeEmoji(pct) {
  const abs = Math.abs(pct);
  if (abs >= 15) return '🔥';
  if (abs >= 10) return '🚀';
  if (abs >= 5) return pct > 0 ? '📈' : '📉';
  return '•';
}

const CATEGORY_EMOJI = {
  '政治': '🏛️', '宏观经济': '💰', '科技': '🤖',
  '加密': '₿', '体育': '⚽', '其他': '📌',
};

function catEmoji(cat) { return CATEGORY_EMOJI[cat] || '📌'; }

function truncate(text, max = 60) {
  return text.length <= max ? text : text.slice(0, max - 3) + '...';
}

// ─── ANSI helpers (terminal mode only) ──────────────────

const ANSI = {
  bold: (s) => `\x1b[1m${s}\x1b[22m`,
  dim: (s) => `\x1b[2m${s}\x1b[22m`,
  red: (s) => `\x1b[31m${s}\x1b[39m`,
  green: (s) => `\x1b[32m${s}\x1b[39m`,
  yellow: (s) => `\x1b[33m${s}\x1b[39m`,
  cyan: (s) => `\x1b[36m${s}\x1b[39m`,
};

function bold(text, mode) {
  return mode === 'terminal' ? ANSI.bold(text) : `*${text}*`;
}

function italic(text, mode) {
  return mode === 'terminal' ? ANSI.dim(text) : `_${text}_`;
}

// ─── Board 1: Trending ──────────────────────────────────

function formatTrending(markets, mode) {
  if (!markets || markets.length === 0) return italic('(无热搜数据)', mode);

  // Group by category
  const byCategory = {};
  for (const m of markets) {
    const cat = m.category || '其他';
    (byCategory[cat] ||= []).push(m);
  }

  // Sort categories by max volume
  const catOrder = Object.keys(byCategory).sort(
    (a, b) => Math.max(...byCategory[b].map(m => m.volume_24h ?? 0))
              - Math.max(...byCategory[a].map(m => m.volume_24h ?? 0))
  );

  const lines = [`🔥 ${bold('板块一：热搜事件 (Trending)', mode)}`, ''];
  let idx = 1;

  for (const cat of catOrder) {
    lines.push(`${bold(`【${catEmoji(cat)} ${cat}】`, mode)}`);
    for (const m of byCategory[cat]) {
      const q = truncate(m.question, 55);
      const chg = (m.change_1d_pct != null && Math.abs(m.change_1d_pct) >= 2)
        ? ` | 24h ${fmtChange(m.change_1d_pct)}`
        : '';
      const multi = m.market_type === 'multi' && m.leading_outcome
        ? ` [${m.outcome_count}选项: ${m.leading_outcome.name} ${fmtProb(m.leading_outcome.probability_pct)}]`
        : '';
      lines.push(`  ${String(idx).padStart(2)}. ${q}`);
      lines.push(`      YES ${fmtProb(m.probability_pct)} | Vol ${fmtVolume(m.volume_24h)}${chg}${multi}`);
      idx++;
    }
    lines.push('');
  }

  return lines.join('\n');
}

// ─── Board 2: Movers ────────────────────────────────────

function formatMovers(markets, mode) {
  if (!markets || markets.length === 0) {
    return `🚀 ${bold('板块二：概率飙升 (Movers)', mode)}\n\n  ${italic('(过去24h无显著概率变动)', mode)}`;
  }

  const lines = [`🚀 ${bold('板块二：概率飙升 (Movers)', mode)}`, ''];

  for (let i = 0; i < markets.length; i++) {
    const m = markets[i];
    const emoji = changeEmoji(m.change_1d_pct);
    const q = truncate(m.question, 55);

    lines.push(`  ${emoji} ${i + 1}. ${bold(q, mode)}`);
    lines.push(`     ├─ YES ${fmtProb(m.probability_pct)} | 24h ${fmtChange(m.change_1d_pct)} | 7d ${fmtChange(m.change_1w_pct)}`);
    lines.push(`     ├─ Vol ${fmtVolume(m.volume_24h)} | ${catEmoji(m.category)} ${m.category}`);
    lines.push(`     └─ 驱动因素: ${italic('待 Analyst 分析', mode)}`);
    lines.push('');
  }

  return lines.join('\n');
}

// ─── Board 3: Certainties ───────────────────────────────

function formatCertainties(markets, mode) {
  // Filter out sports + fully settled (0 liquidity + 100%/0%)
  const filtered = (markets || []).filter(m =>
    m.category !== '体育' &&
    m.liquidity > 0 &&
    m.probability_pct !== 100.0 &&
    m.probability_pct !== 0.0
  );

  if (filtered.length === 0) {
    return `🎯 ${bold('板块三：不确定中的确定性 (Strange Certainties)', mode)}\n\n  ${italic('(无有效确定性候选)', mode)}`;
  }

  const certHigh = get('certainty_high', 85);
  const certLow = get('certainty_low', 15);
  const high = filtered.filter(m => m.probability_pct >= certHigh)
    .sort((a, b) => b.probability_pct - a.probability_pct);
  const low = filtered.filter(m => m.probability_pct <= certLow)
    .sort((a, b) => a.probability_pct - b.probability_pct);

  const lines = [`🎯 ${bold('板块三：不确定中的确定性 (Strange Certainties)', mode)}`, ''];

  if (high.length > 0) {
    lines.push(bold('🔒 高确定性 (YES >' + certHigh + '%)', mode));
    for (const m of high) {
      const chg1m = m.change_1m
        ? ` | 30d ${fmtChange(Math.round(m.change_1m * 1000) / 10)}`
        : '';
      lines.push(`  • ${catEmoji(m.category)} ${truncate(m.question, 55)}`);
      lines.push(`    YES ${fmtProb(m.probability_pct)} | Vol ${fmtVolume(m.volume_24h)}${chg1m}`);
      lines.push(`    ${italic('→ 待 Analyst+Challenger 对抗验证', mode)}`);
      lines.push('');
    }
  }

  if (low.length > 0) {
    lines.push(bold('🔓 低概率 (YES <' + certLow + '%)', mode));
    for (const m of low) {
      lines.push(`  • ${catEmoji(m.category)} ${truncate(m.question, 55)}`);
      lines.push(`    YES ${fmtProb(m.probability_pct)} | Vol ${fmtVolume(m.volume_24h)}`);
      lines.push(`    ${italic('→ 待验证', mode)}`);
      lines.push('');
    }
  }

  return lines.join('\n');
}

// ─── Board 3 Enhanced: Validated Certainties ─────────────

/**
 * Format Board 3 with Analyst reasoning + Challenger verdicts.
 * @param {object[]} certainties - merged certainties from mergeResults()
 * @param {string} [mode='markdown']
 * @returns {string}
 */
export function formatBoard3Enhanced(certainties, mode = 'markdown') {
  if (!certainties || certainties.length === 0) {
    return `🎯 ${bold('板块三：不确定中的确定性 (Strange Certainties)', mode)}\n\n  ${italic('(无经验证的确定性事件)', mode)}`;
  }

  const lines = [
    `🎯 ${bold('板块三：不确定中的确定性 (Strange Certainties)', mode)}`,
    '',
    bold('🔒 经 Analyst+Challenger 验证的确定性事件:', mode),
    '',
  ];

  for (let i = 0; i < certainties.length; i++) {
    const c = certainties[i];
    const stars = '⭐'.repeat(Math.min(c.strange_rating || 0, 5));
    const pct = Math.round((c.probability ?? 0) * 100);
    const verdict = c.verdict || 'PENDING';
    const verdictLabel = verdict === 'CONFIRM' ? 'CONFIRM' :
                         verdict === 'DOWNGRADE' ? 'DOWNGRADE' :
                         verdict === 'REJECT' ? 'REJECT' : 'PENDING';

    lines.push(`  ${i + 1}. ${stars} ${bold(`"${c.event}"`, mode)} — YES ${pct}%`);

    // Analyst reasoning
    if (c.evidence && c.evidence.length > 0) {
      lines.push(`     ├─ Analyst: ${c.evidence[0]}`);
    }

    // Causal chain (from evidence items after the first)
    if (c.evidence && c.evidence.length > 1) {
      lines.push(`     ├─ 因果链: ${c.evidence.slice(1).join(' → ')}`);
    }

    // Challenger counterexample
    if (c.counterexample) {
      const cred = c.counter_credibility ?? 0;
      lines.push(`     ├─ Challenger: 反例 — ${c.counterexample} (可信度: ${cred}%)`);
    }

    // Verdict
    lines.push(`     ├─ 裁决: ${verdictLabel}${c.verdict_reason ? ` (${c.verdict_reason})` : ''}`);

    // Hidden assumptions
    if (c.hidden_assumptions && c.hidden_assumptions.length > 0) {
      lines.push(`     ├─ 隐含假设: ${c.hidden_assumptions.join('; ')}`);
    }

    // Market impact hint
    if (c.market_impact?.equity || c.market_impact?.bond || c.market_impact?.forex) {
      const hint = c.market_impact.equity || c.market_impact.bond || c.market_impact.forex;
      lines.push(`     └─ 对冲视角: ${hint}`);
    } else {
      lines.push(`     └─ 对冲视角: ${italic('(待补充)', mode)}`);
    }

    lines.push('');
  }

  return lines.join('\n');
}

// ─── Board 4: Three-Level Market Impact ──────────────────

/**
 * Format Board 4: equity/bond/forex market impact.
 * @param {object} marketImpact - from mergeResults().market_impact
 * @param {string} [mode='markdown']
 * @returns {string}
 */
export function formatBoard4(marketImpact, mode = 'markdown') {
  if (!marketImpact) {
    return `💹 ${bold('板块四：三级市场影响', mode)}\n\n  ${italic('(无市场影响数据)', mode)}`;
  }

  const lines = [
    `💹 ${bold('板块四：三级市场影响', mode)}`,
    '',
  ];

  // Equity / Futures
  lines.push(bold('【期货 & 股市】', mode));
  if (marketImpact.equity_futures) {
    for (const line of marketImpact.equity_futures.split('\n')) {
      lines.push(`  • ${line.trim()}`);
    }
  } else {
    lines.push(`  • ${italic('(无数据)', mode)}`);
  }

  // Investment signals
  if (marketImpact.investment_signals && marketImpact.investment_signals.length > 0) {
    for (const sig of marketImpact.investment_signals) {
      lines.push(`  • 投资热点: ${sig}`);
    }
  }

  // Risk warnings for equity
  const equityRisks = (marketImpact.risk_warnings || []).filter(r =>
    r.includes('股') || r.includes('估值') || r.includes('equity')
  );
  for (const r of equityRisks) {
    lines.push(`  • ⚠️ 风险: ${r}`);
  }
  lines.push('');

  // Bonds
  lines.push(bold('【债市】', mode));
  if (marketImpact.bonds) {
    for (const line of marketImpact.bonds.split('\n')) {
      lines.push(`  • ${line.trim()}`);
    }
  } else {
    lines.push(`  • ${italic('(无数据)', mode)}`);
  }
  lines.push('');

  // Forex
  lines.push(bold('【汇市】', mode));
  if (marketImpact.forex) {
    for (const line of marketImpact.forex.split('\n')) {
      lines.push(`  • ${line.trim()}`);
    }
  } else {
    lines.push(`  • ${italic('(无数据)', mode)}`);
  }

  // Remaining risk warnings
  const otherRisks = (marketImpact.risk_warnings || []).filter(r =>
    !r.includes('股') && !r.includes('估值') && !r.includes('equity')
  );
  if (otherRisks.length > 0) {
    lines.push('');
    for (const r of otherRisks) {
      lines.push(`  • ⚠️ 风险: ${r}`);
    }
  }

  return lines.join('\n');
}

// ─── Board 5: Watchlist Overlay ──────────────────────────

/**
 * Format Board 5: per-stock watchlist analysis.
 * @param {object[]} watchlistAnalysis - from mergeResults().watchlist
 * @param {string} [mode='markdown']
 * @returns {string}
 */
export function formatBoard5(watchlistAnalysis, mode = 'markdown') {
  if (!watchlistAnalysis || watchlistAnalysis.length === 0) {
    return `📈 ${bold('板块五：自选股 Doctor Strange 分析', mode)}\n\n  ${italic('(无自选股配置)', mode)}`;
  }

  const lines = [
    `📈 ${bold('板块五：自选股 Doctor Strange 分析', mode)}`,
    '',
  ];

  for (const stock of watchlistAnalysis) {
    const sym = stock.symbol || '???';
    const mkt = stock.market || '??';
    const alias = stock.alias || sym;

    lines.push(`${bold(`*${sym}*`, mode)} [${mkt}] — ${alias}`);

    if (stock.macro_context) {
      lines.push(`  • 宏观背景: ${stock.macro_context}`);
    }

    if (stock.sensitivities && stock.sensitivities.length > 0) {
      lines.push(`  • 事件敏感度: ${stock.sensitivities.join(', ')}`);
    }

    const signal = stock.signal || '🟡';
    const judgment = stock.judgment || '待分析';
    lines.push(`  • 综合判断: ${signal} ${judgment}`);

    if (stock.risk) {
      lines.push(`  • ⚠️ 核心风险: ${stock.risk}`);
    }

    lines.push('');
  }

  return lines.join('\n');
}

// ─── Full Reports ───────────────────────────────────────

/**
 * Format a complete scan report.
 * @param {object} scanData - output from gamma.scan()
 * @param {string} [mode='markdown'] - 'markdown' or 'terminal'
 * @returns {string}
 */
export function formatScanReport(scanData, mode = 'markdown') {
  const ts = scanData.timestamp || '';
  let timeStr;
  try {
    timeStr = new Date(ts).toISOString().replace('T', ' ').slice(0, 16) + ' UTC';
  } catch {
    timeStr = ts;
  }

  const divider = '─'.repeat(40);
  const sections = [
    '═'.repeat(50),
    `🔮 ${bold('Doctor Strange — 全球事件概率扫描', mode)}`,
    `📅 ${timeStr} | 扫描 ${scanData.total_markets} 个活跃市场`,
    '═'.repeat(50),
    '',
    formatTrending(scanData.trending, mode),
    divider,
    '',
    formatMovers(scanData.movers, mode),
    divider,
    '',
    formatCertainties(scanData.certainties, mode),
    '',
    divider,
    `📊 统计: ${scanData.trending?.length || 0} 热搜 | ${scanData.movers?.length || 0} 飙升 | ${scanData.certainties?.length || 0} 确定性候选`,
    italic('Phase 2 将增加: Analyst 因果链 + Challenger 对抗验证 + 市场影响 + 自选股叠加', mode),
  ];

  return sections.join('\n');
}

/**
 * Format search results.
 * @param {object[]} markets
 * @param {string} keyword
 * @param {string} [mode='markdown']
 * @returns {string}
 */
// ─── Board 3 Enhanced: Analyst + Challenger Verified ───────

function formatCertaintiesEnhanced(analysis, mode) {
  const certainties = analysis?.certainties || [];
  
  if (certainties.length === 0) {
    return `🎯 ${bold('板块三：不确定中的确定性 (Strange Certainties)', mode)}\n\n  ${italic('(无确定性事件)', mode)}`;
  }

  const lines = [
    `🎯 ${bold('板块三：不确定中的确定性 (Strange Certainties)', mode)}`,
    '',
    bold('🔒 经 Analyst+Challenger 验证的确定性事件:', mode),
    '',
  ];

  for (let i = 0; i < certainties.length; i++) {
    const c = certainties[i];
    const stars = '⭐'.repeat(c.strange_rating || c.strangeRating || 3);
    const q = truncate(c.event || c.question || '', 55);
    const prob = c.probability != null ? (c.probability * 100).toFixed(0) : c.probability_pct || '?';
    const verdict = c.challenger_verdict || c.challengerVerdict || 'PENDING';
    const verdictEmoji = verdict === 'CONFIRM' ? '✅' : verdict === 'DOWNGRADE' ? '⚠️' : '❓';
    
    lines.push(`  ${i + 1}. ${stars} "${q}" — YES ${prob}%`);
    
    if (c.evidence && c.evidence.length > 0) {
      lines.push(`     ├─ Analyst: ${c.evidence[0]?.slice(0, 80) || ''}`);
    }
    
    if (c.counterexample || c.counterExample) {
      const cred = c.challenger_credibility || c.challengerCredibility || '?';
      lines.push(`     ├─ Challenger: 反例 (可信度: ${cred}%)`);
    }
    
    lines.push(`     └─ 裁决: ${verdictEmoji} ${verdict}`);
    lines.push('');
  }

  return lines.join('\n');
}

// ─── Board 4: Market Impact ────────────────────────────────

function formatMarketImpact(analysis, mode) {
  const impact = analysis?.market_impact || {};
  const signals = analysis?.investment_signals || [];
  const risks = analysis?.risk_warnings || [];

  const lines = [
    `💹 ${bold('板块四：三级市场影响 (Market Impact)', mode)}`,
    '',
  ];

  // Equity/Futures
  lines.push(bold('【期货 & 股市】', mode));
  if (impact.equity_futures || impact.equity) {
    lines.push(`  • ${impact.equity_futures || impact.equity}`);
  } else {
    lines.push(`  • ${italic('待分析', mode)}`);
  }
  
  if (signals.length > 0) {
    lines.push(`  • 投资热点: ${signals.slice(0, 3).map(s => s.title || s.name || s).join(', ')}`);
  }
  
  if (risks.length > 0) {
    lines.push(`  • ⚠️ ${italic('风险: ' + risks.slice(0, 2).map(r => r.title || r).join(', '))}`);
  }
  lines.push('');

  // Bonds
  lines.push(bold('【债市】', mode));
  if (impact.bonds) {
    lines.push(`  • ${impact.bonds}`);
  } else {
    lines.push(`  • ${italic('待分析', mode)}`);
  }
  lines.push('');

  // Forex
  lines.push(bold('【汇市】', mode));
  if (impact.forex) {
    lines.push(`  • ${impact.forex}`);
  } else {
    lines.push(`  • ${italic('待分析', mode)}`);
  }
  lines.push('');

  return lines.join('\n');
}

// ─── Board 5: Watchlist Overlay ────────────────────────────

function formatWatchlistOverlay(analysis, mode) {
  const stocks = analysis?.watchlist_analysis || analysis?.stocks || [];

  if (stocks.length === 0) {
    return `📈 ${bold('板块五：自选股 Doctor Strange 分析', mode)}\n\n  ${italic('(无自选股数据)', mode)}`;
  }

  const lines = [
    `📈 ${bold('板块五：自选股 Doctor Strange 分析', mode)}`,
    '',
  ];

  for (const stock of stocks) {
    const symbol = stock.symbol || stock.code || '?';
    const market = stock.market || '';
    const name = stock.name || stock.alias || '';
    const judgment = stock.judgment || stock综合判断 || '🟡';
    const macro = stock.macro_background || stock.宏观背景 || '';
    const sensitivity = stock.sensitivity || stock.事件敏感度 || '';
    const risk = stock.risk || stock.核心风险 || '';
    
    lines.push(`*${symbol}* [${market}] ${name ? `— ${name}` : ''}`);
    if (macro) lines.push(`  • 宏观背景: ${macro}`);
    if (sensitivity) lines.push(`  • 事件敏感度: ${sensitivity}`);
    lines.push(`  • 综合判断: ${judgment}`);
    if (risk) lines.push(`  • ⚠️ 核心风险: ${risk}`);
    lines.push('');
  }

  return lines.join('\n');
}

// ─── Full 5-Board Report ──────────────────────────────────

/**
 * Format a complete Phase 2 report with all 5 boards.
 * @param {object} scanData - output from gamma.scan()
 * @param {object} analysisResult - output from analyzer.mergeResults()
 * @param {string} [mode='markdown'] - 'markdown' or 'terminal'
 * @returns {string}
 */
export function formatFullReport(scanData, analysisResult, mode = 'markdown') {
  const ts = scanData.timestamp || '';
  let timeStr;
  try {
    timeStr = new Date(ts).toISOString().replace('T', ' ').slice(0, 16) + ' UTC';
  } catch {
    timeStr = ts;
  }

  const divider = '─'.repeat(40);
  const sections = [
    '═'.repeat(50),
    `🔮 ${bold('Doctor Strange — 全球事件概率情报', mode)}`,
    `📅 ${timeStr} | 扫描 ${scanData.total_markets} 个活跃市场`,
    '═'.repeat(50),
    '',
    formatTrending(scanData.trending, mode),
    divider,
    '',
    formatMovers(scanData.movers, mode),
    divider,
    '',
    formatCertaintiesEnhanced(analysisResult, mode),
    divider,
    '',
    formatMarketImpact(analysisResult, mode),
    divider,
    '',
    formatWatchlistOverlay(analysisResult, mode),
    '',
    divider,
    `📊 统计: ${scanData.trending?.length || 0} 热搜 | ${scanData.movers?.length || 0} 飙升 | ${analysisResult.certainties?.length || 0} 确定性验证`,
    `⏱️ 分析生成: ${analysisResult.analysis_timestamp ? new Date(analysisResult.analysis_timestamp).toLocaleTimeString() : 'N/A'}`,
  ];

  return sections.join('\n');
}

export function formatSearchResults(markets, keyword, mode = 'markdown') {
  if (!markets || markets.length === 0) {
    return `🔍 搜索 ${bold(`"${keyword}"`, mode)} — 无匹配市场`;
  }

  const lines = [
    `🔍 ${bold(`Doctor Strange — 市场搜索: "${keyword}"`, mode)}`,
    `   找到 ${markets.length} 个匹配市场 ${italic('(仅搜索高成交量市场)', mode)}`,
    '',
  ];

  for (let i = 0; i < markets.length; i++) {
    const m = markets[i];
    const q = truncate(m.question, 60);
    const parts = [`YES ${fmtProb(m.probability_pct)}`, `Vol ${fmtVolume(m.volume_24h)}`];
    if (m.change_1d_pct != null && m.change_1d_pct !== 0) parts.push(`24h ${fmtChange(m.change_1d_pct)}`);
    if (m.end_date) parts.push(`截止 ${m.end_date}`);

    lines.push(`  ${i + 1}. ${catEmoji(m.category)} ${bold(q, mode)}`);
    lines.push(`     ${parts.join(' | ')}`);
    lines.push('');
  }

  return lines.join('\n');
}
