/**
 * Doctor Strange — Config Loader
 * Zero-dependency config management with typed defaults.
 * Single source of truth for all thresholds and parameters.
 */

import { readFileSync } from 'node:fs';
import { join, dirname } from 'node:path';
import { fileURLToPath } from 'node:url';

const __dirname = dirname(fileURLToPath(import.meta.url));
const CONFIG_PATH = join(__dirname, '..', 'config.json');

/** @type {Record<string, any>} */
let _cache = null;

const DEFAULTS = {
  // Thresholds
  movers_threshold: 5,        // 24h change % to qualify as "mover"
  certainty_high: 85,         // probability % above which = high certainty
  certainty_low: 15,          // probability % below which = low certainty
  alert_threshold_pct: 15,    // change % to trigger alert

  // Limits
  scan_limit: 150,            // default markets to scan
  scan_limit_cap: 500,        // hard max regardless of multiplier
  trending_count: 15,         // top N for trending board
  search_limit: 20,           // default search results
  search_fetch_limit: 200,    // markets to pull for local search

  // Data management
  snapshot_retention_days: 30, // auto-cleanup older snapshots
  diff_source: 'api',         // 'api' or 'local'
  filter_settled: true,
  filter_sports: false,

  // Schedule
  cron_interval_hours: 2,
  report_time: '08:30',
  us_supplement_time: '21:00',

  // Models (for Phase 2 LLM orchestration)
  models: {
    scan: 'kimi-coding/k2p5',
    screener: 'kimi-coding/k2p5',
    analyst: 'anthropic/claude-opus-4-6',
    challenger: 'anthropic/claude-opus-4-6',
    watchlist_overlay: 'anthropic/claude-opus-4-6',
    market_search: 'kimi-coding/k2p5',
  },

  // Watchlist
  watchlist: [],
};

/**
 * Load config.json, merge with defaults. Cached after first call.
 * @returns {Record<string, any>}
 */
export function loadConfig() {
  if (_cache) return _cache;

  let userConfig = {};
  try {
    const raw = readFileSync(CONFIG_PATH, 'utf-8');
    userConfig = JSON.parse(raw);
  } catch (err) {
    if (err.code !== 'ENOENT') {
      // File exists but is malformed — warn the user
      process.stderr.write(`[config] ⚠️  Failed to parse config.json: ${err.message}. Using defaults.\n`);
    }
    // File not found → silently use defaults (expected on first run)
  }

  _cache = deepMerge(DEFAULTS, userConfig);
  return _cache;
}

/**
 * Get a config value by dot-notation key.
 * @param {string} key - e.g. 'movers_threshold' or 'models.analyst'
 * @param {any} [fallback] - fallback if key not found
 */
export function get(key, fallback = undefined) {
  const cfg = loadConfig();
  const parts = key.split('.');
  let val = cfg;
  for (const p of parts) {
    if (val == null || typeof val !== 'object') return fallback;
    val = val[p];
  }
  return val !== undefined ? val : fallback;
}

/** Reset cache (useful for tests) */
export function resetConfig() {
  _cache = null;
}

// ─── Helpers ────────────────────────────────────────────

function deepMerge(target, source) {
  const result = { ...target };
  for (const key of Object.keys(source)) {
    // Block prototype pollution
    if (key === '__proto__' || key === 'constructor' || key === 'prototype') continue;
    if (
      source[key] &&
      typeof source[key] === 'object' &&
      !Array.isArray(source[key]) &&
      target[key] &&
      typeof target[key] === 'object' &&
      !Array.isArray(target[key])
    ) {
      result[key] = deepMerge(target[key], source[key]);
    } else {
      result[key] = source[key];
    }
  }
  return result;
}
