#!/usr/bin/env node

/**
 * Doctor Strange — CLI Orchestrator
 * Unified entry: scan / search / report
 * Zero dependencies. ES Modules.
 *
 * Usage:
 *   node src/cli.js scan [--limit 50] [--json] [--mode terminal]
 *   node src/cli.js search --keyword "fed" [--limit 20] [--json]
 *   node src/cli.js report [--limit 80] [--json]
 */

import { loadConfig, get } from './config.js';
import { scan, search } from './gamma.js';
import { saveSnapshot, cleanupSnapshots, detectAlerts, saveReportMeta } from './snapshot.js';
import { saveAlerts } from './alerts.js';
import { formatScanReport, formatSearchResults, formatFullReport } from './formatter.js';
import {
  buildAnalystPrompt, buildChallengerPrompt, buildWatchlistPrompt,
  mergeResults,
} from './analyzer.js';

// ─── Arg Parsing (zero-dep) ─────────────────────────────

function parseArgs(argv) {
  const args = { command: null, keyword: null, limit: null, json: false, mode: 'markdown', skipAnalysis: false };
  const raw = argv.slice(2);

  for (let i = 0; i < raw.length; i++) {
    const arg = raw[i];
    if (!arg.startsWith('-') && !args.command) {
      args.command = arg;
    } else if (!arg.startsWith('-') && args.command && !args.keyword) {
      // Positional keyword after command: "scan iran", "report bitcoin"
      args.keyword = arg;
    } else if (arg === '--keyword' || arg === '-k') {
      const next = raw[++i];
      if (!next || next.startsWith('-')) {
        process.stderr.write('ERROR: --keyword / -k requires a value\n');
        process.exit(1);
      }
      args.keyword = next;
    } else if (arg === '--limit' || arg === '-l') {
      const parsed = parseInt(raw[++i], 10);
      args.limit = (Number.isFinite(parsed) && parsed > 0) ? parsed : null;
    } else if (arg === '--json') {
      args.json = true;
    } else if (arg === '--mode' || arg === '-m') {
      args.mode = raw[++i] || 'markdown';
    } else if (arg === '--skip-analysis') {
      args.skipAnalysis = true;
    } else if (arg === '--help' || arg === '-h') {
      args.command = 'help';
    }
  }

  return args;
}

// ─── Shared Scan Logic (DRY) ────────────────────────────

async function executeScan(reportType, { limit, keyword, json, mode }) {
  const startTime = Date.now();

  const label = keyword ? `scanning "${keyword}"` : 'scanning';
  process.stderr.write(`🔮 Doctor Strange ${label}...\n`);

  const scanData = await scan({ limit: limit || undefined, keyword: keyword || undefined });

  // Save snapshot
  const snapshotFile = saveSnapshot(scanData._allMarkets || [], scanData.timestamp);

  // Cleanup old snapshots
  cleanupSnapshots();

  // Check alerts
  const alerts = detectAlerts();
  if (alerts.length > 0) {
    saveAlerts(alerts);
    process.stderr.write(`⚠️  ${alerts.length} alert(s) detected\n`);
  }

  // Output
  const { _allMarkets, ...output } = scanData; // strip internal field
  if (json) {
    console.log(JSON.stringify(output, null, 2));
  } else {
    console.log(formatScanReport(output, mode));

    if (reportType === 'report') {
      console.log('');
      console.log('─'.repeat(40));
      console.log('💡 使用 `report` (不带 --skip-analysis) 获取完整 5 板块分析');
      console.log('─'.repeat(40));
    }
  }

  // Save meta
  const duration = ((Date.now() - startTime) / 1000).toFixed(1);
  saveReportMeta({
    report_type: reportType,
    duration_seconds: parseFloat(duration),
    markets_scanned: scanData.total_markets,
    movers_found: scanData.movers?.length || 0,
    certainties_found: scanData.certainties?.length || 0,
  });

  process.stderr.write(`✅ ${reportType} complete (${duration}s). Snapshot: ${snapshotFile}\n`);
}

// ─── Commands ───────────────────────────────────────────

async function cmdScan(args) {
  await executeScan('scan', args);
}

async function cmdSearch(args) {
  if (!args.keyword) {
    process.stderr.write('ERROR: --keyword / -k required for search\n');
    process.exit(1);
  }

  process.stderr.write(`🔍 Searching for "${args.keyword}"...\n`);

  const results = await search(args.keyword, { limit: args.limit || undefined });

  if (args.json) {
    console.log(JSON.stringify(results, null, 2));
  } else {
    console.log(formatSearchResults(results, args.keyword, args.mode));
  }
}

async function cmdReport(args) {
  if (args.skipAnalysis) {
    // Fast mode: Phase 1 output only (no LLM analysis)
    await executeScan('report', args);
    return;
  }

  // Full report: scan + build analysis prompts
  const startTime = Date.now();

  const label = args.keyword ? `report "${args.keyword}"` : 'report';
  process.stderr.write(`🔮 Doctor Strange ${label} (with analysis)...\n`);

  const scanData = await scan({ limit: args.limit || undefined, keyword: args.keyword || undefined });

  // Save snapshot
  const snapshotFile = saveSnapshot(scanData._allMarkets || [], scanData.timestamp);
  cleanupSnapshots();

  // Check alerts
  const alerts = detectAlerts();
  if (alerts.length > 0) {
    saveAlerts(alerts);
    process.stderr.write(`⚠️  ${alerts.length} alert(s) detected\n`);
  }

  // Build analysis prompts (LLM calls are NOT made here — prompts are returned for the orchestrator)
  const { _allMarkets, ...output } = scanData;

  process.stderr.write('📝 Building analysis prompts...\n');
  const analystPrompt = buildAnalystPrompt(output);
  const challengerPrompt = buildChallengerPrompt({ certainties: [], market_impact_summary: {}, investment_signals: [], risk_warnings: [], causal_chains: [] }, output);
  const watchlistConfig = { watchlist: get('watchlist') || [], models: get('models') || {} };
  const watchlistPrompt = buildWatchlistPrompt({ certainties: [], market_impact_summary: {}, investment_signals: [], risk_warnings: [], causal_chains: [] }, watchlistConfig);

  // Merge with empty results (prompts are prepared, actual LLM responses will be filled by orchestrator)
  const emptyAnalyst = { certainties: [], market_impact_summary: {}, investment_signals: [], risk_warnings: [], causal_chains: [] };
  const emptyChallenger = { audits: [], summary: '' };
  const emptyWatchlist = { stocks: [] };
  const analysisResult = mergeResults(emptyAnalyst, emptyChallenger, emptyWatchlist);

  if (args.json) {
    console.log(JSON.stringify({
      ...output,
      analysis: analysisResult,
      prompts: {
        analyst: analystPrompt,
        challenger: challengerPrompt,
        watchlist: watchlistPrompt,
      },
    }, null, 2));
  } else {
    console.log(formatFullReport(output, analysisResult, args.mode));
  }

  // Save meta
  const duration = ((Date.now() - startTime) / 1000).toFixed(1);
  saveReportMeta({
    report_type: 'report',
    duration_seconds: parseFloat(duration),
    markets_scanned: scanData.total_markets,
    movers_found: scanData.movers?.length || 0,
    certainties_found: scanData.certainties?.length || 0,
    analysis_prompts_built: true,
  });

  process.stderr.write(`✅ report complete (${duration}s). Snapshot: ${snapshotFile}\n`);
  process.stderr.write('📋 Analysis prompts ready for orchestrator. Use --json to extract.\n');
}

function showHelp() {
  console.log(`
🔮 Doctor Strange — 全球事件概率情报系统

Commands:
  scan [keyword]        Quick scan: trending + movers + certainties
                        Optional keyword narrows to matching markets
  search -k <keyword>   Search markets by keyword (flat list)
  report [keyword]      Full report: scan + analysis prompts (boards 1-5)

Options:
  -l, --limit <n>       Number of markets (default: from config.json)
  -k, --keyword <word>  Search keyword (required for search, optional for scan)
  --skip-analysis       Report without LLM analysis (Phase 1 only, fast)
  --json                Output raw JSON (includes analysis prompts)
  -m, --mode <mode>     Output mode: markdown (default) or terminal
  -h, --help            Show this help

Examples:
  node src/cli.js scan
  node src/cli.js scan iran
  node src/cli.js scan --limit 100 --json
  node src/cli.js search -k "fed"
  node src/cli.js report bitcoin
`);
}

// ─── Main ───────────────────────────────────────────────

async function main() {
  // Initialize config
  loadConfig();

  const args = parseArgs(process.argv);

  switch (args.command) {
    case 'scan':    return cmdScan(args);
    case 'search':  return cmdSearch(args);
    case 'report':  return cmdReport(args);
    case 'help':    return showHelp();
    default:
      showHelp();
      process.exit(args.command ? 1 : 0);
  }
}

// Top-level error boundary — never raw traceback
main().catch(err => {
  process.stderr.write(`\n❌ Doctor Strange error: ${err.message}\n`);
  if (process.env.DS_DEBUG) {
    process.stderr.write(err.stack + '\n');
  }
  process.exit(1);
});
