/**
 * Doctor Strange — Phase 2 Analyzer
 * Prompt builder + result parser for Analyst + Challenger + Watchlist chain
 * Zero dependencies. ES Modules.
 * 
 * This module does NOT call LLM APIs - it builds prompts and parses responses.
 * The actual LLM calls are handled by OpenClaw's agent session.
 */

import { readFileSync } from 'fs';
import { join, dirname } from 'path';
import { fileURLToPath } from 'url';
import { get } from './config.js';

const __dirname = dirname(fileURLToPath(import.meta.url));

// ─── Prompt Loaders ────────────────────────────────────────

function loadPrompt(name) {
  const path = join(__dirname, '..', 'prompts', `${name}.md`);
  return readFileSync(path, 'utf-8');
}

const PROMPTS = {
  analyst: loadPrompt('analyst'),
  challenger: loadPrompt('challenger'),
  watchlist: loadPrompt('watchlist_overlay'),
};

// ─── Data Extractors ────────────────────────────────────────

function extractContext(scanData) {
  const trending = scanData.trending || [];
  const movers = scanData.movers || [];
  const certainties = scanData.certainties || [];
  
  // Get high certainty candidates for Analyst
  const highCert = certainties.filter(m => m.probability_pct >= 85);
  const lowCert = certainties.filter(m => m.probability_pct <= 15);
  
  return {
    trending: trending.slice(0, 15).map(m => ({
      question: m.question,
      probability: m.probability_pct,
      volume: m.volume_24h,
      category: m.category,
      change_1d: m.change_1d_pct,
    })),
    movers: movers.slice(0, 10).map(m => ({
      question: m.question,
      probability: m.probability_pct,
      volume: m.volume_24h,
      change_1d: m.change_1d_pct,
      change_1w: m.change_1w_pct,
    })),
    highCertainty: highCert.slice(0, 10).map(m => ({
      question: m.question,
      probability: m.probability_pct,
      volume: m.volume_24h,
      trend: m.change_1d_pct > 5 ? 'rising' : m.change_1d_pct < -5 ? 'falling' : 'stable',
    })),
    lowCertainty: lowCert.slice(0, 10).map(m => ({
      question: m.question,
      probability: m.probability_pct,
      volume: m.volume_24h,
    })),
  };
}

// ─── Analyst Prompt Builder ────────────────────────────────

export function buildAnalystPrompt(scanData) {
  const ctx = extractContext(scanData);
  
  const userContent = `
# 输入数据

## 热搜事件 (Trending - Top 15)
${JSON.stringify(ctx.trending, null, 2)}

## 概率飙升事件 (Movers - Top 10)
${JSON.stringify(ctx.movers, null, 2)}

## 高确定性候选 (概率 >85%)
${JSON.stringify(ctx.highCertainty, null, 2)}

## 低确定性候选 (概率 <15%)
${JSON.stringify(ctx.lowCertainty, null, 2)}

## 分析要求

请按照 prompts/analyst.md 的要求进行分析，输出 JSON 格式的因果链、确定性事件、市场影响推演。

注意：
- 只输出 JSON，不要输出其他内容
- 因果链必须基于上述数据，不编造
- 每个确定性结论必须包含反例检验
- 用中文输出分析，英文保留原始事件名
`;

  return {
    system: PROMPTS.analyst,
    user: userContent,
  };
}

// ─── Analyst Result Parser ──────────────────────────────────

export function parseAnalystResponse(text) {
  // Try to extract JSON from response
  const jsonMatch = text.match(/```json\n([\s\S]*?)\n```/) 
    || text.match(/```\n([\s\S]*?)\n```/)
    || text.match(/\{[\s\S]*\}/);
  
  if (!jsonMatch) {
    // Return error structure if no JSON found
    return {
      error: 'Failed to parse Analyst response',
      raw: text.slice(0, 500),
      causal_chains: [],
      certainties: [],
      market_impact_summary: {},
      investment_signals: [],
      risk_warnings: [],
    };
  }
  
  try {
    return JSON.parse(jsonMatch[1] || jsonMatch[0]);
  } catch (e) {
    return {
      error: `JSON parse error: ${e.message}`,
      raw: text.slice(0, 500),
    };
  }
}

// ─── Challenger Prompt Builder ─────────────────────────────

export function buildChallengerPrompt(analystResult, scanData) {
  // Extract key conclusions from Analyst result for Challenger to audit
  const conclusions = [];
  
  if (analystResult.certainties) {
    for (const cert of analystResult.certainties.slice(0, 5)) {
      conclusions.push({
        target: cert.event || cert.question,
        probability: cert.probability,
        strange_rating: cert.strange_rating,
        analyst_reasoning: cert.evidence?.[0] || '',
      });
    }
  }
  
  if (analystResult.causal_chains) {
    for (const chain of analystResult.causal_chains.slice(0, 3)) {
      conclusions.push({
        target: chain.relation || chain.name || '因果关系',
        type: 'causal_chain',
        analyst_assessment: chain.confidence || '',
      });
    }
  }
  
  const userContent = `
# Analyst 分析结论 (待审计)

${JSON.stringify(conclusions, null, 2)}

## 原始数据参考
- 热搜事件数: ${scanData.trending?.length || 0}
- 飙升事件数: ${scanData.movers?.length || 0}
- 确定性候选数: ${scanData.certainties?.length || 0}

## 审计要求

请按照 prompts/challenger.md 的要求，对每个结论进行审计：
1. 构造反例
2. 评估可信度 (0-100%)
3. 检查方向性、时间性、隐含假设
4. 输出 CONFIRM / DOWNGRADE / REJECT 裁决

注意：只输出 JSON 格式的审计结果。
`;

  return {
    system: PROMPTS.challenger,
    user: userContent,
  };
}

// ─── Challenger Result Parser ──────────────────────────────

export function parseChallengerResponse(text) {
  const jsonMatch = text.match(/```json\n([\s\S]*?)\n```/)
    || text.match(/```\n([\s\S]*?)\n```/)
    || text.match(/\{[\s\S]*\}/);
  
  if (!jsonMatch) {
    return {
      error: 'Failed to parse Challenger response',
      raw: text.slice(0, 500),
      audits: [],
      summary: '',
    };
  }
  
  try {
    return JSON.parse(jsonMatch[1] || jsonMatch[0]);
  } catch (e) {
    return {
      error: `JSON parse error: ${e.message}`,
      raw: text.slice(0, 500),
    };
  }
}

// ─── Watchlist Prompt Builder ──────────────────────────────

export function buildWatchlistPrompt(analystResult, config) {
  const watchlist = config.watchlist || [];
  
  const watchlistStr = watchlist.map(s => 
    `- ${s.symbol} (${s.market}): ${s.alias || s.symbol} — 敏感事件: ${(s.sensitivity || []).join(', ')}`
  ).join('\n');
  
  const macroSummary = analystResult.market_impact_summary 
    ? JSON.stringify(analystResult.market_impact_summary, null, 2)
    : '无市场影响数据';
  
  const certainties = analystResult.certainties 
    ? analystResult.certainties.map(c => `- ${c.event}: ${(c.probability * 100).toFixed(0)}%`).join('\n')
    : '无确定性事件';
  
  const userContent = `
# 自选股列表
${watchlistStr}

# 确定性事件 (宏观背景)
${certainties}

# 市场影响摘要
${macroSummary}

## 分析要求

请按照 prompts/watchlist_overlay.md 的要求，对每只自选股进行分析：
1. 宏观背景关联
2. 事件敏感度分析
3. 风险矩阵
4. 综合判断 (🟢/🟡/🔴)

注意：只输出 JSON 格式的分析结果。
`;

  // Replace placeholder in prompt
  const systemPrompt = PROMPTS.watchlist.replace('{{WATCHLIST}}', watchlistStr);

  return {
    system: systemPrompt,
    user: userContent,
  };
}

// ─── Watchlist Result Parser ───────────────────────────────

export function parseWatchlistResponse(text) {
  const jsonMatch = text.match(/```json\n([\s\S]*?)\n```/)
    || text.match(/```\n([\s\S]*?)\n```/)
    || text.match(/\{[\s\S]*\}/);
  
  if (!jsonMatch) {
    return {
      error: 'Failed to parse Watchlist response',
      raw: text.slice(0, 500),
      stocks: [],
    };
  }
  
  try {
    const parsed = JSON.parse(jsonMatch[1] || jsonMatch[0]);
    // Normalize to array format
    if (parsed.stocks) {
      return parsed;
    }
    // Single stock case - wrap in array
    return { stocks: [parsed] };
  } catch (e) {
    return {
      error: `JSON parse error: ${e.message}`,
      raw: text.slice(0, 500),
      stocks: [],
    };
  }
}

// ─── Result Merger ─────────────────────────────────────────

export function mergeResults(analystResult, challengerResult, watchlistResult) {
  // Apply Challenger audits to certainties
  const validatedCertainties = [];
  
  if (analystResult.certainties && challengerResult.audits) {
    for (const cert of analystResult.certainties) {
      // Find matching audit
      const audit = challengerResult.audits.find(a => 
        a.target && cert.event && a.target.includes(cert.event.slice(0, 20))
      );
      
      if (audit) {
        // Apply audit verdict
        validatedCertainties.push({
          ...cert,
          challenger_verdict: audit.verdict,
          challenger_credibility: audit.credibility,
          challenger_counterexample: audit.counterexample,
        });
      } else {
        // No audit found - include as-is
        validatedCertainties.push({
          ...cert,
          challenger_verdict: 'PENDING',
        });
      }
    }
  } else {
    // Fallback: use raw certainties
    validatedCertainties.push(...(analystResult.certainties || []));
  }
  
  return {
    // Board 3: Validated Certainties
    certainties: validatedCertainties,
    
    // Board 4: Market Impact
    market_impact: analystResult.market_impact_summary || {},
    investment_signals: analystResult.investment_signals || [],
    risk_warnings: analystResult.risk_warnings || [],
    
    // Board 5: Watchlist
    watchlist_analysis: watchlistResult.stocks || [],
    
    // Metadata
    causal_chains: analystResult.causal_chains || [],
    challenger_summary: challengerResult.summary || '',
    analysis_timestamp: new Date().toISOString(),
  };
}

// ─── Full Analysis Pipeline Builder ────────────────────────

export function buildAnalysisPipeline(scanData, config) {
  // Step 1: Analyst
  const analystPrompt = buildAnalystPrompt(scanData);
  
  // Return pipeline structure (prompt building only, actual LLM calls handled by orchestrator)
  return {
    steps: [
      {
        name: 'analyst',
        model: config.models?.analyst || 'anthropic/claude-opus-4-6',
        prompt: analystPrompt,
        parser: parseAnalystResponse,
      },
      {
        name: 'challenger',
        model: config.models?.challenger || 'anthropic/claude-opus-4-6',
        promptBuilder: (analystResult) => buildChallengerPrompt(analystResult, scanData),
        parser: parseChallengerResponse,
      },
      {
        name: 'watchlist',
        model: config.models?.watchlist_overlay || 'anthropic/claude-opus-4-6',
        promptBuilder: (analystResult) => buildWatchlistPrompt(analystResult, config),
        parser: parseWatchlistResponse,
      },
    ],
    merger: mergeResults,
  };
}
