/**
 * Doctor Strange — Alert Manager
 * Deduplication + atomic append for alerts.
 * Zero dependencies.
 */

import { mkdirSync } from 'node:fs';
import { join, dirname } from 'node:path';
import { fileURLToPath } from 'node:url';
import { atomicWriteJson, safeReadJson } from './snapshot.js';

const __dirname = dirname(fileURLToPath(import.meta.url));
const ALERTS_DIR = join(__dirname, '..', 'data', 'alerts');

/**
 * Build a unique key for deduplication.
 * @param {object} alert
 * @returns {string}
 */
function alertKey(alert) {
  const id = alert.id || alert.question || 'unknown';
  return `${id}|${alert.new_timestamp ?? ''}`;
}

/**
 * Save alerts with deduplication.
 * Merges with existing daily file, skipping duplicates.
 * @param {object[]} newAlerts
 * @returns {string|null} filepath if saved, null if nothing new
 */
export function saveAlerts(newAlerts) {
  if (!newAlerts || newAlerts.length === 0) return null;

  mkdirSync(ALERTS_DIR, { recursive: true });
  const today = new Date().toISOString().slice(0, 10);
  const filepath = join(ALERTS_DIR, `${today}.json`);

  // Load existing alerts for today
  const existing = safeReadJson(filepath, { date: today, alerts: [] });
  const existingKeys = new Set(
    (existing.alerts || []).map(alertKey)
  );

  // Filter out duplicates
  const unique = newAlerts.filter(a => !existingKeys.has(alertKey(a)));
  if (unique.length === 0) return null;

  // Append and save atomically
  const merged = {
    date: today,
    alerts: [...(existing.alerts || []), ...unique],
  };

  atomicWriteJson(filepath, merged);
  process.stderr.write(`[alerts] Saved ${unique.length} new alert(s)\n`);
  return filepath;
}
