# Phase 2 Task: Doctor Strange — Analyst + Challenger + Market Impact + Watchlist

## Overview
Implement Phase 2 of Doctor Strange: boards 3-5 with LLM-powered analysis.
Phase 1 (boards 1-2: trending + movers + basic certainties) is already working.
Phase 2 adds deep analysis via multi-step LLM prompts.

## What Phase 2 Adds

### Board 3 Enhancement: Analyst + Challenger Adversarial Analysis
Currently board 3 just lists high/low probability markets with "待 Analyst+Challenger 对抗验证".
Phase 2 makes this real:
1. **Analyst** receives board 1-3 candidates → outputs causal chains, certainty ratings (⭐), counterexamples, Strange ratings
2. **Challenger** receives Analyst conclusions → constructs counterexamples, evaluates credibility, issues CONFIRM/DOWNGRADE/REJECT verdicts
3. Final output merges both into validated certainties with ratings

### Board 4: Three-Level Market Impact (三级市场影响)
Analyst also produces market impact analysis:
- Equity/Futures impact (sectors affected)
- Bond market (interest rate expectations)
- Forex (currency direction)
- Investment signals + risk warnings

### Board 5: Watchlist Overlay (自选股叠加分析)
Overlay macro analysis onto user's watchlist stocks:
- NVDA (US) — sensitivity: Fed政策, AI监管, 中美关系, 半导体出口管制
- 513300 纳指ETF (CN) — sensitivity: Fed政策, 美股走势, 汇率, 中美关系
Output: 🟢/🟡/🔴 judgment per stock with risk warnings

## Architecture

### New Module: `src/analyzer.js`
This is the core Phase 2 engine. It orchestrates the LLM prompt chain:

```
Step 1: Prepare context from Phase 1 scan data (trending + movers + certainties)
Step 2: Call Analyst prompt → get structured JSON (causal chains, certainties, market impact)
Step 3: Call Challenger prompt → get audit results (CONFIRM/DOWNGRADE/REJECT per conclusion)
Step 4: Call Watchlist Overlay prompt → get per-stock analysis
Step 5: Merge results into final report structure
```

**IMPORTANT: This module does NOT call any LLM API directly.**
Instead, it:
1. Builds the prompt text by reading `prompts/analyst.md`, `prompts/challenger.md`, `prompts/watchlist_overlay.md`
2. Injects the scan data as context into the prompts
3. Returns the prepared prompts + context as structured objects
4. The actual LLM calls will be handled by OpenClaw's agent session (the orchestrator calls the model)

So `analyzer.js` is a **prompt builder + result parser**, not an API caller.

The module should export:
```js
export function buildAnalystPrompt(scanData) → { system, user }
export function parseAnalystResponse(text) → structured object
export function buildChallengerPrompt(analystResult, scanData) → { system, user }
export function parseChallengerResponse(text) → structured object
export function buildWatchlistPrompt(analystResult, config) → { system, user }
export function parseWatchlistResponse(text) → structured object
export function mergeResults(analystResult, challengerResult, watchlistResult) → finalReport
```

### Formatter Updates: `src/formatter.js`
Add new format functions:
- `formatBoard3Enhanced(certainties, mode)` — rated certainties with Analyst reasoning + Challenger verdict
- `formatBoard4(marketImpact, mode)` — equity/bond/forex sections with signals/risks
- `formatBoard5(watchlistAnalysis, mode)` — per-stock cards with 🟢/🟡/🔴
- `formatFullReport(scanData, analysisResult, mode)` — complete 5-board report

### CLI Updates: `src/cli.js`
- `report` command should call analyzer after scan
- Add `--skip-analysis` flag to get Phase 1 output only (fast mode)
- Report meta should include analysis step info

## Prompt Files (Already Exist — Read Only)
- `prompts/analyst.md` — full Analyst prompt with JSON output format
- `prompts/challenger.md` — full Challenger prompt with audit format  
- `prompts/watchlist_overlay.md` — watchlist overlay prompt (has `{{WATCHLIST}}` placeholder)

Read these files to understand the expected input/output formats. Do NOT modify them.

## Config (Read Only)
- `config.json` has model assignments, watchlist, thresholds
- Models: analyst/challenger/watchlist_overlay → `anthropic/claude-opus-4-6`
- Watchlist: NVDA (US), 513300 (CN)

## Implementation Rules

### Code Style
- Zero npm dependencies — native Node.js only (fs, path, crypto)
- ES Modules (import/export)
- Clean, readable code with JSDoc comments
- Error handling: try/catch with meaningful messages, never raw tracebacks

### Git Commits (P1 RULE — CRITICAL)
**Every logical change MUST be its own atomic commit.** Like DB transactions.
Each commit independently revertable. Examples:
```
commit 1: feat(analyzer): add prompt builder for Analyst
commit 2: feat(analyzer): add prompt builder for Challenger  
commit 3: feat(analyzer): add watchlist prompt builder + result merger
commit 4: feat(formatter): add board 3/4/5 format functions
commit 5: feat(cli): wire report command to analyzer pipeline
commit 6: test: verify report command with --skip-analysis
```
**NEVER lump multiple features into one commit. This is a P1 error.**

### File Ownership
Only modify these files:
- `src/analyzer.js` (NEW)
- `src/formatter.js` (ADD functions, don't break existing ones)
- `src/cli.js` (ADD report pipeline, don't break scan/search)

Do NOT modify: `src/gamma.js`, `src/snapshot.js`, `src/alerts.js`, `src/config.js`, `prompts/*`

### Testing
After each module, verify:
1. `node ds.js scan` still works (regression check)
2. `node ds.js scan iran` still works  
3. `node ds.js report --skip-analysis` works (fast mode)
4. `node ds.js report --json` outputs valid JSON with analysis fields

## Expected Output Format

### Board 3 Enhanced Example:
```
🎯 板块三：不确定中的确定性 (Strange Certainties)

🔒 经 Analyst+Challenger 验证的确定性事件:

  1. ⭐⭐⭐⭐⭐ "Fed 2026年至少降息1次" — YES 91%
     ├─ Analyst: 尽管6月降息概率暴跌，全年至少1次概率仍极高
     ├─ 因果链: 就业数据放缓 → Fed 转鸽 → 至少1次降息
     ├─ Challenger: 反例 — 通胀二次反弹 (可信度: 15%)
     ├─ 裁决: CONFIRM (有条件)
     └─ 对冲视角: 利率敏感资产存在修复空间
```

### Board 4 Example:
```
💹 板块四：三级市场影响

【期货 & 股市】
  • 降息推迟 → 科技成长股短期承压
  • 投资热点: AI基础设施, 能源转型
  • ⚠️ 风险: 美股估值历史高位

【债市】
  • 短端利率维持高位
  • 机会: 长债收益率有吸引力

【汇市】
  • 美元走强概率上升
  • ⚠️ 风险: 新兴市场资本外流
```

### Board 5 Example:
```
📈 板块五：自选股 Doctor Strange 分析

*NVDA* [US] — NVIDIA
  • 宏观背景: AI监管确定性利好龙头，降息推迟短期利空
  • 事件敏感度: Fed政策(高), AI监管(高), 中美关系(中)
  • 综合判断: 🟡 震荡偏多，等回调低吸
  • ⚠️ 核心风险: 降息推迟压制估值
```

## Summary
1. Create `src/analyzer.js` — prompt builder + result parser (3 prompt chains + merger)
2. Update `src/formatter.js` — add board 3/4/5 format functions
3. Update `src/cli.js` — wire `report` to analyzer pipeline + `--skip-analysis` flag
4. Atomic commits for each logical unit
5. Regression test: scan/search must still work perfectly

When completely finished, run this command to notify me:
openclaw system event --text "Done: Phase 2 analyzer + formatter + CLI wired. Boards 3-5 implemented." --mode now
