#!/usr/bin/env bash
# DocSync — Documentation Generation Module
# Generates markdown documentation from code analysis

set -euo pipefail

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
SKILL_DIR="$(dirname "$SCRIPT_DIR")"
source "$SCRIPT_DIR/analyze.sh"

# Colors
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
BOLD='\033[1m'
DIM='\033[2m'
NC='\033[0m'

# ─── Config ─────────────────────────────────────────────────────────────────

get_output_dir() {
  echo "${DOCSYNC_OUTPUT_DIR:-docs}"
}

# ─── Single file doc generation ─────────────────────────────────────────────

generate_file_doc() {
  local source_file="$1"
  local output_dir
  output_dir=$(get_output_dir)
  local lang
  lang=$(detect_language "$source_file")
  local basename
  basename=$(basename "$source_file")
  local name="${basename%.*}"
  local dir
  dir=$(dirname "$source_file")

  # Collect symbols
  local functions=() classes=() types=() enums=()

  while IFS=$'\t' read -r type sname lineno filepath; do
    case "$type" in
      function)  functions+=("$sname") ;;
      class)     classes+=("$sname") ;;
      type|interface) types+=("$sname") ;;
      enum)      enums+=("$sname") ;;
    esac
  done < <(parse_symbols "$source_file")

  local total=$(( ${#functions[@]} + ${#classes[@]} + ${#types[@]} + ${#enums[@]} ))

  if [[ "$total" -eq 0 ]]; then
    echo -e "${DIM}  Skipping $source_file (no symbols found)${NC}"
    return 0
  fi

  # Determine output path
  local out_file="$dir/$name.md"
  if [[ -n "${DOCSYNC_OUTPUT_DIR:-}" ]]; then
    mkdir -p "$output_dir/$(dirname "$source_file")"
    out_file="$output_dir/${source_file%.*}.md"
  fi

  # Generate markdown
  {
    echo "# \`$basename\`"
    echo ""
    echo "> Auto-generated by [DocSync](https://docsync.pages.dev) — $(date +%Y-%m-%d)"
    echo ""
    echo "**Language:** $lang  "
    echo "**Path:** \`$source_file\`  "
    echo "**Symbols:** $total ($( [[ ${#functions[@]} -gt 0 ]] && echo "${#functions[@]} functions, " )$( [[ ${#classes[@]} -gt 0 ]] && echo "${#classes[@]} classes, " )$( [[ ${#types[@]} -gt 0 ]] && echo "${#types[@]} types, " )$( [[ ${#enums[@]} -gt 0 ]] && echo "${#enums[@]} enums" ))"
    echo ""
    echo "---"
    echo ""

    # Table of contents
    echo "## Table of Contents"
    echo ""
    if [[ ${#classes[@]} -gt 0 ]]; then
      echo "### Classes"
      for c in "${classes[@]}"; do
        echo "- [\`$c\`](#${c,,})"
      done
      echo ""
    fi
    if [[ ${#functions[@]} -gt 0 ]]; then
      echo "### Functions"
      for f in "${functions[@]}"; do
        echo "- [\`$f\`](#${f,,})"
      done
      echo ""
    fi
    if [[ ${#types[@]} -gt 0 ]]; then
      echo "### Types"
      for t in "${types[@]}"; do
        echo "- [\`$t\`](#${t,,})"
      done
      echo ""
    fi
    if [[ ${#enums[@]} -gt 0 ]]; then
      echo "### Enums"
      for e in "${enums[@]}"; do
        echo "- [\`$e\`](#${e,,})"
      done
      echo ""
    fi

    echo "---"
    echo ""

    # Detail sections
    if [[ ${#classes[@]} -gt 0 ]]; then
      echo "## Classes"
      echo ""
      for c in "${classes[@]}"; do
        echo "### \`$c\`"
        echo ""
        echo "<!-- TODO: Add description for $c -->"
        echo ""
        # Try to extract the class signature from source
        local sig
        sig=$(grep -nP "(class|struct|interface|trait)\s+${c}" "$source_file" | head -1 || true)
        if [[ -n "$sig" ]]; then
          echo "\`\`\`$lang"
          echo "$sig" | sed 's/^[0-9]*://'
          echo "\`\`\`"
          echo ""
        fi
      done
    fi

    if [[ ${#functions[@]} -gt 0 ]]; then
      echo "## Functions"
      echo ""
      for f in "${functions[@]}"; do
        echo "### \`$f\`"
        echo ""
        echo "<!-- TODO: Add description for $f -->"
        echo ""
        local sig
        sig=$(grep -nP "(function|func|fn|def|const|let|var)\s+${f}" "$source_file" | head -1 || true)
        if [[ -n "$sig" ]]; then
          echo "\`\`\`$lang"
          echo "$sig" | sed 's/^[0-9]*://'
          echo "\`\`\`"
          echo ""
        fi
        echo "**Parameters:**"
        echo ""
        echo "| Name | Type | Description |"
        echo "|------|------|-------------|"
        echo "| <!-- TODO --> | <!-- TODO --> | <!-- TODO --> |"
        echo ""
        echo "**Returns:** <!-- TODO -->"
        echo ""
      done
    fi

    if [[ ${#types[@]} -gt 0 ]]; then
      echo "## Types"
      echo ""
      for t in "${types[@]}"; do
        echo "### \`$t\`"
        echo ""
        echo "<!-- TODO: Add description for $t -->"
        echo ""
        local sig
        sig=$(grep -nP "(type|interface)\s+${t}" "$source_file" | head -1 || true)
        if [[ -n "$sig" ]]; then
          echo "\`\`\`$lang"
          echo "$sig" | sed 's/^[0-9]*://'
          echo "\`\`\`"
          echo ""
        fi
      done
    fi

    if [[ ${#enums[@]} -gt 0 ]]; then
      echo "## Enums"
      echo ""
      for e in "${enums[@]}"; do
        echo "### \`$e\`"
        echo ""
        echo "<!-- TODO: Add description for $e -->"
        echo ""
      done
    fi

  } > "$out_file"

  echo -e "  ${GREEN}✓${NC} Generated ${BOLD}$out_file${NC} (${total} symbols)"
}

# ─── Directory doc generation ───────────────────────────────────────────────

generate_directory_readme() {
  local dir="$1"
  local readme="$dir/README.md"
  local project_name
  project_name=$(basename "$(cd "$dir" && pwd)")

  # Collect all source files and their symbols
  local total_files=0
  local total_symbols=0
  local file_list
  file_list=$(mktemp)

  while IFS= read -r file; do
    if is_source_file "$file"; then
      local summary
      summary=$(summarize_file "$file")
      echo "$file|$summary" >> "$file_list"
      ((total_files++)) || true
    fi
  done < <(find "$dir" -maxdepth 3 -type f \
    -not -path '*/node_modules/*' \
    -not -path '*/.git/*' \
    -not -path '*/dist/*' \
    -not -path '*/build/*' \
    -not -path '*/__pycache__/*' \
    -not -path '*/vendor/*' \
    -not -path '*/.venv/*' \
    -not -path '*/target/*' 2>/dev/null | sort)

  {
    echo "# $project_name"
    echo ""
    echo "> Auto-generated by [DocSync](https://docsync.pages.dev) — $(date +%Y-%m-%d)"
    echo ""
    echo "## Overview"
    echo ""
    echo "<!-- TODO: Add project description -->"
    echo ""
    echo "## Project Structure"
    echo ""
    echo "| File | Functions | Classes | Types | Enums |"
    echo "|------|-----------|---------|-------|-------|"

    while IFS='|' read -r file summary; do
      local funcs classes types enums
      funcs=$(echo "$summary" | grep -oP 'functions=\K\d+')
      classes=$(echo "$summary" | grep -oP 'classes=\K\d+')
      types=$(echo "$summary" | grep -oP 'types=\K\d+')
      enums=$(echo "$summary" | grep -oP 'enums=\K\d+')
      local rel_file="${file#$dir/}"
      echo "| \`$rel_file\` | $funcs | $classes | $types | $enums |"
    done < "$file_list"

    echo ""
    echo "**Total:** $total_files source files"
    echo ""
    echo "## Getting Started"
    echo ""
    echo "<!-- TODO: Add setup instructions -->"
    echo ""
    echo "## API Reference"
    echo ""
    echo "See individual file documentation for detailed API reference."
    echo ""

  } > "$readme"

  rm -f "$file_list"
  echo -e "  ${GREEN}✓${NC} Generated ${BOLD}$readme${NC} ($total_files files analyzed)"
}

# ─── Onboarding guide (TEAM tier) ──────────────────────────────────────────

do_onboarding() {
  local target="${1:-.}"
  local output="$target/docs/ONBOARDING.md"
  mkdir -p "$(dirname "$output")"

  local project_name
  project_name=$(basename "$(cd "$target" && pwd)")

  # Analyze project structure
  local lang_counts
  lang_counts=$(mktemp)

  find "$target" -type f \
    -not -path '*/node_modules/*' \
    -not -path '*/.git/*' \
    -not -path '*/dist/*' \
    -not -path '*/build/*' 2>/dev/null | while IFS= read -r f; do
    detect_language "$f"
  done | grep -v "unknown" | sort | uniq -c | sort -rn > "$lang_counts"

  local primary_lang
  primary_lang=$(head -1 "$lang_counts" | awk '{print $2}')

  {
    echo "# Onboarding Guide: $project_name"
    echo ""
    echo "> Auto-generated by [DocSync](https://docsync.pages.dev) — $(date +%Y-%m-%d)"
    echo ""
    echo "## Welcome"
    echo ""
    echo "This guide will help you get up to speed with the **$project_name** codebase."
    echo ""
    echo "## Tech Stack"
    echo ""
    echo "| Language | Files |"
    echo "|----------|-------|"
    while read -r count lang; do
      [[ -z "$lang" ]] && continue
      echo "| $lang | $count |"
    done < "$lang_counts"
    echo ""
    echo "**Primary language:** $primary_lang"
    echo ""
    echo "## Prerequisites"
    echo ""
    echo "<!-- TODO: List required tools, SDKs, and accounts -->"
    echo ""
    echo "## Setup"
    echo ""
    echo "\`\`\`bash"
    echo "# Clone the repository"
    echo "git clone <repo-url>"
    echo "cd $project_name"
    echo ""
    # Detect package manager
    if [[ -f "$target/package.json" ]]; then
      echo "# Install dependencies"
      if [[ -f "$target/pnpm-lock.yaml" ]]; then
        echo "pnpm install"
      elif [[ -f "$target/yarn.lock" ]]; then
        echo "yarn install"
      else
        echo "npm install"
      fi
    elif [[ -f "$target/requirements.txt" ]]; then
      echo "# Install dependencies"
      echo "pip install -r requirements.txt"
    elif [[ -f "$target/Cargo.toml" ]]; then
      echo "# Build the project"
      echo "cargo build"
    elif [[ -f "$target/go.mod" ]]; then
      echo "# Download dependencies"
      echo "go mod download"
    fi
    echo "\`\`\`"
    echo ""
    echo "## Key Directories"
    echo ""
    echo "| Directory | Purpose |"
    echo "|-----------|---------|"
    for d in src lib app api config test tests spec docs scripts; do
      if [[ -d "$target/$d" ]]; then
        echo "| \`$d/\` | <!-- TODO: Describe purpose --> |"
      fi
    done
    echo ""
    echo "## Key Concepts"
    echo ""
    echo "<!-- TODO: List 3-5 key architectural concepts new developers should understand -->"
    echo ""
    echo "## Common Tasks"
    echo ""
    echo "<!-- TODO: Add common development tasks and commands -->"
    echo ""
    echo "## Architecture"
    echo ""
    echo "See [ARCHITECTURE.md](ARCHITECTURE.md) for detailed architecture documentation."
    echo ""
    echo "---"
    echo ""
    echo "*Generated by DocSync. Fill in the TODO sections to complete this guide.*"

  } > "$output"

  rm -f "$lang_counts"
  echo -e "  ${GREEN}✓${NC} Generated ${BOLD}$output${NC}"
}

# ─── Architecture doc (TEAM tier) ──────────────────────────────────────────

do_architecture() {
  local target="${1:-.}"
  local output="$target/docs/ARCHITECTURE.md"
  mkdir -p "$(dirname "$output")"

  local project_name
  project_name=$(basename "$(cd "$target" && pwd)")

  # Find top-level directories
  local top_dirs
  top_dirs=$(find "$target" -maxdepth 1 -type d \
    -not -name '.*' \
    -not -name 'node_modules' \
    -not -name 'dist' \
    -not -name 'build' \
    -not -name '__pycache__' \
    -not -path "$target" 2>/dev/null | sort)

  {
    echo "# Architecture: $project_name"
    echo ""
    echo "> Auto-generated by [DocSync](https://docsync.pages.dev) — $(date +%Y-%m-%d)"
    echo ""
    echo "## High-Level Overview"
    echo ""
    echo "<!-- TODO: Describe the system at a high level -->"
    echo ""
    echo "## Module Map"
    echo ""
    echo "\`\`\`"
    echo "$project_name/"
    while IFS= read -r d; do
      local dirname
      dirname=$(basename "$d")
      local file_count
      file_count=$(find "$d" -type f 2>/dev/null | wc -l | tr -d ' ')
      echo "├── $dirname/    ($file_count files)"
    done <<< "$top_dirs"
    echo "\`\`\`"
    echo ""
    echo "## Components"
    echo ""

    while IFS= read -r d; do
      local dirname
      dirname=$(basename "$d")
      local symbol_count
      symbol_count=$(analyze_directory "$d" 2>/dev/null | wc -l | tr -d ' ')
      echo "### \`$dirname/\`"
      echo ""
      echo "**Symbols:** $symbol_count  "
      echo "**Purpose:** <!-- TODO: Describe this module's responsibility -->"
      echo ""
      echo "**Key files:**"
      echo ""
      find "$d" -maxdepth 1 -type f -name '*.ts' -o -name '*.js' -o -name '*.py' -o -name '*.go' -o -name '*.rs' 2>/dev/null | head -10 | while IFS= read -r f; do
        echo "- \`$(basename "$f")\`"
      done
      echo ""
    done <<< "$top_dirs"

    echo "## Data Flow"
    echo ""
    echo "<!-- TODO: Describe how data flows through the system -->"
    echo ""
    echo "## Dependencies"
    echo ""
    echo "<!-- TODO: List key external dependencies and their purpose -->"
    echo ""
    echo "## Decision Log"
    echo ""
    echo "| Date | Decision | Rationale |"
    echo "|------|----------|-----------|"
    echo "| $(date +%Y-%m-%d) | Initial architecture doc | Auto-generated by DocSync |"
    echo ""
    echo "---"
    echo ""
    echo "*Generated by DocSync. Fill in the TODO sections to complete this document.*"

  } > "$output"

  echo -e "  ${GREEN}✓${NC} Generated ${BOLD}$output${NC}"
}

# ─── Main generation entry point ───────────────────────────────────────────

do_generate() {
  local target="$1"

  if [[ -f "$target" ]]; then
    generate_file_doc "$target"
  elif [[ -d "$target" ]]; then
    echo -e "${BLUE}[DocSync]${NC} Generating docs for directory: ${BOLD}$target${NC}"
    echo ""

    # Generate README for the directory
    generate_directory_readme "$target"

    # Generate individual file docs
    find "$target" -maxdepth 3 -type f \
      -not -path '*/node_modules/*' \
      -not -path '*/.git/*' \
      -not -path '*/dist/*' \
      -not -path '*/build/*' \
      -not -path '*/__pycache__/*' \
      -not -path '*/vendor/*' \
      -not -path '*/.venv/*' \
      -not -path '*/target/*' 2>/dev/null | sort | while IFS= read -r file; do
      if is_source_file "$file"; then
        generate_file_doc "$file"
      fi
    done

    echo ""
    echo -e "${GREEN}[DocSync]${NC} Documentation generation complete."
  else
    echo -e "${RED}[DocSync]${NC} Target not found: $target" >&2
    return 1
  fi
}
