const crypto = require('crypto');
const axios = require('axios');

class XunleiDockerClient {
  constructor(config = {}) {
    this.host = config.host || null;
    this.port = config.port || null;
    this.ssl = config.ssl || false;
    this.deviceId = null;
    this.parentFolderId = null;
    this.baseUrl = null;
    
    // Cache for high version token
    this.cachedHighVersionToken = null;
    this.highVersionTokenCacheTime = 0;
    this.HIGH_VERSION_TOKEN_CACHE_DURATION = 10 * 60 * 1000; // 10 minutes
  }

  /**
   * Initialize the client with configuration
   */
  configure(host, port, ssl = false) {
    this.host = host;
    this.port = port;
    this.ssl = ssl;
    this.baseUrl = `${this.ssl ? 'https' : 'http'}://${host}:${port}`;
  }

  /**
   * Generate MD5 hash
   */
  hex_md5(input) {
    return crypto.createHash('md5').update(input).digest('hex');
  }

  /**
   * Get Xunlei service version directly without using request method
   */
  async getServiceVersion() {
    if (!this.baseUrl) {
      throw new Error('Xunlei service not configured. Please use configure() first.');
    }

    try {
      const headers = {
        'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/90.0.4430.212 Safari/537.36 Edg/90.0.818.66',
        'Accept': '*/*',
        'Accept-Language': 'zh-CN,zh;q=0.9'
      };

      const url = `${this.baseUrl}/webman/3rdparty/pan-xunlei-com/index.cgi/launcher/status`;
      const options = {
        method: 'GET',
        url: url,
        headers,
        timeout: 10000
      };

      const response = await axios(options);
      return response.data.running_version || 'unknown';
    } catch (error) {
      throw new Error(`Failed to get service version: ${error.message}`);
    }
  }

  /**
   * Compare version numbers
   */
  isVersionAtLeast(currentVersion, targetVersion) {
    if (!currentVersion || !targetVersion) return false;
    const currentParts = currentVersion.split('.').map(Number);
    const targetParts = targetVersion.split('.').map(Number);

    for (let i = 0; i < targetParts.length; i++) {
      const current = currentParts[i] || 0;
      const target = targetParts[i] || 0;
      if (current > target) return true;
      if (current < target) return false;
    }
    return currentParts.length >= targetParts.length;
  }

  /**
   * Get high version token from the homepage
   */
  async getHighVersionToken() {
    if (this.cachedHighVersionToken && 
        (Date.now() - this.highVersionTokenCacheTime < this.HIGH_VERSION_TOKEN_CACHE_DURATION)) {
      return this.cachedHighVersionToken;
    }

    try {
      const url = `${this.baseUrl}/webman/3rdparty/pan-xunlei-com/index.cgi/`;
      
      const headers = {
        'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/90.0.4430.212 Safari/537.36 Edg/90.0.818.66'
      };

      const options = {
        method: 'GET',
        url: url,
        headers,
        timeout: 10000
      };

      const response = await axios(options);
      const htmlContent = response.data;
      
      const uiauthRegex = /function uiauth\(value\)\s*{\s*return\s*"([^"]+)"\s*}/;
      const match = htmlContent.match(uiauthRegex);

      if (match && match[1]) {
        this.cachedHighVersionToken = match[1];
        this.highVersionTokenCacheTime = Date.now();
        return this.cachedHighVersionToken;
      }
      return null;
    } catch (error) {
      console.error('Error getting high version token:', error.message);
      return null;
    }
  }

  /**
   * Generate authentication token for Xunlei service
   */
  async generatePanAuth() {
    if (!this.baseUrl) {
      throw new Error('Xunlei service not configured. Please use configure() first.');
    }

    // First try to get the service version
    try {
      const serverVersion = await this.getServiceVersion();
      
      if (serverVersion && this.isVersionAtLeast(serverVersion, "3.21.0")) {
        console.log(`Detected Xunlei version ${serverVersion}, using high version auth method`);
        const token = await this.getHighVersionToken();
        if (token) {
          return token;
        }
        console.warn("High version token not found, falling back to low version method");
      } else if (serverVersion) {
        console.log(`Detected Xunlei version ${serverVersion}, using low version auth method`);
      } else {
        console.log("Cannot determine Xunlei version, using low version auth method");
      }
    } catch (error) {
      console.warn("Error detecting server version, using low version auth method:", error.message);
    }

    // Fallback to low-version method
    const timestamp = Math.floor(Date.now() / 1000);
    const randomString = `${timestamp}yrjmxtpovrzzdqgtbjdncmsywlpmyqcaawbnruddxucykfebpkuseypjegajzzpplmzrejnavcwtvciupgigyrtomdljhtmsljegvutunuizvatwtqdjheituaizfjyfzpbcvhhlaxzfatpgongrqadvixrnvastczwnolznfavqrvmjseiosmvrtcqiapmtzjfihdysqmhaijlpsrssovkpqnjbxuwkhjpfxpoldvqrnlhgdbcpnsilsmydxaxrxjzbdekzmshputmgkedetrcbmcdgljfkpbprvqncixfkavyxoibbuuyqzvcbzdgvipozeplohmcyfornhxzsadavvimivbzexfzhlndddnbywhsvjrotwzarbycpwydvpeqtuigfwzcvoswgpoakuvgdbykdjdcsdlnqskogpbsyceeyaigbgmrbnzixethpvqvvfvdcvjbilxikvklfbkcnfprzhijjnuoovulvigiqvbosnbixeplvnewmyipxuzpvocbvidnzgsrdfkejghvvyizkjlofndcuzvlhdhovpeolsyroljurbplpwbbihmdloahicnqehgjnbthmrljtzovltnlpeibodpjvemhhybmanskbtvdrgkrzoyhsjcexfrcpddoemazkfjwmrbrcloitmdzzkgxwlhnbfpjffrpryljdzdqsbacrjgohzwgbvzgevnqvxppsxqzczfgpuvigjbuhzweyeinukeurkogpotdegqhtsztdinmijjowivciviunhcjhtufzhjlmpqlngslimksdeezdzxihtmaywfvipjctuealhlovmzdodruperyysdhwjbtidwdzusifeepywsmkqbknlgdhextvlheufxivphskqvdtbcjfryxlolujmennakdqjdhtcxwnhknhzlaatuhyofenhdigojyxrluijjxeywnmopsuicglfcqyybbpynpcsnizupumtakwwnjlkfkuooqoqxhjnryylklokmzvmmgjsbbvgmwoucpvzedmqpkmazwhhvxqygrexopkmcdyniqocguykphlngjesqohhuvnkcliuawkzcmvevdbouwzvgmhtavwyhstvqwhcwjluzjopnhuisbsrloavcieskcyqftdhieduduhowgvrkimgdhyszsiknmuzvnrqqlbykbdlixosgxrdunymbixakkmgppteayqmqivxcwawyidpltevotwoxlkrucmluuluatgeskhfsrsebhniwhujpwrpknjxylidtjwebvwmbwayoepootybnlcaoixlgvjmpquxnyomoiopsjxtnorhwnlmonllastiezyvfbbgngjybtgbkxuaqdmkuqwupgzhffuyzgdnahdifaqtfmpysnlesvfoiofxvbtqkiqvdniejbyzugbkursumqddaslhqpkdrjnnsdqfthxtghxhaylgeqnknhqwpammlfnlkjuqevnxesyqsnpufvrbeohphxfabcduuklpkfoiifsqrrbsxkkmdrnkeboprnksfzwmjymjspzsrfjlwneuwzjjwejruubhhqaktxhygtjuhjmtvrklrmxdbbwooxsucmynwgcxhzdctgtchaevmpfiqfwydultmgqnionuendspvdrcctxldnyjlgnsqxaddadxeyvlcifdxksgdhaatsslhcofnxmilljpzdlumfjvcwvjrxegwbwuuwkguydhozqqnuselsoojnsefquuhpijdguofwrcjbuaugyzphkenbyhdstsldybdqsfxjhpgnerbdosbtyzdtrhyvwkzkurnmbgjtzlzcpfsuxussguelnjttmwejhreptwogekfvdsemlkvklcxeuzlboqwbngddexhsmyzqkztvlbgybbfmzbjroajaucykiqvhjrirlgawaessusvulngosviecmbpfgevxqptalguchfzkrrpruwxspggiqokepqpocezcewhyajsgxrqqqeuhwvc`;
    const md5 = this.hex_md5(randomString);
    return `${timestamp}.${md5}`;
  }

  /**
   * Make a request to the Xunlei service
   */
  async request(method, url, data = null, useAuth = true) {
    if (!this.baseUrl) {
      throw new Error('Xunlei service not configured. Please use configure() first.');
    }

    const headers = {
      'DNT': '1',
      'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/90.0.4430.212 Safari/537.36 Edg/90.0.818.66',
      'device-space': '',
      'content-type': 'application/json',
      'Accept': '*/*',
      'Accept-Language': 'zh-CN,zh;q=0.9'
    };

    // Add auth header only if required (some endpoints like version check don't need it)
    if (useAuth) {
      const panAuth = await this.generatePanAuth();
      if (!panAuth) {
        throw new Error('Unable to generate auth token for Xunlei service.');
      }
      headers['pan-auth'] = panAuth;
    }

    const fullUrl = this.baseUrl + url;
    const options = {
      method,
      url: fullUrl,
      headers,
      timeout: 10000 // 10 seconds timeout
    };

    if (method === 'POST' && data) {
      options.data = data;
    }

    try {
      const response = await axios(options);
      return response.data;
    } catch (error) {
      if (error.response) {
        throw new Error(`Request failed with status ${error.response.status}: ${error.response.statusText}`);
      } else if (error.request) {
        throw new Error(`Network error: Unable to reach Xunlei service at ${this.baseUrl}`);
      } else {
        throw new Error(`Error: ${error.message}`);
      }
    }
  }

  /**
   * Get device ID from Xunlei service
   */
  async getDeviceId() {
    try {
      const response = await this.request('GET', '/webman/3rdparty/pan-xunlei-com/index.cgi/drive/v1/tasks?type=user%23runner&device_space=');
      const tasks = response.tasks || [];
      
      if (tasks.length === 0) {
        throw new Error('No remote device is bound to the Xunlei service');
      }

      this.deviceId = tasks[0].params?.target;
      return this.deviceId;
    } catch (error) {
      throw new Error(`Failed to get device ID: ${error.message}`);
    }
  }

  /**
   * Get uncompleted download tasks
   */
  async getUncompletedTasks() {
    if (!this.deviceId) {
      await this.getDeviceId();
    }

    const url = `/webman/3rdparty/pan-xunlei-com/index.cgi/drive/v1/tasks?space=${encodeURIComponent(this.deviceId)}&page_token=&filters=%7B%22phase%22%3A%7B%22in%22%3A%22PHASE_TYPE_PENDING%2CPHASE_TYPE_RUNNING%2CPHASE_TYPE_PAUSED%2CPHASE_TYPE_ERROR%22%7D%2C%22type%22%3A%7B%22in%22%3A%22user%23download-url%2Cuser%23download%22%7D%7D&limit=200&device_space=`;
    
    try {
      const response = await this.request('GET', url);
      return response.tasks ? response.tasks : [];
    } catch (error) {
      throw new Error(`Failed to get uncompleted tasks: ${error.message}`);
    }
  }

  /**
   * Get completed download tasks
   */
  async getCompletedTasks() {
    if (!this.deviceId) {
      await this.getDeviceId();
    }

    const url = `/webman/3rdparty/pan-xunlei-com/index.cgi/drive/v1/tasks?space=${encodeURIComponent(this.deviceId)}&page_token=&filters=%7B%22phase%22%3A%7B%22in%22%3A%22PHASE_TYPE_COMPLETE%22%7D%2C%22type%22%3A%7B%22in%22%3A%22user%23download-url%2Cuser%23download%22%7D%7D&limit=200&device_space=`;
    
    try {
      const response = await this.request('GET', url);
      return response.tasks ? response.tasks : [];
    } catch (error) {
      throw new Error(`Failed to get completed tasks: ${error.message}`);
    }
  }

  /**
   * Extract file list from a magnet link
   */
  async extractFileList(magnetLink) {
    try {
      const response = await this.request('POST', '/webman/3rdparty/pan-xunlei-com/index.cgi/drive/v1/resource/list?device_space=', {
        urls: magnetLink
      });
      return response.list ? response.list : {};
    } catch (error) {
      throw new Error(`Failed to extract file list: ${error.message}`);
    }
  }

  /**
   * Submit a download task
   */
  async submitTask(magnetLink, taskName = '', selectedFiles = []) {
    if (!this.deviceId) {
      await this.getDeviceId();
    }

    // If no files are specified, download all files in the torrent
    let finalTaskFiles = selectedFiles;
    if (selectedFiles.length === 0) {
      try {
        const fileTree = await this.extractFileList(magnetLink);
        const resources = fileTree.resources || [];
        finalTaskFiles = resources.map((resource, index) => ({
          index: resource.file_index || index,
          file_size: resource.file_size || 0,
          file_name: resource.name || `file_${index}`
        }));
      } catch (error) {
        console.warn('Could not extract file list, submitting with all files:', error.message);
        // If we can't extract files, submit with default values
        finalTaskFiles = [{ index: 0, file_size: 0, file_name: 'all_files' }];
      }
    }

    const name = taskName || finalTaskFiles[0]?.file_name || 'Download Task';
    const taskFileCount = finalTaskFiles.length;

    // Get parent folder ID if not already cached
    if (!this.parentFolderId) {
      try {
        const res = await this.request('GET', `/webman/3rdparty/pan-xunlei-com/index.cgi/drive/v1/files?space=${encodeURIComponent(this.deviceId)}&limit=200&parent_id=&filters=%7B%22kind%22%3A%7B%22eq%22%3A%22drive%23folder%22%7D%7D&page_token=&device_space=`);
        this.parentFolderId = res?.files?.[0]?.parent_id || '';
      } catch (error) {
        console.warn('Could not get parent folder ID:', error.message);
        this.parentFolderId = '';
      }
    }

    const submitBody = {
      type: "user#download-url",
      name: name,
      file_name: name,
      file_size: finalTaskFiles.reduce((sum, f) => sum + f.file_size, 0).toString(),
      space: this.deviceId,
      params: {
        target: this.deviceId,
        url: magnetLink,
        total_file_count: taskFileCount.toString(),
        parent_folder_id: this.parentFolderId,
        sub_file_index: finalTaskFiles.map(f => f.index.toString()).join(','),
        file_id: ""
      }
    };

    try {
      const response = await this.request('POST', '/webman/3rdparty/pan-xunlei-com/index.cgi/drive/v1/task?device_space=', submitBody);
      if (response.HttpStatus !== 0) {
        throw new Error(response.error_description || 'Submission failed');
      }
      return { success: true, taskId: response.id, message: 'Task submitted successfully' };
    } catch (error) {
      throw new Error(`Failed to submit task: ${error.message}`);
    }
  }

  /**
   * Test connection to Xunlei service
   */
  async testConnection() {
    try {
      await this.getDeviceId();
      return { success: true, message: 'Connected to Xunlei service successfully' };
    } catch (error) {
      return { success: false, message: error.message };
    }
  }
}

module.exports = XunleiDockerClient;