const XunleiDockerClient = require('./xunlei_docker_client');
const fs = require('fs');
const path = require('path');

class XunleiDockerSkill {
  constructor() {
    this.client = null;
    this.configPath = path.join(__dirname, 'config.json');
    this.loadConfig();
  }

  /**
   * Load configuration from file
   */
  loadConfig() {
    try {
      if (fs.existsSync(this.configPath)) {
        const configData = fs.readFileSync(this.configPath, 'utf8');
        const config = JSON.parse(configData);
        
        this.client = new XunleiDockerClient();
        this.client.configure(config.host, config.port, config.ssl || false);
        console.log('Xunlei Docker skill configuration loaded');
      } else {
        this.client = new XunleiDockerClient();
        console.log('Created new Xunlei Docker client instance');
      }
    } catch (error) {
      console.error('Error loading Xunlei Docker skill configuration:', error);
      this.client = new XunleiDockerClient();
    }
  }

  /**
   * Save configuration to file
   */
  saveConfig(config) {
    try {
      fs.writeFileSync(this.configPath, JSON.stringify(config, null, 2));
      console.log('Xunlei Docker skill configuration saved');
    } catch (error) {
      console.error('Error saving Xunlei Docker skill configuration:', error);
    }
  }

  /**
   * Execute a command based on the message
   */
  async execute(message) {
    const args = message.trim().split(/\s+/);
    const command = args[0].toLowerCase();

    switch (command) {
      case 'xunlei':
        return this.handleXunleiCommand(args.slice(1));
      default:
        return this.help();
    }
  }

  /**
   * Handle xunlei commands
   */
  async handleXunleiCommand(args) {
    if (args.length === 0) {
      return this.help();
    }

    const subCommand = args[0].toLowerCase();

    switch (subCommand) {
      case 'status':
        return this.getStatus();
      case 'submit':
        return this.submitTask(args.slice(1));
      case 'config':
        return this.handleConfigCommand(args.slice(1));
      case 'completed':
        return this.getCompletedTasks();
      case 'version':
        return this.getVersion();
      case 'help':
        return this.help();
      default:
        return `Unknown command: xunlei ${subCommand}. Use 'xunlei help' for available commands.`;
    }
  }

  /**
   * Show help information
   */
  help() {
    return `
Xunlei Docker Downloader Skill Help:

Commands:
  xunlei status                    - Show current download tasks
  xunlei submit <magnet_link>      - Submit a magnet link for download
  xunlei submit <mangle_link> --name <task_name>  - Submit with custom task name
  xunlei config set <host> <port>  - Configure Xunlei service connection
  xunlei config show               - Show current configuration
  xunlei completed                 - Show completed tasks
  xunlei version                   - Show Xunlei service version
  xunlei help                      - Show this help message
    `;
  }

  /**
   * Show current download status
   */
  async getStatus() {
    try {
      const tasks = await this.client.getUncompletedTasks();
      
      if (tasks.length === 0) {
        return 'No active download tasks.';
      }

      let response = `📥 Active Download Tasks (${tasks.length}):\n\n`;
      tasks.forEach((task, index) => {
        const progress = task.progress || 0;
        const speed = task.params?.speed ? Math.round(task.params.speed / 1024 / 1024 * 100) / 100 : 0; // Convert to MB/s
        const fileSize = this.formatFileSize(task.file_size);
        
        response += `${index + 1}. ${task.name}\n`;
        response += `   Progress: ${progress}% | Speed: ${speed} MB/s | Size: ${fileSize}\n`;
        response += `   Updated: ${new Date(task.updated_time).toLocaleString()}\n\n`;
      });

      return response;
    } catch (error) {
      return `Error getting status: ${error.message}`;
    }
  }

  /**
   * Format file size in human-readable form
   */
  formatFileSize(bytes) {
    if (bytes === undefined || bytes === null) return 'Unknown';
    
    const sizes = ['Bytes', 'KB', 'MB', 'GB', 'TB'];
    if (bytes === 0) return '0 Bytes';
    const i = parseInt(Math.floor(Math.log(bytes) / Math.log(1024)));
    return Math.round(bytes / Math.pow(1024, i) * 100) / 100 + ' ' + sizes[i];
  }

  /**
   * Submit a download task
   */
  async submitTask(args) {
    if (args.length === 0) {
      return 'Please provide a magnet link. Usage: xunlei submit <magnet_link>';
    }

    // Extract magnet link and optional task name
    let magnetLink = '';
    let taskName = '';
    
    for (let i = 0; i < args.length; i++) {
      if (args[i] === '--name' && i + 1 < args.length) {
        taskName = args[i + 1];
        i++; // Skip the next argument as it's the task name
      } else if (args[i].startsWith('magnet:?')) {
        magnetLink = args[i];
      }
    }

    if (!magnetLink) {
      return 'No valid magnet link provided. Usage: xunlei submit <magnet_link>';
    }

    try {
      // Analyze the magnet link to identify main content vs ads
      console.log('Analyzing magnet link for content filtering...');
      const fileTree = await this.client.extractFileList(magnetLink);
      
      if (!fileTree.resources || fileTree.resources.length === 0) {
        return 'No files found in magnet link';
      }

      // Get all files from the tree structure
      const allResources = this.getAllFilesFromTree(fileTree.resources);
      
      // Classify files as main content or ads
      const classified = this.classifyFiles(allResources);
      
      console.log(`File analysis complete: ${classified.mainContent.length} main content, ${classified.ads.length} ads`);
      
      // Select only main content files for download
      let selectedFiles = classified.mainContent;
      
      // If no main content identified, use largest files as fallback
      if (selectedFiles.length === 0) {
        console.log('No main content files identified, using largest files as fallback');
        const sortedFiles = classified.all.sort((a, b) => 
          parseInt(b.file_size) - parseInt(a.file_size)
        );
        // Take up to 3 largest files
        selectedFiles = sortedFiles.slice(0, 3);
        console.log(`Selected ${selectedFiles.length} largest files as likely main content`);
      }

      // Prepare file indices for submission
      const selectedFileIndices = selectedFiles.map(file => ({
        index: file.file_index !== undefined ? file.file_index : 0,
        file_size: parseInt(file.file_size),
        file_name: file.name
      }));

      console.log(`Submitting ${selectedFileIndices.length} main content files for download`);
      
      // Create a descriptive task name if not provided
      if (!taskName) {
        taskName = `Filtered Download - ${selectedFiles.length} main content files`;
      }

      const result = await this.client.submitTask(magnetLink, taskName, selectedFileIndices);
      return result.message;
    } catch (error) {
      return `Error submitting task: ${error.message}`;
    }
  }

  /**
   * Get all files recursively from the tree structure
   */
  getAllFilesFromTree(resources, parentPath = '') {
    let allFiles = [];

    for (const resource of resources) {
      if (resource.is_dir && resource.dir && resource.dir.resources) {
        // This is a directory, recursively get files
        const childPath = parentPath ? `${parentPath}/${resource.name}` : resource.name;
        allFiles = allFiles.concat(this.getAllFilesFromTree(resource.dir.resources, childPath));
      } else {
        // This is a file
        allFiles.push({
          ...resource,
          fullPath: parentPath ? `${parentPath}/${resource.name}` : resource.name
        });
      }
    }

    return allFiles;
  }

  /**
   * Classify files as main content, ads, or unknown
   */
  classifyFiles(files) {
    const classified = {
      mainContent: [],
      ads: [],
      unknown: [],
      all: []
    };

    for (const file of files) {
      const classification = this.classifySingleFile(file);
      file.type = classification;
      classified.all.push(file);
      
      if (classification === 'main_content') {
        classified.mainContent.push(file);
      } else if (classification === 'ad') {
        classified.ads.push(file);
      } else {
        classified.unknown.push(file);
      }
    }

    return classified;
  }

  /**
   * Classify a single file as main content, ad, or unknown
   */
  classifySingleFile(file) {
    const fileName = file.name.toLowerCase();
    const fileSize = parseInt(file.file_size) || 0;

    // Check for ad indicators in filename
    const adIndicators = [
      'game', '游戏', '996gg', 'ad', 'advert', 'sample', 'preview', 'trailer', 
      'promo', '宣传', 'demo', '试玩', 'hdwallpaper', '壁纸', '合集', '大全', '龙珠H版', '三国志H版'
    ];

    // Check for main content indicators in filename
    const contentIndicators = [
      'hhd800', 'fhd', 'hd', '1080', '720', '@', '#', '.com@', '.net@', '.org@'
    ];

    // Flag if file name contains ad indicators
    const hasAdIndicator = adIndicators.some(indicator => 
      fileName.includes(indicator.toLowerCase())
    );

    // Flag if file name contains content indicators
    const hasContentIndicator = contentIndicators.some(indicator => 
      fileName.includes(indicator.toLowerCase())
    );

    // Small files (under 10MB) are likely ads or samples if they have ad indicators
    if (fileSize < 10 * 1024 * 1024 && hasAdIndicator) {
      return 'ad';
    }

    // Large files with content indicators are likely main content
    if (fileSize > 100 * 1024 * 1024 && hasContentIndicator) {
      return 'main_content';
    }

    // Files with clear ad indicators regardless of size
    if (hasAdIndicator) {
      return 'ad';
    }

    // Large files without ad indicators are likely main content
    if (fileSize > 500 * 1024 * 1024) {
      return 'main_content';
    }

    // Medium-sized files with content indicators
    if (fileSize > 50 * 1024 * 1024 && hasContentIndicator) {
      return 'main_content';
    }

    // Default to unknown
    return 'unknown';
  }

  /**
   * Handle configuration commands
   */
  async handleConfigCommand(args) {
    if (args.length === 0) {
      return 'Please specify a config sub-command. Usage: xunlei config [set|show]';
    }

    const subCmd = args[0].toLowerCase();

    switch (subCmd) {
      case 'set':
        if (args.length < 3) {
          return 'Usage: xunlei config set <host> <port> [ssl]';
        }
        return this.setConfig(args.slice(1));
      case 'show':
        return this.showConfig();
      default:
        return `Unknown config command: ${subCmd}. Use 'xunlei config set' or 'xunlei config show'.`;
    }
  }

  /**
   * Set configuration
   */
  async setConfig(args) {
    if (args.length < 2) {
      return 'Usage: xunlei config set <host> <port> [ssl]';
    }

    const host = args[0];
    const port = parseInt(args[1]);
    const ssl = args[2] ? args[2].toLowerCase() === 'true' : false;

    if (isNaN(port) || port <= 0 || port > 65535) {
      return 'Invalid port number. Port must be between 1 and 65535.';
    }

    // Test the connection first
    const testClient = new XunleiDockerClient();
    testClient.configure(host, port, ssl);

    try {
      const testResult = await testClient.testConnection();
      if (!testResult.success) {
        return `Connection test failed: ${testResult.message}`;
      }

      // If test passes, save the configuration
      this.client.configure(host, port, ssl);
      this.saveConfig({
        host: host,
        port: port,
        ssl: ssl
      });

      return `Configuration saved successfully. Connected to Xunlei service at ${host}:${port}`;
    } catch (error) {
      return `Failed to connect to Xunlei service: ${error.message}`;
    }
  }

  /**
   * Show current configuration
   */
  showConfig() {
    if (!this.client.host || !this.client.port) {
      return 'Xunlei service is not configured. Use "xunlei config set <host> <port>" to configure.';
    }

    return `Current Xunlei configuration:\n- Host: ${this.client.host}\n- Port: ${this.client.port}\n- SSL: ${this.client.ssl}`;
  }

  /**
   * Get completed tasks
   */
  async getCompletedTasks() {
    try {
      const tasks = await this.client.getCompletedTasks();
      
      if (tasks.length === 0) {
        return 'No completed download tasks.';
      }

      let response = `✅ Completed Download Tasks (${tasks.length}):\n\n`;
      // Show only the most recent 10 tasks
      const recentTasks = tasks.slice(0, 10);
      recentTasks.forEach((task, index) => {
        const fileSize = this.formatFileSize(task.file_size);
        const completedTime = new Date(task.created_time).toLocaleString();
        
        response += `${index + 1}. ${task.name}\n`;
        response += `   Size: ${fileSize} | Completed: ${completedTime}\n\n`;
      });

      if (tasks.length > 10) {
        response += `... and ${tasks.length - 10} more tasks`;
      }

      return response;
    } catch (error) {
      return `Error getting completed tasks: ${error.message}`;
    }
  }

  /**
   * Get Xunlei service version
   */
  async getVersion() {
    try {
      const version = await this.client.getServiceVersion();
      return `Xunlei Service Version: ${version}`;
    } catch (error) {
      return `Error getting service version: ${error.message}`;
    }
  }
}

module.exports = XunleiDockerSkill;