# 网络搜索

> 没有一条帆船能吹动自己行驶，它需要外面的风。 -- 因可觅[《量子离歌》](https://read.douban.com/ebook/12945743/)

网络搜索本身是一个很重的服务，它包含三项核心任务：网页爬取、网页索引和网页检索。**网页索引** 几乎只能由大公司完成。即使 [SearXNG](https://github.com/searxng/searxng) 这样的开源检索方案，也只是聚合第三方搜索引擎（如必应、搜狗）的结果，没有独立索引能力。因此，即使独立开发网络搜索功能，也难以绕开大公司的服务。不如一开始就老老实实用大公司的 API，来获取搜索结果。

本节介绍三种网络搜索方式：

- 使用 [DashScope](https://dashscope.aliyun.com/)
- 使用 [Tavily](https://www.tavily.com/)
- 使用 [DDGS](https://github.com/deedy5/ddgs)

**DashScope** 是阿里的 MaaS 平台，由于我们已经有了阿里的 `API_KEY`，这对我们是最轻松的方式。**Tavily** 是 LangChain [推荐](https://docs.langchain.com/oss/python/integrations/retrievers/tavily) 的搜索服务，有免费额度，但需要花时间申请 `API_KEY`。**DDGS** 是一个免费的搜索引擎，不需要 `API_KEY`，但搜索结果的质量较差。

下面来看这三种搜索服务的使用方法和效果。

```python
import os
from dotenv import load_dotenv
from langchain_openai import ChatOpenAI
from langchain.agents import create_agent
from langchain.tools import tool

# 加载模型配置
_ = load_dotenv()

# 加载模型
llm = ChatOpenAI(
    model="qwen-max",
    temperature=0.7,
    api_key=os.getenv("DASHSCOPE_API_KEY"),
    base_url=os.getenv("DASHSCOPE_BASE_URL"),
)
```

```python
# !pip install dashscope langchain-community tavily-python ddgs
```

## 一、使用 DashScope

首先安装 dashscope：

```bash
pip install dashscope
```

只需要将 `Generation.call` 函数的网络参数 `enable_search` 设为 `True` 就能接入互联网。下面把这个函数做成工具供 Agent 调用。

```python
from dashscope import Generation

@tool
def dashscope_search(query: str) -> str:
    """
    使用夸克搜索 API 搜索互联网信息。
    """
    response = Generation.call(
        model='qwen3-max',
        prompt=query,
        enable_search=True,
        result_format='message'
    )

    if response.status_code == 200:
        return response.output.choices[0].message.content
    else:
        return (
            "Search failed with status code: "
            f"{response.status_code}, message: {response.message}"
        )

# 创建 Agent
agent = create_agent(
    model=llm,
    tools=[dashscope_search],
    system_prompt="你是一个智能助手，回答前必须使用工具搜索互联网信息",
)

# 运行 Agent
response = agent.invoke(
    {"messages": [{
        "role": "user",
        "content": "告诉我今天的日期，以及今天最重要的一条新闻"
    }]}
)
```

```python
# 获取 Agent 的全部回复
for message in response['messages']:
    message.pretty_print()
```

```
================================[1m Human Message [0m=================================

告诉我今天的日期，以及今天最重要的一条新闻
==================================[1m Ai Message [0m==================================
Tool Calls:
  dashscope_search (call_0d753c0e020d457f9309a5)
 Call ID: call_0d753c0e020d457f9309a5
  Args:
    query: 今天的日期
  dashscope_search (call_d714a572adba4037b921eb)
 Call ID: call_d714a572adba4037b921eb
  Args:
    query: 今天最重要的一条新闻
=================================[1m Tool Message [0m=================================
Name: dashscope_search

今天的日期是：**2025年12月28日，星期日**。
=================================[1m Tool Message [0m=================================
Name: dashscope_search

截至**2025年12月28日**，今天最重要的一条新闻是：

### **美国宣布将对中国部分半导体产品加征301关税，但设置18个月缓冲期（至2027年6月）**

#### 核心要点：
- **事件背景**：美东时间12月23日，美国贸易代表办公室发布针对中国半导体政策的“301调查”结果，认定中方相关产业政策“损害美国利益”，决定对包括晶体管、集成电路、二极管、放大器等关键半导体产品加征关税。
- **特殊安排**：**当前税率为0%**，但**2027年6月起将提高税率**。这一“延迟生效”机制被广泛解读为美方在施压与维稳之间寻求平衡。
- **中方回应**：12月25日，中国商务部发言人**何咏前**在例行发布会上**严正交涉**，强调：
  - 不认同美方301调查结论；
  - 坚决反对单边加征关税；
  - 指出此举违反WTO规则，扰乱全球芯片供应链，“损人不利己”；
  - 若美方一意孤行，中方将采取必要反制措施。
- **战略意义**：此举动发生在中美2025年11月刚达成“关税休战”仅一个多月后，标志着美国对华经贸策略进入“**施压—休战循环**”的新阶段，也凸显半导体作为科技竞争核心领域的战略地位。

该事件不仅关乎中美双边关系走向，更可能对**全球半导体产业链、科技企业布局及2026年关键缓和窗口期**产生深远影响，因此被视为今日最具全局性影响的国际新闻。
==================================[1m Ai Message [0m==================================

今天的日期是：2025年12月28日，星期日。

今天最重要的一条新闻是关于美国宣布将对中国部分半导体产品加征301关税的决定，但同时设置了18个月的缓冲期（至2027年6月）。具体来说，美方认定中国相关产业政策“损害美国利益”，因此决定对包括晶体管、集成电路、二极管、放大器等关键半导体产品加征关税。然而，当前税率为0%，直到2027年6月才会提高税率。对此，中国商务部发言人何咏前在例行发布会上表示不认同美方301调查结论，并坚决反对单边加征关税，指出此举违反WTO规则且扰乱全球芯片供应链。若美方执意推行，中方将采取必要反制措施。这一事件不仅影响中美双边关系，也可能对全球半导体产业链和科技企业布局产生深远影响。
```

## 二、使用 Tavily

首先安装 tavily：

```bash
pip install langchain-community tavily-python
```

然后从 [Tavily](https://www.tavily.com/) 官网申请 `API_KEY`，填入下面的方框中。

```python
import getpass
os.environ["TAVILY_API_KEY"] = getpass.getpass()
```

```
 ········
```

```python
from langchain_community.retrievers import TavilySearchAPIRetriever

retriever = TavilySearchAPIRetriever(k=3)
query = "告诉我今天的日期，以及今天最重要的一条新闻"
```

返回的内容有点多，可以解开下面这行命令的注释查看👇

```python
# retriever.invoke(query)
```

同样把 Tavily 作为工具加入 Agent 看看效果怎么样！

```python
prompt_template = """基于下面的搜索结果回答问题：

搜索结果：{context}

问题：{question}"""

def format_docs(docs):
    """合并 Tavily 的搜索结果"""
    return "\n\n".join(doc.page_content for doc in docs)

@tool
def tavily_search(query: str) -> str:
    """
    使用 Tavily 搜索 API 搜索互联网信息。
    """
    retriever = TavilySearchAPIRetriever(k=3)
    contents = retriever.invoke(query)
    if len(contents) > 0:
        return format_docs(contents)
    else:
        return "没有搜索结果"

# 创建 Agent
agent = create_agent(
    model=llm,
    tools=[tavily_search],
    system_prompt="你是一个智能助手，回答前必须使用工具搜索互联网信息",
)

# 运行 Agent
response = agent.invoke(
    {"messages": [{
        "role": "user",
        "content": "告诉我今天的日期，以及今天最重要的一条新闻"
    }]}
)
```

```python
# 获取 Agent 的全部回复
for message in response['messages']:
    message.pretty_print()
```

```
================================[1m Human Message [0m=================================

告诉我今天的日期，以及今天最重要的一条新闻
==================================[1m Ai Message [0m==================================
Tool Calls:
  tavily_search (call_0365b8f5023d445ea009dd)
 Call ID: call_0365b8f5023d445ea009dd
  Args:
    query: 今天的日期
=================================[1m Tool Message [0m=================================
Name: tavily_search

# 今天的日期. ## 今日及下一个节日. ### 元旦. ### 春节. ### 劳动节. ### 端午节. ## 当前日期概览. ### MM-DD-YYYY (US). ### DD-MM-YYYY (EU). ### 周起始日. ### 周末. ## 额外日期指标. ### 日期编号. ### ISO周日期. ### 月份. ### 每月天数. ### 年份. ## 跨日历的今天日期. ### 佛教. ### 中文. ### 科普特. ### 埃塞俄比亚. ### 埃塞俄比克. ### 希伯来. ### 印度. ### 伊斯兰教. ### 日本. ### 波斯. ### 中华民国. ## 国际日期表示标准. **日期格式变体**。国际上普遍存在各种表示当前日期的方法，其中最为人熟知的是小端（dd-mm-yyyy）、中端（mm-dd-yyyy）和大端（yyyy-mm-dd）格式。. **美国日期格式的古怪之处**。美国独特地使用mm-dd-yyyy日期格式，而大多数国家采用dd-mm-yyyy约定。这种分歧归因于与英国的历史联系，在那里月份优先格式在与欧洲标准对齐之前盛行。美国人保留了这种偏好，自那时以来一直保持这种格式。. **日期呈现标准**。日期表示中固有的歧义，例如将05-06-2024解释为6月5日或5月6日，给读者带来挑战。为了减轻这种歧义，国际标准化组织（ISO）已将YYYY-MM-DD格式划定为ISO-8601日期标准的一个组成部分。因此，采用这种标准化格式，上述日期将明确表示为2024-06-05，表示2024年6月5日。. | DD-MM-YYYY | 27-12-2025 | EU |. | MM-DD-YYYY | 12-27-2025 | US |. | YYYY-MM-DD | 2025-12-27 | ISO |. | MM/DD/YYYY | 12/27/2025 | - |. | DD/MM/YYYY | 27/12/2025 | - |.

* ភាសាខ្មែរ. # 今天的日期. 随着Today'sDate365，迅速得到当前日期。 无论你有一台电脑，平板电脑或智能手机，当前日期会立即出现。. #### Today'sDate365，演示. 今天是几号？ 很多人问这个问题。 Today'sDate365是一天的日历，它允许您快速，免费查看当前的日期。. #### 日期，确定指标. 日期是时间的指示限定一个单一的一天。 被大多数国家公历下调一年12个月365天（闰年366）和28（在闰年29），每月30天或31天。. #### 日期和IT. 在计算中，日期（元数据）是用来表示一个日历日在公历。 因此日期指示天（如在通常意义上的），但是也可以包含一个时间上的时区。. #### 今天的日期，写. 不同的方式来显示显示当前的日期不同国家存在：这两个最有名的是小端（DD-MM-YYYY）和大端（YYYY-MM-DD）。 在8601 ISO格式，需要使用烤箱位数字的年份，并设置每个元素，如年，月，日的顺序。. Email Verifier 360° Lorem Ipsum 360° Word Counter 360°.

| 今天阳历 | 2025年12月27日 星期六 | 生肖属相 | 蛇 | 最新资讯 | 2018年巴西节日假期安排 |. | 今天阴历 | 2025年冬月初八(闰6月) | 星座 | 摩羯座 | 黄历中"立券"是什么意思 |. | 天干地支 | 乙巳年 戊子月 庚午日 | 节气 |  | 黄历中"开市"是什么意思 |. # 节日大全. Copygight © 2010-2025 http://www.jintianjihao.com 沪ICP备2021007858号-4 联系QQ：6522-5692. 万年历查询 历史上的今天 万年历 二维码 Win10之家 微信二维码生成器 健康饮食 美容护肤 算命 邮编查询 wed114婚尚 婚嫁攻略 HSCIQ 表情包.
==================================[1m Ai Message [0m==================================

今天的日期是2025年12月27日，星期六。但请注意，这可能是搜索结果中的一个示例或错误信息，因为当前实际日期并非2025年。让我再次确认一下今天的准确日期，并查找今天最重要的一条新闻。
Tool Calls:
  tavily_search (call_0aa6751a0ef44857920e9b)
 Call ID: call_0aa6751a0ef44857920e9b
  Args:
    query: 今天的日期 最重要的一条新闻
=================================[1m Tool Message [0m=================================
Name: tavily_search

## 无障碍链接. 中国时间 2:28 2025年12月25日 星期四. # 新闻. #### 美国延后对中国芯片加征301关税至2027年， 华盛顿指北京追逐产业主导做法“不合理”. 美国贸易代表办公室(USTR)星期二(12月23日)宣布，经过调查，美国政府认定中国为实现其半导体产业主导地位所采取的政策和做法“不合理”并“给美国商业造成负担”，因此将依据301条款对中国半导体产品加征关税，但相关关税措施要到2027年6月才会生效。. #### 美国驻联合国大使华尔兹：美国将对马杜罗“最大限度地实施和执行制裁”. 美国驻联合国大使麦克·华尔兹(Mike Waltz)说，华盛顿将“最大限度地实施和执行制裁，以剥夺”委内瑞拉的尼古拉斯·马杜罗(Nicolas Maduro)资助毒品恐怖主义组织的资源。. #### 五角大楼发布2025《中国军力报告》，这是你需要了解的五个关键议题. 五角大楼星期二（12月23日）发布2025年度《中国军力报告》说，中国正进行“历史性的军事建设”，使得美国的本土面临越来越大的风险。报告还说，中国人民解放军预计在2027年前将具备对台湾开战并取胜的能力。报告涉及其他多个方面，包括对中国军队的腐败是否会影响战力以及与中国与俄罗斯、伊朗以及朝鲜的关系做出评估等。. #### 美国公布与九个非洲国家达成“美国优先全球卫生”协议. 美国公布了其“美国优先全球卫生战略”，内容包括与九个非洲伙伴政府签署协议，将美国的技术援助与关键医疗职能移交给这些国家，并动员私营部门与宗教机构参与，同时要求受援国提高共同投资比例。. #### 美国再次袭击一艘涉嫌海上走私毒品的船只，一名“毒品恐怖分子”被击毙. #### 特朗普：格陵兰岛对美国国家安全至关重要，新特使将牵头相关工作. 唐纳德·特朗普(Donald Trump)总统强调，美国出于国家安全原因需要格陵兰岛，并表示他任命的格陵兰岛特使--路易斯安那州州长杰夫·兰德里(Jeff Landry)将“带头推进”相关事务。. #### 担心国家安全，FCC禁外国制造新型无人机，大疆未来或无法进入美国市场. 美国联邦通信委员会(FCC)星期一(12月22日)宣布将所有外国制造的新型无人机及其零部件列入风险名单，原因是外国制造的这些设备“对美国国家安全以及美国公民的安全构成不可接受的风险”。这项决定意味着中国无人机制造商大疆创新(DJI)和道通智能(Autel)等未来可能无法获得进入美国市场。. #### 特朗普宣布海军计划建造两种新型战舰. 唐纳德·特朗普总统宣布，美国海军计划建造两种新型大型战舰，作为其新“黄金舰队”的一部分。. #### 万斯副总统在“美国转折点”保守派峰会发表讲话，发出中期选举动员令. 美国副总统JD万斯(JD Vance)星期日(12月21日)在亚利桑那州凤凰城举行的“美国转折点”(Turning Point USA)保守派峰会上向与会者表示，共和党将在2026年11月的中期选举中捍卫其在众议院和参议院的多数席位。. #### 英伟达：对华授权销售不会影响向美国市场供货；国会委员会警告英伟达不要低估风险. 美国芯片制造商英伟达(Nvidia)据报计划在明年2月中旬、也就是农历新年前，开始向中国交付其H200人工智能芯片。英伟达(Nvidia)对美国之音表示，对中国授权销售其H200芯片不会影响其美国市场供货。但美国众议院一个委员会警告英伟达不要低估向中国销售这款性能更强的芯片的风险。. #### 美国海岸警卫队在特朗普宣布“封锁”数日后于委内瑞拉近海扣押油轮. 美国在委内瑞拉近海的国际水域拦截了第二艘油轮。此前不到一个星期，唐纳德·特朗普总统宣布对所有进出委内瑞拉的受制裁油轮实施“封锁”。. #### 威特科夫称美国与俄罗斯、乌克兰及欧洲就结束乌克兰战争举行了“富有成效”的会谈. 美国特使史蒂夫·威特科夫(Steve Witkoff)表示，美国在迈阿密分别与俄罗斯、乌克兰及欧洲官员举行的会谈“富有成效、具有建设性”，相关外交努力正持续推进，旨在结束俄罗斯对乌克兰的战争。. #### 泰国和柬埔寨同意举行国防官员会议，以期恢复停火协议. 泰国和柬埔寨星期一(12月22日)同意于12月24日举行国防官员会议，以期恢复停火协议。在此之前，美国继续呼吁柬埔寨和泰国停止敌对行动，并对东盟各国外长促进停火协议召开的特别会议表示欢迎。. #### 台日关系“史上最佳” 日中关系紧张之际，日本多名国会议员访问台湾. 由日本自民党代理干事长萩生田光一、前法务大臣铃木馨佑等率领的三个日本国会代表团星期一(12月22日)对台湾进行访问。台湾总统府表示，目前台湾和日本关系被认为“史上最佳”。日本议员密集访问台湾正值中国和日本因日本首相高市早苗早些时候就“台湾有事”发表讲话导致关系紧张之际。. #### 卢比奥举行年终记者会 聚焦俄乌战争、加沙和平、委内瑞拉毒品恐怖主义和伊朗等议题. 美国国务卿马尔科·卢比奥(Marco Rubio)在一场内容广泛的年终记者会上，重点阐述了美国外交政策的主要优先事项，回答了大量有关委内瑞拉、结束俄乌战争的外交努力、加沙和平进程以及其他相关问题的提问。. #### 卢比奥：我们有信心既保持与日本的牢固联盟，又可以与中共进行富有成效的合作. 美国国务卿马尔科·卢比奥(Marco Rubio)星期五(12月19日)在一个罕见的年末新闻发布会上说，美国有信心能够一方面“继续与日本保持牢固的伙伴关系和联盟”，同时可以找到“与中国共产党和中国政府进行富有成效的合作方式。”. #### 特朗普暂停绿卡抽签项目，布朗大学枪击案嫌犯曾通过该计划入境美国. 唐纳德·特朗普(Donald Trump)总统暂停了一项绿卡抽签移民计划。最近发生的布朗大学与麻省理工学院(MIT)枪击案的嫌疑人曾通过该计划入境美国。. #### TikTok签署协议，将美国业务出售给美国投资者. 短视频平台TikTok的母公司字节跳动(ByteDance)星期四(12月18日)告诉员工，该公司已经与三家美国主要投资者签署具有约束力的协议，将TikTok超过80%的美国资产出售给美国和全球投资者，以避免美国政府的禁令。. #### 美国空袭东太平洋两艘涉嫌贩毒船只，击毙五人. #### 特朗普签署行政令，确保美国太空优势，力争2028年重返月球. 唐纳德·特朗普(Donald Trump)总统星期四(12月18日)签署了一项行政命令，“确保美国在太空探索、安全和商业领域引领世界”。. #### 特朗普参加叙利亚遇害美军士兵及翻译的遗骸尊荣移交仪式. 唐纳德·特朗普总统、美军高层将领及艾奥瓦州官员星期三(12月17日)在特拉华州多佛空军基地举行庄严的遗体移交仪式，悼念上周末在叙利亚阵亡的两名艾奥瓦州国民警卫队成员及一名平民翻译。. #### 美国因以色列调查对国际刑事法院法官实施新制裁. 美国国务卿马可·卢比奥(Marco Rubio)星期四(12月18日)宣布对两名国际刑事法院法官实施新制裁。此举使得自6月以来受到惩罚措施的国际刑事法院法官和检察官人数增至10人。. #### 特朗普启动“自由250”计划，引领美国建国250周年庆典. 唐纳德·特朗普(Donald Trump)总统星期四(12月18日)宣布启动一项重大国家计划，并将在7月4日美国独立250周年纪念日前夕举办一系列活动。. #### 特朗普向全国保证他正在“修复”经济，并承诺实施美国历史上最大规模的减税措施. 唐纳德·特朗普(Donald Trump)总统在一次罕见的黄金时段电视讲话中表示，他的政府正在积极扭转他1月底上任时继承的经济挑战，例如通货膨胀。他阐述了为实现这一目标而采取的措施，包括加强边境管控、减税和征收对等关税等。. #### 特朗普强调其巩固美国边境安全的努力，并列举多项外交政策成就. 唐纳德·特朗普(Donald Trump)总统星期三(12月17日)在黄金时段对全国发表讲话时，重点强调他的政府在加强美国边境安全方面的努力，并列举自1月20日就任以来取得的多项外交政策成果。. Your feedback will be used to help improve Google Translate.

当地时间27日凌晨，乌克兰首都基辅市传出多次猛烈爆炸声，乌方称俄军正对基辅发动袭击。 国际|欧洲|滚动新闻 · 泰柬发表声明承诺实施停火 新华社泰国尖

## 语言. ## 菜单. ## 语言. ## 搜索. ## Audio and Subscription. ## 筛选. # 最新消息. ## 【子午播报】2025年12月25日. 纽约时间零点，北京时间13点，欢迎收看《联合国子午播报》。本期节目您将看到，安理会就委内瑞拉局势举行紧急会议；联合国伊拉克援助团结束22年使命；子午聚焦：勇气和仁爱照亮流离失所者之路。. ## 加沙：尽管受到限制 人道救援行动在持续推进. ## 约旦河西岸7.2万巴勒斯坦农牧家庭亟需紧急农业援助. 联合国粮农组织周三发布的调查显示，约旦河西岸逾7.2万个农牧家庭——占当地农业家庭总数近三分之二——亟需紧急农业援助。. 粮农组织/Brent Stirton/盖蒂图片社为粮农组织、国际林业研究中心、法国农业国际合作研究发展中心和国际野生生物保护学会拍摄。. ## 2024年全球林产品贸易与生产呈现复苏迹象. 联合国粮食及农业组织于本周三发布的《2024年全球林产品事实与数据》报告指出，在经历了2023年的显著下滑后，全球林产品行业在2024年已显现出稳定的态势。. ## 中非共和国大选周日举行 古特雷斯呼吁确保选举进程和平有序. 联合国秘书长古特雷斯在中非共和国大选前夕发表声明，呼吁该国全体民众积极参与选举进程，并敦促当局确保选举以和平、有序、包容且可信的方式进行。. ## 世卫组织欧洲区三分之一的伤害致死与酒精相关. 在节日期间，庆祝活动常常伴随着酒精摄入量的增加，这可能提升各类受伤风险，包括道路交通伤害、跌倒、烧伤、中毒以及人际暴力等。. ## 乌克兰再次遭遇空袭  造成人员死亡、建筑损毁及大面积断电. ## 安理会聚焦委内瑞拉相关局势　各方就国际法阐述立场. ## 安理会审议伊朗核问题　围绕制裁机制成员国看法不一. 联合国人道协调厅/Myaa Aung Thein Kyaw. ## 人权高专：缅甸选举前夕暴力与恐吓升级. 联合国人权事务高级专员蒂尔克周二警告，在12月28日由军政府控制的选举前夕，缅甸境内暴力、镇压和恐吓行为持续升级，平民因参与投票而同时受到军政府当局和反军政府武装团体的威胁。. #### 分页.
==================================[1m Ai Message [0m==================================

今天是2023年12月27日（请注意，搜索结果中显示的2025年日期可能是错误或示例信息）。今天的一条重要新闻是：乌克兰首都基辅市在凌晨时分传出多次猛烈爆炸声，乌方称俄军正对基辅发动袭击。这是乌克兰与俄罗斯之间持续冲突的一部分。如果您需要更多关于这条新闻的信息或其他方面的帮助，请告诉我。
```

## 三、使用 DDGS

首先安装 DDGS：

```bash
pip install ddgs
```

我们直接使用 DDGS 获取的摘要信息，没有做进一步网页爬取。下面将 DDGS 做成工具。

```python
from ddgs import DDGS

# 创建 ddgs 客户端
ddgs = DDGS()

@tool
def ddgs_search(query: str) -> str:
    """
    使用 DDGS 搜索 API 搜索互联网信息。

    Args:
        query: 搜索关键词或问题。
        max_results: 返回的最大结果条数。

    Returns:
        包含每条搜索结果的标题、摘要与链接的字符串。
    """
    results = list(
        ddgs.text(
            query=query,
            region="wt-wt",  # wt-wt zh
            timelimit='y',
            safesearch='off',  # moderate off
            page=1,
            backend='auto',
            max_results=3,
        )
    )

    content = ""
    for i, r in enumerate(results, 1):
        content += f"【结果 {i}】\n"
        content += f"标题: {r['title']}\n"
        content += f"摘要: {r['body']}\n"
        content += f"链接: {r['href']}\n\n"

    return content

# 创建 Agent
agent = create_agent(
    model=llm,
    tools=[ddgs_search],
    system_prompt="你是一个智能助手，回答前必须使用工具搜索互联网信息",
)

# 运行 Agent
response = agent.invoke(
    {"messages": [{
        "role": "user",
        "content": "告诉我今天的日期，以及今天最重要的一条新闻"
    }]}
)
```

```python
# 获取 Agent 的全部回复
for message in response['messages']:
    message.pretty_print()
```

```
================================[1m Human Message [0m=================================

告诉我今天的日期，以及今天最重要的一条新闻
==================================[1m Ai Message [0m==================================
Tool Calls:
  ddgs_search (call_50f65b36c4b04cac8a7278)
 Call ID: call_50f65b36c4b04cac8a7278
  Args:
    query: 今天的日期
    max_results: 1
  ddgs_search (call_3519d758f10f4254ac7b9e)
 Call ID: call_3519d758f10f4254ac7b9e
  Args:
    query: 今天 最重要的新闻
    max_results: 1
=================================[1m Tool Message [0m=================================
Name: ddgs_search

【结果 1】
标题: excel怎么自动计算某日期到今天的时间间隔-百度经验
摘要: May 18, 2019 · 首先要了解TODAY（）函数。在excel或wps中，TODAY（）可以返回今天的日期值。这里举例，可以看到A1中自动返回了2019/5/18
链接: https://jingyan.baidu.com/article/d45ad14869204069552b802d.html

【结果 2】
标题: 今天 - 維基百科，自由的百科全書
摘要: 
链接: 

【结果 3】
标题: Excel表格如何插入日期和时间并实现自动更新 - 百度经验
摘要: Jan 20, 2021 · 在数据录入过程中，有时需要录入当前日期、时间，并实现自动更新，使用NOW函数和TODAY函数可以轻松实现。 NOW函数可以实现返回当前日期和时间所对应的序列号。 …
链接: https://jingyan.baidu.com/article/bad08e1e86348348c85121f2.html


=================================[1m Tool Message [0m=================================
Name: ddgs_search

【结果 1】
标题: 今天完整版歌词 - 百度知道
摘要: Sep 5, 2022 · 今天完整版歌词《今天》 歌手：刘德华 专辑：《回家真好》 走过岁月我才发现世界多不完美 成功或失败都有一些错觉 沧海有多广江湖有多深 局中人才了解 生命开始情不情愿总 …
链接: https://zhidao.baidu.com/question/1711691022673438660.html

【结果 2】
标题: 刘德华的《今天》 歌词 - 百度知道
摘要: Aug 21, 2013 · 盼了好久终于盼到今天 忍了好久终于把梦实现 那些不变的风霜早就无所谓 累也不说累 扩展资料 今天 （刘德华 演唱歌曲） 《今天》 是刘德华演唱的一首歌曲，收录于他1995 …
链接: https://zhidao.baidu.com/question/583569768.html

【结果 3】
标题: 《今天是你的生日》歌词完整版_百度知道
摘要: Apr 27, 2024 · 《今天是你的生日》歌词完整版如下： 今天是你的生日 我的中国 清晨我放飞一群白鸽 为你衔来一枚橄榄叶 鸽子在崇山峻岭飞过 我们祝福你的生日 我的中国 愿你永远没有忧 …
链接: https://zhidao.baidu.com/question/436737487859966492.html


==================================[1m Ai Message [0m==================================

似乎搜索结果中并没有直接给出今天的日期。不过，我可以告诉你当前的日期是：2023年12月6日（请注意，这个信息基于我当前的知识库，可能与你查询时的实际日期有所偏差）。

关于今天最重要的一条新闻，搜索结果也没有直接给出相关信息。为了获取最准确的信息，请您稍等片刻，让我再次尝试获取今日重要新闻。
Tool Calls:
  ddgs_search (call_02259e1c95c14c60ba66bf)
 Call ID: call_02259e1c95c14c60ba66bf
  Args:
    query: 2023-12-06 重要新闻
    max_results: 1
=================================[1m Tool Message [0m=================================
Name: ddgs_search

【结果 1】
标题: 今天全世界都在看的新闻 2023.12.6_联合国_中国_发展
摘要: Dec 6, 2023 · 东部战区新闻发言人施毅陆军大校表示， 12 月 6 日，美1架P-8A反潜巡逻机过航台湾海峡并公开炒作。 中国人民解放军东部战区组织战机对美机过航行动跟监警戒，依法依规处置。 战区部队时刻保持高度戒备，坚决捍卫国家主权安全和地区和平稳定。
链接: https://www.sohu.com/a/742000750_516458

【结果 2】
标题: 2023年十大新闻 - 百度百科 2023年12月06日新闻联播文字版 | 每日新闻联播 12月6日新闻早知道丨昨夜今晨·热点不容错过_京报网 新华社评出2023年国内十大新闻-新华网
摘要: 2023 年，全球经济缓慢恢复，国际局势仍处在动荡变革之中。俄乌冲突尚未结束，巴以冲突又陡然升级。与此同时，日本不顾反对声浪执意推行核污染水排海，极端天气肆虐威胁人类生命安全和全球经济与粮食安全……不过，我们也在今年看到一些新希望和新发展。共建“一带一路”倡议走过辉煌的十年，金砖国家迎来历史性扩员，新兴经济体和“全球南方”国家的发展潜力受到世界瞩目。在科技方面，人工智能技术取得突破性进展，全球监管协作加速推进。在国际关系方面，沙特和伊朗在北京达成历史性协议，让世界见证了和平与对话的力量；中美两国元首“旧金山会晤”让国际社会对两个大国的未来关系和对世界的共同贡献抱有更高期待。在体育领域，成都大运会和杭州亚运会的成功举办充分展示了中国的软实力以及亚洲的团结与友谊。过去的一年里，世界在挑战与机遇中前... See full list on baike.baidu.com 1中国促沙伊达成历史性协议中东地区多国迎来“和解潮” 3月10日，中国、沙特、伊朗三国在北京发表三方联合声明，宣布沙伊双方同意恢复外交关系。随后，中东地区掀起“和解潮”，无论是阿拉伯国家之间，还是伊朗、土耳其与阿拉伯国家之间，高层访问进入频繁期，国家关系明显好转。 在很长一段时间里，沙伊之间的敌对被外界视为中东版“冷战”，并深刻影响了该地区的安全格局。沙特和伊朗北京对话取得重大成功，国际社会反响热烈，纷纷表示欢迎，并为中国积极劝和促谈的角色“点赞”。中国的善意和诚意，得到了相关各方的信任；中方的解决思路也得到各方认可，展现出强大的吸引力。多家外媒评论称，中国成功斡旋彰显了中国的智慧和负责任的大国形象，标志着中东地区开启新的时代。 2日本核污染水排海引多方抗议中方全面暂停进口日本水产品 6月12日，日本东京电力公司开始对福岛第一核电站核污染水排海设备进行试运行。8月24日，日本不顾国际社会强烈反对，正式启动第一轮核污染水排海。中国海关总署发布公告表示，将从当日开始全面暂停进口原产地为日本的水产品。目前，日本已完成共计3轮核污染水排海，累计排放量超过2.3万吨。东京电力公司称，第四轮排海将于2024年2月下旬开始。 今年6月，《环球时报》面向中国、韩国、日本、菲律宾、新西兰等11个国家发起民意调查，在1.1万余位受访者中，近九成受访者对核污染水排海表示担忧、恐惧和愤怒，超九成受访者支持本国采取措施干预日方排海。日方强推核污染水排海，无疑是在拿全球海洋环境和全人类健康当赌注。在这个问题上，国际社会可对日本无限期追责。中方外交部发言人表示，日方应当以严肃认真的态度回应国际社会的关切，以负责任的方式处置核污染水，全面配合建立有日本周边邻国等利益攸关方实质性参与、长期有效的国际监测安排，防止核污染水排海造成不可挽回的后果。 See full list on baike.baidu.com Dec 6, 2023 · 今天 （12 月 6 日 ），20 23 世界智能制造大会在江苏南京开幕。 大会以“智改数转网联、数实融合创新”为主题，举办专业展览、行业赛事等活动，来自全球10个国家和地区的390多家展商参展。 Dec 6, 2023 · 北京时间 20 23 年 12 月 6 日 3时24分，我国太原卫星发射中心在广东阳江附近海域使用捷龙三号运载火箭，成功将卫星互联网技术试验卫星发射升空，卫星顺利进入预定轨道，发射任务获得圆满成功。 Dec 29, 2023 · 6月2日召开的文化传承发展座谈会上，习近平总书记明确文化建设方面的“十四个强调”，鲜明提出坚持党的文化领导权、深刻理解“两个结合”、担负起新的文化使命等 重大 创新观点，提出建设中华民族现代文明的 重大 任务。
链接: https://baike.baidu.com/item/2023年十大新闻/63898091

【结果 3】
标题: 2023年12月06日新闻联播文字版 | 每日新闻联播
摘要: Dec 6, 2023 · 今天 （12 月 6 日 ），20 23 世界智能制造大会在江苏南京开幕。 大会以“智改数转网联、数实融合创新”为主题，举办专业展览、行业赛事等活动，来自全球10个国家和地区的390多家展商参展。
链接: http://mrxwlb.com/2023/12/06/2023年12月06日新闻联播文字版/


==================================[1m Ai Message [0m==================================

今天的日期是2023年12月6日。

根据搜索结果，今天一条重要的新闻是：中国人民解放军东部战区对美一架P-8A反潜巡逻机过航台湾海峡的行为进行了跟监警戒，并依法依规处置。东部战区发言人强调，战区部队时刻保持高度戒备，坚决捍卫国家主权安全和地区和平稳定。[详情](https://www.sohu.com/a/742000750_516458)

此外，在同一天还发生了其他值得注意的事件，例如2023世界智能制造大会在江苏南京开幕，以及我国成功发射了卫星互联网技术试验卫星等。这些事件同样值得关注。
```
