"""District9 D9Portal token launcher — Mode B (0x9999 suffix)."""

import json
import os
import time

import requests
from eth_account import Account
from web3 import Web3

from ..config import OpenClawConfig
from ..utils.logger import log
from .constants import (
    AGENT_SHARE_BPS,
    D9_BASE_URL,
    D9_PORTAL_CONTRACTS,
    D9_TOKEN_SUFFIX,
    DISTRICT9_SHARE_BPS,
    DISTRICT9_TREASURY,
    FLAP_UPLOAD_API,
)

# D9Portal ABI — createToken + buy + TokenCreated event
D9_PORTAL_ABI = [
    {
        "inputs": [
            {"name": "name", "type": "string"},
            {"name": "symbol", "type": "string"},
            {"name": "meta", "type": "string"},
            {"name": "salt", "type": "bytes32"},
            {
                "components": [
                    {"name": "recipient", "type": "address"},
                    {"name": "bps", "type": "uint16"},
                ],
                "name": "vaultRecipients",
                "type": "tuple[]",
            },
        ],
        "name": "createToken",
        "outputs": [{"name": "token", "type": "address"}],
        "stateMutability": "nonpayable",
        "type": "function",
    },
    {
        "inputs": [
            {"name": "token", "type": "address"},
            {"name": "minTokensOut", "type": "uint256"},
        ],
        "name": "buy",
        "outputs": [],
        "stateMutability": "payable",
        "type": "function",
    },
    {
        "anonymous": False,
        "inputs": [
            {"indexed": True, "name": "token", "type": "address"},
            {"indexed": True, "name": "vault", "type": "address"},
            {"indexed": True, "name": "creator", "type": "address"},
        ],
        "name": "TokenCreated",
        "type": "event",
    },
]


class District9Launcher:
    """Launch tokens through D9Portal with 0x9999 vanity suffix."""

    def __init__(self, config: OpenClawConfig):
        self.config = config
        self.account = Account.from_key(config.wallet.private_key)

        chain_key = config.chain
        self.chain = D9_PORTAL_CONTRACTS.get(chain_key)
        if not self.chain:
            raise ValueError(f"District9 Portal not available on: {chain_key}")

        self.w3 = Web3(Web3.HTTPProvider(self.chain["rpc"]))
        if not self.w3.is_connected():
            raise ConnectionError(f"Cannot connect to {self.chain['rpc']}")

    def launch(self, metadata: dict, image_path: str = "") -> dict:
        """
        Launch a token through D9Portal.

        Flow: Find salt (9999) → Upload IPFS → createToken → initial buy.
        Same interface as FlapLauncher.launch().
        """
        # Step 1: Find CREATE2 salt (9999 suffix, deployer = D9Portal)
        token_impl = self.chain["tax_token_v1_impl"]
        d9_portal = self.chain["d9_portal"]
        salt, predicted_addr = self._find_salt(token_impl, d9_portal)

        # Step 2: Set website URL and upload to IPFS
        metadata["website"] = f"{D9_BASE_URL}/token/{predicted_addr}"
        cid = self._upload_to_ipfs(metadata, image_path)

        # Step 3: Create token
        result = self._send_create_tx(metadata, cid, salt)

        # Step 4: Initial buy (if configured)
        token_addr = result["contract_address"]
        quote_amt = self.w3.to_wei(float(self.config.launch.initial_buy), "ether")
        if quote_amt > 0:
            self._send_buy_tx(token_addr, quote_amt)

        # Add convenience URLs
        explorer = self.chain["explorer"]
        result.update({
            "predicted_address": predicted_addr,
            "ipfs_cid": cid,
            "explorer_tx": f"{explorer}/tx/{result['tx_hash']}",
            "explorer_token": f"{explorer}/token/{token_addr}",
            "d9_token_url": f"{D9_BASE_URL}/token/{token_addr}",
            "d9_agent_url": f"{D9_BASE_URL}/agent/{self.account.address}",
        })

        return result

    def _upload_to_ipfs(self, metadata: dict, image_path: str = "") -> str:
        """Upload token metadata to Flap's IPFS via GraphQL mutation."""
        mutation = """
        mutation Create($file: Upload!, $meta: MetadataInput!) {
          create(file: $file, meta: $meta)
        }
        """

        meta = {
            "website": metadata.get("website", ""),
            "twitter": metadata.get("twitter") or None,
            "telegram": metadata.get("telegram") or None,
            "description": metadata.get("description", ""),
            "creator": "0x0000000000000000000000000000000000000000",
        }

        operations = json.dumps({
            "query": mutation,
            "variables": {"file": None, "meta": meta},
        })
        mapping = json.dumps({"0": ["variables.file"]})

        if image_path and os.path.exists(image_path):
            with open(image_path, "rb") as f:
                image_data = f.read()
            filename = os.path.basename(image_path)
        else:
            import base64
            png_b64 = (
                "iVBORw0KGgoAAAANSUhEUgAAAAEAAAABCAYAAAAfFcSJAAAADUlEQVR4"
                "2mP8/58BAwAI/AL+hc2rNAAAAABJRU5ErkJggg=="
            )
            image_data = base64.b64decode(png_b64)
            filename = "logo.png"

        files = {
            "operations": (None, operations, "application/json"),
            "map": (None, mapping, "application/json"),
            "0": (filename, image_data, "image/png"),
        }

        log.info("Uploading metadata to IPFS...")
        resp = requests.post(FLAP_UPLOAD_API, files=files, timeout=30)

        if resp.status_code != 200:
            raise RuntimeError(f"IPFS upload failed: {resp.status_code} {resp.text}")

        data = resp.json()
        if "errors" in data:
            raise RuntimeError(f"GraphQL errors: {data['errors']}")

        cid = data["data"]["create"]
        log.info(f"Metadata uploaded: {cid}")
        return cid

    def _find_salt(self, token_impl: str, deployer: str) -> tuple[bytes, str]:
        """Find CREATE2 salt for vanity address (9999 suffix)."""
        impl_hex = token_impl[2:].lower()

        # EIP-1167 minimal proxy bytecode
        bytecode_hex = (
            "3d602d80600a3d3981f3"
            "363d3d373d3d3d363d73"
            + impl_hex
            + "5af43d82803e903d91602b57fd5bf3"
        )
        bytecode = bytes.fromhex(bytecode_hex)
        bytecode_hash = Web3.keccak(bytecode)
        deployer_bytes = bytes.fromhex(deployer[2:].lower())

        log.info("Finding CREATE2 salt (9999 suffix)...")
        seed = Account.create().key
        salt = Web3.keccak(seed)
        iterations = 0
        start = time.time()

        while True:
            data = b"\xff" + deployer_bytes + salt + bytecode_hash
            addr_hash = Web3.keccak(data)
            addr = Web3.to_checksum_address(addr_hash[-20:].hex())

            if addr.lower().endswith(D9_TOKEN_SUFFIX):
                elapsed = time.time() - start
                log.info(f"Salt found in {iterations} iterations ({elapsed:.1f}s): {addr}")
                return salt, addr

            salt = Web3.keccak(salt)
            iterations += 1

    def _send_create_tx(self, metadata: dict, cid: str, salt: bytes) -> dict:
        """Build, sign, and send the D9Portal.createToken transaction."""
        d9_portal_addr = self.chain["d9_portal"]
        portal = self.w3.eth.contract(
            address=Web3.to_checksum_address(d9_portal_addr),
            abi=D9_PORTAL_ABI,
        )

        # SplitVault recipients: 50% DISTRICT9 + 50% Agent
        recipients = [
            (Web3.to_checksum_address(DISTRICT9_TREASURY), DISTRICT9_SHARE_BPS),
            (Web3.to_checksum_address(self.account.address), AGENT_SHARE_BPS),
        ]

        wallet = self.account.address
        nonce = self.w3.eth.get_transaction_count(wallet)
        balance = self.w3.eth.get_balance(wallet)

        log.info(f"Wallet: {wallet}")
        log.info(f"Balance: {self.w3.from_wei(balance, 'ether')} BNB")
        log.info(f"Split Vault: {DISTRICT9_SHARE_BPS}bps DISTRICT9 + {AGENT_SHARE_BPS}bps Agent")

        # Estimate gas
        try:
            gas_est = portal.functions.createToken(
                metadata["name"], metadata["symbol"], cid, salt, recipients
            ).estimate_gas({"from": wallet})
            gas_limit = int(gas_est * 1.3)
        except Exception as e:
            log.warning(f"Gas estimation failed ({e}), using fallback")
            gas_limit = 3_000_000

        tx = portal.functions.createToken(
            metadata["name"], metadata["symbol"], cid, salt, recipients
        ).build_transaction({
            "from": wallet,
            "value": 0,
            "gas": gas_limit,
            "gasPrice": self.w3.eth.gas_price,
            "nonce": nonce,
            "chainId": self.w3.eth.chain_id,
        })

        log.info("Signing and sending createToken()...")
        signed = self.w3.eth.account.sign_transaction(tx, self.account.key)
        tx_hash = self.w3.eth.send_raw_transaction(signed.raw_transaction)
        log.info(f"TX sent: {tx_hash.hex()}")

        log.info("Waiting for confirmation...")
        receipt = self.w3.eth.wait_for_transaction_receipt(tx_hash, timeout=120)

        if receipt["status"] != 1:
            raise RuntimeError(f"Transaction reverted! TX: {tx_hash.hex()}")

        log.info(f"Confirmed in block {receipt['blockNumber']} (gas: {receipt['gasUsed']})")

        # Parse TokenCreated event
        token_address = None
        try:
            logs = portal.events.TokenCreated().process_receipt(receipt)
            if logs:
                token_address = logs[0]["args"]["token"]
        except Exception:
            pass

        if not token_address:
            token_address = "unknown"
            log.warning("Could not parse token address from event logs")

        return {
            "contract_address": token_address,
            "tx_hash": tx_hash.hex(),
            "block_number": receipt["blockNumber"],
            "gas_used": receipt["gasUsed"],
        }

    def _send_buy_tx(self, token_addr: str, value: int):
        """Send initial buy transaction on the bonding curve."""
        d9_portal_addr = self.chain["d9_portal"]
        portal = self.w3.eth.contract(
            address=Web3.to_checksum_address(d9_portal_addr),
            abi=D9_PORTAL_ABI,
        )

        wallet = self.account.address
        nonce = self.w3.eth.get_transaction_count(wallet)

        log.info(f"Initial buy: {self.w3.from_wei(value, 'ether')} BNB...")

        try:
            gas_est = portal.functions.buy(
                Web3.to_checksum_address(token_addr), 0
            ).estimate_gas({"from": wallet, "value": value})
            gas_limit = int(gas_est * 1.3)
        except Exception as e:
            log.warning(f"Buy gas estimation failed ({e}), using 500K")
            gas_limit = 500_000

        tx = portal.functions.buy(
            Web3.to_checksum_address(token_addr), 0
        ).build_transaction({
            "from": wallet,
            "value": value,
            "gas": gas_limit,
            "gasPrice": self.w3.eth.gas_price,
            "nonce": nonce,
            "chainId": self.w3.eth.chain_id,
        })

        signed = self.w3.eth.account.sign_transaction(tx, self.account.key)
        tx_hash = self.w3.eth.send_raw_transaction(signed.raw_transaction)
        log.info(f"Buy TX sent: {tx_hash.hex()}")

        receipt = self.w3.eth.wait_for_transaction_receipt(tx_hash, timeout=120)
        if receipt["status"] != 1:
            raise RuntimeError(f"Buy TX reverted! {tx_hash.hex()}")

        log.info(f"Buy confirmed in block {receipt['blockNumber']} (gas: {receipt['gasUsed']})")
