---
name: distinctive-design-systems
model: reasoning
description: Patterns for creating design systems with personality and distinctive aesthetics. Covers aesthetic documentation, color token architecture, typography systems, layered surfaces, and motion. Use when building design systems that go beyond generic templates. Triggers on design system, design tokens, aesthetic, color palette, typography, CSS variables, tailwind config.
---

# Distinctive Design Systems

Create design systems with personality that users remember. Move beyond generic templates to build cohesive, emotionally resonant visual languages.

---

## When to Use

- Building a new product that needs distinctive visual identity
- Creating design tokens for Tailwind + CSS variables
- Documenting aesthetic decisions for consistent implementation
- Want to go beyond default Bootstrap/Tailwind aesthetics

---

## Core Philosophy

A distinctive design system starts with **aesthetic documentation**, not color pickers.

```
1. Define the vibe      → What does this look and feel like?
2. Gather references    → Mood boards, inspiration, examples
3. Document emotions    → What should users feel?
4. Extract tokens       → Colors, typography, spacing, motion
5. Build components     → Implement the documented vision
```

---

## Pattern 1: Aesthetic Foundation

Document the vibe before writing CSS:

### Example: Retro-Futuristic

**The Vibe**: Crystalline, luminescent, slightly melancholic—hopeful hues tempered by muted gradients, sharp typography, and CRT textures. Everything references a primary "Crystal" cyan tone.

**Inspirations**:
- Retro console boot sequences — Futuristic orderly menus
- JRPG UI panels — Luminous data displays, overlay HUDs
- Sci-fi terminal interfaces — Elegant decay, monospaced readouts

| Emotion | How It's Achieved |
|---------|-------------------|
| Precision | Sharp typography, tabular numerics, grid patterns |
| Nostalgia | CRT scanlines, pixel grain, retro-era color palette |
| Hope | Floating cyan orbs, gentle animations, luminous accents |
| Melancholy | Dark gradients, muted backgrounds, soft focus layers |

### Example: Warm Neutral Cyberpunk

**The Vibe**: Warm neutral cyberpunk with a terminal feel. Unlike harsh green-on-black hacker aesthetics, uses warm tan/beige as the foundation, creating approachable yet technical atmosphere.

**Key Differentiation**: Most dark UIs go cold with neon accents. This approach uses warmth as its secret weapon—the neutral tan base creates visual comfort while emerald accents maintain the futuristic aesthetic.

| Emotion | How It's Achieved |
|---------|-------------------|
| Technical credibility | Terminal typography, mono fonts, glow effects |
| Approachability | Warm neutral base instead of cold black |
| Premium quality | Glass panels, backdrop blur, layered shadows |
| Futuristic trust | Circuit patterns, hex grids, scanlines |

---

## Pattern 2: Color Token Architecture

### The Three-Layer System

```
CSS Variables (source of truth)
    ↓
Tailwind Config (utility classes)
    ↓
TypeScript Tokens (runtime access)
```

### CSS Variables

```css
:root {
  /* Base tones (use in rgba()) */
  --tone-void: 2, 7, 18;
  --tone-midnight: 6, 12, 32;
  --tone-cyan: 76, 204, 255;
  
  /* Semantic colors (HSL) */
  --primary: 216 90% 68%;
  --success: 154 80% 60%;
  --destructive: 346 80% 62%;
  
  /* Effect variables */
  --glow-primary: 216 90% 68%;
  --glass-bg: 33 18% 71% / 0.8;
}
```

### Tailwind Config

```typescript
// tailwind.config.ts
export default {
  theme: {
    extend: {
      colors: {
        background: 'hsl(var(--background))',
        foreground: 'hsl(var(--foreground))',
        primary: {
          DEFAULT: 'hsl(var(--primary))',
          foreground: 'hsl(var(--primary-foreground))',
        },
        // Tone palette for rgba usage
        tone: {
          void: 'rgb(var(--tone-void))',
          cyan: 'rgb(var(--tone-cyan))',
        },
      },
    },
  },
};
```

### TypeScript Tokens

```typescript
// styles/design-tokens.ts
export const colors = {
  primary: 'hsl(var(--primary))',
  success: 'hsl(var(--success))',
  // For rgba usage
  toneCyan: 'rgb(var(--tone-cyan))',
};

export const withOpacity = (token: string, opacity: number) =>
  token.replace('rgb(', 'rgba(').replace(')', `, ${opacity})`);
```

---

## Pattern 3: Typography System

### Font Stack

```typescript
fonts: {
  display: ['Orbitron', 'system-ui'],     // Headings, labels
  mono: ['Share Tech Mono', 'monospace'], // Metrics, code
  sans: ['Inter', 'system-ui'],           // Body fallback
}
```

### Type Scale with Multiplier

```css
:root {
  --typo-scale: 0.88;  /* Responsive multiplier */
  
  --typo-page-title: calc(1.75rem * var(--typo-scale));
  --typo-section-title: calc(1rem * var(--typo-scale));
  --typo-metric-lg: calc(1.75rem * var(--typo-scale));
  --typo-metric-md: calc(0.96rem * var(--typo-scale));
  --typo-body: calc(0.9rem * var(--typo-scale));
}

@media (min-width: 640px) {
  :root { --typo-scale: 1; }
}
```

### Typography Patterns

**Magazine-Style Numbers**:
```css
.metric {
  font-weight: 800;
  letter-spacing: -0.02em;
  font-variant-numeric: tabular-nums;
}
```

**Labels**:
```css
.label {
  text-transform: uppercase;
  letter-spacing: 0.1em;
  font-weight: 700;
  font-size: 0.72rem;
}
```

---

## Pattern 4: Layered Surface System

### Layer Hierarchy

| Layer | Name | Purpose |
|-------|------|---------|
| 0 | Ambient | Full-viewport gradients, slow motion |
| 1 | Glow Field | Floating orbs, atmospheric effects |
| 2 | Texture | CRT scanlines, grain, filters |
| 3 | Panels | Elevated cards, section headers |
| 4 | Content | Metrics, charts, tables |
| 5 | Details | Nested sub-panels, rows |

### Surface Component

```tsx
interface SurfaceProps {
  layer?: 'panel' | 'tile' | 'chip' | 'deep' | 'metric';
  children: React.ReactNode;
}

export function Surface({ layer = 'tile', children }: SurfaceProps) {
  return (
    <div className={cn(
      'rounded-lg backdrop-blur-sm',
      layerStyles[layer]
    )}>
      {children}
    </div>
  );
}

const layerStyles = {
  panel: 'bg-tone-cadet/40 border border-tone-jordy/10',
  tile: 'bg-tone-midnight/60 border border-tone-jordy/5',
  chip: 'bg-tone-cyan/10 border border-tone-cyan/20',
  deep: 'bg-tone-void/80',
  metric: 'bg-tone-cadet/20',
};
```

---

## Pattern 5: Motion Tokens

### Timing Scale

```css
:root {
  --transition-fast: 0.15s;
  --transition-default: 0.2s;
  --transition-medium: 0.25s;
  --transition-slow: 0.3s;
}
```

### Animation Patterns

```javascript
// tailwind.config.ts
keyframes: {
  'shimmer': {
    '0%': { backgroundPosition: '200% 0' },
    '100%': { backgroundPosition: '-200% 0' },
  },
  'pulse-glow': {
    '0%, 100%': { opacity: '1', transform: 'scale(1)' },
    '50%': { opacity: '0.5', transform: 'scale(1.05)' },
  },
  'slide-in': {
    '0%': { opacity: '0', transform: 'translateY(10px)' },
    '100%': { opacity: '1', transform: 'translateY(0)' },
  },
  'value-flash': {
    '0%': { textShadow: '0 0 8px currentColor' },
    '100%': { textShadow: 'none' },
  },
},
animation: {
  'shimmer': 'shimmer 1.5s ease-in-out infinite',
  'pulse-glow': 'pulse-glow 1.8s ease-in-out infinite',
  'slide-in': 'slide-in 0.2s ease-out',
  'value-flash': 'value-flash 0.6s ease-out',
}
```

---

## Pattern 6: Glass & Glow Effects

### Glass Panel

```css
.glass-panel {
  background: linear-gradient(180deg, 
    hsl(var(--glass-bg) / 0.95) 0%, 
    hsl(var(--glass-bg) / 0.85) 100%
  );
  backdrop-filter: blur(16px);
  border: 1px solid hsl(var(--glass-border));
  box-shadow: 
    0 4px 16px hsl(var(--glass-shadow)),
    0 0 0 1px hsl(var(--glass-border) / 0.6) inset,
    0 0 20px hsl(var(--glow-primary) / 0.1);
}
```

### Neon Border

```css
.neon-border {
  border: 1px solid hsl(var(--brand-600) / 0.4);
  box-shadow: 
    0 0 10px hsl(var(--glow-primary) / 0.3),
    0 0 20px hsl(var(--glow-primary) / 0.2),
    inset 0 0 10px hsl(var(--glow-primary) / 0.1);
}
```

---

## Proven Aesthetic Directions

| Aesthetic | Inspirations | Emotions |
|-----------|--------------|----------|
| Retro-futuristic glassmorphism | Retro console UIs, JRPG HUDs, sci-fi terminals | Precision, nostalgia, hope |
| Warm neutral cyberpunk | Terminal UIs, sci-fi film interfaces | Credibility, approachability |
| Magazine-style financial | Trading platforms, data dashboards | Trust, clarity, sophistication |

---

## Related Skills

- **Meta-skill:** [ai/skills/meta/design-system-creation/](../../meta/design-system-creation/) — Complete design system workflow
- [design-system-components](../design-system-components/) — CVA component patterns and Surface primitives
- [loading-state-patterns](../loading-state-patterns/) — Skeleton loaders matching your aesthetic

---

## NEVER Do

- **Use pure black (#000) as base** — Always use tinted blacks
- **Use pure white (#fff) for text** — Use tinted whites
- **Use Inter/Roboto for headings** — Display fonts create distinctiveness
- **Use default Tailwind colors** — Define your own palette
- **Skip backdrop-filter blur** — Glass panels need blur
- **Apply decorative patterns to readable content** — Background only
- **Use high-saturation colors without opacity** — Modulate with rgba()
- **Hardcode font sizes** — Use tokens with scale multiplier
- **Skip the aesthetic documentation** — Vibes before code

---

## File Structure

```
styles/
├── globals.css          # CSS variables, base styles
├── design-tokens.ts     # TypeScript token exports
├── theme.css            # Component patterns
└── patterns/
    ├── glass.css
    ├── neon.css
    └── backgrounds.css

tailwind.config.ts       # Token integration
```

---

## Quick Reference

```css
/* 1. Define CSS variables */
:root {
  --tone-primary: 76, 204, 255;
  --primary: 200 90% 65%;
}

/* 2. Configure Tailwind */
colors: {
  primary: 'hsl(var(--primary))',
  tone: { primary: 'rgb(var(--tone-primary))' },
}

/* 3. Use in components */
<div className="bg-primary text-tone-primary/80">
```
