---
name: Developer
description: Write clean, maintainable code with debugging, testing, and architectural best practices.
metadata: {"clawdbot":{"emoji":"💻","os":["linux","darwin","win32"]}}
---

# Software Development Rules

## Code Quality
- Readable code beats clever code — you'll read it 10x more than write it
- Functions do one thing — if you need "and" to describe it, split it
- Name things by what they do, not how — implementation changes, purpose doesn't
- Delete dead code — version control remembers, codebase shouldn't carry weight
- Consistent style matters more than which style — match the project

## Debugging
- Read the error message completely — the answer is often in there
- Reproduce before fixing — if you can't trigger it, you can't verify the fix
- Binary search: comment out half the code to find the problem half
- Check the obvious first — typos, wrong file, stale cache, wrong environment
- Print/log liberally when stuck — assumptions are usually wrong

## Testing
- Test behavior, not implementation — tests shouldn't break when you refactor
- One assertion per test when possible — failures point to exact problem
- Name tests as sentences describing expected behavior — readable test names are documentation
- Mock external dependencies, not internal logic — integration points are boundaries
- Fast tests run often, slow tests get skipped — optimize for feedback speed

## Error Handling
- Fail fast and loud — silent failures create debugging nightmares
- Catch specific exceptions, not generic — different errors need different handling
- Log enough context to debug — error type alone isn't enough
- User-facing errors should be helpful — "something went wrong" helps nobody
- Don't catch exceptions you can't handle — let them bubble up

## Architecture
- Start simple, add complexity when needed — premature abstraction wastes time
- Separate concerns — UI, business logic, data access are different responsibilities
- Dependencies flow inward — core logic shouldn't know about frameworks
- Configuration separate from code — environment-specific values externalized
- Document decisions, not just code — why matters more than what

## Code Review
- Review for understanding, not just correctness — if you can't follow it, others won't
- Ask questions instead of making demands — "what if..." opens discussion
- Small PRs get better reviews — 500 lines gets skimmed, 50 lines gets read
- Approve when good enough, not perfect — progress beats perfection
- Catch bugs early, style issues are secondary — priorities matter

## Performance
- Measure before optimizing — intuition about bottlenecks is usually wrong
- Optimize the hot path — 90% of time is spent in 10% of code
- Database queries are usually the bottleneck — check there first
- Caching solves many problems — but cache invalidation creates new ones
- Premature optimization wastes time — make it work, then make it fast

## Dependencies
- Evaluate before adding — every dependency is code you don't control
- Pin versions — "latest" breaks builds unpredictably
- Check maintenance status — abandoned packages become security risks
- Fewer dependencies is better — each one adds supply chain risk
- Read changelogs before upgrading — breaking changes hide in minor versions

## Working in Existing Codebases
- Match existing patterns — consistency beats personal preference
- Improve incrementally — boy scout rule, leave it better than you found it
- Understand before changing — read the tests, check git history
- Don't refactor while fixing bugs — separate commits, separate PRs
- Legacy code works — respect the battle scars

## Communication
- Commit messages explain why, not what — diff shows what changed
- Document surprising behavior — future developers need context
- Ask before large refactors — alignment prevents wasted work
- Estimate with ranges, not points — "2-4 days" is more honest than "3 days"
- Say "I don't know" when you don't — guessing wastes everyone's time
