---
name: design-system-creation
model: reasoning
description: Complete workflow for creating distinctive design systems from scratch. Orchestrates aesthetic documentation, token architecture, components, and motion. Use when starting a new design system or refactoring an existing one. Triggers on create design system, design tokens, design system setup, visual identity, theming.
---

# Design System Creation (Meta-Skill)

Complete workflow for creating distinctive design systems with personality.


## Installation

### OpenClaw / Moltbot / Clawbot

```bash
npx clawhub@latest install design-system-creation
```


---

## When to Use

- Starting a new product that needs visual identity
- Refactoring an existing design with scattered styles
- Building a component library with design tokens
- Want to go beyond generic Tailwind/Bootstrap aesthetics

---

## Workflow Overview

```
1. Aesthetic Foundation   → Document the vibe before code
2. Color Token System     → CSS variables + Tailwind + TypeScript
3. Typography System      → Font stack + scale + patterns
4. Surface Components     → Layered primitives with CVA
5. Motion Tokens          → Consistent animation timing
6. Loading States         → Skeleton + shimmer patterns
```

---

## Step 1: Aesthetic Foundation

**Read:** `ai/skills/design-systems/distinctive-design-systems`

Before writing CSS, document the aesthetic:

```markdown
## The Vibe
[1-2 sentences describing the visual feel]

## Inspirations
- [Reference 1 - what to take from it]
- [Reference 2 - what to take from it]

## Emotions to Evoke
| Emotion | How It's Achieved |
|---------|-------------------|
| [X] | [specific technique] |
| [Y] | [specific technique] |
```

### Proven Directions to Consider

| Aesthetic | Characteristics |
|-----------|----------------|
| Retro-futuristic | CRT textures, glow effects, monospace fonts |
| Warm cyberpunk | Tan/beige base, emerald accents, glass panels |
| Magazine financial | Bold typography, dark theme, gradient text |

---

## Step 2: Color Token Architecture

**Read:** `ai/skills/design-systems/distinctive-design-systems`

Create the three-layer token system:

```css
/* 1. CSS Variables (source of truth) */
:root {
  --tone-primary: 76, 204, 255;
  --primary: 200 90% 65%;
  --success: 154 80% 60%;
  --destructive: 346 80% 62%;
}
```

```typescript
// 2. Tailwind Config
colors: {
  primary: 'hsl(var(--primary))',
  tone: { primary: 'rgb(var(--tone-primary))' },
}
```

```typescript
// 3. TypeScript Tokens (optional)
export const colors = {
  primary: 'hsl(var(--primary))',
};
```

---

## Step 3: Typography System

**Read:** `ai/skills/design-systems/distinctive-design-systems`

Define fonts and scale:

```css
:root {
  --font-display: 'Orbitron', system-ui;
  --font-mono: 'Share Tech Mono', monospace;
  --font-sans: 'Inter', system-ui;
  
  --typo-scale: 0.88;  /* Mobile */
}

@media (min-width: 640px) {
  :root { --typo-scale: 1; }
}
```

### Typography Patterns

```css
/* Magazine-style numbers */
.metric { font-weight: 800; letter-spacing: -0.02em; font-variant-numeric: tabular-nums; }

/* Labels */
.label { text-transform: uppercase; letter-spacing: 0.1em; font-weight: 700; }
```

---

## Step 4: Surface Components

**Read:** `ai/skills/design-systems/design-system-components`

Build layered surface primitives with CVA:

```tsx
const surfaceVariants = cva(
  'rounded-lg backdrop-blur-sm transition-colors',
  {
    variants: {
      layer: {
        panel: 'bg-tone-cadet/40 border border-tone-jordy/10',
        tile: 'bg-tone-midnight/60 border border-tone-jordy/5',
        chip: 'bg-tone-cyan/10 border border-tone-cyan/20 rounded-full',
      },
    },
  }
);

export function Surface({ layer, children }: SurfaceProps) {
  return <div className={surfaceVariants({ layer })}>{children}</div>;
}
```

---

## Step 5: Motion Tokens

**Read:** `ai/skills/design-systems/distinctive-design-systems`

Define consistent animation timing:

```javascript
// tailwind.config.ts
keyframes: {
  'shimmer': { '0%': { backgroundPosition: '200% 0' }, '100%': { backgroundPosition: '-200% 0' } },
  'pulse-glow': { '0%, 100%': { opacity: '1' }, '50%': { opacity: '0.5' } },
  'slide-in': { '0%': { opacity: '0', transform: 'translateY(10px)' }, '100%': { opacity: '1', transform: 'translateY(0)' } },
},
animation: {
  'shimmer': 'shimmer 1.5s ease-in-out infinite',
  'pulse-glow': 'pulse-glow 1.8s ease-in-out infinite',
  'slide-in': 'slide-in 0.2s ease-out',
}
```

---

## Step 6: Loading States

**Read:** `ai/skills/design-systems/loading-state-patterns`

Create skeleton components that match your aesthetic:

```tsx
export function Skeleton({ className }: { className?: string }) {
  return (
    <div
      className={cn(
        'rounded-md bg-muted animate-shimmer',
        'bg-gradient-to-r from-muted via-muted-foreground/10 to-muted bg-[length:200%_100%]',
        className
      )}
    />
  );
}
```

---

## Component Skills Reference

| Skill | Purpose |
|-------|---------|
| `distinctive-design-systems` | Aesthetic foundation, tokens |
| `design-system-components` | Surface primitives, CVA |
| `animated-financial-display` | Number animations |
| `loading-state-patterns` | Skeletons, shimmer |
| `financial-data-visualization` | Chart theming |

---

## File Structure

```
styles/
├── globals.css          # CSS variables, base styles
├── design-tokens.ts     # TypeScript exports
└── theme.css            # Component patterns

tailwind.config.ts       # Token integration

components/
├── ui/
│   ├── surface.tsx      # Surface primitive
│   ├── skeleton.tsx     # Loading states
│   └── button.tsx       # Variant components
```

---

## NEVER Do

- **Never start with code before documenting aesthetic** — Vibes before CSS
- **Never use pure black (#000) as base** — Always use tinted blacks for depth
- **Never use pure white (#fff) for text** — Use tinted whites that match the palette
- **Never skip design tokens in favor of hardcoded values** — Tokens prevent drift
- **Never create components without variant system** — Use CVA or similar for consistency
- **Never use Inter/Roboto for headings** — Display fonts create distinctiveness
- **Never use default Tailwind colors** — Define your own palette
- **Never skip backdrop-filter blur on glass** — Glass panels need blur to work
- **Never apply decorative patterns to readable content** — Background decoration only
- **Never use high-saturation colors without opacity** — Modulate with rgba()

---

## Checklist

- [ ] Document aesthetic foundation (vibe, inspirations, emotions)
- [ ] Create color token system (CSS + Tailwind + TS)
- [ ] Define typography stack and scale
- [ ] Build Surface primitive component
- [ ] Add motion tokens and animations
- [ ] Create loading state components
- [ ] Document anti-patterns (what NOT to do)
