---
name: design-integration
description: >-
  Design a Uniswap integration architecture. Use when user is building a
  project that needs to integrate Uniswap and wants recommendations on
  integration method (Trading API vs SDK vs direct contract), architecture
  patterns, required dependencies, and security considerations.
allowed-tools: >-
  Read, Write, Edit, Glob, Grep,
  Task(subagent_type:integration-architect),
  mcp__uniswap__get_supported_chains
model: opus
---

# Design Integration

## Overview

Designs a complete Uniswap integration architecture for any project by delegating to the `integration-architect` agent. Produces a comprehensive blueprint covering: integration method recommendation (Trading API vs Universal Router SDK vs direct contract vs V4 hooks), component architecture, dependency list with versions, security review with mitigations, and ordered implementation plan.

## When to Use

Activate when the user asks:

- "Help me integrate Uniswap"
- "Design a swap integration"
- "Architecture for a DEX aggregator"
- "How to build an arbitrage bot"
- "Uniswap integration plan"
- "How should I add swaps to my dApp?"
- "Best way to integrate Uniswap in my backend"
- "Architecture for a DeFi protocol that uses Uniswap"
- "What SDK should I use for Uniswap?"

## Parameters

| Parameter | Required | Default | Description |
| --- | --- | --- | --- |
| projectType | Yes | -- | Type of project (e.g., "DeFi dashboard", "arb bot", "wallet app", "DeFi protocol") |
| functionality | Yes | -- | Required Uniswap functionality: "swap", "LP", "both", or "custom" |
| environment | No | Auto-detect | Execution environment: "frontend", "backend", "smart contract", or "full-stack" |
| chains | No | ethereum | Target chain(s): single chain name or comma-separated list |
| scale | No | -- | Expected scale: transaction volume, concurrent users, latency requirements |

## Workflow

1. **Extract parameters** from the user's request: identify the project type, required functionality, target environment, chains, and scale requirements. If the user has an existing codebase, note it for the agent to analyze.

2. **Delegate to integration-architect**: Invoke `Task(subagent_type:integration-architect)` with the full context. The integration-architect will:
   - Understand the project context and requirements
   - Evaluate integration methods and recommend primary + fallback
   - Design the component architecture with data flow
   - Identify all dependencies (NPM packages, APIs, infrastructure)
   - Perform a security review with attack vectors and mitigations
   - Produce an ordered implementation plan with effort estimates

3. **Present the blueprint** to the user covering:
   - Integration method recommendation with clear rationale
   - Architecture overview (components, data flow, error handling)
   - Complete dependency list with versions and purposes
   - Security considerations with specific mitigations
   - Step-by-step implementation order with effort estimates
   - Overall complexity assessment

## Output Format

Present a structured integration blueprint:

```text
Integration Blueprint: DeFi Dashboard with Swap (Ethereum + Base)

  Recommended Method: Trading API (primary), Universal Router SDK (fallback)
  Rationale: Trading API provides optimized routing with minimal frontend
             complexity. Fallback to SDK for advanced use cases or API downtime.

  Architecture:
    User -> QuoteService -> ApprovalManager -> SwapExecutor -> Wallet -> Chain
    
    Components:
      QuoteService:      Fetches and caches quotes from Trading API
      ApprovalManager:   Manages Permit2 approvals and allowances
      SwapExecutor:      Constructs and submits swap transactions
      ChainManager:      Multi-chain config and RPC connections

  Dependencies:
    @uniswap/sdk-core    ^5.8.0    Token and price primitives
    viem                 ^2.21.0   Ethereum client
    wagmi                ^2.14.0   React wallet hooks
    @tanstack/react-query ^5.0.0   Data fetching and caching

  Security:
    - Stale quotes: Refresh every 15s, enforce deadline (block.timestamp + 300s)
    - Approvals: Permit2 with exact amounts and 30-min expiry
    - Key management: wagmi wallet connection only, never handle keys

  Implementation Order:
    1. Wallet connection (wagmi)           — 1 day
    2. Chain configuration                 — 0.5 days
    3. Quote fetching service              — 1 day
    4. Permit2 approval flow               — 1 day
    5. Swap execution and tracking         — 1.5 days
    6. Error handling and retry logic      — 1 day

  Complexity: Medium
```

## Important Notes

- This skill delegates entirely to the `integration-architect` agent -- it does not call MCP tools directly.
- The blueprint is tailored to the specific project type and requirements -- not a generic template.
- For existing codebases, the agent will analyze the current code and recommend integration patterns that fit the existing architecture.
- The implementation order considers dependencies between components -- follow the order for smoothest development.
- Always uses viem (not ethers.js) and Permit2 (not legacy approve) per project conventions.

## Error Handling

| Error | User-Facing Message | Suggested Action |
| --- | --- | --- |
| `VAGUE_PROJECT` | "Need more detail about your project to recommend an integration approach." | Describe project type and what Uniswap functionality is needed |
| `DEPRECATED_APPROACH` | "The requested integration method is deprecated. Recommending alternatives." | Follow the updated recommendation |
| `UNSUPPORTED_CHAIN` | "Uniswap is not deployed on the specified chain." | Choose from the 11 supported chains |
| `CONFLICTING_REQUIREMENTS` | "The requirements conflict (e.g., frontend + direct contract calls)." | Clarify the execution environment and adjust expectations |
