#!/usr/bin/env bash
# DepGuard — Policy & Compliance Module (TEAM tier)
# Enforces license policies and generates compliance reports

set -euo pipefail

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
source "$SCRIPT_DIR/scanner.sh"

RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
BOLD='\033[1m'
DIM='\033[2m'
NC='\033[0m'

# ─── Load policy config ────────────────────────────────────────────────────

get_blocked_licenses() {
  local config="${HOME}/.openclaw/openclaw.json"
  if [[ -f "$config" ]] && command -v node &>/dev/null; then
    node -e "
try {
  const cfg = require('$config');
  const blocked = cfg?.skills?.entries?.depguard?.config?.blockedLicenses || ['GPL-3.0', 'AGPL-3.0'];
  console.log(blocked.join('\n'));
} catch(e) {
  console.log('GPL-3.0\nAGPL-3.0');
}
" 2>/dev/null
  else
    echo -e "GPL-3.0\nAGPL-3.0"
  fi
}

get_allowed_licenses() {
  local config="${HOME}/.openclaw/openclaw.json"
  if [[ -f "$config" ]] && command -v node &>/dev/null; then
    node -e "
try {
  const cfg = require('$config');
  const allowed = cfg?.skills?.entries?.depguard?.config?.allowedLicenses || ['MIT', 'Apache-2.0', 'BSD-2-Clause', 'BSD-3-Clause', 'ISC'];
  console.log(allowed.join('\n'));
} catch(e) {
  console.log('MIT\nApache-2.0\nBSD-2-Clause\nBSD-3-Clause\nISC');
}
" 2>/dev/null
  else
    echo -e "MIT\nApache-2.0\nBSD-2-Clause\nBSD-3-Clause\nISC"
  fi
}

# ─── Policy enforcement ────────────────────────────────────────────────────

do_policy() {
  local dir="${1:-.}"
  local blocked
  blocked=$(get_blocked_licenses)
  local violations=0

  echo -e "${BOLD}━━━ DepGuard Policy Check ━━━${NC}"
  echo ""
  echo -e "Blocked licenses: ${RED}$(echo "$blocked" | tr '\n' ', ' | sed 's/,$//')${NC}"
  echo ""

  # Get all licenses
  local license_data
  license_data=$(mktemp)

  local managers
  managers=$(detect_package_managers "$dir")

  while IFS= read -r mgr; do
    case "$mgr" in
      npm|yarn|pnpm) scan_npm_licenses "$dir" >> "$license_data" ;;
      pip) scan_pip_licenses "$dir" >> "$license_data" ;;
    esac
  done <<< "$managers"

  # Check each package against policy
  while IFS=$'\t' read -r name ver lic; do
    [[ -z "$name" ]] && continue

    if echo "$blocked" | grep -qFx "$lic"; then
      echo -e "  ${RED}✗ BLOCKED${NC} ${BOLD}$name${NC}@$ver — $lic"
      ((violations++)) || true
    fi

    if [[ "$lic" == "UNKNOWN" || -z "$lic" ]]; then
      echo -e "  ${YELLOW}⚠ UNKNOWN${NC} ${BOLD}$name${NC}@$ver — license not specified"
      ((violations++)) || true
    fi
  done < "$license_data"

  echo ""
  if [[ "$violations" -gt 0 ]]; then
    echo -e "${RED}${BOLD}Policy violations: $violations${NC}"
    echo -e "Fix these before deploying to production."
    rm -f "$license_data"
    return 1
  else
    echo -e "${GREEN}${BOLD}All dependencies comply with policy.${NC}"
    rm -f "$license_data"
    return 0
  fi
}

# ─── Compliance report ─────────────────────────────────────────────────────

do_compliance() {
  local dir="${1:-.}"
  local output="$dir/COMPLIANCE-REPORT.md"
  local project_name
  project_name=$(basename "$(cd "$dir" && pwd)")

  local license_data
  license_data=$(mktemp)

  local managers
  managers=$(detect_package_managers "$dir")

  while IFS= read -r mgr; do
    case "$mgr" in
      npm|yarn|pnpm) scan_npm_licenses "$dir" >> "$license_data" ;;
      pip) scan_pip_licenses "$dir" >> "$license_data" ;;
    esac
  done <<< "$managers"

  local permissive=0 copyleft=0 unknown=0 other_count=0

  local permissive_list="" copyleft_list="" unknown_list="" other_list=""

  while IFS=$'\t' read -r name ver lic; do
    [[ -z "$name" ]] && continue
    local cat
    cat=$(categorize_license "$lic")
    case "$cat" in
      permissive)
        ((permissive++)) || true
        permissive_list+="| $name | $ver | $lic | Permissive | Low |"$'\n'
        ;;
      copyleft)
        ((copyleft++)) || true
        copyleft_list+="| $name | $ver | $lic | Copyleft | **High** |"$'\n'
        ;;
      unknown)
        ((unknown++)) || true
        unknown_list+="| $name | $ver | ${lic:-UNKNOWN} | Unknown | **Critical** |"$'\n'
        ;;
      other)
        ((other_count++)) || true
        other_list+="| $name | $ver | $lic | Other | Medium |"$'\n'
        ;;
    esac
  done < "$license_data"

  local total=$((permissive + copyleft + unknown + other_count))

  {
    echo "# License Compliance Report: $project_name"
    echo ""
    echo "> Generated by [DepGuard](https://depguard.pages.dev) — $(date +%Y-%m-%d)"
    echo ""
    echo "## Summary"
    echo ""
    echo "| Category | Count | Risk Level |"
    echo "|----------|-------|------------|"
    echo "| Permissive (MIT, Apache, BSD, ISC) | $permissive | Low |"
    echo "| Copyleft (GPL, AGPL, LGPL) | $copyleft | High |"
    echo "| Unknown | $unknown | Critical |"
    echo "| Other | $other_count | Medium |"
    echo "| **Total** | **$total** | |"
    echo ""

    if [[ "$copyleft" -gt 0 || "$unknown" -gt 0 ]]; then
      echo "## ⚠ Action Required"
      echo ""
      if [[ "$unknown" -gt 0 ]]; then
        echo "### Unknown Licenses ($unknown packages)"
        echo ""
        echo "These packages have no detectable license. They may be proprietary or incompatible with your project."
        echo ""
      fi
      if [[ "$copyleft" -gt 0 ]]; then
        echo "### Copyleft Licenses ($copyleft packages)"
        echo ""
        echo "These packages use copyleft licenses that may require your code to be open-sourced. Review for compliance."
        echo ""
      fi
    fi

    echo "## Full Dependency List"
    echo ""
    echo "| Package | Version | License | Category | Risk |"
    echo "|---------|---------|---------|----------|------|"

    # Critical first, then high, then medium, then low
    [[ -n "$unknown_list" ]] && echo -n "$unknown_list"
    [[ -n "$copyleft_list" ]] && echo -n "$copyleft_list"
    [[ -n "$other_list" ]] && echo -n "$other_list"
    [[ -n "$permissive_list" ]] && echo -n "$permissive_list"

    echo ""
    echo "## Policy Configuration"
    echo ""
    echo "Current policy (from \`~/.openclaw/openclaw.json\`):"
    echo ""
    echo "- **Blocked licenses:** $(get_blocked_licenses | tr '\n' ', ' | sed 's/,$//')"
    echo "- **Allowed licenses:** $(get_allowed_licenses | tr '\n' ', ' | sed 's/,$//')"
    echo ""
    echo "---"
    echo ""
    echo "*Report generated by [DepGuard](https://depguard.pages.dev)*"

  } > "$output"

  rm -f "$license_data"
  echo -e "${GREEN}[DepGuard]${NC} Compliance report written to ${BOLD}$output${NC}"
}
